%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This is the main program that replicates section 6 in 
% Yamamoto, Y. (2018) "Bootstrap inference for impulse response functions in factor-augmented vector
% autoregressions "
% Journal of Applied Econometrics

clear all;
clc;

% 1) Parameter values
p             = 13; 
r             = 3;  
m             = 1;  
l             = 0; 
wild          = 0;
hmax          = 48;
nrep          = 3000;
k             = r+m;
c             = 0.1;

% 2) Data loading
load('nsbalpanel.txt','-ascii')  % same data as Bernanke et al. (2005)
T             = size(nsbalpanel,1);
stdffr        = std(nsbalpanel(:,77));
X0            = (nsbalpanel-repmat(mean(nsbalpanel,1),T,1))./repmat(std(nsbalpanel,1),T,1);
Y             = X0(:,77); % FFR
XIND          = [77;16;107;74;26;62]; % variables of interest
X             = X0(:,[1:76 78:end]);

% 3) Estimation
[T,N]         = size(X);
[F,L,u,V,C,Z] = pcestim(X,Y,m,r);
LC            = [L C];
if m==0
    shock     = [zeros(1,r-1) 1];
    S         = F;
else
    shock     = [zeros(1,r) .25/stdffr];
    S         = [F Y];
end
[A,e]         = varestim(S,p);
Ve            = (e'*e)/(T-p*(r+m));
P             = chol(Ve)';

% 4) Bootstrap inference
irfh_favar    = zeros(hmax,N);
ucl0_favar    = zeros(hmax,N);
ucl1_favar    = zeros(hmax,N);
lcl0_favar    = zeros(hmax,N);
lcl1_favar    = zeros(hmax,N);
 
for i=1:6
  xind        = XIND(i,1);
  disp([xind])      
  pivot0      = zeros(hmax,nrep);
  pivot1      = zeros(hmax,nrep);
  irfh        = irfgen(shock,A,P,L,C,xind,hmax,p,r,m);

% bootstrap replications
for rep=1:nrep
      % bootstrap data generation
      S0b0       = btvargen(A,S,e,p);
      S0b1       = btvargen(A,S,e,p);
      F0b0       = S0b0(:,1:r);
      F0b1       = S0b1(:,1:r);          
      if m==0
         Yb0=[];Yb1=[];
      else
         Yb0     = S0b0(:,r+1:end);
         Yb1     = S0b1(:,r+1:end);
      end
      
      if m==0
          Xb0    = btfmgen(L',F0b0,u,wild);
          Xb1    = btfmgen(L',F0b1,u,wild);    
      else
          Zb0    = btfmgen(L',F0b0(:,1:r),u,wild);
          Zb1    = btfmgen(L',F0b1(:,1:r),u,wild);
          Xb0    = btfmgen(C',Yb0,Z,wild);
          Xb1    = btfmgen(C',Yb1,Z,wild);
      end
      
     % factor estimation
     if m==0
         [Fb0,Lb0,Ub0,Cb0]  = pcestim_nofac(Xb0,[],F0b0,0);
         [Fb1,Lb1,Ub1,Cb1]  = pcestim(Xb1,[],0,r);
     else
         [Fb0,Lb0,Ub0,Cb0]  = pcestim_nofac(Zb0,[],F0b0,0);
         [Fb1,Lb1,Ub1,Cb1]  = pcestim(Zb1,[],0,r);
         Cb0                = (Xb0'*Yb0)/(Yb0'*Yb0);
         Cb1                = (Xb1'*Yb1)/(Yb1'*Yb1);
     end
     
     % VAR estimation
     Sb0         = [Fb0 Yb0(l+1:end,:)];
     Sb1         = [Fb1 Yb1(l+1:end,:)];
     [Ab0,eb0]   = varestim(Sb0,p);
     [Ab1,eb1]   = varestim(Sb1,p);
     LCb0        = [Lb0 Cb0];
     LCb1        = [Lb1 Cb1];

     Vb0         = (eb0'*eb0)/(T-p*(r+m));
     Vb1         = (eb1'*eb1)/(T-p*(r+m));
     Pb0         = chol(Vb0)';
     Pb1         = chol(Vb1)';
        
    % impulse response estimation
    irfb10       = irfgen(shock,Ab0,Pb0,Lb0,Cb0,xind,hmax,p,r,m);
    irfb11       = irfgen(shock,Ab1,Pb1,Lb1,Cb1,xind,hmax,p,r,m);
    irfb10       = signcorrect(irfh,irfb10,2);
    irfb11       = signcorrect(irfh,irfb11,2);
        
    % confidence intervals 
    pivot0(:,rep)  = irfb10 -irfh;
    pivot1(:,rep)  = irfb11 -irfh;
end
  
 irfh_favar(:,xind)=irfh;
 lcl0_favar(:,xind)=irfh-quantile(pivot0',1-c/2)';
 lcl1_favar(:,xind)=irfh-quantile(pivot1',1-c/2)';
 ucl0_favar(:,xind)=irfh-quantile(pivot0',c/2)';
 ucl1_favar(:,xind)=irfh-quantile(pivot1',c/2)';

end

zax=zeros(hmax+1,1);
Resp=[];

figure(1)

subplot(3,2,1)
id=77;
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,0);
plot([resp1 zax]);
title('FFR')
Resp=[Resp resp1];

subplot(3,2,2)
id=16;
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,1);
plot([resp1 zax]);
legend('A-lower','B-lower','est','B-upper','A-upper')
title('IP')
Resp=[Resp resp1];

subplot(3,2,3)
id=107; 
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,1);
plot([resp1 zax]);
title('CPI')
Resp=[Resp resp1];

subplot(3,2,4)
id=74;
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,1);
plot([resp1 zax]);
title('Exchange rate YEN')
Resp=[Resp resp1];

subplot(3,2,5)
id=26;
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,0);
plot([resp1 zax]);
title('Unemployment rate')
Resp=[Resp resp1];

subplot(3,2,6)
id=62;
resp=[zeros(1,5);[lcl1_favar(:,id) lcl0_favar(:,id) irfh_favar(:,id) ucl0_favar(:,id) ucl1_favar(:,id)]];
resp1=backlevel(resp,0);
plot([resp1 zax]);
title('New orders')
Resp=[Resp resp1];


