# Functions used in "Flexible Estimation of Demand Systems: a Copula Approach"
# Mateo Velasquez-Giraldo,
# Gustavo Canavire-Bacarreza,
# Kim P. Huynh,
# David T. Jacho-Chavez

# Misc functions -------------------------------------------------------------------

# Compute Slutsky matrix (exact)
slutsky_mat <- function(shares_fn, ds_dp_fn, ds_dw_fn, v2p_fn, est, p, y){
  
  p <- as.matrix(p)
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p_fn, list("params"=est))
  
  # Compute estimated shares
  s_est <- do.call( shares_fn, c(par,list(y=y,p=t(p) )) )
  
  # Compute price derivatives of shares
  ds_dp <- do.call(ds_dp_fn, c(par, list(y=y, p=t(p)) ))
  
  # Compute wealth derivatives of shares
  ds_dw <- do.call( ds_dw_fn, c(par, list(y=y, p=t(p)) ))
  
  # Compute marshalian demand
  x <- s_est / t(p) * y
  
  # Derivative on prices
  pmat <- matrix(rep(p,length(p)), nrow = length(p), byrow = F)
  DpX <- y / pmat^2 *
    ( ds_dp*pmat - diag(as.numeric(s_est)))
  
  # Derivative on wealth
  DwX <- 1/p*(ds_dw*y + t(s_est))
  
  # Slutsky matrix
  slu <- DpX + DwX %*% x

  return(slu)
  
}

# Compute Slutsky matrix (numeric)
slutsky_mat_num <- function(iu_fn, v2p, est, y, p){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Find price derivatives numerically
  Dp <- jacobian( function(x) do.call(marsh_demand_num,
                                      c( list(iu_fn=iu_fn, v2p=v2p, est=est),
                                         list(y = y, p = x))),
                  x = matrix(p, nrow=1))
  
  # Find wealth derivatives numerically
  Dw <- jacobian( function(x) do.call(marsh_demand_num,
                                      c( list(iu_fn=iu_fn, v2p=v2p, est=est),
                                         list(y = x, p = matrix(p, nrow=1)))),
                  x = y)
  
  x <- marsh_demand_num(iu_fn, v2p, est, p = matrix(p, nrow=1), y)
  
  Slu <- Dp + Dw %*% x
  
  return(Slu)
  
}

# Obtain marshailian uncompensated price elasticities
marsh_elast <- function(ds_dp_fn, shares_fn, v2p, est, p, y){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Compute price derivatives of shares
  ds_dp <- do.call(ds_dp_fn, c(par, list(y=y, p=p)))
  
  # Compute estimated shares
  s_est <- do.call( shares_fn, c(par,list(y=y,p=p)) )
  
  # Compute elasticities
  m_el <- matrix(rep(p,3), nrow = 3, byrow = TRUE)/matrix(rep(s_est,3), nrow = 3, byrow = FALSE)*ds_dp -
    diag(3)
  
  return(m_el)
  
}

# Obtain income elasticities
income_elast <- function(ds_dy_fn, shares_fn, v2p, est, p, y){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Compute income derivatives of shares
  ds_dy <- do.call(ds_dy_fn, c(par, list(y=y, p=p)))
  
  # Compute estimated shares
  s_est <- do.call( shares_fn, c(par,list(y=y,p=p)) )
  
  # Compute elasticities
  inc_el <- rep(1,3) + y/s_est * t(ds_dy)
  colnames(inc_el) <- paste("s",1:3, sep ="")
  
  return(inc_el)
  
}

# Compute demand given parameters, prices and income
marsh_demand <- function(shares_fn, v2p, est, p, y){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Compute shares
  shares <- do.call(shares_fn, c(par, list(y = y, p = p)) )
  
  # Find demand
  m_demand <- shares * matrix(rep(y, ncol(p)), ncol = ncol(p), byrow = FALSE) / p
  
  return(m_demand)
  
}

# Obtain estimated budget shares and count the number
# of negatives.
count_negatives <- function(shares, v2p, est, data, n){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Add prices and y to parameter list
  par$y <- data$y
  par$p <- data$p 
  
  # Get estimated shares
  s_est <- do.call( shares, par)
  
  # Check positivity for every row
  neg <- apply(s_est, 1, function(x) sum( x < 0) > 0)
  
  # Return the total of rows where there are shares under 0
  return(sum(neg))
  
}

# Check monotonicity
check_mono <- function(iu_fn, v2p, est, data){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p, list("params"=est))
  
  # Add prices and y to parameter list
  y <- data$y
  p <- data$p 
  
  # Compute price derivatives at data points
  grad <- matrix(NA, nrow = nrow(p), ncol = ncol(p))
  for (i in 1:nrow(p)){
    
    grad[i,] <- jacobian( function(x) do.call(iu_fn, c(par, list(y = y[i], p = x))),
                          x = matrix(p[i,], nrow=1))
    
  }
  
  # Check if any derivative is non-negative
  nneg <- apply(grad, 1, function(x) sum( x >= 0) > 0)
  
  return(sum(nneg))
  
}

# Check curvature
check_curv <- function(shares_fn, ds_dp_fn, ds_dw_fn, v2p_fn, iu_fn, est, data, numeric = F){
  
  # Turn the list of estimates into parameters
  par <- do.call(v2p_fn, list("params"=est))
  
  # Get income and prices
  y <- data$y
  p <- data$p 
  
  # Check negative semi definiteness
  if (numeric) {
    
    nsd <- sapply( 1:nrow(p),
                   function(x) curv_cond(slutsky_mat_num(iu_fn = iu_fn,
                                                         v2p = v2p_fn,
                                                         est = est,
                                                         y = y[x],
                                                         p = p[x,])
                   )
    )
    
  } else {
    
    nsd <- sapply( 1:nrow(p),
                   function(x) curv_cond(slutsky_mat(shares_fn,
                                                     ds_dp_fn,
                                                     ds_dw_fn,
                                                     v2p_fn,
                                                     est,
                                                     p[x,],
                                                     y[x]))
    )
    
  }
  
  
  return( nrow(p) - sum(nsd) )
  
}

# Check curvature conditions of a Slutsky Matrix.
curv_cond <- function(mat){
  
  # Check symmetry
  if ( max(abs(mat - t(mat))) <= 10^(-10) ){
    
    # Make symmetric
    mat <- (mat + t(mat))/2
    
    # Check negative definiteness
    if (is.negative.semi.definite(mat)){
      
      return(T)
      
    } else {
      
      return(F)
      
    }
    
  } else {
    
    return(F)
    
  }
  
}

# Estimate gamma from paraeters of restricted models using
# K's estimates. (aids and quaids)
gamma <- function( par, v2p ){
  
  par <- do.call(v2p, list(par))
  
  return(c("gam11"=par$gamma[1,1],
           "gam12"=par$gamma[1,2],
           "gam22"=par$gamma[2,2]))
  
}

# Estimate gamma standard errors using the delta method (aids and quaids)
gamma_se <- function( par, cov, v2p_fn){
  
  g <- jacobian(function(x) gamma(x, v2p_fn), par)
  
  gam_cov <- g%*% cov %*% t(g)
  
  return( sqrt( diag(gam_cov) ) )
  
}

# Estimate elasticities' standard errors using the delta method
elast_se <- function(ds_dp_fn, ds_dy_fn, shares_fn, v2p_fn, est, p, y){
  
  # Find positions of coef in fullcoef
  pos <- match(names(est@coef),names(est@fullcoef))
  
  # S.E for marshalian price elasticities
  Dmarsh <- jacobian( function(x) as.vector(do.call(marsh_elast,
                                                    list(ds_dp_fn=ds_dp_fn,
                                                         shares_fn=shares_fn,
                                                         v2p = v2p_fn,
                                                         est = replace(est@fullcoef,pos, x),
                                                         p = p,
                                                         y = y))),
                      x = est@coef)
  
  marsh_cov <- Dmarsh %*% est@vcov %*% t(Dmarsh)
  se_marsh <- sqrt(diag(marsh_cov))
  se_marsh <- matrix(se_marsh, nrow = sqrt(length(se_marsh)))
  
  # S.E for income elasticities
  Dinc <- jacobian( function(x) as.vector(do.call(income_elast,
                                                  list(ds_dy_fn=ds_dy_fn,
                                                       shares_fn=shares_fn,
                                                       v2p = v2p_fn,
                                                       est = replace(est@fullcoef,pos, x),
                                                       p = p,
                                                       y = y))),
                    x = est@coef)
  
  inc_cov <- Dinc %*% est@vcov %*% t(Dinc)
  se_inc <- sqrt(diag(inc_cov))
  
  return(list("marsh_elast_se"=se_marsh,"wealth_elast_se"=se_inc))
  
}

# [(-Inf,Inf) -> Range ] transormation applied to copula parameters
# so that optimization is unrestricted.
cop_param_transform <- function(cop,cop_par){
  
  if (cop$type == "ellip"){
    
    rho <- 2 / ( 1 + exp(-cop_par) ) - 1
    
    cop$copula@parameters <- rho
    
  } else if (cop$type == "arch") {
    
    theta <- switch(cop$fam,
                    clayton =  0.001 + exp(cop_par), # Theta \in (0, Inf)
                    frank = exp(cop_par), # Theta \in (0, Inf)
                    gumbel = 1 + exp(cop_par)) # Theta \in (1,Inf)
    cop$copula@parameters <- theta
    
  }
  
  return(cop)
  
}

# Compute the location parameter of a skewed normal
# given its scale and shape parameters so that its mean is 0.
mean_zero_xi <- function(om,alpha){
  
  return( -1 * om*alpha/sqrt(1+alpha^2)*sqrt(2/pi) )
  
}

# Log-density of zero-mean skew normal
logdsn_0 <- function(x, om, alph){
  
  xi <- mean_zero_xi(om, alph)
  ld <- dsn(x,
            xi=xi,
            omega=om,
            alpha=alph,
            tau=0,
            log = T)
  
  return(ld)
  
}

# CDF of zero-mean skew normal
psn_0 <- function(x, om, alph){
  
  xi <- mean_zero_xi(om, alph)
  u <- psn(x,
           xi=xi,
           omega=om,
           alpha=alph,
           tau=0)
  
  return(u)
  
}

# AIDS -----------------------------------------------------------------

# AIDS shares:
# Obtain predicted shares using estimated parameters
aids_shares <- function(alpha0, alpha, gamma, beta, y, p){
  
  # Transform prices
  p <- log(p)
  
  # Construct Ln a
  ln_a <- alpha0 + p %*% alpha + 0.5 * rowSums( (p%*%gamma) * p )
  
  # Construct estimated s
  s_est <- matrix( rep( alpha, nrow(p) ) , ncol = length(alpha), byrow = TRUE ) +
    p %*% gamma +
    matrix( log(y) - ln_a, ncol = 1) %*% t(beta)
  
  return(s_est)
  
}

# AIDS indirect utility
# Obtain the maximum utility attainable with p and y, using
# parameter estimates
aids_iu <- function(alpha0, alpha, gamma, beta0 = 1,beta, y, p){
  
  # Transform prices
  p <- log(p)
  
  # Construct Ln a
  ln_a <- alpha0 + p %*% alpha + 0.5 * rowSums( (p%*%gamma) * p )
  
  prod <- exp(rowSums(p*matrix(rep(-beta, nrow(p)), nrow = nrow(p), byrow = TRUE)))
  
  # Compute indirect utility
  h <- (log(y) - ln_a)*beta0^(-1)*prod
  
  return(h)
  
  
}

# Share derivatives with respect to price
ds_dp_aids <- function(alpha0, alpha, gamma, beta, y, p){
  
  p <- as.matrix(p)
  
  gam0 <- gamma
  diag(gam0) <- rep(0,length(alpha))
  
  dlna_dp <- alpha/t(p) + diag(gamma)*t(log(p)/p) + rowSums(  t(1/p)%*%(log(p)) * gam0 )
  
  ds_dp <- gamma/matrix(rep(p,3), ncol = 3, byrow=TRUE) -
    matrix(rep(beta,3), ncol =3, byrow=FALSE) * matrix(rep(dlna_dp,3), ncol =3, byrow=TRUE)
  
  colnames(ds_dp) <- paste("p",1:3,sep="")
  rownames(ds_dp) <- paste("s",1:3,sep="")
  
  return(ds_dp)
  
}

# Share derivatives with respect to wealth
ds_dy_aids <- function(alpha0, alpha, gamma, beta, y, p){
  
  ds_dy <- as.matrix(beta/y)
  rownames(ds_dy) <- paste("s",c(1:3), sep ="")
  
  return(ds_dy)
  
}

# Vector to parameter conversion for AIDS
v2p_aids <- function(params, n=3){
  
  alpha0 <- params[1]
  alpha <- params[2:n]
  
  gamma_v <- params[ (n+1):(n + (n-1)*n/2)]
  
  beta <- params[ (n + (n-1)*n/2 + 1):(n + (n-1)*n/2 + n - 1)]
  
  # Construct gamma and omega
  gamma <- matrix(0, nrow = n-1, ncol = n-1)
  gamma[ upper.tri(gamma, diag = TRUE) ] <- gamma_v
  gamma <- gamma + t(gamma)
  diag(gamma) <- diag(gamma)/2
  
  # Expand alpha
  alpha[n] <- 1 - sum(alpha)
  
  # Expand beta
  beta[n] <- 0 - sum(beta)
  
  # Expand gamma
  gamma <- rbind(gamma, 0 - colSums(gamma))
  gamma <- cbind(gamma, 0 - rowSums(gamma))
  
  return(list("alpha0" = alpha0,
              "alpha" = as.matrix(alpha, ncol = 1),
              "gamma" = gamma,
              "beta" = as.matrix(beta, ncol = 1) ))
  
}

# Vector to parameter conversion for restricted AIDS
v2p_aidsR <- function( params ){
  
  n <- 3
  
  alpha0 <- params[1]
  alpha <- params[2:n]
  
  K_v <- params[ (n+1):(n + (n-1)*n/2)]
  
  beta <- matrix(params[ (n + (n-1)*n/2 + 1):(n + (n-1)*n/2 + n - 1)], ncol = 1)
  
  # Construct K
  K <- matrix(0, nrow = n-1, ncol = n-1)
  K[ lower.tri(K, diag = TRUE) ] <- K_v
  
  # Expand alpha
  alpha[n] <- 1 - sum(alpha)
  
  # Expand beta
  beta[n] <- 0 - sum(beta)
  
  # Construct gamma
  Slu <- - 1 * K %*% t(K)
  
  gamma <- matrix(0, nrow = n-1, ncol = n-1)
  for ( i in 1:(n-1)){
    
    for( j in i:(n-1)){
      
      gamma[i,j] <- Slu[i,j] - 
        (alpha[j] - beta[j]*alpha0)*(alpha[i] - beta[i]*alpha0) +
        beta[i] * beta[j] * alpha0
      
      if (j==i){
        
        gamma[i,j] <- gamma[i,j] + (alpha[i] - beta[i]*alpha0)
        
      }
      
      gamma[j,i] <- gamma[i,j]
      
    }
    
  }
  
  # Expand gamma
  gamma <- rbind(gamma, 0 - colSums(gamma))
  gamma <- cbind(gamma, 0 - rowSums(gamma))
  
  return(list("alpha0" = alpha0,
              "alpha" = alpha,
              "gamma" = gamma,
              "beta" = beta))
  
}


# QUAIDS --------------------------------------------------------------------

# QUAIDS shares:
# Obtain predicted shares using estimated parameters
quaids_shares <- function( alpha0, alpha, gamma, beta, lambda, y, p ){
  
  p <- log(p)
  
  # Construct Ln a
  ln_a <- alpha0 + p %*% alpha + 0.5 * rowSums( (p%*%gamma) * p )
  
  # Construct b
  b <- exp(p %*% beta)
  
  # Construct estimated s
  s_est <- matrix( rep( alpha, nrow(p) ) , ncol = length(alpha), byrow = TRUE ) +
    p %*% gamma +
    matrix( log(y) - ln_a, ncol = 1) %*% t(beta) +
    matrix( (log(y) - ln_a)^2/b ) %*% t(lambda)
  
  return(s_est)
  
}

# QUAIDS indirect utility
# Obtain the maximum utility attainable with p and y, using
# parameter estimates
quaids_iu <- function(alpha0, alpha, gamma, beta, lambda, y, p){
  
  # Transform prices
  p <- log(p)
  
  # Construct Ln a
  ln_a <- alpha0 + p %*% alpha + 0.5 * rowSums( (p%*%gamma) * p )
  
  # Construct b
  b <- exp(rowSums(p*matrix(rep(beta, nrow(p)), nrow = nrow(p), byrow = TRUE)))
  
  # Construct lambda(p)
  l_p <- p %*% lambda
  
  # Compute indirect utility
  h <- exp(1/( ( b/(log(y) - ln_a) ) + l_p ))
  
  return(h)
  
  
}

# Share derivatives with respect to price
ds_dp_quaids <- function(alpha0, alpha, gamma, beta, lambda, y, p){
  
  
  p <- as.matrix(p)
  
  # Construct Ln a
  ln_a <- alpha0 + log(p) %*% alpha + 0.5 * rowSums( (log(p)%*%gamma) * log(p) )
  
  # Find the b product
  b <- exp( log(p) %*% beta )
  
  # Find the derivative of log(a)
  gam0 <- gamma
  diag(gam0) <- rep(0,length(alpha))
  dlna_dp <- alpha/t(p) + diag(gamma)*t(log(p)/p) + rowSums(  t(1/p)%*%(log(p)) * gam0 )
  
  # Derivative of b
  db_dp <- b %*% ( t(beta) / p )
  
  ds_dp <- gamma/matrix(rep(p,3), ncol = 3, byrow=TRUE) -
    matrix(rep(beta,3), ncol =3, byrow=FALSE) * matrix(rep(dlna_dp,3), ncol =3, byrow=TRUE) - 
    matrix(rep(lambda,3), ncol =3, byrow=FALSE) * (
      
      as.numeric(b^(-2)*(log(y) - ln_a)^2) * matrix(rep(db_dp,3), ncol =3, byrow=TRUE) +
        
        as.numeric(b^(-1)*2*(log(y) - ln_a)) * matrix(rep(dlna_dp,3), ncol =3, byrow=TRUE)
      
    )
  
  colnames(ds_dp) <- paste("p",1:3,sep="")
  rownames(ds_dp) <- paste("s",1:3,sep="")
  
  return(ds_dp)
  
}

# Share derivatives with respect to wealth
ds_dy_quaids <- function(alpha0, alpha, gamma, beta, lambda, y, p){
  
  p <- as.matrix(p)
  
  # Construct Ln a
  ln_a <- alpha0 + log(p) %*% alpha + 0.5 * rowSums( (log(p)%*%gamma) * log(p) )
  
  # Find the b product
  b <- exp( log(p) %*% beta )
  
  # Compute wealth derivatives
  ds_dy <- beta/y + lambda/as.vector(b) * as.vector(2 * (log(y) - ln_a)*1/y)
  names(ds_dy) <- paste("s",c(1:3), sep ="")
  
  return(ds_dy)
  
}

# Vector to parameter conversion for QUAIDS
v2p_quaids <- function( params, n=3 ){
  
  alpha0 <- params[1]
  alpha <- params[2:n]
  
  gamma_v <- params[ (n+1):(n + (n-1)*n/2)]
  
  beta <- matrix(params[ (n + (n-1)*n/2 + 1):(n + (n-1)*n/2 + n - 1)], ncol = 1)
  
  lambda <- matrix( params[(n + (n-1)*n/2 + n):(n + (n-1)*n/2 + 2*n - 2)], ncol = 1 )
  
  # Construct gamma
  gamma <- matrix(0, nrow = n-1, ncol = n-1)
  gamma[ upper.tri(gamma, diag = TRUE) ] <- gamma_v
  gamma <- gamma + t(gamma)
  diag(gamma) <- diag(gamma)/2
  
  # Expand alpha
  alpha[n] <- 1 - sum(alpha)
  
  # Expand beta
  beta[n] <- 0 - sum(beta)
  
  # Expand lambda
  lambda[n] <- 0 - sum(lambda)
  
  # Expand gamma
  gamma <- rbind(gamma, 0 - colSums(gamma))
  gamma <- cbind(gamma, 0 - rowSums(gamma))
  
  return( list("alpha0" = alpha0,
               "alpha"= alpha,
               "gamma"=gamma,
               "beta"=beta,
               "lambda"=lambda) )
  
}

# Vector to parameter conversion for restricted AIDS
v2p_quaidsR <- function( params ){
  
  n <- 3
  
  alpha0 <- params[1]
  alpha <- params[2:n]
  
  K_v <- params[ (n+1):(n + (n-1)*n/2)]
  
  beta <- matrix(params[ (n + (n-1)*n/2 + 1):(n + (n-1)*n/2 + n - 1)], ncol = 1)
  
  lambda <- matrix( params[(n + (n-1)*n/2 + n):(n + (n-1)*n/2 + 2*n - 2)], ncol = 1 )
  
  # Construct K
  K <- matrix(0, nrow = n-1, ncol = n-1)
  K[ lower.tri(K, diag = TRUE) ] <- K_v
  
  # Expand alpha
  alpha[n] <- 1 - sum(alpha)
  
  # Expand beta
  beta[n] <- 0 - sum(beta)
  
  # Expand lambda
  lambda[n] <- 0 - sum(lambda)
  
  # Construct gamma
  Slu <- - 1 * K %*% t(K)
  
  gamma <- matrix(0, nrow = n-1, ncol = n-1)
  for ( i in 1:(n-1)){
    
    for( j in i:(n-1)){
      
      gamma[i,j] <- Slu[i,j] + alpha0*beta[i]*beta[j] -
        alpha0^2*beta[i]*lambda[j] - alpha0^2*beta[j]*lambda[i] + 2*alpha0^3*lambda[i]*lambda[j] - alpha[i]*alpha[j] +
        alpha0*alpha[i]*beta[j] + alpha0*alpha[j]*beta[i] - alpha0^2*alpha[i]*lambda[j] - alpha0^2*alpha[j]*lambda[i] -
        alpha0^2*beta[i]*beta[j] + alpha0^3*beta[i]*lambda[j] + alpha0^3*beta[j]*lambda[i] - alpha0^4*lambda[i]*lambda[j]
      
      if (j==i){
        
        gamma[i,j] <- gamma[i,j] -
          ( alpha0*beta[i] - alpha0^2*lambda[i] - alpha[i] )
        
      }
      
      gamma[j,i] <- gamma[i,j]
      
    }
    
  }
  
  #Expand gamma
  gamma <- rbind(gamma, 0 - colSums(gamma))
  gamma <- cbind(gamma, 0 - rowSums(gamma))
  
  return(list("alpha0" = alpha0,
              "alpha" = alpha,
              "gamma" = gamma,
              "beta" = beta,
              "lambda" = lambda))
  
}

# Minflex Laurent ----------------------------------------------------------

# Minflex Laurent shares:
# Obtain predicted shares using estimated parameters
ml_shares <- function(a, A, B, y, p){
  
  # Compute normalized prices v and transformations
  v <- p/y
  w <- sqrt(v)
  wbar <- 1/w
  
  # Create a version of A with no diagonal
  A_ij <- A
  diag(A_ij) <- rep(0,3)
  
  # Estimated shares
  s_est <- matrix(rep(a, nrow(p)), nrow = nrow(p), byrow = TRUE) * w +
    matrix(rep(diag(A), nrow(p)), nrow = nrow(p), byrow = TRUE) * v +
    (w %*% A_ij) * w +
    (wbar %*% B) * wbar
  
  s_est <- s_est/rowSums(s_est)
  
  return(s_est)
  
}

# Minflex Laurent indirect utility
# Obtain the maximum utility attainable with p and y, using
# parameter estimates
ml_iu <- function(a0=1,a, A, B, y, p){
  
  # Compute normalized prices v and transformations
  v <- p/y
  w <- sqrt(v)
  wbar <- 1/w
  
  # Create a version of A with no diagonal
  A_ij <- A
  diag(A_ij) <- rep(0,3)
  
  # Compute h
  h <- matrix(rep(a0,nrow(p)),ncol=1)+
    2*w%*%a + 
    rowSums( v * matrix(rep(diag(A), nrow(p)), nrow = nrow(p), byrow = TRUE)) +
    rowSums( (w%*%A_ij) * w ) -
    rowSums( (wbar%*%B) * wbar )
  
  return(1/h)
  
}

# Share derivatives with respect to price
ds_dp_ml <- function(a, A, B, y, p){
  
  
  p <- as.matrix(p)
  
  # Compute normalized prices v and transformations
  v <- p/y
  w <- sqrt(v)
  wbar <- 1/w
  
  # Create a version of A with no diagonal
  A_ij <- A
  diag(A_ij) <- rep(0,3)
  
  # Copute top
  top <- t(a) * w + diag(A) * v + (w %*% A_ij) * w + (wbar %*% B) * wbar
  top <- matrix(rep(top, 3), nrow = 3, byrow = FALSE)
  
  # Compute bottom
  bot <- sum(diag(top))
  
  # D(top)/D(vi) diag
  dtop_dv_D <- 0.5 * a * t(wbar) + diag(A) +
    0.5 * t(wbar) * (A_ij %*% t(w)) -
    0.5 * t(v^(-3/2)) * (B %*% t(wbar))
  # D(bot)/D(vi) diag
  dbot_dv_D <- 0.5*t(wbar)*a + diag(A) + t(wbar)*(A_ij %*% t(w)) - t(v^(-3/2)) * (B %*% t(wbar))
  
  # OFF DIAGONAL DERIVATIVES
  # D(top)/D(vi)
  dtop_dv <- 0.5*A*matrix(rep(wbar, 3), nrow = 3, byrow = TRUE)*matrix(rep(w, 3), nrow = 3, byrow = FALSE) -
    0.5*B*matrix(rep(v^(-3/2), 3), nrow = 3, byrow = TRUE)*matrix(rep(wbar, 3), nrow = 3, byrow = FALSE)
  diag(dtop_dv) <- dtop_dv_D
  
  # As the bottom is the same for every share, the derivative matrix is a row copy of its diagonal 
  dbot_dv <- matrix(rep(dbot_dv_D, 3), nrow = 3, byrow = TRUE)
  
  # Apply derivative of the quotient
  ds_dv <- (dtop_dv*bot - top*dbot_dv)/bot^2
  
  # Apply the chain rule
  ds_dp <- ds_dv / y
  
  colnames(ds_dp) <- paste("p",1:3,sep="")
  rownames(ds_dp) <- paste("s",1:3,sep="")
  
  return(ds_dp)
  
}

# Share derivatives with respect to wealth
ds_dy_ml <- function(a, A, B, y, p){
  
  p <- as.matrix(p)
  
  # Compute normalized prices v and transformations
  v <- p/y
  w <- sqrt(v)
  wbar <- 1/w
  
  # Create a version of A with no diagonal
  A_ij <- A
  diag(A_ij) <- rep(0,3)
  
  # Copute top
  top <- t(a) * w + diag(A) * v + (w %*% A_ij) * w + (wbar %*% B) * wbar
  top <- matrix(rep(top, 3), nrow = 3, byrow = FALSE)
  
  # Compute bottom
  bot <- sum(diag(top))
  
  # D(top)/D(vi) diag
  dtop_dv_D <- 0.5 * a * t(wbar) + diag(A) +
    0.5 * t(wbar) * (A_ij %*% t(w)) -
    0.5 * t(v^(-3/2)) * (B %*% t(wbar))
  # D(bot)/D(vi) diag
  dbot_dv_D <- 0.5*t(wbar)*a + diag(A) + t(wbar)*(A_ij %*% t(w)) - t(v^(-3/2)) * (B %*% t(wbar))
  
  # OFF DIAGONAL DERIVATIVES
  # D(top)/D(vi)
  dtop_dv <- 0.5*A*matrix(rep(wbar, 3), nrow = 3, byrow = TRUE)*matrix(rep(w, 3), nrow = 3, byrow = FALSE) -
    0.5*B*matrix(rep(v^(-3/2), 3), nrow = 3, byrow = TRUE)*matrix(rep(wbar, 3), nrow = 3, byrow = FALSE)
  diag(dtop_dv) <- dtop_dv_D
  
  # As the bottom is the same for every share, the derivative matrix is a row copy of its diagonal 
  dbot_dv <- matrix(rep(dbot_dv_D, 3), nrow = 3, byrow = TRUE)
  
  # Apply derivative of the quotient
  ds_dv <- (dtop_dv*bot - top*dbot_dv)/bot^2
  
  # Find dvj/dy
  dv_dy <- - p / y^2
  
  # Add up using ds_i/dy = sum_j_( dsi/dvj * dvj/dy )
  ds_dy <- ds_dv %*% t(dv_dy)
  
  rownames(ds_dy) <- paste("s",1:3, sep="")
  
  return(ds_dy)
  
}

# Vector to parameter conversion for Minflex Laurent
v2p_ml <- function(params){
  
  a <- as.matrix(params[1:3], ncol = 1)
  
  # A
  A <- matrix(0, nrow = 3, ncol = 3)
  A[1,1] <- params[4]
  A[1,3] <- params[5]^2
  A[3,3] <- params[6]
  A[2,2] <- params[7]
  # Symetry
  A[3,1] <- A[1,3]
  
  # B
  B <- matrix(0, nrow = 3, ncol = 3)
  B[1,2] <- params[8]^2
  B[2,3] <- params[9]^2
  # Symetry
  B[2,1] <- B[1,2]
  B[3,2] <- B[2,3]
  
  return(list("a"=a, "A"=A, "B"=B))
  
}

# Vector to parameter conversion for restricted Minflex Laurent
v2p_ml_r <- function(params){
  
  a <- as.matrix(params[1:3]^2, ncol = 1)
  
  # A
  A <- matrix(0, nrow = 3, ncol = 3)
  A[1,1] <- params[4]^2
  A[1,3] <- params[5]^2
  A[3,3] <- params[6]^2
  A[2,2] <- params[7]^2
  # Symetry
  A[3,1] <- A[1,3]
  
  # B
  B <- matrix(0, nrow = 3, ncol = 3)
  B[1,2] <- params[8]^2
  B[2,3] <- params[9]^2
  # Symetry
  B[2,1] <- B[1,2]
  B[3,2] <- B[2,3]
  
  return(list("a"=a, "A"=A, "B"=B))
  
}

# Parameter normalization for unrestricted Minflex Laurent
normalize_ml <- function(params){
  # Nomalize so that all parameters in the structural form sum to 1
  
  # Which parameters are squared
  sq <- c(F,F,F,F,T,F,F,T,T)
  
  # Number of times each parameter appears
  reps <-c(1,1,1,1,2,1,1,2,2)
  
  tot <- params
  tot[sq] <- params[sq]^2
  
  tot <- sum(tot * reps)
  
  params[sq] <- params[sq] / sqrt(tot)
  params[!sq] <- params[!sq] / tot
  
  
  
  return(params) 
}

# Parameter normalization for restricted Minflex Laurent
normalize_ml_r <- function(params){
  # Nomalize so that all parameters in the structural form sum to 1
  
  # Which parameters are squared
  sq <- c(T,T,T,T,T,T,T,T,T)
  
  # Number of times each parameter appears
  reps <-c(1,1,1,1,2,1,1,2,2)
  
  tot <- params
  tot[sq] <- params[sq]^2
  
  tot <- sum(tot * reps)
  
  params[sq] <- params[sq] / sqrt(tot)
  params[!sq] <- params[!sq] / tot
  
  return(params)
}

# Estimation funtions ----

# Log-likelihood of errors given a copula, marginals, and its parameters.
ll_fun_copula_res <- function(cop_par, M_par, cop, err, margins, transform = "no"){
  
  # Flip second error term if needed
  if (transform == "yes") {
    
    err[,2] <- -1*err[,2]
    
  }
  
  # Find densities and pseudo-observations: u's
  d <- tryCatch(sapply(1:ncol(err),
                       function(i) do.call(margins[[i]]$log_dfun, args = c(list(err[,i]),
                                                                           as.numeric(M_par[[i]])))),
                error = function(err) return(NaN))
  
  u <- tryCatch(sapply(1:ncol(err),
                       function(i) do.call(margins[[i]]$pfun, args = c(list(err[,i]),
                                                                       as.numeric(M_par[[i]])))),
                error = function(err) return(NaN))
  
  
  # Create copula
  cop <- cop_param_transform(cop,cop_par)
  
  # Copula densities return -Inf for u=0. And this can happen often when parameters make a distribution
  # move to the right of data (P(X<=x)=0). To avoid discontinuity, bound u.
  u <- pmax(u,10^-15)
  u <- pmin(u, 1- 10^-15)
  
  # Copula log-likelihood:
  # Try to compute the log-density, if it fails (usually through parameters out of bounds
  # it is set to -Inf)
  ll_c <- tryCatch(dCopula(u, cop$copula, log=T),
                   error = function(err) return(NaN))
  s_c <- sum(ll_c)
  s_d <- sum(d)
  
  s <- c(s_c,s_d)
  
  if (any(is.nan(s))){
    print(Inf)
  }
  
  # Frank copula sometimes throws density values of +Inf when
  # theta is big
  if ( any(is.nan(s)) | any(s == Inf) ){
    return(NaN)
  } else {
    return( -sum(s) )
  }
  
}

# Log-likelihood of demand data given a model (copula, marginals, demand) and its parameters.
ll_fun_copula <- function(mod_par, cop_par, M_par, cop, dat, margins, mod, transform = "no"){
  
  # Construct model parameters from vector
  par <- do.call(mod$v2p_fn, list("params"=mod_par))
  
  # Construct estimated shares
  par$y <- dat$y
  par$p <- dat$p
  s_est <- do.call(mod$shares_fn, par)
  
  # Find error terms
  err <- as.matrix( dat$s - s_est )
  err <- err[, mod$sel]
  
  # Find and return the log-likelihood of errors.
  ll <- ll_fun_copula_res(cop_par, M_par, cop, err, margins,transform = transform)
  
  return(ll)
  
}

# Estimates the specified model-copula-selection-marginals combination
wraper_est <- function(db,model,m1,m2,cop_fam,sel1,sel2,transform = "no",
                       quiet = TRUE, Num = F){
  
  tic()
  
  # Convert arguments to appropriate types as apply turns args to strings
  quiet <- as.logical(quiet)
  sel1 <- as.numeric(sel1)
  sel2 <- as.numeric(sel2)
  
  
  # Load and process data
  filename <- switch(db,
                     single="Single_Members.txt",
                     married="Married_Couples_without_Children.txt",
                     child="Married_Couples_with_One_Child.txt")
  data <- fread( paste( "data/", filename, sep="") )
  y <- data$y
  p <- as.matrix( subset( data, select = grep("p[0-9]+", names(data)) ) )
  s <- as.matrix( subset( data, select = grep("s[0-9]+", names(data)) ) )
  
  dat <- list("y" = y, "p" = p, "s" = s)
  
  # Create model structure
  mod <- switch(model,
                
                aids = list(shares_fn = aids_shares,
                            v2p_fn = v2p_aids,
                            iu_fn = aids_iu,
                            norm_fn = NULL,
                            ds_dp_fn = ds_dp_aids,
                            ds_dy_fn = ds_dy_aids),
                
                aids_r = list(shares_fn = aids_shares,
                              v2p_fn = v2p_aidsR,
                              iu_fn = aids_iu,
                              norm_fn = NULL,
                              ds_dp_fn = ds_dp_aids,
                              ds_dy_fn = ds_dy_aids),
                
                quaids = list(shares_fn = quaids_shares,
                              v2p_fn = v2p_quaids,
                              iu_fn = quaids_iu,
                              norm_fn = NULL,
                              ds_dp_fn = ds_dp_quaids,
                              ds_dy_fn = ds_dy_quaids),
                
                quaids_r = list(shares_fn = quaids_shares,
                                v2p_fn = v2p_quaidsR,
                                iu_fn = quaids_iu,
                                norm_fn = NULL,
                                ds_dp_fn = ds_dp_quaids,
                                ds_dy_fn = ds_dy_quaids),
                
                ml = list(shares_fn = ml_shares,
                          v2p_fn = v2p_ml,
                          iu_fn = ml_iu,
                          norm_fn = normalize_ml,
                          ds_dp_fn = ds_dp_ml,
                          ds_dy_fn = ds_dy_ml),
                
                ml_r = list(shares_fn = ml_shares,
                            v2p_fn = v2p_ml_r,
                            iu_fn = ml_iu,
                            norm_fn = normalize_ml_r,
                            ds_dp_fn = ds_dp_ml,
                            ds_dy_fn = ds_dy_ml))
  
  # Add selection to model
  mod$sel <- c(sel1,sel2)
  
  # Construct marginals
  margins <- list(NULL,NULL)
  m_start <- list(NULL,NULL)
  mtypes <- c(m1,m2)
  for (i in 1:2) {
    
    margins[[i]] <- switch(mtypes[i],
                           n = list("log_dfun" = function(x, sig) dnorm(x,
                                                                        mean = 0,
                                                                        sd = exp(sig),
                                                                        log = T),
                                    "pfun" = function(q, sig) pnorm(q,
                                                                    mean = 0,
                                                                    sd = exp(sig))),
                           
                           sn = list("log_dfun" = function(x, om, alph) logdsn_0(x,
                                                                                 om=exp(om),
                                                                                 alph=alph),
                                     "pfun" = function(q, om, alph) psn_0(q,
                                                                          om=exp(om),
                                                                          alph=alph)))
    
    m_start[[i]] <- switch(mtypes[i],
                           n = c("sig"=0),
                           sn = c("om"=0,"alph"=0))
    
    names(m_start[[i]]) <- paste("mar_",i,"_",names(m_start[[i]]), sep ="")
    
  }
  names(margins) <- c("M1","M2")
  
  
  # Create copula
  cop <- list()
  
  if ( cop_fam == "normal" ){
    
    cop$copula <- ellipCopula(dim = 2,
                              dispstr = "un",
                              family = cop_fam)
    cop$type <- "ellip"
    
  } else {
    
    cop$copula <- archmCopula(dim = 2,
                              family = cop_fam)
    cop$type <- "arch"
    
  }
  cop$family <- cop_fam
  cop_start <- c("Cop_p" = -1)
  
  
  # Number of free parameters in the conditional mean model
  n1 <- switch(model,
               aids = 8, aids_r = 8,
               quaids = 10, quaids_r = 10,
               ml = 9, ml_r = 9)
  
  # Check if a startpoint exists
  if (file.exists( paste("start_points/cs_",db,"_",model,".RData",sep="") )){
    
    # Load mle estimates to use them as starting values
    load(paste("start_points/cs_",db,"_",model,".RData",sep=""))
    # The results are loaded in an object called est0
    
    # Extract initial values for the conditional mean model
    start <- est0@fullcoef[1:n1]
    # Extract residuals
    res_0 <- est0@resid
    
    # Delete the loaded object
    rm(est0)
    
  } else {
    
    # Create starting vector
    start <- rep(0.5,n1)
    start <- switch (model,
                     aids =  c(alpha0 = 0,alpha1=0,alpha2=0,gam11=0,gam12=0,gam22=0,beta1=0,beta2=0),
                     aids_r = c(alpha0 = 0,alpha1=0,alpha2=0,K11=1,K21=1,K22=1,beta1=0,beta2=0),
                     quaids = c(alpha0 = 0,alpha1=0,alpha2=0,gam11=0,gam12=0,gam22=0,beta1=0,beta2=0,lam1=0, lam2=0),
                     quaids_r = c(alpha0 = 0,alpha1=0,alpha2=0,K11=1,K21=1,K22=1,beta1=0,beta2=0,lam1=0, lam2=0),
                     ml = c(a1 = 0.1, a2=0.1,a3=0.1,A11=0.1,A13=0.1,A33=0.1,A22=0.1,B12=0.1, B23=0.1),
                     ml_r = c(a1 = 0.1, a2=0.1,a3=0.1,A11=0.1,A13=0.1,A33=0.1,A22=0.1,B12=0.1, B23=0.1)
    )
    
    # Initial residuals are the budget shares
    res_0 <- s
    
  }
  
  # Set up parameter lengths (to find them easily in)
  # the full parameter vector
  n2 <- length(cop_start)
  n3 <- length(m_start[[1]])
  n4 <- length(m_start[[2]])
  
  # This chunk leaves the mean (demand system) parameters fixed at the
  # previously estimated values and optimizes over the marginals' and copula's
  # parameters to get a starting point.
  # Setup parameter lengths
  
  # Satarting parameter for copula estimation
  start_m_c <- c(cop_start, m_start[[1]], m_start[[2]])
  
  # Set up ll (copula and margins only) function
  ll_m_c <- function(x) ll_fun_copula_res(cop_par = x[1:n2],
                                          M_par = list("M1"= x[(n2+1):(n2+n3)],
                                                       "M2"= x[(n2+n3+1):(n2+n3+n4)]),
                                          cop = cop,
                                          err = res_0[,mod$sel],
                                          margins = margins,
                                          transform = transform)
  parnames(ll_m_c) <- names(start_m_c)
  
  # Set up optimization options
  
  if (quiet) {
    
    print_level <- 0
    
  } else {
    
    print_level <- 1
    
  }
  
  global_opts <- list( "algorithm" = "NLOPT_GN_DIRECT_L",
                       "maxeval" = 5000,
                       "print_level" = print_level,
                       "ftol_rel"=10^-10)
  ub <- rep(20, length(start_m_c))
  lb <- -ub
  
  # Optimize ll function
  initial <- nloptr( x0 = start_m_c,
                     eval_f = function(x) ll_m_c(x),
                     lb = lb,
                     ub = ub,
                     opts = global_opts)
  initial <- initial$solution
  names(initial) <- names(start_m_c)
  
  # Set up estimation for full model.
  # starting point is built with past (normal) estimates
  # for model parameters and the copula & margins parameters
  # we just found.
  start <- c(start, initial)
  ll <- function(x) ll_fun_copula(mod_par = x[1:n1],
                                  cop_par = x[(n1+1):(n1+n2)],
                                  M_par = list("M1"= x[(n1+n2+1):(n1+n2+n3)],
                                               "M2"= x[(n1+n2+n3+1):(n1+n2+n3+n4)]),
                                  cop = cop,
                                  dat = dat,
                                  margins = margins,
                                  mod = mod,
                                  transform = transform)
  parnames(ll) <- names(start)
  
  # Maximum estimation loops
  max_loops <- 10
  
  second_ord <- FALSE
  loops <- 1
  while (!second_ord & loops <= max_loops) {
    
    # Scale model parameters to sum to 1 for ML models
    # atthe beggining of every
    # Iteration, before fixing the first (improves scaling issues)
    if (model == "ml" || model == "ml_r") {
      start[1:n1] <- mod$norm_fn(start[1:n1])
    }
    
    
    ub <- pmax(2*abs(start), rep(10^-2,length(start)))
    lb <- -ub
    
    # Fix a parameter in ML models
    if (model == "ml" || model == "ml_r") {
      ub[1] <- start[1]
      lb[1] <- start[1]
    }
    
    # SBPLX is good for finding initial solutions. We use it 
    # initially and switch to COBYLA
    if (loops == 1){
      
      # Setup optimization options for NLOPTR
      global_opts <- list( "algorithm" = "NLOPT_LN_SBPLX",
                           "maxeval" = 5000,
                           "print_level" = print_level,
                           "ftol_rel"=10^-20)
      
    } else {
      
      # Setup optimization options for NLOPTR
      global_opts <- list( "algorithm" = "NLOPT_LN_COBYLA",
                           "maxeval" = 15000,
                           "print_level" = print_level,
                           "ftol_rel"=10^-20)
      
    }
    
    # Optimize with nloptr
    e1 <- nloptr( x0 = start,
                  eval_f = function(x) ll(x),
                  lb = lb,
                  ub = ub,
                  opts = global_opts)
    
    e1 <- e1$solution
    names(e1) <- names(start)
    
    # If we are woring with the Minflex Laurent,
    # fix a parameter
    if (model == "ml" || model == "ml_r") {
      fixed <- list("a1"=as.numeric(e1[1]))
    } else {
      fixed <- NULL
    }
    
    # Optimize with mle2:
    # BFGS often fails as it is gradient based. If it does,
    # we switch to Nelder-Mead
    est <- tryCatch(mle2(minuslogl = ll,
                         start = e1,
                         vecpar = T,
                         parnames = names(start),
                         method = "BFGS",
                         fixed = fixed,
                         control = list(trace = print_level, REPORT = 1, maxit = 200),
                         skip.hessian = F,
                         use.ginv = T),
                    
                    error = function(error){
                      
                      print("BFGS Failure, Switching to Nelder-Mead")
                      
                      x <- mle2(minuslogl = ll,
                                start = e1,
                                vecpar = T,
                                parnames = names(start),
                                method = "Nelder-Mead",
                                fixed = fixed,
                                control = list(trace = print_level, REPORT = 1, maxit = 100),
                                skip.hessian = F,
                                use.ginv = T)
                      
                      return(x)
                    }
                    
    )
    
    # Check conditions on the Hessian.
    if ( !any(is.nan(abs(est@details$hessian))) ){
      
      if ( !any(abs(est@details$hessian)==Inf) ){
        
        if (rcond(est@details$hessian) != 0){
          
          if ( !any(diag(est@vcov)<0) ){
            
            second_ord <- TRUE
            
          }
          
        }
        
      }
    }
    
    # Set the starting values for next iteration on current estimates.
    start <- est@fullcoef
    
    loops <- loops + 1
    
  }
  
  # Carry out regularity checks
  tic()
  regularity <- regChecks(model = mod, data = dat, par = est@fullcoef[1:n1], Num = Num)
  time <- toc()
  attributes(est)$Reg_checks_time <-  time$toc - time$tic
  attributes(est)$Reg_vio <- regularity
  
  # Save residuals
  s_est <- do.call( mod$shares_fn,
                    c(mod$v2p_fn(est@fullcoef[1:n1]),
                      list(y=y,p = p)))
  
  res <- as.matrix( dat$s - s_est )
  attributes(est)$resid <- res
  
  # Akaike and Bayesian information criteria
  attributes(est)$AIC <- AIC(est)
  attributes(est)$BIC <- BIC(est)
  
  # Assign data name
  attributes(est)$Data <- db
  
  # Assign selection
  attributes(est)$Selection <- mod$sel
  
  # Assign copula family
  attributes(est)$Copula <- cop_fam
  
  # Assign marginals
  attributes(est)$Margins <- c(m1,m2)
  
  # Find and store elasticities
  # Get mean price values
  mean_p <- colMeans(dat$p)
  mean_p <- t(as.matrix(mean_p))
  
  # Marshalian elasticity matrix
  attributes(est)$Marsh_elast <- marsh_elast(mod$ds_dp_fn,
                                             mod$shares_fn,
                                             mod$v2p_fn,
                                             est@fullcoef,
                                             p = mean_p,
                                             y = mean(dat$y))
  
  # Income elasticities
  attributes(est)$Income_elast <- income_elast(mod$ds_dy_fn,
                                               mod$shares_fn,
                                               mod$v2p_fn,
                                               est@fullcoef,
                                               p = mean_p,
                                               y = mean(dat$y))
  
  # Add the standard errors of elasticities.
  se <- elast_se(mod$ds_dp_fn, mod$ds_dy_fn, mod$shares_fn, mod$v2p_fn, est, mean_p, mean(dat$y))
  attributes(est)$Elast_SE <- se
  
  # Store time
  attributes(est)$Time_est <- toc(quiet = TRUE)
  
  # Store second order condition
  attributes(est)$Second_Order <- second_ord
  
  # Compute copula fit measure ----
  
  # Get marginal parameters
  M_par <- list("M1"= est@fullcoef[(n1+n2+1):(n1+n2+n3)],
                "M2"= est@fullcoef[(n1+n2+n3+1):(n1+n2+n3+n4)])
  
  # Get copula parameters
  cop_par <- est@fullcoef[(n1+1):(n1+n2)]
  cop <- cop_param_transform(cop,cop_par)
  
  # Compute copula fit function
  copfit <- copula_fit(res[,mod$sel],margins,M_par,cop,transform)
  # Store
  attributes(est)$copfit <- copfit
  
  return(est)
  
}

# Result analysis ----

# Regularity checks
regChecks <- function(model, data, par, Num = F){
  
  neg <- count_negatives(shares=model$shares_fn,
                         v2p=model$v2p_fn,
                         est=par,
                         data=data,
                         n=3)
  
  mono <- check_mono(model$iu_fn, model$v2p_fn, par, data)
  
  curv <- check_curv(shares_fn = model$shares_fn,
                     ds_dp_fn = model$ds_dp_fn,
                     ds_dw_fn = model$ds_dy_fn,
                     v2p_fn = model$v2p_fn,
                     iu_fn = model$iu_fn,
                     est = par,
                     data = data,
                     numeric = Num)
  
  return(c("Negativity"=neg, "Monotonicity"=mono, "Curvature"=curv))
  
}

# Compute the BIC of a set of estimates
BIC <- function(est){
  
  n <- nrow(est@resid)
  k <- length(est@coef)
  logL <- -est@min
  
  return(log(n)*k - 2*logL)
  
  
}

# Compute the empirical copula decribed in Trivedi Eq.4.9
empir_copula <- function(data){
  
  # Data must be a two-column matrix.
  
  # Number of observations
  n <- nrow(data)
  # Compute impirical copula cummulative value of each row
  cop <- apply(data, MARGIN = 1, FUN = function(x) sum(data[,1]<=x[1] & data[,2]<=x[2])/n)
  
  # Return the vector of empirical copula values
  return(cop)
  
}

# Copula fitness measure in Trivedi Eq 4.10
copula_fit <- function(res,margins,M_par,cop,transform = F){
  
  if (transform == "yes") {
    res[,2] <- -1*res[,2]
  }
  
  # Pseudo-observations
  u_emp <- apply(res, MARGIN = 2, FUN = function(x) rank(x)/(length(x)+1))
  # Empirical copula
  emp_cop <- empir_copula(u_emp)
  
  # Parametric estimated pseudo-observations
  u_par <- tryCatch(sapply(1:ncol(res),
                           function(i) do.call(margins[[i]]$pfun, args = c(list(res[,i]),
                                                                           as.numeric(M_par[[i]])))),
                    error = function(res) return(NaN))
  
  # Parametric estimated copula values
  par_cop <- pCopula(u_par, cop$copula)
  
  return(sum((emp_cop - par_cop)^2)/length(emp_cop))
  
}