function [g_ineq,g_eq] = moments_theta(theta,J1,J2,KMSoptions)
%% USER-SPECIFIED FUNCTION: Moment function that depends on parameter only
% The moment functions are in the form
%
%       E_P[m(W_i,theta)] = E_P[f(W_i)] + g(theta)
%
% where
%
%       E_P[m_j(W_i,theta)] <= 0 for j = 1,...,J1
%       E_P[m_j(W_i,theta)] = 0  for j = J1+1,...,J1+J2
%
% This function computes the model-implied component moment (in)equalities,
% g_j(theta).
%
% The user needs to specify this function.  An example is given below.
%
% INPUT:
%   theta         d-by-1 vector of parameters
%
%   J1            Integer number of moment inequalities
%
%   J2.           Integer number of moment equalities
%
%   KMSoptions.   This is a structure of additional inputs.  The user can
%                 add parameters to KMSoptions, say KMSoptions.params,
%                 and call KMSoptions.params in the user-specified
%                 functions.
%                 For example, in the 2x2 entry game, we include the
%                 support for the covariates and the probability that
%                 a particular support point occurs.
%
% OUTPUT:
%   g_ineq    J1-by-1 vector of moment inequalities g_j(theta) for j=1,...,J1
%
%   g_eq      2*J2-by-1 vector of moment inequalities g_j(theta) for j=1,...,J2
%             Note that we re-write the moment equalities as two moment
%             inequalities.  Thus, we have
%             f_eq = [g(theta) ; - g(theta)], where g(theta) is the
%             vector of moment equalities.
%
% NOTE: User does not need to specify J1 nor J2 here, since it is already
% computed in moments_w(W,KMSoptions).

% Load ancilliary parameters
uH = KMSoptions.uH;
xx = KMSoptions.xx;

% Extract parameters
a = theta(1);
g = theta(2);
b = theta(3:end);

% Define moment inequalities
L = (1+kron(exp(-xx*b),g*uH.^a)).^(-1/g);
Lm = (1+kron(exp(-xx*b),g*uH(2:end).^a)).^(-1/g) ...
        - (1+kron(exp(-xx*b),g*uH(1:end-1).^a)).^(-1/g);
g_ineq = [-L; L; Lm; 1-L; L-1];
    
% There are no moment equalities, so output the empty set.
g_eq = [];
