function KMS_EndCensAppl(DGP,yyvec,ks,knopt,alpha,ncpus)

% This code estimates confidence sets for benefit elasticity using SIPP data
% similar to Chetty(2008) with KMS method, c-dependence and piecewise
% constant hazard
%   DGP             DGP = 1, 5 regressors
%                   DGP = 2, 3 regressors
%   yyvec           yyvec(1):yyvec(2):yyvec(3) = grid for y
%   ks              number of steps in the integrated hazard 
%   knopt           Slackness sequence contstant (see def. of kappa below)
%   alpha           Significance level
%   ncpus           Number of CPUs used
%   KMSoptions      Structure of options set by user.

%% Key Parameters:
method      = 'KMS';    % Method - either AS or KMS
component   = ks + 2;        % Component of theta to build confidence interval around
name        = strcat('_cdep_DGP',num2str(DGP),'_coverage',num2str(100*(1-alpha)),...
                     '_k',num2str(knopt),'_ymax',num2str(yyvec(3)),'_ystep',num2str(yyvec(2)));
KMSoptions  = KMSoptions();

%% Load data
dname = sprintf('../SIPP1spdat%d',DGP);
load(dname); n = size(data,1);
x = data(:,3:end-1);

%% Get the steps and starting values for hazard step sizes
if ks == 10
  % Starting step sizes chosen to fit Lam(y) = y^1.5 (estimate from Weibull model under indep. cens.)
  a0 = [5.6410   17.2525   24.1870   29.4640   33.9161   37.8433   41.3973   44.6683   47.7148   50.5778];
  steps = 1:5:46;
elseif ks == 5
  a0 = [14.2672   47.5452   67.5698   82.6531   95.3378];      
  steps = 1:10:46;
elseif ks == 8
  a0 = [5.6410   17.2525   24.1870   29.4640   33.9161   37.8433   41.3973   44.6683];      
  steps = 1:6:47;  
else
  error('Number %d of steps not supported',ks)
end
KMSoptions.steps = steps;

%% Extract/Save Information to KMSoptions, and set seed
KMSoptions.DGP = DGP;
KMSoptions.n = n;
KMSoptions.component = component;
KMSoptions.num_cores = ncpus;
KMSoptions.seed =  1;
stream = RandStream('mlfg6331_64','Seed',1);
RandStream.setGlobalStream(stream)

%% Parameters
type = 'two-sided';                 % Two-sided or one sided test?  Set to 'one-sided-UB' or 'one-sided-LB' or 'two-sided'
kappa = @(n) sqrt(knopt*log(n));    % Kappa function
phi   = NaN;                        % Default GMS function      

%% Parameters that depend on DGP
if DGP == 1
    c = 0.26283;
elseif DGP == 2    
    c = 0.15946;
end
KMSoptions.c = c;
    
xx = unique(x,'rows');          % Discrete X values
KMSoptions.xx = xx;
kk = size(xx,1); KMSoptions.kk = kk;
KMSoptions.uH = [yyvec(1):yyvec(2):yyvec(3)]';   % Discrete evaluation points for duration Y
xname = sprintf('../x00_%d',DGP); load(xname);
theta_0 = [a0 x00(2:end-1)]';     % Initial parameter vector
p = zeros(size(theta_0,1),1);   % Projection direction
p(component) = 1;               % Projection direction
KMSoptions.S =  0;              % Rho Polytope Constraints
xs = size(x,2);
LB_theta = [zeros(1,ks) 1e-12 -20*ones(1,xs)]';  % Lower bound on parameter space
UB_theta = [1000*ones(1,ks) 20*ones(1,1+xs)]';    % Upper bound on parameter space
A_theta = [];                   % No polytope constraints
b_theta = [];                   % No polytope constraints
CVXGEN_name = [];               % CVXGEN file name

t1 = tic;
[KMS_confidence_interval,KMS_output] = KMS_0_Main(data(:,1:end-1),theta_0,...
    p,[],LB_theta,UB_theta,A_theta,b_theta,alpha,type,method,kappa,phi,CVXGEN_name,KMSoptions)
totaltime_KMS = toc(t1)

%% Save  KMS
%date = datestr(now, 'yyyy_mm_dd_HH_MM_SS');
filename = strcat('Results/IDset',name,'.mat');
save(filename)

