

# Procedures for implementing method of "Decomposing Economic Mobility Transition Matrices" - Richey and Rosburg

# This is a modified version of code originally written by Christoph Rothe for his "Decomposing the Composition Effect" Paper in JBES 2016
# His original code is available on his website (www.christophrothe.net), we thank him for making this available.
# This version modifies his code to the transition matrix setting and also uses a different copula estimation procedure.

# Before running the 'call' code, you need to 'source' this file.

# Load required packages
library(foreign)
library(Hmisc)
library(mvtnorm)
library(copula)
library(survey)
library(xtable)
library(clusterGeneration)
library(corpcor)
 
# Modify the function wtd.quantile such that it has the properties described in Rothe's JBES paper. 
#        This is Rothe's alteration.
# NOTE: we do not use weights, but this has been left in case others wish to switch the copula estimation and use weights for their application

wtd.quantile <- function (x, weights = NULL, probs = c(0, 0.25, 0.5, 0.75, 1),
                          type = c("quantile", "(i-1)/(n-1)", "i/(n+1)", "i/n"), normwt = FALSE,
                          na.rm = TRUE)
{
      if (!length(weights))
            return(quantile(x, probs = probs, na.rm = na.rm))
      type <- match.arg(type)
      if (any(probs < 0 | probs > 1))
            stop("Probabilities must be between 0 and 1 inclusive")
      nams <- paste(format(round(probs * 100, if (length(probs) >
                                                        1) 2 - log10(diff(range(probs))) else 2)), "%", sep = "")
      if (type == "quantile") {
            w <- wtd.table(x, weights, na.rm = na.rm, normwt = normwt,
                           type = "list")
            x <- w$x
            wts <- w$sum.of.weights
            n <- sum(wts)
            order <- 1 + (n - 1) * probs
            low <- pmax(floor(order), 1)
            high <- pmin(low + 1, n)
            order <- order%%1
            allq <- approx(cumsum(wts), x, xout = c(low, high), method = "constant",
                           f = 1, rule = 2)$y
            k <- length(probs)
            quantiles <- (1 - order) * allq[1:k] + order * allq[-(1:k)]
            names(quantiles) <- nams
            return(quantiles)
      }
      w <- wtd.Ecdf(x, weights, na.rm = na.rm, type = type, normwt = normwt)
      structure(approx(w$ecdf, w$x, xout = probs, rule = 2, f=1,method = "constant",ties="ordered")$y,
                names = nams)
}



# decomp.object is a function that estimates the copula and the conditional CDF.
# It is later used to create an object on which the actual decomposition is then
# performed.

decomp.object <- function(formula, covariate, dat, weights = NA) {
      
      
      # get variable name
      outname <- all.vars(formula)[1]
      covname <- all.vars(formula)[-1]
      pure.covname <- all.vars(covariate)
      
      # Define List objects
      m.cdf <- list(NA)
      j.cdf <- list(NA)
      n.corr <- list(NA)
      copula.para <- list(NA)
      norm.cop <- list(NA)
      cond.cdf <- list(NA)
      dat.modmat <- list(NA)
      dat.pure <- list(NA)
      y <- list(NA)
      n <- list(NA)
      new.dat <- list(NA)
      
      ygrid <- quantile(c(dat[[1]][,outname],dat[[2]][,outname],dat[[3]][,outname],dat[[4]][,outname]), c(seq(0.001,.999,length=101))  )
      
      dat.pure[[1]] <- model.matrix(covariate, data = dat[[1]])[,-1]
      dat.pure[[2]] <- model.matrix(covariate, data = dat[[2]])[,-1]
      dat.pure[[3]] <- model.matrix(covariate, data = dat[[3]])[,-1]
      dat.pure[[4]] <- model.matrix(covariate, data = dat[[4]])[,-1]
      
      pb <- txtProgressBar(min = 0, max = 6, style = 3)
      
      for (i in 1:4) {
            
            # determine sample size
            n[[i]] <- dim(dat[[i]])[1]
            
            # check for presence of weights. if false assign 1/n
            if (sum(is.na(weights[[i]]))>0) {weights[[i]] <- rep(1,n[[i]])}
            
            # Prepare some data for output later
            y[[i]] <- dat[[i]][,outname]
            
            ## Estimate Conditional CDFs
            
            dat.modmat[[i]] <- model.matrix(formula, data = dat[[i]])[,-1] 
            cond.cdf[[i]] <- matrix(NA,ncol=dim(dat.modmat[[i]])[2]+1,nrow=length(ygrid))
            
            for (j in 2:(length(ygrid)-1)){
                  dat[[i]][,outname] <-  I(y[[i]] <= ygrid[j])
                  design <- svydesign(~0, data=dat[[i]], weights=~weights[[i]])
                  cond.cdf[[i]][j,] <- svyglm(formula, data=dat[[i]], design=design,
                                              family=quasibinomial(link="probit"),
                                              start=(if (exists("mod")) {coef(mod)} ),
                                              control=list(epsilon=1e-8,maxit=100,trace=FALSE))$coef
                  setTxtProgressBar(pb, 3*(i-1) + j/length(ygrid))
            }
            dat[[i]][,outname] <- y[[i]]
            
            
              ## Estimate Marginal CDFs
              ## Randomly determine at which points to evaluate the joint CDF. Here, the evaluation is performed at 500 randomly selected
              ## points. With smaller datasets than the one used in the paper, one could choose
              ## a higher number here.
            
            ss <- sample(1:length(dat.pure[[i]][,1]), min(c(length(dat.pure[[i]][,1]),500)),prob=weights[[i]], replace=FALSE)
            
            m.cdf[[i]] <- dat.pure[[i]]
            for (j in 1:dim(dat.pure[[i]])[2]) {
                  wecdf <- wtd.Ecdf(dat.pure[[i]][,j], w=weights[[i]], normwt=TRUE)
                  m.cdf[[i]][,j] <- approx(wecdf$x[-1], wecdf$ecdf[-1], xout = dat.pure[[i]][,j],  f=1, yleft=0, yright=1, method = "constant")$y
            }
            m.cdf[[i]] <-  m.cdf[[i]][ss,]
            
            ## Estimate Copula Functions
              ## NOTE: this is a key difference in our code versus Rothe's code (See Rothe's original code for his method;
              ## available at his website -- www.christophrothe.net)
            
            p.data <- pobs(dat.pure[[i]]) # creates the psuedo data
            dim.par <- ((dim(p.data)[2])^2 - (dim(p.data)[2]))/2 # dimension length of parameters of gaussian copula
            
            # NOTE: this next step is the point at which one can choose to use a normal or t copula. Simply comment
            #       one and use the other. As noted in a manuscript footnote, we used the normal for our baseline estimation but
            #       the 't' for a robustness check. 
            
            n.copu <- normalCopula(rep(0.5,dim.par), dim = dim(p.data)[2], dispstr = "un") #creates generic copula to use in estimation as object
            #n.copu <- tCopula(rep(0.5,dim.par), dim = dim(p.data)[2], dispstr = "un") # Robust check swaps out for a tCopula with df estimated by fitCopula
            
            vars.mat <- fitCopula(n.copu,p.data)  #fits copula model
            corr.dat <- slot(vars.mat,"copula") #save actual copula from 'fitcopula' object to use in data generation
            
            ## Generate hypothetical data for decomposition later.
            new.dat[[i]] <- list(NA)
            new.dat[[i]][[1]] <- rCopula(500000,corr.dat) #draws from copula to match with data from group of choice 
            new.dat[[i]][[2]] <- new.dat[[i]][[1]]
            new.dat[[i]][[3]] <- new.dat[[i]][[1]]
            new.dat[[i]][[4]] <- new.dat[[i]][[1]]
            for (l in 1:dim(new.dat[[i]][[1]])[2]) {
                  new.dat[[i]][[1]][,l] <- wtd.quantile(dat.pure[[1]][,l], new.dat[[i]][[1]][,l],type="i/n", w=weights[[1]], normwt=TRUE)
                  new.dat[[i]][[2]][,l] <- wtd.quantile(dat.pure[[2]][,l], new.dat[[i]][[2]][,l],type="i/n", w=weights[[2]], normwt=TRUE)
                  new.dat[[i]][[3]][,l] <- wtd.quantile(dat.pure[[3]][,l], new.dat[[i]][[3]][,l],type="i/n", w=weights[[3]], normwt=TRUE)
                  new.dat[[i]][[4]][,l] <- wtd.quantile(dat.pure[[4]][,l], new.dat[[i]][[4]][,l],type="i/n", w=weights[[4]], normwt=TRUE)
                  setTxtProgressBar(pb, 3*(i-1) + 2+l/dim(new.dat[[i]][[1]])[2])
            }
      }
      close(pb)
      
      list(norm.cop, dat.pure, cond.cdf, ygrid, outname, formula, weights,  pure.covname, y, new.dat)
      
}


## The function decomp performs the computation of the various elements of the
## detailed decompositions. The arguments conditional, copula and marginals
## specify from which group the respective object is to be taken.

decomp <- function(decomp.obj, conditional = 1, copula =1, marginals = c(1,2)) {
      
      dat.pure <- decomp.obj[[2]]
      cond.cdf <- decomp.obj[[3]]
      ygrid <- decomp.obj[[4]]
      outname <- decomp.obj[[5]]
      formula <-  decomp.obj[[6]]
      weights <-  decomp.obj[[7]]
      y <- decomp.obj[[9]]
      new.dat <- decomp.obj[[10]]
      
      i <- conditional     ## conditional
      j <- copula          ## copula
      k <- marginals       ## marginal
      
      ## Create New Covariate Data (for numerical integration)
      
      newxdat <- new.dat[[j]][[1]]
      for (l in 1:dim(newxdat)[2]) {
            newxdat[,l] <- new.dat[[j]][[k[l]]][,l]
      }
      
      newxdat <- data.frame(rep(1,dim(newxdat)[1]),newxdat)
      colnames(newxdat) <- c(outname,colnames(dat.pure[[1]]))
      newevdat <- model.matrix(formula, data = newxdat)
      
      ## Compute Counterfactual Outcome CDF
      cdf.out <- 1/dim(dat.pure[[i]])[1]
      for (l in 2:(length(ygrid)-1)) {
            cdf.out[l] <- mean(pnorm(as.vector(newevdat%*%cond.cdf[[i]][l,])))
      }
      cdf.out[length(ygrid)] <- 1
      
      pseudo.data <- approx(x=sort(cdf.out), y=ygrid, xout=seq(0,1,length=50000), method="linear", rule=2)$y
      ## can compute summary measures from pseudo data.
      
      
      
}


full.decomp <- function(formula, covariate, dat, weights = NA) {
      
      
      test <<- decomp.object(formula, covariate, dat, weights)
      varnames  <- test[[8]]
      nvar <- length(varnames)
      
      # Now we use the different estimated components to simulate our various counterfactual transition matrices
      
      v <- list(NA)
      v[[1]] <- matrix(0,ncol=4,nrow=5+sum(nvar:1)) # index results
      v[[2]] <- matrix(0,ncol=4,nrow=5+sum(nvar:1)) # Top quartile results
      v[[3]] <- matrix(0,ncol=4,nrow=5+sum(nvar:1)) # Bottom quartile results
      v[[4]] <- matrix(0,ncol=4,nrow=4) # Empirical Matrix from simulated data
      v[[5]] <- matrix(0,ncol=4,nrow=4) # Counterfactual Matrix
      v[[6]] <- matrix(0, ncol = 2, nrow = 4) # empirical and counterfactual Ms
      v[[7]] <- matrix(0,ncol=4,nrow=4) # Composition Effect Matrix
      v[[8]] <- matrix(0,ncol=4,nrow=4) # Structure Effect Matrix

      # Here we use the 'decomp' function to create the various counterfactual distributions of 
      # childrens incomes for each group (quartile of parents' income).
      # These will then be pieced together in different ways (according to paper)
      # in order to simulate counterfacual transition matrices.
      
      d.treat <- decomp(test,conditional = 4, copula =4, marginals = rep(4,nvar)) ## COULD ALSO USE RAW SAMPLE VALUE
      
      d.counter.3 <- decomp(test,conditional = 3, copula =4, marginals = rep(4,nvar)) ## COULD ALSO USE VALUE THAT CIRCUMVENTS COPULA ESTIMATION
      d.counter.m.3 <- decomp(test,conditional = 3, copula =3, marginals = rep(4,nvar))
      d.base.3 <- decomp(test,conditional = 3, copula =3, marginals = rep(3,nvar)) ## COULD ALSO USE RAW SAMPLE VALUE
      
      d.counter.2 <- decomp(test,conditional = 2, copula =4, marginals = rep(4,nvar)) ## COULD ALSO USE VALUE THAT CIRCUMVENTS COPULA ESTIMATION
      d.counter.m.2 <- decomp(test,conditional = 2, copula =2, marginals = rep(4,nvar))
      d.base.2 <- decomp(test,conditional = 2, copula =2, marginals = rep(2,nvar)) ## COULD ALSO USE RAW SAMPLE VALUE
      
      d.counter.1 <- decomp(test,conditional = 1, copula =4, marginals = rep(4,nvar)) ## COULD ALSO USE VALUE THAT CIRCUMVENTS COPULA ESTIMATION
      d.counter.m.1 <- decomp(test,conditional = 1, copula =1, marginals = rep(4,nvar))
      d.base.1 <- decomp(test,conditional = 1, copula =1, marginals = rep(1,nvar)) ## COULD ALSO USE RAW SAMPLE VALUE
      
      
      
      # c.none is the independent transition matrix
      c.none <- c(d.treat, d.treat, d.treat, d.treat)
      t.mat.none <- matrix(nrow = 4, ncol = 4)
      for(j in c(0.25, 0.5, 0.75, 1)){
            
            j.temp <- quantile(c.none, probs = j)
            j.temp2 <- quantile(c.none, probs = (j - 0.25))
            t.mat.none[1,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)
            t.mat.none[2,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)     
            t.mat.none[3,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)
            t.mat.none[4,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)
            
      }    
      
      # calculate various indices of the matrix (see call function for details)
      none.m1 <- 0
      none.m2 <- 0
      none.m3 <- 0
      none.m4 <- 0
      
      temp <- 0
      for(i in 1:4){ temp <- temp + t.mat.none[i,i] }
      none.m1 <- (4 - temp)/3
      
      none.m2 <-  1 - abs(eigen(t.mat.none)$values[2])
      
      temp <- 0
      for(i in 1:4){
            for(j in 1:4){temp <- temp + 0.25*t.mat.none[i,j]*abs(i-j)}
      }
      none.m3 <- temp
      
      pi <- matrix(c(.25,0,0,0,0,.25,0,0,0,0,.25,0,0,0,0,.25), ncol = 4, nrow = 4)
      t.mat.none.sym <- 0.5*(t.mat.none + pi%*%t(t.mat.none)%*%solve(pi))
      none.m4 <- 1 - abs(eigen(t.mat.none.sym)$values[2])
      
      # c.full is the simulated empirical matrix
      c.full <- c(d.treat, d.base.3, d.base.2, d.base.1)
      t.mat.full <- matrix(nrow = 4, ncol = 4)
      for(j in c(0.25, 0.5, 0.75, 1)){
            
            j.temp <- quantile(c.full, probs = j)
            j.temp2 <- quantile(c.full, probs = (j - 0.25))
            t.mat.full[1,j*4] <-  length(d.base.1[d.base.1 >= (j.temp2) & d.base.1 <= j.temp])/length(d.base.1)
            t.mat.full[2,j*4] <-  length(d.base.2[d.base.2 >= (j.temp2) & d.base.2 <= j.temp])/length(d.base.2)
            t.mat.full[3,j*4] <-  length(d.base.3[d.base.3 >= (j.temp2) & d.base.3 <= j.temp])/length(d.base.3)
            t.mat.full[4,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)       
            
      }   
      
      full.m1 <- 0
      full.m2 <- 0
      full.m3 <- 0
      full.m4 <- 0
      
      temp <- 0
      for(i in 1:4){ temp <- temp + t.mat.full[i,i] }
      full.m1 <- (4 - temp)/3
      
      full.m2 <-  1 - abs(eigen(t.mat.full)$values[2])
      
      temp <- 0
      for(i in 1:4){
            for(j in 1:4){temp <- temp + 0.25*t.mat.full[i,j]*abs(i-j)}
      }
      full.m3 <- temp
      
      t.mat.full.sym <- 0.5*(t.mat.full + pi%*%t(t.mat.full)%*%solve(pi))
      full.m4 <- 1 - abs(eigen(t.mat.full.sym)$values[2])
      
      # c.str is the counterfactual matrix that closes the full composition effect 
      c.str <- c(d.treat, d.counter.3, d.counter.2, d.counter.1)
      t.mat.str <- matrix(nrow = 4, ncol = 4)
      for(j in c(0.25, 0.5, 0.75, 1)){
            
            j.temp <- quantile(c.str, probs = j)
            j.temp2 <- quantile(c.str, probs = (j - 0.25))
            t.mat.str[1,j*4] <-  length(d.counter.1[d.counter.1 >= (j.temp2) & d.counter.1 <= j.temp])/length(d.counter.1)
            t.mat.str[2,j*4] <-  length(d.counter.2[d.counter.2 >= (j.temp2) & d.counter.2 <= j.temp])/length(d.counter.2)
            t.mat.str[3,j*4] <-  length(d.counter.3[d.counter.3 >= (j.temp2) & d.counter.3 <= j.temp])/length(d.counter.3)
            t.mat.str[4,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)       
            
      }     
      
      str.m1 <- 0
      str.m2 <- 0
      str.m3 <- 0
      str.m4 <- 0
      
      temp <- 0
      for(i in 1:4){ temp <- temp + t.mat.str[i,i] }
      str.m1 <- (4 - temp)/3
      
      str.m2 <-  1 - abs(eigen(t.mat.str)$values[2])
      
      temp <- 0
      for(i in 1:4){
            for(j in 1:4){temp <- temp + 0.25*t.mat.str[i,j]*abs(i-j)}
      }
      str.m3 <- temp
      
      t.mat.str.sym <- 0.5*(t.mat.str + pi%*%t(t.mat.str)%*%solve(pi))
      str.m4 <- 1 - abs(eigen(t.mat.str.sym)$values[2])
        
      # c.dep is the counterfactual matrix that only closes the covariate (marginal) distribution gaps of the composition effect (ie not the dependence aspect)
      c.dep <- c(d.treat, d.counter.m.3, d.counter.m.2, d.counter.m.1)
      t.mat.dep <- matrix(nrow = 4, ncol = 4)
      for(j in c(0.25, 0.5, 0.75, 1)){
            
            j.temp <- quantile(c.dep, probs = j)
            j.temp2 <- quantile(c.dep, probs = (j - 0.25))
            t.mat.dep[1,j*4] <-  length(d.counter.m.1[d.counter.m.1 >= (j.temp2) & d.counter.m.1 <= j.temp])/length(d.counter.m.1)
            t.mat.dep[2,j*4] <-  length(d.counter.m.2[d.counter.m.2 >= (j.temp2) & d.counter.m.2 <= j.temp])/length(d.counter.m.2)
            t.mat.dep[3,j*4] <-  length(d.counter.m.3[d.counter.m.3 >= (j.temp2) & d.counter.m.3 <= j.temp])/length(d.counter.m.3)
            t.mat.dep[4,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)       
            
      }    
      
      dep.m1 <- 0
      dep.m2 <- 0
      dep.m3 <- 0
      dep.m4 <- 0
      
      temp <- 0
      for(i in 1:4){ temp <- temp + t.mat.dep[i,i] }
      dep.m1 <- (4 - temp)/3
      
      dep.m2 <-  1 - abs(eigen(t.mat.dep)$values[2])
      
      temp <- 0
      for(i in 1:4){
            for(j in 1:4){temp <- temp + 0.25*t.mat.dep[i,j]*abs(i-j)}
      }
      dep.m3 <- temp
      
      t.mat.dep.sym <- 0.5*(t.mat.dep + pi%*%t(t.mat.dep)%*%solve(pi))
      dep.m4 <- 1 - abs(eigen(t.mat.dep.sym)$values[2])
      
      # With the counterfactuals above, we can do the aggregate decomp as well as identify the dependence and total marginal effects
      
      v[[1]][1,] <- cbind(none.m1, none.m2, none.m3, none.m4) - cbind(full.m1, full.m2, full.m3, full.m4)
      v[[1]][2,] <- cbind(none.m1, none.m2, none.m3, none.m4) - cbind(str.m1, str.m2, str.m3, str.m4)
      v[[1]][3,] <- cbind(str.m1, str.m2, str.m3, str.m4) - cbind(full.m1, full.m2, full.m3, full.m4)
      v[[1]][4,] <- cbind(str.m1, str.m2, str.m3, str.m4) - cbind(dep.m1, dep.m2, dep.m3, dep.m4)
      v[[1]][5,] <- cbind(dep.m1, dep.m2, dep.m3, dep.m4) - cbind(full.m1, full.m2, full.m3, full.m4)
      
      v[[2]][1,] <- t.mat.none[4,] - t.mat.full[4,]
      v[[2]][2,] <- t.mat.none[4,] - t.mat.str[4,]
      v[[2]][3,] <- t.mat.str[4,] - t.mat.full[4,]
      v[[2]][4,] <- t.mat.str[4,] - t.mat.dep[4,]
      v[[2]][5,] <- t.mat.dep[4,] - t.mat.full[4,]
      
      v[[3]][1,] <- t.mat.none[1,] - t.mat.full[1,]
      v[[3]][2,] <- t.mat.none[1,] - t.mat.str[1,]
      v[[3]][3,] <- t.mat.str[1,] - t.mat.full[1,]
      v[[3]][4,] <- t.mat.str[1,] - t.mat.dep[1,]
      v[[3]][5,] <- t.mat.dep[1,] - t.mat.full[1,]
      
      v[[4]]<- t.mat.full

      
      v[[5]] <- t.mat.str
      
      v[[6]][1,1] <- full.m1
      v[[6]][2,1] <- full.m2
      v[[6]][3,1] <- full.m3
      v[[6]][4,1] <- full.m4
      v[[6]][1,2] <- str.m1
      v[[6]][2,2] <- str.m2
      v[[6]][3,2] <- str.m3
      v[[6]][4,2] <- str.m4
      

      
      # Now loop through and alter one marginal at a time to get the 'direct' marginal effects of the composition effect
      
      base.marginals.1 <-  rep(1,nvar)
      base.marginals.2 <-  rep(2,nvar)
      base.marginals.3 <-  rep(3,nvar)
      
      for (i in 1:nvar) {
            mm1 <- base.marginals.1
            mm2 <- base.marginals.2
            mm3 <- base.marginals.3
            
            mm1[i] <- mm1[i]+3
            mm2[i] <- mm2[i]+2
            mm3[i] <- mm3[i]+1
            
            d.counter.m.m.1 <- decomp(test,conditional = 1, copula =1, marginals = mm1)
            d.counter.m.m.2 <- decomp(test,conditional = 2, copula =2, marginals = mm2)
            d.counter.m.m.3 <- decomp(test,conditional = 3, copula =3, marginals = mm3)
            
            c.dep <- c(d.treat, d.counter.m.m.1, d.counter.m.m.3, d.counter.m.m.2)
            t.mat.dep <- matrix(nrow = 4, ncol = 4)
            for(j in c(0.25, 0.5, 0.75, 1)){
                  
                  j.temp <- quantile(c.dep, probs = j)
                  j.temp2 <- quantile(c.dep, probs = (j - 0.25))
                  t.mat.dep[1,j*4] <-  length(d.counter.m.m.1[d.counter.m.m.1 >= (j.temp2) & d.counter.m.m.1 <= j.temp])/length(d.counter.m.m.1)
                  t.mat.dep[2,j*4] <-  length(d.counter.m.m.2[d.counter.m.m.2 >= (j.temp2) & d.counter.m.m.2 <= j.temp])/length(d.counter.m.m.2)
                  t.mat.dep[3,j*4] <-  length(d.counter.m.m.3[d.counter.m.m.3 >= (j.temp2) & d.counter.m.m.3 <= j.temp])/length(d.counter.m.m.3)
                  t.mat.dep[4,j*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)       
                  
            }
            
            dep.m1 <- 0
            dep.m2 <- 0
            dep.m3 <- 0
            dep.m4 <- 0
            
            temp <- 0
            for(k in 1:4){ temp <- temp + t.mat.dep[k,k] }
            dep.m1 <- (4 - temp)/3
            
            dep.m2 <-  1 - abs(eigen(t.mat.dep)$values[2])
            
            temp <- 0
            for(k in 1:4){
                  for(j in 1:4){temp <- temp + 0.25*t.mat.dep[k,j]*abs(k-j)}
            }
            dep.m3 <- temp
            
            t.mat.dep.sym <- 0.5*(t.mat.dep + pi%*%t(t.mat.dep)%*%solve(pi))
            dep.m4 <- 1 - abs(eigen(t.mat.dep.sym)$values[2])
            
            v[[1]][5+i,] <- cbind(dep.m1, dep.m2, dep.m3, dep.m4) - cbind(full.m1, full.m2, full.m3, full.m4)
            
            v[[2]][5+i,] <- t.mat.dep[4,] - t.mat.full[4,]
            
            v[[3]][5+i,] <- t.mat.dep[1,] - t.mat.full[1,]
 
      }
      
      # Now loop through and calculate all two-way interactions to the marginal effects
      
      for (i in 1:(nvar-1)) {
            for (j in 1:(nvar-i)) {
                  mm1 <- base.marginals.1
                  mm2 <- base.marginals.2
                  mm3 <- base.marginals.3
                  
                  mm1[c(i,i+j)] <- mm1[c(i,i+j)]+c(3,3)
                  mm2[c(i,i+j)] <- mm2[c(i,i+j)]+c(2,2)
                  mm3[c(i,i+j)] <- mm3[c(i,i+j)]+c(1,1)
                  
                  d.counter.m.m1 <- decomp(test,conditional = 1, copula =1, marginals = mm1)
                  d.counter.m.m2 <- decomp(test,conditional = 2, copula =2, marginals = mm2)
                  d.counter.m.m3 <- decomp(test,conditional = 3, copula =3, marginals = mm3)
                  
                  c.dep <- c(d.treat, d.counter.m.m3, d.counter.m.m2, d.counter.m.m1)
                  t.mat.dep <- matrix(nrow = 4, ncol = 4)
                  for(k in c(0.25, 0.5, 0.75, 1)){
                        
                        j.temp <- wtd.quantile(c.dep, probs = k)
                        j.temp2 <- wtd.quantile(c.dep, probs = (k - 0.25))
                        t.mat.dep[1,k*4] <-  length(d.counter.m.m1[d.counter.m.m1 >= (j.temp2) & d.counter.m.m1 <= j.temp])/length(d.counter.m.m1)
                        t.mat.dep[2,k*4] <-  length(d.counter.m.m2[d.counter.m.m2 >= (j.temp2) & d.counter.m.m2 <= j.temp])/length(d.counter.m.m2)
                        t.mat.dep[3,k*4] <-  length(d.counter.m.m3[d.counter.m.m3 >= (j.temp2) & d.counter.m.m3 <= j.temp])/length(d.counter.m.m3)
                        t.mat.dep[4,k*4] <-  length(d.treat[d.treat >= (j.temp2) & d.treat <= j.temp])/length(d.treat)       
                        
                  }
                  
                  dep.m1 <- 0
                  dep.m2 <- 0
                  dep.m3 <- 0
                  dep.m4 <- 0
                  
                  temp <- 0
                  for(k in 1:4){ temp <- temp + t.mat.dep[k,k] }
                  dep.m1 <- (4 - temp)/3
                  
                  dep.m2 <-  1 - abs(eigen(t.mat.dep)$values[2])
                  
                  temp <- 0
                  for(k in 1:4){
                        for(l in 1:4){temp <- temp + 0.25*t.mat.dep[k,l]*abs(k-l)}
                  }
                  dep.m3 <- temp
                  
                  t.mat.dep.sym <- 0.5*(t.mat.dep + pi%*%t(t.mat.dep)%*%solve(pi))
                  dep.m4 <- 1 - abs(eigen(t.mat.dep.sym)$values[2])
                  
                  v[[1]][(5+nvar)+ cumsum(c(0,((nvar-1)):2 ))[i]  + j, ] <- cbind(dep.m1, dep.m2, dep.m3, dep.m4) - 
                                                                            cbind(full.m1, full.m2, full.m3, full.m4) - 
                                                                            v[[1]][5+i,] - v[[1]][5+i+j,]
                  
                  v[[2]][(5+nvar)+ cumsum(c(0,((nvar-1)):2 ))[i]  + j, ] <- t.mat.dep[4,] - t.mat.full[4,] - v[[2]][5+i,] - v[[2]][5+i+j,]
                  
                  v[[3]][(5+nvar)+ cumsum(c(0,((nvar-1)):2 ))[i]  + j, ] <- t.mat.dep[1,] - t.mat.full[1,] - v[[3]][5+i,] - v[[3]][5+i+j,]
 
                  
                  varnames <- c(varnames,paste(varnames[c(i,i+j)],collapse =":"))
            }
      }
      
      # Now piece together and create objects for output
      
      out <- list(NA)
      
      out[[1]] <- as.matrix(rbind(v[[1]][1,], v[[1]][2,], v[[1]][3,], v[[1]][4,], v[[1]][5,], v[[1]][-(1:5),] ))
      rownames(out[[1]]) <- c("total","structure","composition", "dependence", "marginal",varnames)
      colnames(out[[1]])<- c("M1","M2","M3", "M4")
      
      out[[2]] <- as.matrix(rbind(v[[2]][1,], v[[2]][2,], v[[2]][3,], v[[2]][4,], v[[2]][5,], v[[2]][-(1:5),] ))
      rownames(out[[2]]) <- c("total","structure","composition", "dependence", "marginal",varnames)
      colnames(out[[2]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out[[3]] <- as.matrix(rbind(v[[3]][1,], v[[3]][2,], v[[3]][3,], v[[3]][4,], v[[3]][5,], v[[3]][-(1:5),] ))
      rownames(out[[3]]) <- c("total","structure","composition", "dependence", "marginal",varnames)
      colnames(out[[3]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out[[4]] <- as.matrix(v[[4]])
      rownames(out[[4]]) <- c("P - 1st Q", "p - 2nd Q", "P - 3rd Q", "P - 4th Q")
      colnames(out[[4]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out[[5]] <- as.matrix(v[[5]])
      rownames(out[[5]]) <- c("P - 1st Q", "p - 2nd Q", "P - 3rd Q", "P - 4th Q")
      colnames(out[[5]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out[[6]] <- as.matrix(v[[6]])
      rownames(out[[6]]) <- c("Trace", "Eigen", "Bartholomew", "Sym. Eigen")
      colnames(out[[6]])<- c("Empirical","Counterfactual")
      
      out[[7]] <- as.matrix(v[[4]] - v[[5]]) # composition
      rownames(out[[7]]) <- c("P - 1st Q", "p - 2nd Q", "P - 3rd Q", "P - 4th Q")
      colnames(out[[7]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out[[8]] <- as.matrix(v[[5]] - 0.25) # structure
      rownames(out[[8]]) <- c("P - 1st Q", "p - 2nd Q", "P - 3rd Q", "P - 4th Q")
      colnames(out[[8]])<- c("1st Q","2nd Q","3rd Q", "4th Q")
      
      out
      
}
