function [z,S,accept_z] = OPS_z_S(draw,priors,data)
%Output:
%   z: (TxN)
%   S: (TxN)
% Observation equation: y_t = H*z_t + e_t
% Measurement equation: z_t = M_t + F*z_tm1 + v_t
% e_t ~ N(0,R)
% v_t ~ N(0,Q)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Define the model
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

accept_z = zeros(priors.N,priors.T);

z = draw.z;
S = draw.S;

Y = data.y;
x = data.x;
K = (priors.N + priors.J) * priors.P; % dimension of state vector
K_y = size(Y,1); % diminesion of observation vector
T = size(Y,2);

% H: (K_y x K) loadings
H = zeros(K_y,K);

H(1:+ priors.J,priors.N + 1:priors.N + priors.J) = eye(priors.J);

% M: (K x T) time-varying intercept in the measurement equation
Msmall = zeros(K,1);
Msmall(1:priors.N+priors.J,1) = draw.C;
M = repmat(Msmall,1,T); % Need to add X part

% F: (K x K) AR coefficient matrix in measurement equation
F = zeros(K,K);
F(1:priors.N+priors.J,:) = draw.B;
for p = 1:priors.P-1
    x_start = (priors.N + priors.J) * p + 1;
    x_stop = (priors.N + priors.J) * (p + 1);
    y_start = (priors.N + priors.J) * (p-1) + 1;
    y_stop = (priors.N + priors.J) * p;
    F(x_start:x_stop,y_start:y_stop) = eye(priors.N + priors.J);
end

% R: (N x N) covariance matrix for e_t
R = zeros(priors.J,priors.J);

% Q: (K x K) covariance matrix for v_t
Q = zeros(K,K);
Q(1:priors.N+priors.J,1:priors.N+priors.J) = draw.omega;
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initialization
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Initial Distribution for z_0|0 based on unconditional mean
C0 =  M(:,1);
C0 = zeros(size(M,1),1);
z_0_0 = (eye(K)-F)\C0;
P_0_0_vec = (eye(size(Q(:),1)) - kron(F,F))\Q(:);
P_0_0 = reshape(P_0_0_vec,K,K);

z_tm1_tm1 = z_0_0;
P_tm1_tm1 = P_0_0;

% Initial Distribution for P_S0|0
Pr_Stm1_tm1 = 0.5 * ones(2,priors.N);

beta = [draw.C, draw.B];

for t = 1:priors.T
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Regime Candidate Draw
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Calculate transition prob
    if t < (priors.P + 1)
        X = 1;
        for p = 1:priors.P
            if t-p > 0
                X = [X, z(:,t-p)', data.y(:,t-p)'];
            else
                X = [X, z_0_0((priors.N+priors.J)*(p-t)+1:(priors.N+priors.J)*(p-t)+(priors.N+priors.J),1)'];
            end
        end
    else
        X = 1;
        for p = 1:priors.P
            X = [X, z(:,t-p)', data.y(:,t-p)'];
        end
    end
    yhat = beta*X';
    
    % HAMILTON FILTER %
    S_star = zeros(1,priors.N);
    for n = 1:priors.N
        Pr_Sntm1_tm1 = Pr_Stm1_tm1(:,n);
        Pr_Snt_0 = normcdf(0,yhat(n,1),sqrt(draw.omega(n,n)));
        Pr_Snt_1 = 1-normcdf(0,yhat(n,1),sqrt(draw.omega(n,n)));
        Pr_Snt_Stm1 = [Pr_Snt_0; Pr_Snt_1]; % Transition Probability
        Pr_Snt_tm1(1,:) = Pr_Snt_Stm1(1,:) * Pr_Sntm1_tm1;
        Pr_Snt_tm1(2,:) = Pr_Snt_Stm1(2,:) * Pr_Sntm1_tm1;
        log_f_xnt_Snt = zeros(2,1);
        % Regime 1
        if t == 1
            error = x(n,t) - draw.mu(n,1)/(1-draw.rho(n,1));
        else
            error = x(n,t) - draw.rho(n,1) * x(n,t-1) - draw.mu(n,1);
            if (n==3) && (priors.Mex_break == 1)
                error = error - priors.Dum(t,1) * draw.h;
            end
        end
        term1 = 2 * pi * draw.sigsq(n,1);
        term2 = draw.sigsq(n,1) \ -0.5 * (error^2);
        log_f_xnt_Snt(1,1) = -0.5 * log(term1) + term2;
        % Regime 2
        if t == 1
            error = x(n,t) - draw.mu(n,2)/(1-draw.rho(n,1));
        else
            error = x(n,t) - draw.rho(n,1) * x(n,t-1)  - draw.mu(n,2);
            if (n==3) && (priors.Mex_break == 1)
                error = error - priors.Dum(t,1) * draw.h;
            end
        end
        term1 = 2 * pi * draw.sigsq(n,1);
        term2 = draw.sigsq(n,1) \ -0.5 * (error^2);
        log_f_xnt_Snt(2,1) = -0.5 * log(term1) + term2;
        % Numerical Stabilization correction (Fruhwirth-Schnatter p.322)
        scale = max(log_f_xnt_Snt);
        f_xnt_Snt_tm1 = exp(log_f_xnt_Snt - scale * ones(2,1));
        f_xnt_tm1 = f_xnt_Snt_tm1' * Pr_Snt_tm1;
        %         LL = scale + log(f_xnt_tm1);  % Log-likelihood
        Pr_Snt_t = f_xnt_Snt_tm1 .* Pr_Snt_tm1 / f_xnt_tm1;
        % Draw candidate Snt
        u = unifrnd(0,1) * ones(2,1);
        PR_ = cumsum(Pr_Snt_t);
        S_star(1,n) = find(le(u,PR_),1);
        Pr_Stm1_tm1(:,n) = Pr_Snt_t;
    end
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % State Candidate Draw (Kalman)
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % Prediction
    z_t_tm1 = M(:,t) + F * z_tm1_tm1;
    P_t_tm1 = F * P_tm1_tm1 * F' + Q;
    
    % Updating
    Y_t_tm1 = H * z_t_tm1; % Prediction
    n_t_tm1 = Y(:,t) - Y_t_tm1; % Predicition Error
    f_t_tm1 = H * P_t_tm1 * H' + R; % Predicition Error Covariance
    K_t = P_t_tm1 * H' / f_t_tm1; % Kalman gain
    
    z_t_t = z_t_tm1 + K_t * n_t_tm1; % Updated state estimate
    P_t_t = P_t_tm1 - K_t * H * P_t_tm1; % Updated covariance estimate
    
    % Q is singular (with the first J x J block postive definite)
    J = priors.N;
    
    lbnd = [-Inf; 1e-10]; %lower bounds for each regime
    ubnd = [0; Inf]; %upper bounds for each regime
    lbndt = lbnd(S_star(1,:));
    ubndt = ubnd(S_star(1,:));
    
    % Rejection sampler to ensure z matches S
    accept = 0; wait = 0;
    while accept == 0
        z_star = z_t_t(1:J,1) + chol(P_t_t(1:J,1:J))' * randn(J,1); % Alternative: z_full = mvnrnd(z_t_T(1:J,1), P_t_T(1:J,1:J))';
        %         z_star = z_full(1:J,1);
        S_hat = double(z_star > 0) + 1;
        if all(S_hat == S_star(1,:)') == 1
            accept = 1;
        else
            wait = wait + 1;
        end
        if wait > 1e4
            z_star = z_t_t(1:J,1) + mvrandn(lbndt - z_t_t(1:J,1),ubndt - z_t_t(1:J,1),P_t_t(1:J,1:J),1);
            accept = 1;
        end
    end
    z_tm1_tm1 = z_t_t;
    P_tm1_tm1 = P_t_t;
    
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    % MH Acceptance Probability
    %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
    Omega = diag(draw.sigsq);
    invOmega = diag(1./draw.sigsq);
    Si = S(:,t)' + 1;
    for n = 1:priors.N
        mu_star(n,1) = draw.mu(n,S_star(1,n));
        mu_i(n,1) = draw.mu(n,Si(1,n));
    end
    if t == 1
        e_star = x(:,t) - mu_star(:,1)./(1-draw.rho);
        e_i = x(:,t) - mu_i(:,1)./(1-draw.rho);
    else
        e_star = x(:,t) - draw.rho .* x(:,t-1) - mu_star;
        e_i = x(:,t) - draw.rho .* x(:,t-1) - mu_i;
        if (n==3) && (priors.Mex_break == 1)
            e_star = e_star - priors.Dum(t,1) * draw.h;
            e_i = e_i - priors.Dum(t,1) * draw.h;
        end
    end
    f_x_Sstar = (det(2*pi*Omega))^(-0.5) * exp(-0.5 * e_star' * invOmega * e_star) ;
    f_x_Si = (det(2*pi*Omega))^(-0.5) * exp(-0.5 * e_i' * invOmega * e_i) ;
    
    z_var_star = z;
    z_var_star(:,t) = z_star;
    CB = [draw.C, draw.B];
    Y_var_star = [z_var_star; data.y]';
    Y_var = [z; data.y]';
    X_var = ones(size(Y_var,1),1);
    X_var_star = ones(size(Y_var_star,1),1);
    for p = 1:priors.P
        X_var = [X_var,lag0(Y_var,p)];
        X_var_star = [X_var_star,lag0(Y_var_star,p)];
    end
    u_i = Y_var - X_var * CB';
    u_star = Y_var_star - X_var_star * CB';
    
    iSig_u_star = draw.omega \ u_star';
    iSig_u_i = draw.omega \ u_i';
    
    term1 = (det(2*pi*draw.omega))^(-0.5);
    term2_star = -0.5 * sum(sum(u_star .* iSig_u_star'));
    term2_i = -0.5 * sum(sum(u_star .* iSig_u_i'));

    f_y_zstar = log(term1) + term2_star;
    f_y_zi = log(term1) + term2_i;
    
    f_yz_star = log(f_x_Sstar) + f_y_zstar;
    f_yz_i = log(f_x_Si) + f_y_zi;
    
    a = exp(f_yz_star - f_yz_i);
    
    u = rand(1,1);
    if a > u
        z(:,t) = z_star;
        S(:,t) = S_star(1,:)'-1;
        accept_z(:,t) = 1;
    end
end



end

