library(ggplot2)
library(ggfortify)
# to Seasonally Adjust AIIH
library(seasonal) # for seas and final


# set the working directory in the data folder
main_path <- getwd()
setwd(paste0(main_path, "/data"))
fig_path <- paste0(main_path, "/figures")


# prod --------------------------------------------------------------------

# ONS data
gva <- read.csv("GVA-ABMM.csv", header = FALSE, skip = 80)
wfj <- read.csv("WFJ-DYDC.csv", header = FALSE, skip = 69)
wrgtp <- read.csv("WRGTP-LOJU.csv", header = FALSE, skip = 69)
ecg <- read.csv("ECG-G6NQ.csv", header = FALSE, skip = 29)
ela <- read.csv("ELA-G6NT.csv", header = FALSE, skip = 29)

gva <- ts(gva[, 2], start = c(1955, 01), frequency = 4)
wfj <- ts(wfj[, 2], start = c(1959, 02), frequency = 4)
wrgtp <- ts(wrgtp[, 2], start = c(1959, 02), frequency = 4)
ecg <- ts(ecg[, 2], start = c(1999, 01), frequency = 4)
ela <- ts(ela[, 2], start = c(1999, 01), frequency = 4)

prod_data <- ts.union(gva, wfj, wrgtp, ecg, ela)
eps <- prod_data[, "wfj"] - prod_data[, "wrgtp"] - prod_data[, "ecg"] - prod_data[, "ela"]
eps <- ts(eps[!is.na(eps)], start = c(1999, 01), frequency = 4)

# PSS2001 data
data1 <- read.delim("earn1.dat", sep = " ", header = FALSE, stringsAsFactors = FALSE)
data1 <- unlist(data.frame(t(data1)))
data1 <- data1[!is.na(data1)]
data1 <- ifelse(data1 == "99999999.9", NA, data1)
data1 <- data.frame(matrix(data1, ncol = 9, byrow = TRUE))
colnames(data1) <- c("w", "union", "ur", "prod", "constant", "d7475", "d7579", "wedge", "time_trend")
data1 <- ts(data1, start = c(1969, 01), frequency = 4)

data2 <- read.delim("earn2.dat", sep = " ", header = FALSE, stringsAsFactors = FALSE)
data2 <- unlist(data.frame(t(data2)))
data2 <- data2[!is.na(data2)]
data2 <- ifelse(data2 == "99999999.9", NA, data2)
data2 <- data.frame(matrix(data2, ncol = 26, byrow = TRUE))
colnames(data2) <- c("ERPR", "UDEN", "GDPMS", "FCAS", "CGGS", "PBRENT", "YOG", "RXD", "GDPA", "CGG",
                     "TYEM", "EENIC", "WFP", "YPROM", "YMF", "EMF", "ENMF", "EMPNIC", "NIS", "OCR",
                     "ILOU", "WFEMP", "RPIX", "PYNONG", "TE", "TD")
empl <- ts(c(data2$EMF + data2$ENMF), start = c(1969, 01), frequency = 4)

# Using EPS we have limited data starting from 1999
# It seems ok to merge the PSS2001 employment data (EMF+ENMF) with the EPS series as defined in Office for Budget Responsibility 2013.
eps_extended <- ts(c(empl, rep(NA, 4), eps), start = c(1969, 01), frequency = 4)
dataset_extended <- ts.union(gva, eps_extended)

prod <- prod_data[, "gva"] / (prod_data[, "wfj"] - prod_data[, "wrgtp"] - 
                                  prod_data[, "ecg"] - prod_data[, "ela"])
prod_extended <- dataset_extended[, "gva"] / dataset_extended[, "eps_extended"]

# impute the eps_extended
eps_extended <- ts(spline(c(eps_extended), xmin = 1, xmax = length(eps_extended), n = length(eps_extended))$y,
                   start = start(eps_extended), frequency = frequency(eps_extended))

# final prod data
dataset_extended <- ts.intersect(gva, eps = eps_extended)
dataset_extended_names <- colnames(dataset_extended)
prod_extended <- dataset_extended[, "gva"] / dataset_extended[, "eps"]
dataset_extended <- ts.intersect(dataset_extended, prod_extended)
colnames(dataset_extended) <- c(dataset_extended_names, "prod")


# defl --------------------------------------------------------------------

# ONS data
pgva <- read.csv("PVGA-CGBV.csv", header = FALSE, skip = 81)
pgva <- ts(pgva[, 2], start = c(1955, 01), frequency = 4)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, pgva)
colnames(dataset_extended) <- c(dataset_extended_names, "pgva")


# rpix --------------------------------------------------------------------

# ONS data
prxmip_ons <- read.csv("PRXMIP-CHMK.csv", header = FALSE, skip = 41, nrows = 133)
rpix <- ts(data2[,"RPIX"], start = c(1969, 01), frequency = 4)
# merge with PSS2001 data for (1969-1986)
prxmip <- ts(c(rpix[which((time(rpix) >= 1969.0) & (time(rpix) <= 1986.75))], prxmip_ons[, 2]), 
             start = c(1969, 01), frequency = 4)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, prxmip)
colnames(dataset_extended) <- c(dataset_extended_names, "prxmip")


# erstr -------------------------------------------------------------------

# Employers Tax Rate
# note: erstr = 1 + TE

# ONS data
empsc <- read.csv("EMPSC-ROYK.csv", header = FALSE, skip = 80)
nis <- read.csv("NIS-GTAY.csv", header = FALSE, skip = 81)
wfp <- read.csv("WFP-DTWL.csv", header = FALSE, skip = 80)

empsc <- ts(empsc[, 2], start = c(1955, 01), frequency = 4)
# nis is 0 from 1985:Q2 and on, so it's ok to extrapolate on the next 2 quarters adding a zero value to each
nis <- ts(c(nis[93:nrow(nis),2],0,0), start = c(1969, 01), frequency = 4)
wfp <- ts(wfp[, 2], start = c(1955, 01), frequency = 4)
erstr <- ts.intersect(1 + (empsc + nis) / wfp)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, erstr, empsc, nis, wfp)
colnames(dataset_extended) <- c(dataset_extended_names, "erstr", "empsc", "nis", "wfp")

# compare Employers Tax Rate in PSS2001 and OBR2013
pss_obr <- ts(data.frame(obr2013 = dataset_extended[1:(29*4),"erstr"], 
                         pss2001 = 1 + data2[,"TE"]), 
              start = start(data1), frequency = 4)
autoplot(pss_obr, facets = FALSE, main = "Employers Tax Rate") +
    labs(fill = "variables")


# eestr -------------------------------------------------------------------

# Employees Tax Rate
# note: eestr = 1 - TD

# DBBQ was used instead of the DBBO (specified in obr2013). The SA version of DBBO.
tyem_ons <- read.csv("TYEM-DBBQ.csv", header = FALSE, skip = 41)
tyem_pss <- ts(data2[,"TYEM"], start = c(1969,01), frequency = 4)
aiih <- read.csv("AIIH-AIIH.csv", header = FALSE, skip = 82, nrows = 297)
empnic <- read.csv("EMPNIC-CEAN.csv", header = FALSE, skip = 8)

# merge TYEM PSS2001 data from (1969-1986) with obr2013 ONS data from (1987-2019)
tyem <- ts(c(tyem_pss[which((time(tyem_pss) >= 1969.0) & (time(tyem_pss) <= 1986.75))], tyem_ons[, 2]), start = c(1969, 01), frequency = 4)
aiih <- ts(aiih[, 2], start = c(1946, 01), frequency = 4)

empnic <- ts(empnic[, 2], start = c(1987), frequency = 1)
# empnic extrapolation by simple ARIMA. The ARIMA(0,1,1) found to be appropriate.
empnic_arima <- arima(empnic, order=c(0,1,1),method="ML")
empnic_pred <- predict(empnic_arima, n.ahead = 1)
empnic_lower <- c(empnic, empnic_pred$pred[1]-2*empnic_pred$se[1]) # -2SD
empnic_upper <- c(empnic, empnic_pred$pred[1]+2*empnic_pred$se[1]) # +2SD
empnic <- c(empnic, empnic_pred$pred[1])

eenic_pss <- ts(data2[,"EENIC"], start = c(1969, 01), frequency = 4)

# create a Seasonally Adjusted AIIH
aiih_sa <- seas(aiih)
aiih_sa <- final(aiih_sa)

# for EENIC, use PSS2001 data for (1969-1986) and OBR formula for (1987-2017)
# using seasonally adjusted quarterly aiih and quarterized empnic

empnic_q = ts(rep(empnic/4, each = 4), start = c(1987, 01), frequency = 4)
empnic_lower_q = ts(rep(empnic_lower/4, each = 4), start = c(1987, 01), frequency = 4)
empnic_upper_q = ts(rep(empnic_upper/4, each = 4), start = c(1987, 01), frequency = 4)
end_time_empnic_q <- as.numeric(paste0(end(empnic_q)[1], ifelse(end(empnic_q)[2] == 1, ".0",
                                                                ifelse(end(empnic_q)[2] == 2, ".25",
                                                                       ifelse(end(empnic_q)[2] == 3, ".50", ".75"))), collapse = ""))

eenic <- c(eenic_pss[which((time(eenic_pss) >= 1969.0) & (time(eenic_pss)) <= 1986.75)], 
           aiih_sa[which((time(aiih_sa) >= 1987.0) & (time(aiih_sa) <= end_time_empnic_q))] - 
               empnic_q[which((time(empnic_q) >= 1987.0) & (time(empnic_q) <= end_time_empnic_q))])
eenic_lower <- c(eenic_pss[which((time(eenic_pss) >= 1969.0) & (time(eenic_pss)) <= 1986.75)], 
                 aiih_sa[which((time(aiih_sa) >= 1987.0) & (time(aiih_sa) <= end_time_empnic_q))] - 
                     empnic_lower_q[which((time(empnic_lower_q) >= 1987.0) & (time(empnic_lower_q) <= end_time_empnic_q))])
eenic_upper <- c(eenic_pss[which((time(eenic_pss) >= 1969.0) & (time(eenic_pss)) <= 1986.75)], 
                 aiih_sa[which((time(aiih_sa) >= 1987.0) & (time(aiih_sa) <= end_time_empnic_q))] - 
                     empnic_upper_q[which((time(empnic_upper_q) >= 1987.0) & (time(empnic_upper_q) <= end_time_empnic_q))])

eenic <- ts(eenic, start = c(1969, 01), frequency = 4)
eenic_lower <- ts(eenic_lower, start = c(1969, 01), frequency = 4)
eenic_upper <- ts(eenic_upper, start = c(1969, 01), frequency = 4)

eestr <- ts.intersect(1 - (tyem + eenic*0.914) / wfp)
eestr_lower <- ts.intersect(1 - (tyem + eenic_lower*0.914) / wfp)
eestr_upper <- ts.intersect(1 - (tyem + eenic_upper*0.914) / wfp)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, eestr, eestr_lower, eestr_upper, tyem, eenic, eenic_lower, eenic_upper)
colnames(dataset_extended) <- c(dataset_extended_names, "eestr", "eestr_lower", "eestr_upper", "tyem", "eenic", "eenic_lower", "eenic_upper")


# wedge -------------------------------------------------------------------

## incorporating in wedge the:
# Employers Tax Rate (erstr)
# Employees Tax Rate (eestr)
# PRXMIP
# PGVA
## following the PSS2001 formula for wedge, in contrast with OBR2013

# log(erstr) + log(eestr) - log(prxmip/pgva)
# =
# log(erstr*eestr*pgva/prxmip)
wedge <- dataset_extended[,"erstr"] * 
    dataset_extended[,"eestr"] * 
    dataset_extended[,"pgva"] / 
    dataset_extended[,"prxmip"]
wedge_lower <- dataset_extended[,"erstr"] * 
    dataset_extended[,"eestr_lower"] * 
    dataset_extended[,"pgva"] / 
    dataset_extended[,"prxmip"]
wedge_upper <- dataset_extended[,"erstr"] * 
    dataset_extended[,"eestr_upper"] * 
    dataset_extended[,"pgva"] / 
    dataset_extended[,"prxmip"]

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, wedge, wedge_lower, wedge_upper)
colnames(dataset_extended) <- c(dataset_extended_names, "wedge", "wedge_lower", "wedge_upper")

autoplot(dataset_extended[,c("wedge_lower", "wedge", "wedge_upper")], facets = FALSE)

# compare wedge components PSS2001 vs wide reprication
erstr_TE <- ts.intersect(pss=ts(1 + data2[,'TE'], start = c(1969,01), frequency = 4), dataset_extended[,"erstr"])
autoplot(erstr_TE, facets = FALSE, main = "employers tax rate vs 1+TE")

eestr_TD <- ts.intersect(pss=ts(1 - data2[,'TD'], start = c(1969,01), frequency = 4), dataset_extended[,"eestr"])
autoplot(eestr_TD, facets = FALSE, main = "employees tax rate vs 1+TD")

prxmip_RPIX <- ts.intersect(pss=ts(data2[,'RPIX'], start = c(1969,01), frequency = 4), dataset_extended[,"prxmip"])
autoplot(prxmip_RPIX, facets = FALSE, main = "PRXMIP vs PRIX")

pgva_PYNONG <- ts.intersect(pss=ts(data2[,'PYNONG'], start = c(1969,01), frequency = 4), dataset_extended[,"pgva"])
autoplot(pgva_PYNONG, facets = FALSE, main = "PGVA vs PYNONG")

# the change in scale is due to the different deflator (PGVA instead of PYNONG)
wedge_wedge <- ts.intersect(pss = data1[,'wedge'], log(dataset_extended[,"wedge"]))
autoplot(wedge_wedge, facets = FALSE, main = "wedge")


wedge_wide <- ts.intersect(PYNONG=log(ts(dataset_extended[,"erstr"][1:116] * 
                                             dataset_extended[,"eestr"][1:116] *
                                             data2[,'PYNONG'] /
                                             dataset_extended[,"prxmip"][1:116], start = c(1969,01), frequency = 4)),
                           PGVA=log(wedge))

wedge_narrow_wide_comparison <-  ts.intersect(wedge_narrow = data1[,'wedge'], wedge_wide)
autoplot(wedge_narrow_wide_comparison, facets = FALSE)
data_plot <- data.frame(Time = rep(time(wedge_narrow_wide_comparison), 3),
                        values = c(wedge_narrow_wide_comparison[,"wedge_narrow"], wedge_narrow_wide_comparison[,"wedge_wide.PYNONG"],
                                   wedge_narrow_wide_comparison[,"wedge_wide.PGVA"]),
                        variable = c(rep("wedge_narrow", nrow(wedge_narrow_wide_comparison)), rep("wedge_wide.PYNONG", nrow(wedge_narrow_wide_comparison)),
                                     rep("wedge_wide.PGVA", nrow(wedge_narrow_wide_comparison))))
#pdf(paste0(fig_path, "/wedge-deflators.pdf"), width=5.6, height=4)
ggplot(data = data_plot, aes(x = Time, y = values, group = variable)) +
   geom_line(aes(linetype = variable)) +
   labs(y = 'Wedge', linetype = "Variables") +
   theme(legend.position="bottom") +
   scale_x_continuous(breaks = seq(1972, 1997, by = 2))
# dev.off()

deflators <- ts.intersect(PYNONG = ts(data2[,'PYNONG'], start = c(1969,01), frequency = 4),
                          PGVA = dataset_extended[,"pgva"])
autoplot(deflators, facets = FALSE)
data_plot <- data.frame(Time = rep(time(deflators), 2),
                        values = c(deflators[,"PGVA"], deflators[,"PYNONG"]),
                        variable = c(rep("PGVA", nrow(deflators)), rep("PYNONG", nrow(deflators))))
#pdf(paste0(fig_path, "/deflators-lvl.pdf"), width=5.6, height=4)
ggplot(data = data_plot, aes(x = Time, y = values, group = variable)) +
   geom_line(aes(linetype = variable)) +
   labs(y = 'Deflators', linetype = "Variables") +
   theme(legend.position="bottom") +
   scale_x_continuous(breaks = seq(1972, 1997, by = 2))
# dev.off()

deflators.log <- ts.intersect(log.PYNONG = log(ts(data2[,'PYNONG'], start = c(1969,01), frequency = 4)),
                              log.PGVA = log(dataset_extended[,"pgva"]))
autoplot(deflators.log, facets = FALSE)
data_plot <- data.frame(Time = rep(time(deflators.log), 2),
                        values = c(deflators.log[,"log.PGVA"], deflators.log[,"log.PYNONG"]),
                        variable = c(rep("log.PGVA", nrow(deflators.log)), rep("log.PYNONG", nrow(deflators.log))))
#pdf(paste0(fig_path, "/deflators-log.pdf"), width=5.6, height=4)
ggplot(data = data_plot, aes(x = Time, y = values, group = variable)) +
   geom_line(aes(linetype = variable)) +
   labs(y = 'Log Scale', linetype = "Variables") +
   theme(legend.position="bottom") +
   scale_x_continuous(breaks = seq(1972, 1997, by = 2))
# dev.off()


# ur ----------------------------------------------------------------------

# ONS data
# starts in 1971 instead of 1969
lfsur <- read.csv("LFSUR-MGSX.csv", header = FALSE, skip = 57, nrows = 197)
lfsur <- ts(lfsur[, 2], start = c(1971, 01), frequency = 4)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, lfsur)
colnames(dataset_extended) <- c(dataset_extended_names, "ur")

# ur in pss2001 is probably not correct between 1969Q1 and around 1974
# especially between 1969Q1 and 1972Q2
# so we decide not to replicate the missing data in ur for the wide sense replication (1970Q1 - 1970Q4)
ur_wide_narrow <- ts.intersect(log(dataset_extended[,'ur']), data1[,"ur"])
colnames(ur_wide_narrow) <- c('ur_wide', "ur_narrow")
autoplot(ur_wide_narrow)


# union -------------------------------------------------------------------

# PSS2001 data or Department of Employment Gazette or OBR
uden <- ts(c(data2[, "UDEN"], rep(data2[nrow(data2), "UDEN"], 88)), start = c(1969, 01), frequency = 4)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, uden)
colnames(dataset_extended) <- c(dataset_extended_names, "union")


# dummies -----------------------------------------------------------------

d7475 <- ts(c(data1[, "d7475"], rep(data1[nrow(data1), "d7475"], 88)), start = c(1969, 01), frequency = 4)
d7579 <- ts(c(data1[, "d7579"], rep(data1[nrow(data1), "d7579"], 88)), start = c(1969, 01), frequency = 4)

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, d7475, d7579)
colnames(dataset_extended) <- c(dataset_extended_names, "d7475", "d7579")


# w -----------------------------------------------------------------------

# ONS data
erpr <- ts.intersect((wfp / dataset_extended[,"eps"])*1000)
w <- erpr / dataset_extended[,"pgva"]
w_pgva <- w
w_pynong <- ts.intersect(erpr / ts(data2[,"PYNONG"], start = c(1969, 01), frequency = 4))

wpgva_wpynong_wpss <- ts.intersect(w_pgva = log(w_pgva), 
                                   w_pynong = log(w_pynong), 
                                   w_pss = data1[,"w"],
                                   erpr_pynong = ts(log(data2[,"ERPR"] / data2[,"PYNONG"]), start = c(1969, 01), frequency = 4))

autoplot(wpgva_wpynong_wpss[,-3])
autoplot(wpgva_wpynong_wpss[,-c(1,2)])
autoplot(wpgva_wpynong_wpss[,-c(2,4)])

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, w)
colnames(dataset_extended) <- c(dataset_extended_names, "w")


# w comparison
w_comparison <- ts.intersect(w_wide_PGVA = log(w_pgva), 
                             w_wide_PYNONG = log(w_pynong), 
                             w_pss = data1[,"w"],
                             w_narrow = ts(log(data2[,"ERPR"] / data2[,"PYNONG"]), star = c(1969, 01), frequency = 4))

# the flactuation between the narrow and the wide wage series looks similar. The deflator changes the scale.
autoplot(w_comparison[,-3])
autoplot(w_comparison[,-3], facet = FALSE)
data_plot <- data.frame(Time = rep(time(w_comparison[,-3]), 3),
                        values = c(w_comparison[,-3][,"w_wide_PGVA"], w_comparison[,-3][,"w_wide_PYNONG"],
                                   w_comparison[,-3][,"w_narrow"]),
                        variable = c(rep("w_wide_PGVA", nrow(w_comparison[,-3])), rep("w_wide_PYNONG", nrow(w_comparison[,-3])),
                                     rep("w_narrow", nrow(w_comparison[,-3]))))
#pdf(paste0(fig_path, "/w-deflators.pdf"), width=5.6, height=4)
ggplot(data = data_plot, aes(x = Time, y = values, group = variable)) +
   geom_line(aes(linetype = variable)) +
   labs(y = 'Real wages (w)', linetype = "Variables") +
   theme(legend.position="bottom") +
   scale_x_continuous(breaks = seq(1972, 1997, by = 2))
# dev.off()

# the pre-calculated wage series in PSS looks smoother and much different in scale and trend.
# We are not sure what is the purpose of this series in the dataset, but as turned out replicating the paper example,
# they actually construct and use the wage series as: ERPR/PYNONG
autoplot(w_comparison[,-c(1,2)])
autoplot(w_comparison[,-c(1,2)], facet = FALSE)


# UnionR ------------------------------------------------------------------

UnionMembers <- read.csv("TradeUnionMembers.csv", header = FALSE, skip = 4, sep = ";", nrows = 128)
UnionMembers <- cbind(rep(UnionMembers[,1], each=4), rep(UnionMembers[,2], each=4))
UnionMembers <- ts(UnionMembers[, 2], start = c(1892, 01), frequency = 4)
UnionMembers <- window(UnionMembers, start = c(1971, 01))
UnionR <- (UnionMembers*1000)/dataset_extended[,"eps"]

dataset_extended_names <- colnames(dataset_extended)
dataset_extended <- ts.intersect(dataset_extended, UnionR)
colnames(dataset_extended) <- c(dataset_extended_names, "UnionR")


# final dataset -----------------------------------------------------------

data_wide <- dataset_extended[, c("w", "prod", "ur")]
# choose between: "wedge", "wedge_lower", "wedge_upper"
# (the "wedge_lower" and "wedge_upper" for sensitivity analysis purposes only)
data_wide <- ts.intersect(data_wide, dataset_extended[, "wedge"])
data_wide <- ts.intersect(data_wide, dataset_extended[, c("union", "d7475", "d7579", "UnionR")])
colnames(data_wide) <- c("w", "Prod", "UR", "Wedge", "Union", "D7475", "D7579", "UnionR")
data_wide[, c("w", "Prod", "UR", "Wedge", "Union", "UnionR")] <- log(data_wide[, c("w", "Prod", "UR", "Wedge", "Union", "UnionR")])

# write the data in csv
data_wide_names <- colnames(data_wide)
data_clean_wide <- ts.intersect(time(data_wide), data_wide)
colnames(data_clean_wide) <- c("time", data_wide_names)
write.csv(data_clean_wide, file = "data_clean_wide.csv", row.names = FALSE)

data_wide_97Q4 <- ts(data_wide[1:length(seq(1971, 1997.75, by=0.25)),], start = start(data_wide), frequency = 4)

# using PYNONG as deflator instead of PGVA
data_wide_97Q4_PYNONG <- ts.intersect(w_pynong, dataset_extended[, c("prod", "ur")], exp(wedge_wide[,"PYNONG"]),
                                      dataset_extended[, c("union", "d7475", "d7579")])
colnames(data_wide_97Q4_PYNONG) <- c("w", "Prod", "UR", "Wedge", "Union", "D7475", "D7579")
data_wide_97Q4_PYNONG[, c("w", "Prod", "UR", "Wedge", "Union")] <- log(data_wide_97Q4_PYNONG[, c("w", "Prod", "UR", "Wedge", "Union")])

rm(data1, data2, dataset_extended, eestr_TD, empnic_arima,
   empnic_pred, erstr_TE, pgva_PYNONG, prod_data, prxmip_ons,
   prxmip_RPIX, pss_obr, tyem_ons, ur_wide_narrow, wedge_wedge,
   wpgva_wpynong_wpss, aiih, aiih_sa, d7475, d7579, data_wide_names,
   dataset_extended_names, ecg, eenic, eenic_lower, eenic_pss, eenic_upper,
   eestr, eestr_lower, eestr_upper, ela, empl, empnic, empnic_lower,
   empnic_lower_q, empnic_q, empnic_upper, empnic_upper_q,
   empsc, end_time_empnic_q, eps, eps_extended, erpr, erstr,
   gva, lfsur, nis, pgva, prod, prod_extended, prxmip, rpix,
   tyem, tyem_pss, uden, w, w_pgva, w_pynong, wedge, wedge_lower,
   wedge_upper, wfj, wfp, wrgtp, data_clean_wide, deflators, deflators.log,
   w_comparison, wedge_narrow_wide_comparison, wedge_wide, UnionR,
   UnionMembers, data_plot)

setwd(main_path)
