### R code from vignette source 'OnlineAppendixToJAE_MS10278.snw'

###################################################
### code chunk number 1: init
###################################################
library(plm)
library(lmtest)
library(xtable)
library(reshape)


###################################################
### code chunk number 2: data
###################################################
data(RDSpillovers, package="pder")
# alt.: read in Eberhardt's Stata dataset
#library(foreign)
#ehs0<-read.dta("./data/EberhardtHelmersStrauss_RDspillover.dta")
ehs <- pdata.frame(RDSpillovers, index=c("id","year"))


###################################################
### code chunk number 3: fisherdef
###################################################
## def. Fisher test function for any residual test
source("./functions/fishertest.R")


###################################################
### code chunk number 4: artests
###################################################
## joint panel AR(l) test on residuals
source("./functions/partest.R")


###################################################
### code chunk number 5: moddef
###################################################
fm <- lny ~ lnl + lnk + lnrd


###################################################
### code chunk number 6: t5.pols
###################################################

## pooled model with time fixed effects (done as within/time)
tfemod<-plm(fm, ehs, model="within", effect="time")
## alt.: pols<-plm(update(fm, .~.+as.factor(year)), data=ehs, model="pooling")
#summary(tfemod)


###################################################
### code chunk number 7: t5.2fe
###################################################
## 2way Fe
fe2mod<-plm(fm, ehs, model="within", effect="twoway")
#coeftest(fe2mod)


###################################################
### code chunk number 8: t5.fd
###################################################

## FD
fdmod0<-plm(fm, ehs, model="fd")
#coeftest(fdmod0)

## no, EHS do  "first difference with time fixed effects" model
fdmod<-plm(update(fm, .~.+as.factor(year)-1), ehs, model="fd")
#coeftest(fdmod)
## <-- problem here is that they drop time1, so time2 gets dropped as well!
## 'reg d.lny d.lnl d.lnk d.lnrd time2-time26, robust nocons' gives very same
## results as here

## their version:
ehs0 <- RDSpillovers
ehs2 <- ehs0[ehs0$year>1980,]
fdmod.wrong <- plm(update(fm, .~.+as.factor(year)-1), ehs2,
                   index=c("id","year"), model="fd")
#coeftest(fdmod)

## alt. FD (see below, CD test)
fdfm <- diff(lny) ~ diff(lnl) + diff(lnk) + diff(lnrd)
fdamod0 <- plm(fdfm, data=ehs, model="p")
fdamod <- plm(update(fdfm, . ~ . + as.factor(year) - 1),
              data=ehs, model="p")
fdamod.wrong <- plm(update(fdfm, . ~ . + as.factor(year) - 1), ehs2,
                    index=c("id","year"), model="p")


###################################################
### code chunk number 9: t5.ccep
###################################################

## CCEP
ccepmod<-pcce(fm, ehs, model="p", trend=FALSE)
#summary(ccepmod)


###################################################
### code chunk number 10: t5.augccep
###################################################

## CCEP with year FEs
#ccepmodt<-pcce(update(fm, .~.+as.factor(year)), ehs, model="p", trend=FALSE)
# summary(ccepmod)
# fails for insufficient number of time periods, singular matrix

## Try group-demeaning the data in advance
dati<-data.frame(ehs)
dati<-dati[order(dati$id, dati$year),]

gfemod <- plm(fm, dati, index=c("id","year"), effect="time")
myx<-model.matrix(gfemod)
myy<-pmodel.response(gfemod)
demdati <- cbind(dati[,c("id","year")], myx, myy)
dimnames(demdati)[[2]][6]<-"lny"
gd.ehs <- pdata.frame(demdati)

ccepmodt<-pcce(update(fm,.~.-1), gd.ehs, model="p", trend=FALSE)
#summary(ccepmodt)
## not very same results, but enormous collinearity here, and groups too
## short for estimation etc.



###################################################
### code chunk number 11: t5.ccept
###################################################
## CCEP with time dummies, through generalized inverse
library(MASS) # for ginv()
source("./functions/pcce2.R") # ad-hoc modification of pcce()
ccepmodt <- pcce2(update(fm, .~.+as.factor(year)-1) , ehs,
                  model="p", trend=FALSE)

## Not same results as EHS, but who knows what happens here!?
## The point is that adding time dummies to a generalized FE model is
## unwarranted.


###################################################
### code chunk number 12: t5.badccep
###################################################

## Comparison to EHS' procedure: interacting each cross-sectional average
## with individual dummies

## make full formula
fullfm<-lny~lnl+lnk+lnrd+as.factor(id)+
  (Between(lny, effect="time")+Between(lnl, effect="time")+
   Between(lnk, effect="time")+Between(lnrd, effect="time")):
  as.factor(id)-1

## extract only relevant coefs (model takes some time to run)
fullccep <- plm(fullfm, ehs, model="pooling")
summ.fullccep <- coef(summary(plm(fullfm, ehs, model="pooling")))[1:3,]
#tstats.fullccep <- coef(fullccep)[1:3]/sqrt(diag(vcov(fullccep)[1:3,1:3]))

## not exactly as in EHS, matches CCEP exactly instead!
## SEs are different


###################################################
### code chunk number 13: t5.badccept
###################################################

## Comparison to EHS' procedure: interacting each cross-sectional average
## with individual dummies

## make full formula
fullfmt<-lny~lnl+lnk+lnrd+as.factor(id)+as.factor(year)+
  (Between(lny, effect="time")+Between(lnl, effect="time")+
   Between(lnk, effect="time")+Between(lnrd, effect="time")):
  as.factor(id)-1

## extract only relevant coefs (model takes some time to run)
fullccept <- plm(fullfmt, ehs, model="pooling")
tstats.fullccept <- coef(fullccept)[1:3]/sqrt(diag(vcov(fullccept)[1:3,1:3]))
## exactly as in EHS, SEs are different



###################################################
### code chunk number 14: ccepses
###################################################
ccepmoda <- pcce(fm, ehs, model="p")

## calc vcovs

## nonparametric vcov ex Pesaran 2006, Th.3
ccep.vc <- vcov(ccepmoda)
## clustered standard errors sub homogeneity
ccep.vchc <- vcovHC(ccepmoda)
## clustered SEs under homogeneity ex Th. 4
ccep.vcnw <- vcovNW(ccepmoda)
## SCC version, allowing for residual XS dependence
ccep.vcscc <- vcovSCC(ccepmoda)

## pairs- cluster-bootstrapped SEs based on the distribution of betas
## (bootstrap-se in Cameron, Gelbach and Miller RES 2008, App.B.2)

## vcovBoot
## make wrappers vcovCB, vcovWB for type="pairs" or "wild"
source("./functions/vcovBoot.R")
set.seed(123)
ccep.vcwb <- vcovBoot(ccepmoda, type="wild")
ccep.vcpb <- vcovBoot(ccepmoda, type="pairs")

## the wild bootstrap correctly reproduces clustered errors when applied to the dummy-based CCEP, so that the following are close:
# fullccep.vcwb <- coeftest(fullccep, vcovBoot) # takes 10'
# fullccep.vchc <- coeftest(fullccep, vcovHC)[1:3, ]

## but then, it is less time consuming to do clustering directly

## The problem remains: how appropriate is this w.r.t. the native
## nonparametric estimator from CCEP? SEs are almost exactly double



###################################################
### code chunk number 15: t5.cdtest
###################################################

## CD test

cd.tfe <- pcdtest(tfemod)
cd.fe2 <- pcdtest(fe2mod)
## these ok

#pcdtest(fdmod) # breaks

## alt. FD model (FDs in formula):
cd.fd <- pcdtest(fdamod)
## NB original drops one period too much!
## perfect if compared with the adjusted version:
#****************     FD-OLS w/ year FE     *****************
#reg d.lny d.lnl d.lnk d.lnrd time2-time26, robust nocons
#predict eFD2 if e(sample), res
#xtcd eFD2

## tests as in paper, on models with dummy interactions
cd.ccep <- pcdtest(fullccep)
cd.ccept <- pcdtest(fullccept) # both perfect

## tests on correct models
cd.ccep.corr <- pcdtest(ccepmod) # exact equivalence
cd.ccept.corr <- pcdtest(ccepmodt) # bigger difference, but see above


###################################################
### code chunk number 16: t5.cips
###################################################

## CIPS (PCADF) test
cips.tfe <- cipstest(resid(tfemod), lags=2, model="cmg", type="drift")
cips.fe2 <- cipstest(resid(fe2mod), lags=2, model="cmg", type="drift")
cips.fd <- cipstest(resid(fdamod), lags=2, model="cmg", type="drift")
cips.ccep <- cipstest(resid(ccepmod), lags=1, model="cmg", type="drift")
cips.ccept <- cipstest(resid(ccepmodt), lags=1, model="cmg", type="drift")


###################################################
### code chunk number 17: t5.crs
###################################################

## CRS (constant rate of substitution) hypothesis:
crs<-"lnl+lnk+lnrd=1"

library(car)

crs.tfe.corr <- lht(tfemod, crs, vcov=vcovHC) # corrected version
crs.tfe <- lht(tfemod, crs, vcov=function(x) vcovHC(x, method="white1"))
             # ok with White, does not reject with Arellano (clustering)
crs.fe2.corr <- lht(fe2mod, crs, vcov=vcovHC)
crs.fe2 <- lht(fe2mod, crs, vcov=function(x) vcovHC(x, method="white1"))
                                        # tab5 is with Arellano covariance
crs.fd.corr <- lht(fdamod,"diff(lnl)+diff(lnk)+diff(lnrd)=1", vcov=vcovHC)
crs.fd <- lht(fdamod, "diff(lnl)+diff(lnk)+diff(lnrd)=1",
              vcov=function(x) vcovHC(x, method="white1"))
                                        # "fd" model breaks here
crs.ccep.corr <- lht(ccepmod, crs)
crs.ccep <- lht(fullccep, crs)
crs.ccept.corr <- lht(ccepmodt, crs)
crs.ccept <- lht(fullccept, crs)
## not so similar... must depend from (robust) vcov used.


###################################################
### code chunk number 18: t5.rmse
###################################################

rmse.tfe <- sqrt(mean(resid(tfemod)^2))
rmse.fe2 <- sqrt(mean(resid(fe2mod)^2))
rmse.fd <- sqrt(mean(resid(fdamod)^2))
rmse.ccep <- sqrt(mean(resid(ccepmod)^2))
rmse.ccept <- sqrt(mean(resid(ccepmodt)^2)) # very bad! ??



###################################################
### code chunk number 19: t5.mtests
###################################################
source("./functions/mtest3.R")

m1.tfe <- mtest3(tfemod)
m2.tfe <- mtest3(tfemod, order=2)
m1.fe2 <- mtest3(fe2mod)
m2.fe2 <- mtest3(fe2mod, order=2)
m1.fd <- mtest3(fdmod0) # fdmod and fdmoda bomb because of the time dummies
m2.fd <- mtest3(fdmod0, order=2) # idem
m1.ccep <- mtest3(fullccep)
m2.ccep <- mtest3(fullccep, order=2)
m1.ccept <- NA # fails because of numerical problems
m2.ccept <- NA # idem



###################################################
### code chunk number 20: t5.ar
###################################################
## serial correlation tests as wald tests

ar1.tfe <- partest(tfemod, l=1)
ar1.fe2 <- partest(fe2mod, l=1)
ar1.fd <- partest(fdamod, l=1)
ar1.ccep <- partest(ccepmod, l=1)
#ar1.ccept <- partest(ccepmodt, l=1)



###################################################
### code chunk number 21: tab5
###################################################

tab5 <- matrix(ncol=5, nrow=12)
dimnames(tab5) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS", "AB-AR(1)", "AB-AR(2)", "CD", "CIPS",
                         "RMSE"),
                       c("POLS", "2FE", "FD", "CCEP", "CCEPt"))

## populate coefs
tab5[c(1,3,5), 1] <- coef(tfemod)
tab5[c(1,3,5), 2] <- coef(fe2mod)
tab5[c(1,3,5), 3] <- coef(fdmod.wrong)[1:3] #coef(fdamod)[1:3]
tab5[c(1,3,5), 4] <- summ.fullccep[,1]  #coef(ccepmod)
tab5[c(1,3,5), 5] <- coef(fullccept)[1:3] #coef(ccepmodt)

## SEs
myvcov <- function(x) vcovHC(x, method="white1")
tab5[c(2,4,6), 1] <- coeftest(tfemod, vcov=myvcov)[,3]
tab5[c(2,4,6), 2] <- coeftest(fe2mod, vcov=myvcov)[,3]
tab5[c(2,4,6), 3] <- coeftest(fdamod, vcov=myvcov)[1:3, 3]
tab5[c(2,4,6), 4] <- summ.fullccep[,3] #coeftest(ccepmod)[,3]
tab5[c(2,4,6), 5] <- tstats.fullccept #coeftest(ccepmodt)[,3]

## CRS test
tab5["CRS", ] <- c(crs.tfe[2,4], crs.fe2[2,4], crs.fd[2,4],
                   lht(fullccep, crs)[2,4], lht(fullccept, crs)[2,4])

## m test
tab5["AB-AR(1)", ] <- c(m1.tfe$p.value, m1.fe2$p.value, m1.fd$p.value,
                        m1.ccep$p.value, NA)
tab5["AB-AR(2)", ] <- c(m2.tfe$p.value, m2.fe2$p.value, m2.fd$p.value,
                        m2.ccep$p.value, NA)

## CD test
tab5["CD", ] <- c(cd.tfe$p.value, cd.fe2$p.value, cd.fd$p.value,
                  pcdtest(fullccep)$p.value,
                  pcdtest(fullccept)$p.value)

## CIPS test
tab5["CIPS", ] <- c(cips.tfe$p.value, cips.fe2$p.value, cips.fd$p.value,
                    cipstest(resid(fullccep),
                             lags=1, model="cmg", type="drift")$p.value,
                    cipstest(resid(fullccept),
                             lags=1, model="cmg", type="drift")$p.value)
#tab5["CIPS", ] <- ifelse(tab5["CIPS",]>0.05, "I(1)", "I(0)")

## RMSE
tab5["RMSE", ] <- c(rmse.tfe, rmse.fe2, rmse.fd,
                    sqrt(mean(resid(fullccep)^2)),
                    sqrt(mean(resid(fullccept)^2)))

xtable(tab5, digits=3,
       caption="Static homogeneous models (Table 5 in EHS).
   POLS: pooled OLS with time FEs; 2FE: two-way fixed effects;
   FD: first differences with time FEs; CCEP: pooled common
   correlated effects.
   CRS: Wald test for $H_0$ of constant returns to scale;
   AB-AR(1) and (2): Arellano-Bond (1991) $m_1$ and $m_2$ tests for residual
   serial correlation; CIPS: Pesaran (2007) unit root test:
   $H_A$ is stationarity, upper limits are reported for the $p$-value.
   RMSE: root mean square error.")


###################################################
### code chunk number 22: t5.war
###################################################
## Alternate AR tests: Wooldridge procedure
source("./functions/wartest.R")

## apply to models accounting for induced correlation from transform:
## H0 here is that the original errors are uncorr.

## vcovHC version (clustering, no XS corr.)
war1.tfe <- wartest(tfemod, 1)
war1.fe2 <- wartest(fe2mod, 1, hyp="fe")
war1.fd <- wartest(fdamod, 1, hyp="fd")
war1.ccep <- wartest(ccepmod, 1)
#war1.ccept <- wartest(ccepmodt, 1)

war2.tfe <- wartest(tfemod, 2)
war2.fe2 <- wartest(fe2mod, 2, hyp="fe")
war2.fd <- wartest(fdamod, 2, hyp="ols") # fd does not induce MA at 2 order
war2.ccep <- wartest(ccepmod, 2)
#war2.ccept <- wartest(ccepmodt, 2)


## vcovSCC version (accounting for XS corr. and AR)
war1.tfe.scc <- wartest(tfemod, 1, vcov=vcovSCC)
war1.fe2.scc <- wartest(fe2mod, 1, hyp="fe", vcov=vcovSCC)
war1.fd.scc <- wartest(fdamod, 1, hyp="fd", vcov=vcovSCC)
war1.ccep.scc <- wartest(ccepmod, 1, vcov=vcovSCC)
#war1.ccept <- wartest(ccepmodt, 1, vcov=vcovSCC)

war2.tfe.scc <- wartest(tfemod, 2, vcov=vcovSCC)
war2.fe2.scc <- wartest(fe2mod, 2, hyp="fe", vcov=vcovSCC)
war2.fd.scc <- wartest(fdamod, 2, hyp="ols", vcov=vcovSCC) # as above
war2.ccep.scc <- wartest(ccepmod, 2, vcov=vcovSCC)
#war2.ccept <- wartest(ccepmodt, 2, vcov=vcovSCC)



###################################################
### code chunk number 23: t5.frees
###################################################
source("./functions/pcdtest3.R")

## Frees rank-based test
frees.tfe <- pcdtest3(tfemod, test="frees", exact=FALSE)
frees.fe2 <- pcdtest3(fe2mod, test="frees", exact=FALSE)
frees.fd <- pcdtest3(fdamod, test="frees", exact=FALSE)
frees.ccep <- pcdtest3(ccepmod, test="frees", exact=FALSE)
frees.ccept <- pcdtest3(ccepmodt, test="frees", exact=FALSE)

## relative and absolute rho:
rho.tfe <- pcdtest(tfemod, test="rho")
rho.fe2 <- pcdtest(fe2mod, test="rho")
rho.fd <- pcdtest(fdamod, test="rho")
rho.ccep <- pcdtest(ccepmod, test="rho")
rho.ccept <- pcdtest(ccepmodt, test="rho")

absrho.tfe <- pcdtest(tfemod, test="absrho")
absrho.fe2 <- pcdtest(fe2mod, test="absrho")
absrho.fd <- pcdtest(fdamod, test="absrho")
absrho.ccep <- pcdtest(ccepmod, test="absrho")
absrho.ccept <- pcdtest(ccepmodt, test="absrho")



###################################################
### code chunk number 24: tab5corr
###################################################

tab5corr <- matrix(ncol=5, nrow=13)
dimnames(tab5corr) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                             "CRS", "W-AR(1)", "W-AR(2)", "CD", "Frees",
                             "CIPS", "RMSE"),
                       c("POLS", "2FE", "FD", "CCEP", "CCEPt"))

## populate coefs
tab5corr[c(1,3,5), 1] <- coef(tfemod)
tab5corr[c(1,3,5), 2] <- coef(fe2mod)
tab5corr[c(1,3,5), 3] <- coef(fdamod)[1:3]
tab5corr[c(1,3,5), 4] <- coef(ccepmod)
tab5corr[c(1,3,5), 5] <- coef(ccepmodt)[1:3]

## SEs
myvcov <- function(x) vcovHC(x, method="arellano")
tab5corr[c(2,4,6), 1] <- coeftest(tfemod, vcov=myvcov)[,3]
tab5corr[c(2,4,6), 2] <- coeftest(fe2mod, vcov=myvcov)[,3]
tab5corr[c(2,4,6), 3] <- coeftest(fdamod, vcov=myvcov)[1:3, 3]
tab5corr[c(2,4,6), 4] <- coeftest(ccepmod)[,3]
tab5corr[c(2,4,6), 5] <- coeftest(ccepmodt)[1:3,3]

## CRS tests are different from tab5 because of the robust vcovs used

## CRS test
tab5corr["CRS", ] <- c(crs.tfe.corr[2,4], crs.fe2.corr[2,4],
                       crs.fd.corr[2,4],  crs.ccep.corr[2,4],
                       crs.ccept.corr[2,4])

## m test
tab5corr["W-AR(1)", ] <- c(war1.tfe[2, 4], war1.fe2[2, 4],
                           war1.fd[2, 4], war1.ccep[2, 4], NA)
tab5corr["W-AR(2)", ] <- c(war2.tfe[2, 4], war2.fe2[2, 4],
                           war2.fd[2, 4], war2.ccep[2, 4], NA)

## Residuals are ok but for CCEPt, so CD, AR, CIPs and RMSE do not change
## for the first 4 models; for CCEPt resid(ccepmodt) are taken instead of
## resid(fullccept)

## CD test
## (residuals are identical but for ccept)
tab5corr["CD", ] <- c(cd.tfe$p.value, cd.fe2$p.value, cd.fd$p.value,
                      cd.ccep$p.value, pcdtest(ccepmodt)$p.value)

## substitute with Frees' test
tab5corr["Frees", ] <- c(frees.tfe$p.value, frees.fe2$p.value,
                         frees.fd$p.value, frees.ccep$p.value,
                         frees.ccept$p.value)

## CIPS test
tab5corr["CIPS", ] <- c(cips.tfe$p.value, cips.fe2$p.value,
                        cips.fd$p.value,  cips.ccep$p.value,
                        cips.ccept$p.value)
#tab5corr["CIPS", ] <- ifelse(tab5corr["CIPS",]>0.05, "I(1)", "I(0)")

## RMSE
tab5corr["RMSE", ] <- c(rmse.tfe, rmse.fe2, rmse.fd, rmse.ccep, rmse.ccept)

xtable(tab5corr[-10, -5], digits=3,
       caption="Static homogeneous models, modified table.
   $t$-statistics in parentheses, constructed from clustered (POLS,
   2FE, FD) viz. nonparametric (CCEP) standard errors.
   W-AR(p) test: Wooldridge (2010) test for $H_0$ of no residual serial
   correlation at the p-th order, robustified; Frees (1995) test for
   $H_0$ of cross-sectionally independent residuals; CRS, CIPS, RMSE:
   see preceding Table", label="tab:ehs5")


###################################################
### code chunk number 25: t5.cdp
###################################################
############# local (spatial) dependence diagnostics ############

## make proximity matrices in land and sector
ids <- unique(ehs$id)
n <- length(ids)
cntw <- matrix(0, ncol=n, nrow=n)
secw <- matrix(0, ncol=n, nrow=n)

## make cntw
for(i in 1:n) {
  ## all equal, so take the first
  icnt <- ehs[ehs$id==ids[i], "country"][1]
  for(j in 1:n) {
    jcnt <- ehs[ehs$id==ids[j], "country"][1]
    if(icnt==jcnt) cntw[i,j] <- 1
  }
}
## eliminate self-neighbours
diag(cntw) <- 0
rownames(cntw) <- colnames(cntw) <- ids

## make secw
for(i in 1:n) {
  ## all equal, so take the first
  icnt <- ehs[ehs$id==ids[i], "sector"][1]
  for(j in 1:n) {
    jcnt <- ehs[ehs$id==ids[j], "sector"][1]
    if(icnt==jcnt) secw[i,j] <- 1
  }
}
## eliminate self-neighbours
diag(secw) <- 0
rownames(secw) <- colnames(secw) <- ids

## assessment of the two forms of "local" dependence
cd1.sec.tfe <- pcdtest(tfemod, w=secw)
cd1.sec.fe2 <- pcdtest(fe2mod, w=secw)
cd1.sec.fd <- pcdtest(fdamod, w=secw)
cd1.sec.ccep <- pcdtest(ccepmod, w=secw)
cd1.sec.ccept <- pcdtest(ccepmodt, w=secw)

cd1.cnt.tfe <- pcdtest(tfemod, w=cntw)
cd1.cnt.fe2 <- pcdtest(fe2mod, w=cntw)
cd1.cnt.fd <- pcdtest(fdamod, w=cntw)
cd1.cnt.ccep <- pcdtest(ccepmod, w=cntw)
cd1.cnt.ccept <- pcdtest(ccepmodt, w=cntw)


###################################################
### code chunk number 26: t5.rw
###################################################
library(splm)
rw.sec.tfe <- rwtest(tfemod, w=secw, rep=999)
rw.sec.fe2 <- rwtest(fe2mod, w=secw, rep=999)
rw.sec.fd <- rwtest(fdamod, w=secw, rep=999)
rw.sec.ccep <- rwtest(ccepmod, w=secw, rep=999)
rw.sec.ccept <- rwtest(ccepmodt, w=secw, rep=999)

rw.cnt.tfe <- rwtest(tfemod, w=cntw, rep=999)
rw.cnt.fe2 <- rwtest(fe2mod, w=cntw, rep=999)
rw.cnt.fd <- rwtest(fdamod, w=cntw, rep=999)
rw.cnt.ccep <- rwtest(ccepmod, w=cntw, rep=999)
rw.cnt.ccept <- rwtest(ccepmodt, w=cntw, rep=999)


###################################################
### code chunk number 27: tab5bis
###################################################
tab5bis <- matrix(ncol=5, nrow=8)
dimnames(tab5bis) <- list(c("Global CD", "   ",
                            "CD(1), sector", "", "CD(1), country", " ",
                            "RW, sector", "RW, country"),
                       c("POLS", "2FE", "FD", "CCEP", "CCEPt"))

tab5bis[1, ] <- c(cd.tfe$statistic, cd.fe2$statistic,
                  cd.fd$statistic, cd.ccep$statistic,
                  cd.ccept$statistic)
tab5bis[2, ] <- c(cd.tfe$p.value, cd.fe2$p.value,
                  cd.fd$p.value, cd.ccep$p.value,
                  cd.ccept$p.value)
tab5bis[3, ] <- c(cd1.sec.tfe$statistic, cd1.sec.fe2$statistic,
                  cd1.sec.fd$statistic, cd1.sec.ccep$statistic,
                  cd1.sec.ccept$statistic)
tab5bis[4, ] <- c(cd1.sec.tfe$p.value, cd1.sec.fe2$p.value,
                  cd1.sec.fd$p.value, cd1.sec.ccep$p.value,
                  cd1.sec.ccept$p.value)
tab5bis[5, ] <- c(cd1.cnt.tfe$statistic, cd1.cnt.fe2$statistic,
                  cd1.cnt.fd$statistic, cd1.cnt.ccep$statistic,
                  cd1.cnt.ccept$statistic)
tab5bis[6, ] <- c(cd1.cnt.tfe$p.value, cd1.cnt.fe2$p.value,
                  cd1.cnt.fd$p.value, cd1.cnt.ccep$p.value,
                  cd1.cnt.ccept$p.value)

#xtable(tab5bis, digits=3, caption="Global vs. local CD tests.")
## substituted by tab5ter


###################################################
### code chunk number 28: tab5ter0
###################################################
tab5ter <- matrix(ncol=5, nrow=12)
dimnames(tab5ter) <- list(c("Global CD", "   ", "avg. rho",
                            "avg. |rho|", "Frees rank test", "    ",
                            "CD(1), sector", "", "CD(1), country", " ",
                            "RW, sector", "RW, country"),
                          c("POLS", "2FE", "FD", "CCEP", "CCEPt"))

tab5ter[1, ] <- c(cd.tfe$statistic, cd.fe2$statistic,
                  cd.fd$statistic, cd.ccep$statistic,
                  cd.ccept$statistic)
tab5ter[2, ] <- c(cd.tfe$p.value, cd.fe2$p.value,
                  cd.fd$p.value, cd.ccep$p.value,
                  cd.ccept$p.value)
tab5ter[3, ] <- c(rho.tfe$statistic, rho.fe2$statistic,
                  rho.fd$statistic, rho.ccep$statistic,
                  rho.ccept$statistic)
tab5ter[4, ] <- c(absrho.tfe$statistic, absrho.fe2$statistic,
                  absrho.fd$statistic, absrho.ccep$statistic,
                  absrho.ccept$statistic)
tab5ter[5, ] <- c(frees.tfe$statistic, frees.fe2$statistic,
                  frees.fd$statistic, frees.ccep$statistic,
                  frees.ccept$statistic)
tab5ter[6, ] <- c(frees.tfe$p.value, frees.fe2$p.value,
                  frees.fd$p.value, frees.ccep$p.value,
                  frees.ccept$p.value)
tab5ter[7, ] <- c(cd1.sec.tfe$statistic, cd1.sec.fe2$statistic,
                  cd1.sec.fd$statistic, cd1.sec.ccep$statistic,
                  cd1.sec.ccept$statistic)
tab5ter[8, ] <- c(cd1.sec.tfe$p.value, cd1.sec.fe2$p.value,
                  cd1.sec.fd$p.value, cd1.sec.ccep$p.value,
                  cd1.sec.ccept$p.value)
tab5ter[9, ] <- c(cd1.cnt.tfe$statistic, cd1.cnt.fe2$statistic,
                  cd1.cnt.fd$statistic, cd1.cnt.ccep$statistic,
                  cd1.cnt.ccept$statistic)
tab5ter[10, ] <- c(cd1.cnt.tfe$p.value, cd1.cnt.fe2$p.value,
                  cd1.cnt.fd$p.value, cd1.cnt.ccep$p.value,
                  cd1.cnt.ccept$p.value)
tab5ter[11, ] <- c(rw.sec.tfe$p.value, rw.sec.fe2$p.value,
                  rw.sec.fd$p.value, rw.sec.ccep$p.value,
                  rw.sec.ccept$p.value)
tab5ter[12, ] <- c(rw.cnt.tfe$p.value, rw.cnt.fe2$p.value,
                  rw.cnt.fd$p.value, rw.cnt.ccep$p.value,
                  rw.cnt.ccept$p.value)



###################################################
### code chunk number 29: tab5tera
###################################################
## assess factor structure of the residuals through exponent of XSD:
tab5tera <- matrix(nrow=3, ncol=4)
dimnames(tab5tera) <- list(c("a.o.lwr95", "a.o", "a.o.upr95"),
                           c("POLS", "2FE", "FD", "CCEP"))

source("./functions/cdexp5.R")
## controlling for spatial dependence:
source("./functions/dineof.R")

tab5tera[2,1] <- cdexp(resid(tfemod), persistent=TRUE, vf="nw")$a.thrtilde
tab5tera[2,2] <- cdexp(resid(fe2mod), persistent=TRUE, vf="nw")$a.thrtilde
tab5tera[2,3] <- cdexp(resid(fdamod), persistent=TRUE, vf="nw")$a.thrtilde
tab5tera[2,4] <- cdexp(resid(ccepmod), persistent=TRUE, vf="nw")$a.thrtilde
tab5tera[c(1,3),1] <- cdexp(resid(tfemod), persistent=TRUE, vf="nw")$cb.a.thrtilde
tab5tera[c(1,3),2] <- cdexp(resid(fe2mod), persistent=TRUE, vf="nw")$cb.a.thrtilde
tab5tera[c(1,3),3] <- cdexp(resid(fdamod), persistent=TRUE, vf="nw")$cb.a.thrtilde
tab5tera[c(1,3),4] <- cdexp(resid(ccepmod), persistent=TRUE, vf="nw")$cb.a.thrtilde

#round(tab5tera,3)
## NB the exponent is moot if the variable is centered cross-section on zero means;
## hence the exponent is not computable for TFE, FE2, FD+TFE and any model where
## time dummies are added. For the CCEP (see referee), it confirms the appropriate
## defactoring has occurred.

## calc only for CCEP, in different flavours:
tab5terb <- matrix(nrow=6, ncol=5)
dimnames(tab5terb) <- list(c("None", "Persistent factors, NW",
                             "Persistent factors, AR",
                             "Spatial correlation (reduce)",
                             "Spatial correlation (interpolate)",
                             "Both (NW + reduce)"),
                           c("a.o.lwr95", "a.o", "a.o.upr95", "N.alfa", "N"))
tab5terb[1,2] <- cdexp(resid(ccepmod), persistent=FALSE, vf="nw")$a.thrtilde
tab5terb[2,2] <- cdexp(resid(ccepmod), persistent=TRUE, vf="nw")$a.thrtilde
tab5terb[3,2] <- cdexp(resid(ccepmod), persistent=TRUE, vf="ar")$a.thrtilde
tab5terb[4,2] <- cdexp(resid(ccepmod), persistent=FALSE,
                       spat.err=TRUE, balance="reduce", vf="nw")$a.thrtilde
tab5terb[5,2] <- cdexp(resid(ccepmod), persistent=FALSE,
                       spat.err=TRUE, balance="interpolate", vf="nw")$a.thrtilde
tab5terb[6,2] <- cdexp(resid(ccepmod), persistent=TRUE,
                       spat.err=TRUE, balance="reduce", vf="nw")$a.thrtilde

tab5terb[1, c(1,3)] <- cdexp(resid(ccepmod), persistent=FALSE, vf="nw")$cb.a.thrtilde
tab5terb[2, c(1,3)] <- cdexp(resid(ccepmod), persistent=TRUE, vf="nw")$cb.a.thrtilde
tab5terb[3, c(1,3)] <- cdexp(resid(ccepmod), persistent=TRUE, vf="ar")$cb.a.thrtilde
tab5terb[4, c(1,3)] <- cdexp(resid(ccepmod), persistent=FALSE,
                             spat.err=TRUE, balance="reduce", vf="nw")$cb.a.thrtilde
tab5terb[5, c(1,3)] <- cdexp(resid(ccepmod), persistent=FALSE,
                             spat.err=TRUE, balance="interpolate", vf="nw")$cb.a.thrtilde
tab5terb[6, c(1,3)] <- cdexp(resid(ccepmod), persistent=TRUE,
                             spat.err=TRUE, balance="reduce", vf="nw")$cb.a.thrtilde

tab5terb[1,4] <- cdexp(resid(ccepmod), persistent=FALSE, vf="nw")$ggg.t
tab5terb[2,4] <- cdexp(resid(ccepmod), persistent=TRUE, vf="nw")$ggg.t
tab5terb[3,4] <- cdexp(resid(ccepmod), persistent=TRUE, vf="ar")$ggg.t
tab5terb[4,4] <- cdexp(resid(ccepmod), persistent=FALSE,
                       spat.err=TRUE, balance="reduce", vf="nw")$ggg.t
tab5terb[5,4] <- cdexp(resid(ccepmod), persistent=FALSE,
                       spat.err=TRUE, balance="interpolate", vf="nw")$ggg.t
tab5terb[6,4] <- cdexp(resid(ccepmod), persistent=TRUE,
                       spat.err=TRUE, balance="reduce", vf="nw")$ggg.t

tab5terb[,5] <- pdim(resid(ccepmod))$nT$n


###################################################
### code chunk number 30: tabSEs
###################################################
## make table of alternative SEs for Tab. 5 (4th column)

tabSEs <- matrix(ncol=9, nrow=7)
dimnames(tabSEs) <- list(c("EHS", "Nonparametric", "HC (cluster by firm)", "Newey-West",
                           "Driscoll-Kraay", "Wild bootstrap", "Pairs bootstrap"),
                         c("ln Lit", " ", "  " ,"ln Kit", "   ", "   ",
                           "ln Rit", "   ", "    "))

## populate
tabSEs[1, c(1, 4, 7)] <- summ.fullccep[,2]
tabSEs[2, c(1, 4, 7)] <- sqrt(diag(ccep.vc))
tabSEs[3, c(1, 4, 7)] <- sqrt(diag(ccep.vchc))
tabSEs[4, c(1, 4, 7)] <- sqrt(diag(ccep.vcnw))
tabSEs[5, c(1, 4, 7)] <- sqrt(diag(ccep.vcscc))
tabSEs[6, c(1, 4, 7)] <- sqrt(diag(ccep.vcwb))
tabSEs[7, c(1, 4, 7)] <- sqrt(diag(ccep.vcpb))

## p-values from t-stats
tabSEs[ , c(2,5,8)] <- pnorm(rep(as.numeric(coef(ccepmoda)), each=7)/tabSEs[, c(1, 4, 7)],
                             lower.tail=FALSE)*2

## add bootstrapped p-values
source("./functions/vcovTBoot.R")
set.seed(12345)

## wild t-bootstrap
#wtb <- coeftest(ccepmoda, vcovTBoot)

## pairs t-bootstrap
#ptb <- coeftest(ccepmoda, vcov=function(x) vcovTBoot(x, type="pairs", reps=1000))

tabSEs <- rbind(tabSEs, c(NA, 0, NA, NA, 0.002, NA, NA, 0.020, NA))
tabSEs <- rbind(tabSEs, c(NA, 0, NA, NA, 0, NA, NA, 0.016, NA))
row.names(tabSEs)[8] <- "Wild t-bootstrap (HC)"
row.names(tabSEs)[9] <- "Pairs t-bootstrap (HC)"

## set rounding
tabSEs[, c(1,4,7)] <- round(tabSEs[, c(1,4,7)], 3)
tabSEs[, c(1,4,7)+1] <- round(tabSEs[, c(1,4,7)+1], 5)

## make stars
mystarfun <- function(x) {
    bins <- unclass(cut(x, breaks=c(-Inf, 0.001, 0.01, 0.05, 0.1, Inf)))
    mstars <- c("***", "**", "*", ".", "")
    return(mstars[bins])
}
tabSEs[, c(3,6,9)] <- mystarfun(tabSEs[, c(2,5,8)])

xtable(tabSEs, digits=3,
       caption="Original (EHS) vs. alternative standard errors for the CCEP
  model from Table 5 in EHS, with corresponding significance
  diagnostics. Nonparametric: as in Pesaran 2006, Th.3; HC, NW and DK:
  'sandwich' estimates as in Pesaran 2006, Th.4, with the 'meat'
  calculated by the respective methods; bootstrap ($M=999$): 'pairs' or
  'wild' bootstrap estimates, clustered by firm, as defined in
  CGM; $t$-bootstrap: 'pairs' or 'wild' resampling of
  cluster-robust $t$-statistics. For each variable, first column: standard
  error; second column: p-value for the $t$-test.", label="tab:ccepses")


###################################################
### code chunk number 31: tab5quater
###################################################
tab5quater <- matrix(ncol=4, nrow=4)
dimnames(tab5quater) <- list(c("W AR(1), HC",
                               "W AR(2), HC",
                               "W AR(1), SCC",
                               "W AR(2), SCC"),
                             c("POLS", "2FE", "FD", "CCEP"))

tab5quater[1, ] <- c(war1.tfe[2,4], war1.fe2[2,4], war1.fd[2,4],
                     war1.ccep[2,4])
tab5quater[2, ] <- c(war2.tfe[2,4], war2.fe2[2,4], war2.fd[2,4],
                     war2.ccep[2,4])
tab5quater[3, ] <- c(war1.tfe.scc[2,4], war1.fe2.scc[2,4],
                     war1.fd.scc[2,4],  war1.ccep.scc[2,4])
tab5quater[4, ] <- c(war2.tfe.scc[2,4], war2.fe2.scc[2,4],
                     war2.fd.scc[2,4],  war2.ccep.scc[2,4])

xtable(tab5quater, digits=3, caption="Wooldridge's informal test for serial correlation, with either HC (Arellano) or SCC (Driscoll-Kraay) covariance; H0: no serial correlation at given order in untransformed errors.", label="tab:wootest")


###################################################
### code chunk number 32: tab5ter
###################################################
xtable(tab5ter[ , 1:4], digits=3,
       caption="Comparison of cross section dependence diagnostics for the
  models in Table 5. Global CD is Pesaran's (2015) test; $rho$
  is the mean of pairwise correlation coefficients; $|rho|$
  the mean of their absolute values; Frees (1995) is the rank test for
  cross-sectional dependence; CD(1) sector and, respectively,
  country are the local CD tests (Pesaran, 2004) when defining
  neighbourhood as sharing either the same industrial sector or the
  same country. RW is Millo's (2017) randomization test for spatial
  correlation robust to common factors (pseudo-$p$ values from 999
  random draws are reported).",
       label="tab:xsd")


###################################################
### code chunk number 33: tab5terb
###################################################
#round(tab5terb,3)
## N.alfa gives an assessment of how many cross-sectional units still react
## to changes in some common factor
xtable(tab5terb, digits=c(0,3,3,3,0,0),
       caption="Exponent of cross-sectional dependence estimated on the CCEP residuals, allowing for (in row order): no persistence or spatial residual correlation; persistence, Newey-West style variance estimator; persistence, variance estimated through AR(4); spatial correlation in remainder errors, data balanced by reduction; spatial correlation, balanced by interpolation; both persistent factors (NW) and spatial correlation (by reduction). In column order: lower 95 percent bound, central estimate and upper bound for the bias-corrected version of the exponent; estimated number of significant factor loadings; total number of cross-sectional units (memorandum item).", label="tab:exp")


###################################################
### code chunk number 34: dynmodsdef
###################################################
dfm <- lny ~ lag(lny) + lnl + lag(lnl) + lnk + lag(lnk) + lnrd + lag(lnrd)


###################################################
### code chunk number 35: t6.tfe
###################################################

dpols <- plm(update(dfm, .~.+as.factor(year)-1), data=ehs, model="pooling")
#summary(dpols)

dpolsa <- plm(dfm, ehs, effect="time")
#summary(dpolsa)
## equivalent to dpols: use this for comfactest() w/o specifying k,
## because it has the right variables


###################################################
### code chunk number 36: t6.fe2
###################################################
# dfe <- plm(dfm, data=ehs, model="within") # (not in final paper)
# summary(dfe) # (not in final paper)

d2fe <- plm(dfm, data=ehs, model="within", effect="twoways")
#summary(d2fe)


###################################################
### code chunk number 37: t6.bb
###################################################

# ab not in final version
gmmfm0 <- lny ~ lag(lny) + lag(lnl, 0:1) + lag(lnk, 0:1) +
             lag(lnrd, 0:1) | lag(lny, 3:6) + lag(lnl, 3:6) +
             lag(lnk, 3:6) + lag(lnrd, 3:6)
gmmfm <- lny ~ lag(lny) + lag(lnl, 0:1) + lag(lnk, 0:1) +
             lag(lnrd, 0:1) | lag(lny, 3:99) + lag(lnl, 3:99) +
             lag(lnk, 3:99) + lag(lnrd, 3:99)
#dab <- pgmm(formula = gmmfm, data = ehs, effect = "twoways",
#    model = "onestep", transformation = "d", collapse=FALSE)
#summary(dab)
# bb
#dbb <- pgmm(formula = gmmfm, data = ehs, effect = "twoways",
 #   model = "twostep", transformation = "ld")

## coefs for BB are reasonably similar to DPOLS and 2FE if keeping
## instruments down to manageable level (but still singular matrix)

## alt.: onestep model
#dbba <- pgmm(formula = gmmfm, data = ehs, effect = "twoways",
#    model = "onestep", transformation = "ld")

## this here once for all, takes time
#sdbb <- summary(dbb)


###################################################
### code chunk number 38: bb.alt
###################################################
## alt. pgmm function from Nina Schoenfelder, see email 9 April 2015
source("./functions/mypgmm.R")

#adbb <- mypgmm(formula = gmmfm, data = ehs, effect = "twoways",
#               model = "twostep", transformation = "ld", collapse=T)
#sadbb <- summary(adbb)
## collapse instruments
cadbb <- mypgmm(formula = gmmfm0, data = ehs, effect = "twoways",
               model = "twostep", transformation = "ld", collapse=T)
scadbb <- summary(cadbb)

## take this one as BB
dbb <- cadbb
sdbb <- scadbb


###################################################
### code chunk number 39: t6.ccep
###################################################

dccep <- pcce(dfm, data=ehs, model="p")
#summary(dccep)



###################################################
### code chunk number 40: t6.badccep
###################################################

## make full formula
dfullfm<-lny ~ lag(lny) + lag(lnl,0:1) + lag(lnk,0:1) + lag(lnrd,0:1) +
    as.factor(id) +
    (Between(lny, effect="time") + Between(lnl, effect="time") +
     Between(lnk, effect="time") + Between(lnrd, effect="time")):
    as.factor(id) +
    (Between(lag(lny), effect="time") + Between(lag(lnl), effect="time") +
     Between(lag(lnk), effect="time") + Between(lag(lnrd), effect="time")):
    as.factor(id) - 1

## extract only relevant coefs (model takes some time to run)
dfullccep <- plm(dfullfm, ehs, model="pooling")

tstats.dfullccep <- coef(dfullccep)[1:7]/
    sqrt(diag(vcov(fullccep)[1:7,1:7]))



###################################################
### code chunk number 41: t6.badccept
###################################################

## make full formula
dfullfmt<-lny ~ lag(lny) + lag(lnl,0:1) + lag(lnk,0:1) + lag(lnrd,0:1) +
    as.factor(id) + as.factor(year) +
    (Between(lny, effect="time") + Between(lnl, effect="time") +
     Between(lnk, effect="time") + Between(lnrd, effect="time")):
    as.factor(id) +
    (Between(lag(lny), effect="time") + Between(lag(lnl), effect="time") +
     Between(lag(lnk), effect="time") + Between(lag(lnrd), effect="time")):
    as.factor(id) - 1

## extract only relevant coefs (model takes some time to run)
dfullccept <- plm(dfullfmt, ehs, model="pooling")

tstats.dfullccept <- coef(dfullccept)[1:7]/sqrt(diag(vcov(fullccept)[1:7,1:7]))
## exactly as in EHS, SEs are different


###################################################
### code chunk number 42: t6.cdtest
###################################################
source("./functions/pres.pgmm.R")

## CD test

## Residuals are the same for original and corrected models, bar ccept
cd.dpols <- pcdtest(dpols)
cd.d2fe <- pcdtest(d2fe)
cd.dbb <- pcdtest(pres.pgmm(dbb))
## resid(bb) is not compliant for structural reasons (it is a list,
## and it has length depending on the estimation method (difference or
## system)
cd.dccep <- pcdtest(dccep)
cd.dccept <- pcdtest(dfullccept) #list(statistic=NA, p.value=NA)


###################################################
### code chunk number 43: t6.cips
###################################################

## CIPS (PCADF) test
cips.dpols <- cipstest(resid(dpols), lags=1, model="cmg", type="drift")
cips.d2fe <- cipstest(resid(d2fe), lags=1, model="cmg", type="drift")
cips.dbb <- list(statistic=NA, p.value=NA) #cipstest(resid(dbb), lags=1, model="cmg", type="drift")
cips.dccep <- cipstest(resid(dccep), lags=1, model="cmg", type="drift")
cips.dccept <-  cipstest(resid(dfullccept), lags=1, model="cmg",
                         type="drift")


###################################################
### code chunk number 44: t6.rmse
###################################################

rmse.dpols <- sqrt(mean(resid(dpols)^2))
rmse.d2fe <- sqrt(mean(resid(d2fe)^2))
rmse.dbb <- sqrt(mean(pres.pgmm(dbb)^2))
rmse.dccep <- sqrt(mean(resid(dccep)^2))
rmse.dccept <- sqrt(mean(resid(dfullccept)^2))



###################################################
### code chunk number 45: t6.cdp
###################################################
############# local (spatial) dependence diagnostics ############

## use proximity matrices in land and sector done above

## assessment of the two forms of "local" dependence
cd1.sec.dpols <- pcdtest(dpols, w=secw)
cd1.sec.d2fe <- pcdtest(d2fe, w=secw)
cd1.sec.dbb <- pcdtest(pres.pgmm(dbb), w=secw)
cd1.sec.dccep <- pcdtest(dccep, w=secw)
cd1.sec.dccept <- list(statistic=NA, p.value=NA) #pcdtest(dccept, w=secw)

cd1.cnt.dpols <- pcdtest(dpols, w=cntw)
cd1.cnt.d2fe <- pcdtest(d2fe, w=cntw)
cd1.cnt.dbb <- pcdtest(pres.pgmm(dbb), w=cntw)
cd1.cnt.dccep <- pcdtest(dccep, w=cntw)
cd1.cnt.dccept <- list(statistic=NA, p.value=NA) #pcdtest(dccept, w=cntw)


###################################################
### code chunk number 46: tab6bis
###################################################
tab6bis <- matrix(ncol=5, nrow=6)
dimnames(tab6bis) <- list(c("Global CD", "   ",
                            "CD(1), sector", "", "CD(1), country", " "),
                       c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

tab6bis[1, ] <- c(cd.dpols$statistic, cd.d2fe$statistic,
                  cd.dbb$statistic, cd.dccep$statistic,
                  cd.dccept$statistic)
tab6bis[2, ] <- c(cd.dpols$p.value, cd.d2fe$p.value,
                  cd.dbb$p.value, cd.dccep$p.value,
                  cd.dccept$p.value)
tab6bis[3, ] <- c(cd1.sec.dpols$statistic, cd1.sec.d2fe$statistic,
                  cd1.sec.dbb$statistic, cd1.sec.dccep$statistic,
                  cd1.sec.dccept$statistic)
tab6bis[4, ] <- c(cd1.sec.dpols$p.value, cd1.sec.d2fe$p.value,
                  cd1.sec.dbb$p.value, cd1.sec.dccep$p.value,
                  cd1.sec.dccept$p.value)
tab6bis[5, ] <- c(cd1.cnt.dpols$statistic, cd1.cnt.d2fe$statistic,
                  cd1.cnt.dbb$statistic, cd1.cnt.dccep$statistic,
                  cd1.cnt.dccept$statistic)
tab6bis[6, ] <- c(cd1.cnt.dpols$p.value, cd1.cnt.d2fe$p.value,
                  cd1.cnt.dbb$p.value, cd1.cnt.dccep$p.value,
                  cd1.cnt.dccept$p.value)


###################################################
### code chunk number 47: t6.comfactest
###################################################
## COMFAC tests:
source("./functions/comfactest.R")

## do all models: at least for the test, if it accepts then also coefs

comfac.dpols <- comfactest(dpolsa) # alt.: comfactest(dpols, k=7)
comfac.dpols.corr <- comfactest(dpolsa, vcov=vcovHC)
comfac.d2fe <- comfactest(d2fe) # slight diff. due to ss correction
comfac.d2fe.corr <- comfactest(d2fe, vcov=vcovHC)
comfac.dbb <- comfactest(dbb, k=7) # slight diff, qualitatively OK
## but:
comfac.dbb.hc <- comfactest(dbb, vcov=vcovHC, k=7) ## accepts!!
comfac.dccep <- comfactest(dfullccep, k=7)
comfac.dccep.corr <- comfactest(dccep) # qualitatively different result (accepts!)
comfac.dccept <- comfactest(dfullccept, k=7)
comfac.dccept.corr <- NA


###################################################
### code chunk number 48: tab6d
###################################################
## replicate Table of dynamic model "as is"

tab6d <- matrix(ncol=5, nrow=7*2+4)
dimnames(tab6d) <- list(c("ln Yit-1", "", "ln Lit", " ", "ln Lit-1", "  ",
                         "ln Kit", "   ", "ln Kit-1", "    ",
                         "ln Rit", "     ","ln Rit-1", "      ",
                         "COMFAC", "CD", "CIPS", "RMSE"),
                       c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6d[c(1,3,5,7,9,11,13), 1] <- coef(dpols)[1:7] ## because of time d.
tab6d[c(1,3,5,7,9,11,13), 2] <- coef(d2fe)
tab6d[c(1,3,5,7,9,11,13), 3] <- coef(dbb)[1:7] ## because of time dummies
tab6d[c(1,3,5,7,9,11,13), 4] <- coef(dfullccep)[1:7]
tab6d[c(1,3,5,7,9,11,13), 5] <- coef(dfullccept)[1:7]   #coef(dccepmodt)

## SEs
myvcov <- function(x) vcovHC(x, method="white1")
tab6d[c(2,4,6,8,10,12,14), 1] <- coeftest(dpols)[1:7,3]
tab6d[c(2,4,6,8,10,12,14), 2] <- coeftest(d2fe)[,3]
tab6d[c(2,4,6,8,10,12,14), 3] <- coef(sdbb)[, 3]
## coeftest does not work! not compliant!
tab6d[c(2,4,6,8,10,12,14), 4] <- coeftest(dfullccep)[1:7, 3]
tab6d[c(2,4,6,8,10,12,14), 5] <- tstats.dfullccept


## CD test
tab6d["CD", ] <- c(cd.dpols$p.value, cd.d2fe$p.value, cd.dbb$p.value,
                  cd.dccep$p.value, cd.dccept$p.value)

## CIPS test
tab6d["CIPS", ] <- c(cips.dpols$p.value, cips.d2fe$p.value, cips.dbb$p.value, cips.dccep$p.value, cips.dccept$p.value)

## RMSE
tab6d["RMSE", ] <- c(rmse.dpols, rmse.d2fe, rmse.dbb, rmse.dccep,
                     rmse.dccept)

## COMFAC test
tab6d["COMFAC", ] <- c(comfac.dpols$test$p.value,
                       comfac.d2fe$test$p.value, comfac.dbb$test$p.value,
                       comfac.dccep$test$p.value,
                       comfac.dccept$test$p.value)

xtable(tab6d, digits=3, caption="Dynamic homogeneous models. Replication of unreported table, basis for Table 6 in EHS. Diagnostics denoted as in the preceding tables.")


###################################################
### code chunk number 49: t6.frees
###################################################

## Frees rank-based test
frees.dpols <- pcdtest3(dpols, test="frees", exact=FALSE)
frees.d2fe <- pcdtest3(d2fe, test="frees", exact=FALSE)
frees.dbb <- pcdtest3(pres.pgmm(dbb), test="frees", exact=FALSE)
frees.dccep <- pcdtest3(dccep, test="frees", exact=FALSE)
frees.dccept <- pcdtest3(dfullccept, test="frees", exact=FALSE)

## relative and absolute rho:
rho.dpols <- pcdtest(dpols, test="rho")
rho.d2fe <- pcdtest(d2fe, test="rho")
rho.dbb <- pcdtest(pres.pgmm(dbb), test="rho")
rho.dccep <- pcdtest(dccep, test="rho")
rho.dccept <- pcdtest(dfullccept, test="rho") # not used

absrho.dpols <- pcdtest(dpols, test="absrho")
absrho.d2fe <- pcdtest(d2fe, test="absrho")
absrho.dbb <- pcdtest(pres.pgmm(dbb), test="absrho")
absrho.dccep <- pcdtest(dccep, test="absrho")
absrho.dccept <- pcdtest(dfullccept, test="absrho") # not used


###################################################
### code chunk number 50: tab6d.corr
###################################################

tab6d.corr <- matrix(ncol=5, nrow=7*2+4)
dimnames(tab6d.corr) <- list(c("ln Yit-1", "", "ln Lit", " ", "ln Lit-1",
                               "  ", "ln Kit", "   ", "ln Kit-1", "    ",
                               "ln Rit", "     ","ln Rit-1", "      ",
                               "COMFAC", "Frees", "CIPS", "RMSE"),
                             c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6d.corr[c(1,3,5,7,9,11,13), 1] <- coef(dpols)[1:7] ## because of time d.
tab6d.corr[c(1,3,5,7,9,11,13), 2] <- coef(d2fe)
tab6d.corr[c(1,3,5,7,9,11,13), 3] <- coef(dbb)[1:7] ## because of time dummies
tab6d.corr[c(1,3,5,7,9,11,13), 4] <- coef(dccep)
tab6d.corr[c(1,3,5,7,9,11,13), 5] <- coef(dfullccept)[1:7]   #coef(dccepmodt)

## SEs
tab6d.corr[c(2,4,6,8,10,12,14), 1] <- coeftest(dpols, vcov=vcovHC)[1:7,3]
tab6d.corr[c(2,4,6,8,10,12,14), 2] <- coeftest(d2fe, vcov=vcovHC)[,3]
tab6d.corr[c(2,4,6,8,10,12,14), 3] <- coef(sdbb)[, 3]
## coeftest does not work! not compliant!
tab6d.corr[c(2,4,6,8,10,12,14), 4] <- coeftest(dccep)[,3]
tab6d.corr[c(2,4,6,8,10,12,14), 5] <- tstats.dfullccept


## Frees test
tab6d.corr["Frees", ] <- c(frees.dpols$p.value, frees.d2fe$p.value,
                           frees.dbb$p.value, frees.dccep$p.value,
                           frees.dccept$p.value)

## CIPS test
tab6d.corr["CIPS", ] <- c(cips.dpols$p.value, cips.d2fe$p.value, cips.dbb$p.value, cips.dccep$p.value, cips.dccept$p.value)

## RMSE
tab6d.corr["RMSE", ] <- c(rmse.dpols, rmse.d2fe, rmse.dbb, rmse.dccep,
                     rmse.dccept)

## COMFAC test
tab6d.corr["COMFAC", ] <- c(comfac.dpols.corr$test$p.value,
                            comfac.d2fe.corr$test$p.value,
                            comfac.dbb$test$p.value,
                            comfac.dccep.corr$test$p.value, NA) #cd.dccept$p.value)

xtable(tab6d.corr[, -5], digits=3,
       caption="Dynamic homogeneous models, modified table. Diagnostics denoted as in the preceding tables.")


###################################################
### code chunk number 51: t6.unrestrictedlr
###################################################
## Unrestricted long-run coefs:
library(msm) # for delta method
source("./functions/lrcoef.R")

## replicates exactly the dynamic OLS model:
##
## Stata> reg lny lnl l.lnl lnk l.lnk lnrd l.lnrd l.lny
## Stata> nlcom ((_b[lnl ]+_b[l.lnl])/(1-_b[l.lny ])) ((_b[lnk ]+_b[l.lnk ])/(1-_b[l.lny ])) ((_b[lnrd]+_b[l.lnrd])/(1-_b[l.lny ]))
##
## R> mymod<-plm(dfm, ehs, model="p")
## R> lrcoef(mymod)

myvcov <- function(x) vcovHC(x, method="white1")
lr.dpols <- lrcoef(dpolsa, vcov=myvcov)
lr.dpols.corr <- lrcoef(dpolsa, vcov=vcovHC)
lr.dbb <- lrcoef(dbb, k=7)              # very similar to EHS
lr.dccep <- lrcoef(dfullccep)[1:3, ]
lr.dccep.corr <- lrcoef(dccep)          # only for comparison, see COMFAC
                # Different, even qualitatively (not sign. any more)
lr.dccept <- lrcoef(dfullccept)[1:3, ]

## other models, use unrestricted long-term coefs and vcov



###################################################
### code chunk number 52: t6.crstests.r
###################################################

## CRS tests: (must use coefs and vcovs from long-run models)

## long-run 2FE and CCEP models, use restricted b and vcov:
source("./functions/comfacvcov.R")

crstest <- function(x, names=c("lnl","lnk","lnrd"),
                    lrvcov=comfacvcov, ...) {
    ## performs a Wald test for CRS
    ## notice: 'names' can also be an integer vector of indices
    bvb <- lrvcov(x, ...)
    b <- bvb$coef[names]
    vb <- bvb$vcov[names, names]
    R <- rep(1, length(names))
    r <- 1
    stat <- crossprod(R%*%b-r, solve(crossprod(R,vb)%*%R, R%*%b-r))
    pval <- pchisq(stat, df=1, lower.tail=FALSE)
    return(list(statistic=stat, p.value=pval))
}

crs.d2fe <- crstest(d2fe, lrvcov=comfacvcov)
crs.d2fe.corr <- crstest(d2fe, lrvcov=comfacvcov, vcov=vcovHC)
crs.dccep.corr <- crstest(dccep)
crs.dbb.hc <- crstest(dbb, names=dimnames(lr.dbb)[[1]],
                      vcov=vcovHC)



###################################################
### code chunk number 53: t6.crstests.u
###################################################

## CRS tests: (must use coefs and vcovs from long-run models)

## long-run POLS etc. models, use unrestricted b and vcov:
source("./functions/lrvcov.R")

crs.dpols <- crstest(dpolsa, lrvcov=lrvcov, vcov=myvcov) # as in paper
crs.dpols.corr <- crstest(dpolsa, lrvcov=lrvcov, vcov=vcovHC)
crs.dbb <- crstest(dbb, names=dimnames(lr.dbb)[[1]],
                   lrvcov=comfacvcov)
crs.dccep <- crstest(dfullccep, names=1:3, k=8, lrvcov=lrvcov)
crs.dccept <- crstest(dfullccept, names=1:3, k=8, lrvcov=lrvcov)


###################################################
### code chunk number 54: tab6A
###################################################

tab6A <- matrix(ncol=5, nrow=3*2+1)
dimnames(tab6A) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS"),
                       c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6A[c(1,3,5), 1] <- lr.dpols[1:3, 1]
tab6A[c(1,3,5), 2] <- NA
tab6A[c(1,3,5), 3] <- lr.dbb[1:3, 1]
tab6A[c(1,3,5), 4] <- lr.dccep[1:3, 1]  # only for comparison/replication
tab6A[c(1,3,5), 5] <- lr.dccept[1:3, 1]

## SEs
tab6A[c(2,4,6), 1] <- lr.dpols[1:3, 3]
tab6A[c(2,4,6), 2] <- NA
tab6A[c(2,4,6), 3] <- lr.dbb[1:3, 3]
tab6A[c(2,4,6), 4] <- lr.dccep[1:3, 3]  # only for comparison/replication
tab6A[c(2,4,6), 5] <- lr.dccept[1:3, 3]

## CRS test
tab6A["CRS", ] <- c(crs.dpols$p.value, NA, crs.dbb$p.value,
                    crs.dccep$p.value, crs.dccept$p.value)



###################################################
### code chunk number 55: tab6B
###################################################

tab6B <- matrix(ncol=5, nrow=3*2+1)
dimnames(tab6B) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS"),
                       c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6B[c(1,3,5), 1] <- NA
tab6B[c(1,3,5), 2] <- comfac.d2fe$model[1:3, 1]
tab6B[c(1,3,5), 3] <- NA # comfac.dbb$model[1:3, 1]
tab6B[c(1,3,5), 4] <- NA # comfac.dccep$model[1:3, 1]
tab6B[c(1,3,5), 5] <- NA

## SEs
myvcov <- function(x) vcovHC(x, method="white1")
tab6B[c(2,4,6), 1] <- NA
tab6B[c(2,4,6), 2] <- comfac.d2fe$model[1:3, 3]
tab6B[c(2,4,6), 3] <- NA # comfac.dbb$model[1:3, 3]
tab6B[c(2,4,6), 4] <- NA # comfac.dccep$model[1:3, 3]
tab6B[c(2,4,6), 5] <- NA

## CRS test
tab6B["CRS", ] <- c(NA, crs.d2fe$p.value, NA, NA, NA) # crs.dbb.hc$p.value,
                  #  crs.dccep$p.value, NA)



###################################################
### code chunk number 56: tab6
###################################################
xtable(tab6A, digits=3, caption="Dynamic homogeneous models, unrestricted long-run solutions (Replication of Table 6, Panel A in EHS).")
xtable(tab6B, digits=3, caption="Dynamic homogeneous models, long run solutions imposing COMFAC restriction (Replication of Table 6, Panel B in EHS).")


###################################################
### code chunk number 57: tab6Acorr
###################################################

tab6Acorr <- matrix(ncol=5, nrow=3*2+1)
dimnames(tab6Acorr) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS"),
                            c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6Acorr[c(1,3,5), 1] <- lr.dpols[1:3, 1]
tab6Acorr[c(1,3,5), 2] <- NA
tab6Acorr[c(1,3,5), 3] <- lr.dbb[1:3, 1]
tab6Acorr[c(1,3,5), 4] <- NA # lr.dccep[1:3, 1]
tab6Acorr[c(1,3,5), 5] <- NA

## SEs
tab6Acorr[c(2,4,6), 1] <- lr.dpols[1:3, 3]
tab6Acorr[c(2,4,6), 2] <- NA
tab6Acorr[c(2,4,6), 3] <- lr.dbb[1:3, 3]
tab6Acorr[c(2,4,6), 4] <- NA #lr.dccep[1:3, 3]
tab6Acorr[c(2,4,6), 5] <- NA

## CRS test
tab6Acorr["CRS", ] <- c(crs.dpols$p.value, NA, crs.dbb$p.value, NA, NA)



###################################################
### code chunk number 58: tab6Bcorr
###################################################

tab6Bcorr <- matrix(ncol=5, nrow=3*2+1)
dimnames(tab6Bcorr) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS"),
                       c("POLS", "2FE", "BB", "CCEP", "CCEPt"))

## populate coefs
tab6Bcorr[c(1,3,5), 1] <- NA
tab6Bcorr[c(1,3,5), 2] <- comfac.d2fe$model[1:3, 1]
tab6Bcorr[c(1,3,5), 3] <- NA # comfac.dbb$model[1:3, 1]
tab6Bcorr[c(1,3,5), 4] <- comfac.dccep.corr$model[1:3, 1]
tab6Bcorr[c(1,3,5), 5] <- NA

## SEs
myvcov <- function(x) vcovHC(x, method="white1")
tab6Bcorr[c(2,4,6), 1] <- NA
tab6Bcorr[c(2,4,6), 2] <- comfac.d2fe$model[1:3, 3]
tab6Bcorr[c(2,4,6), 3] <- NA # comfac.dbb$model[1:3, 3]
tab6Bcorr[c(2,4,6), 4] <- comfac.dccep.corr$model[1:3, 3]
tab6Bcorr[c(2,4,6), 5] <- NA

## CRS test
tab6Bcorr["CRS", ] <- c(NA, crs.d2fe.corr$p.value, NA, #crs.dbb.hc$p.value,
                        crs.dccep.corr$p.value, NA)



###################################################
### code chunk number 59: tab6corr
###################################################
xtable(tab6Acorr[,-5], digits=3, caption="Dynamic homogeneous models, unrestricted long-run solutions, modified version of Panel A.")
xtable(tab6Bcorr[,-5], digits=3, caption="Dynamic homogeneous models, long run solutions imposing COMFAC restriction, modified version of Panel B.")


###################################################
### code chunk number 60: t7.mg
###################################################

## PMG
mgmod0<-pmg(fm, ehs, model="mg")
#summary(mgmod0) # not in paper

## PMG with trend
mgmod<-pmg(fm, ehs, model="mg", trend=TRUE)
#summary(mgmod)


###################################################
### code chunk number 61: t7.dmg
###################################################

## DMG
dmgmod<-pmg(fm, ehs, model="dmg", trend=FALSE)
#summary(dmgmod)
## (notice trend=FALSE)


###################################################
### code chunk number 62: t7.cmg
###################################################

## CCE MG
## without trends:
cmgmod<-pmg(fm, ehs, model="cmg", trend=FALSE)
#summary(cmgmod)


###################################################
### code chunk number 63: t7.cmgt
###################################################

## with trends:
cmgmodt<-pmg(fm, ehs, model="cmg", trend=TRUE)
#summary(cmgmodt)


###################################################
### code chunk number 64: t7.cmga
###################################################
cmgmoda<-pcce(fm, ehs, model="mg", trend=FALSE)
#summary(cmgmoda)


###################################################
### code chunk number 65: t7.cmgat
###################################################
cmgmodat<-pcce(fm, ehs, model="mg", trend=TRUE)
#summary(cmgmodat)


###################################################
### code chunk number 66: t7.lb
###################################################
## Fisher-type Ljung-Box test (combination of individual AR(1) tests)
lb.mg <- fishertest(resid(mgmod),
                    test=function(x) Box.test(x, type="Ljung-Box"))
lb.dmg <- fishertest(resid(dmgmod),
                    test=function(x) Box.test(x, type="Ljung-Box"))
lb.cmg <- fishertest(resid(cmgmod),
                    test=function(x) Box.test(x, type="Ljung-Box"))
lb.cmgt <- fishertest(resid(cmgmodt),
                    test=function(x) Box.test(x, type="Ljung-Box"))


###################################################
### code chunk number 67: t7.cd
###################################################
## CD tests
cd.mg <- pcdtest(mgmod)
cd.dmg <- pcdtest(dmgmod)
cd.cmg <- pcdtest(cmgmod)
cd.cmgt <- pcdtest(cmgmodt)
cd.cmga <- pcdtest(cmgmoda)
cd.cmgat <- pcdtest(cmgmodat)


###################################################
### code chunk number 68: t7.frees
###################################################
## Frees rank-based test
frees.mg <- pcdtest3(resid(mgmod), test="frees", exact=FALSE)
frees.dmg <- pcdtest3(resid(dmgmod), test="frees", exact=FALSE)
frees.cmg <- pcdtest3(resid(cmgmod), test="frees", exact=FALSE)
frees.cmgt <- pcdtest3(resid(cmgmodt), test="frees", exact=FALSE)

## relative and absolute rho:
rho.mg <- pcdtest(mgmod, test="rho")
rho.dmg <- pcdtest(dmgmod, test="rho")
rho.cmg <- pcdtest(cmgmod, test="rho")
rho.cmgt <- pcdtest(cmgmodt, test="rho")

absrho.mg <- pcdtest(mgmod, test="absrho")
absrho.dmg <- pcdtest(dmgmod, test="absrho")
absrho.cmg <- pcdtest(cmgmod, test="absrho")
absrho.cmgt <- pcdtest(cmgmodt, test="absrho")


###################################################
### code chunk number 69: t7.cdp
###################################################

## assessment of the two forms of "local" dependence
cd1.sec.mg <- pcdtest(mgmod, w=secw)
cd1.sec.dmg <- pcdtest(dmgmod, w=secw)
cd1.sec.cmg <- pcdtest(cmgmod, w=secw)
cd1.sec.cmgt <- pcdtest(cmgmodt, w=secw)

cd1.cnt.mg <- pcdtest(mgmod, w=cntw)
cd1.cnt.dmg <- pcdtest(dmgmod, w=cntw)
cd1.cnt.cmg <- pcdtest(cmgmod, w=cntw)
cd1.cnt.cmgt <- pcdtest(cmgmodt, w=cntw)


###################################################
### code chunk number 70: tab7bis
###################################################
tab7bis <- matrix(ncol=4, nrow=6)
dimnames(tab7bis) <- list(c("Global CD", "   ",
                            "CD(1), sector", "", "CD(1), country", " "),
                       c("MG", "CDMG", "CMG", "CMGt"))

tab7bis[1, ] <- c(cd.mg$statistic, cd.dmg$statistic,
                  cd.cmg$statistic, cd.cmgt$statistic)
tab7bis[2, ] <- c(cd.mg$p.value, cd.dmg$p.value,
                  cd.cmg$p.value, cd.cmgt$p.value)
tab7bis[3, ] <- c(cd1.sec.mg$statistic, cd1.sec.dmg$statistic,
                  cd1.sec.cmg$statistic, cd1.sec.cmgt$statistic)
tab7bis[4, ] <- c(cd1.sec.mg$p.value, cd1.sec.dmg$p.value,
                  cd1.sec.cmg$p.value, cd1.sec.cmgt$p.value)
tab7bis[5, ] <- c(cd1.cnt.mg$statistic, cd1.cnt.dmg$statistic,
                  cd1.cnt.cmg$statistic, cd1.cnt.cmgt$statistic)
tab7bis[6, ] <- c(cd1.cnt.mg$p.value, cd1.cnt.dmg$p.value,
                  cd1.cnt.cmg$p.value, cd1.cnt.cmgt$p.value)


###################################################
### code chunk number 71: t7.cips
###################################################

## CIPS (PCADF) test
cips.mg <- cipstest(resid(mgmod), lags=1, model="cmg", type="drift")
cips.dmg <- cipstest(resid(dmgmod), lags=1, model="cmg", type="drift")
cips.cmg <- cipstest(resid(cmgmod), lags=1, model="cmg", type="drift")
cips.cmgt <- cipstest(resid(cmgmodt), lags=1, model="cmg", type="drift")


###################################################
### code chunk number 72: t7.crs
###################################################

## CRS (constant rate of substitution) hypothesis:
crs <- "lnl+lnk+lnrd=1"

#library(car)

crs.mg <- lht(mgmod, crs)
crs.dmg <- lht(dmgmod, crs)
crs.cmg <- lht(cmgmod, crs)
crs.cmgt <- lht(cmgmodt, crs)


###################################################
### code chunk number 73: t7.rmse
###################################################

rmse.mg <- sqrt(mean(resid(mgmod)^2))
rmse.dmg <- sqrt(mean(resid(dmgmod)^2))
rmse.cmg <- sqrt(mean(resid(cmgmod)^2))
rmse.cmgt <- sqrt(mean(resid(cmgmodt)^2))


###################################################
### code chunk number 74: tab7
###################################################

tab7 <- matrix(ncol=4, nrow=11)
dimnames(tab7) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS", "Ljung-box AR(1)", "CD", "CIPS", "RMSE"),
                       c("MG", "CDMG", "CMG", "CMGt"))

## populate coefs
tab7[c(1,3,5), 1] <- coef(mgmod)[c("lnl", "lnk", "lnrd")]
tab7[c(1,3,5), 2] <- coef(dmgmod)[c("lnl", "lnk", "lnrd")]
tab7[c(1,3,5), 3] <- coef(cmgmod)[c("lnl", "lnk", "lnrd")]
tab7[c(1,3,5), 4] <- coef(cmgmodt)[c("lnl", "lnk", "lnrd")]

## SEs
tab7[c(2,4,6), 1] <- coeftest(mgmod)[c("lnl", "lnk", "lnrd"),3]
tab7[c(2,4,6), 2] <- coeftest(dmgmod)[c("lnl", "lnk", "lnrd"),3]
tab7[c(2,4,6), 3] <- coeftest(cmgmod)[c("lnl", "lnk", "lnrd"),3]
tab7[c(2,4,6), 4] <- coeftest(cmgmodt)[c("lnl", "lnk", "lnrd"),3]

## CRS test
tab7["CRS", ] <- c(crs.mg[2,4], crs.dmg[2,4], crs.cmg[2,4],
                   crs.cmgt[2,4])

## Ljung-Box-Fisher test
tab7["Ljung-box AR(1)", ] <- c(lb.mg$p.value, lb.dmg$p.value,
                               lb.cmg$p.value, lb.cmgt$p.value)

## CD test
tab7["CD", ] <- c(cd.mg$p.value, cd.dmg$p.value, cd.cmg$p.value,
                  cd.cmgt$p.value)

## CIPS test
tab7["CIPS", ] <- c(cips.mg$p.value, cips.dmg$p.value, cips.cmg$p.value,
                  cips.cmgt$p.value)

## RMSE
tab7["RMSE", ] <- c(rmse.mg, rmse.dmg, rmse.cmg, rmse.cmgt)

xtable(tab7, digits=3, caption="Static heterogeneous models (Table 7 in EHS). Diagnostics denoted as in the preceding tables, but for Ljung-Box AR(1) which is a Fisher-type combination of individual Ljung-Box tests for serial correlation.")


###################################################
### code chunk number 75: t7.war
###################################################

## vcovHC version (clustering, no XS corr.)
war1.mg <- wartest(mgmod, 1)
war1.dmg <- wartest(dmgmod, 1)
war1.cmg <- wartest(cmgmod, 1)
war1.cmgt <- wartest(cmgmodt, 1)

war2.mg <- wartest(mgmod, 2)
war2.dmg <- wartest(dmgmod, 2)
war2.cmg <- wartest(cmgmod, 2)
war2.cmgt <- wartest(cmgmodt, 2)

## vcovSCC version (accounting for XS corr. and AR)
war1.mg.scc <- wartest(mgmod, 1, vcov=vcovSCC)
war1.dmg.scc <- wartest(dmgmod, 1, vcov=vcovSCC)
war1.cmg.scc <- wartest(cmgmod, 1, vcov=vcovSCC)
war1.cmgt.scc <- wartest(cmgmodt, 1, vcov=vcovSCC)

war2.mg.scc <- wartest(mgmod, 2, vcov=vcovSCC)
war2.dmg.scc <- wartest(dmgmod, 2, vcov=vcovSCC)
war2.cmg.scc <- wartest(cmgmod, 2, vcov=vcovSCC)
war2.cmgt.scc <- wartest(cmgmodt, 2, vcov=vcovSCC)



###################################################
### code chunk number 76: tab7quater
###################################################
tab7quater <- matrix(ncol=4, nrow=4)
dimnames(tab7quater) <- list(c("W AR(1), HC",
                               "W AR(2), HC",
                               "W AR(1), SCC",
                               "W AR(2), SCC"),
                             c("MG", "CDMG", "CMG", "CMGt"))

tab7quater[1, ] <- c(war1.mg[2,4], war1.dmg[2,4], war1.cmg[2,4],
                     war1.cmgt[2,4])
tab7quater[2, ] <- c(war2.mg[2,4], war2.dmg[2,4], war2.cmg[2,4],
                     war2.cmgt[2,4])
tab7quater[3, ] <- c(war1.mg.scc[2,4], war1.dmg.scc[2,4],
                     war1.cmg.scc[2,4],  war1.cmgt.scc[2,4])
tab7quater[4, ] <- c(war2.mg.scc[2,4], war2.dmg.scc[2,4],
                     war2.cmg.scc[2,4],  war2.cmgt.scc[2,4])

#xtable(tab7quater, digits=3, caption="Wooldridge's informal test for serial correlation; H0: no serial correlation at given order in untransformed errors.")


###################################################
### code chunk number 77: t7.rw
###################################################
library(splm)
rw.sec.mg <- rwtest(mgmod, w=secw, rep=999)
rw.sec.dmg <- rwtest(dmgmod, w=secw, rep=999)
rw.sec.cmg <- rwtest(cmgmod, w=secw, rep=999)
rw.sec.cmgt <- rwtest(cmgmodt, w=secw, rep=999)

rw.cnt.mg <- rwtest(mgmod, w=cntw, rep=999)
rw.cnt.dmg <- rwtest(dmgmod, w=cntw, rep=999)
rw.cnt.cmg <- rwtest(cmgmod, w=cntw, rep=999)
rw.cnt.cmgt <- rwtest(cmgmodt, w=cntw, rep=999)


###################################################
### code chunk number 78: tab7ter
###################################################
tab7ter <- matrix(ncol=4, nrow=12)
dimnames(tab7ter) <- list(c("Global CD", "   ", "avg. rho",
                            "avg. |rho|", "Frees rank test", "    ",
                            "CD(1), sector", "", "CD(1), country", " ",
                            "RW, sector", "RW, country"),
                          c("MG", "CDMG", "CMG", "CMGt"))

tab7ter[1, ] <- c(cd.mg$statistic, cd.dmg$statistic,
                  cd.cmg$statistic, cd.cmgt$statistic)
tab7ter[2, ] <- c(cd.mg$p.value, cd.dmg$p.value,
                  cd.cmg$p.value, cd.cmgt$p.value)
tab7ter[3, ] <- c(rho.mg$statistic, rho.dmg$statistic,
                  rho.cmg$statistic, rho.cmgt$statistic)
tab7ter[4, ] <- c(absrho.mg$statistic, absrho.dmg$statistic,
                  absrho.cmg$statistic, absrho.cmgt$statistic)
tab7ter[5, ] <- c(frees.mg$statistic, frees.dmg$statistic,
                  frees.cmg$statistic, frees.cmgt$statistic)
tab7ter[6, ] <- c(frees.mg$p.value, frees.dmg$p.value,
                  frees.cmg$p.value, frees.cmgt$p.value)
tab7ter[7, ] <- c(cd1.sec.mg$statistic, cd1.sec.dmg$statistic,
                  cd1.sec.cmg$statistic, cd1.sec.cmgt$statistic)
tab7ter[8, ] <- c(cd1.sec.mg$p.value, cd1.sec.dmg$p.value,
                  cd1.sec.cmg$p.value, cd1.sec.cmgt$p.value)
tab7ter[9, ] <- c(cd1.cnt.mg$statistic, cd1.cnt.dmg$statistic,
                  cd1.cnt.cmg$statistic, cd1.cnt.cmgt$statistic)
tab7ter[10, ] <- c(cd1.cnt.mg$p.value, cd1.cnt.dmg$p.value,
                  cd1.cnt.cmg$p.value, cd1.cnt.cmgt$p.value)
tab7ter[11, ] <- c(rw.sec.mg$p.value, rw.sec.dmg$p.value,
                  rw.sec.cmg$p.value, rw.sec.cmgt$p.value)
tab7ter[12, ] <- c(rw.cnt.mg$p.value, rw.cnt.dmg$p.value,
                  rw.cnt.cmg$p.value, rw.cnt.cmgt$p.value)
xtable(tab7ter, digits=3,
       caption="Comparison of cross section dependence diagnostics for the
  models in Table 7. Global CD test, $rho$, $|rho|$, Frees rank test, CD(1) sector and, respectively, country, and RW randomization test defined as in Table \ref{tab:xsd}.")


###################################################
### code chunk number 79: t8.mg
###################################################
d.mg <- pmg(dfm, data=ehs, model="mg", trend=TRUE)
#summary(d.mg)

## not in paper:
#d.mg0 <- pmg(dfm, data=ehs, model="mg", trend=FALSE)
#summary(d.mg0)
## this reproduces the version w/o trend one gets
## with 'xtmg lny lnl l1_lnl lnk l1_lnk lnrd l1_lnrd l1_lny, res(r_mg0)'



###################################################
### code chunk number 80: t8.dmg
###################################################
d.dmga <- pmg(dfm, gd.ehs, m="mg", trend=F)
#summary(d.dmga)
## this one is ok, differences after 5th dec.
d.dmgat <- pmg(dfm, gd.ehs, m="mg", trend=T)
#summary(d.dmgat)


###################################################
### code chunk number 81: t8.cmg
###################################################
## CCEs

## Reduce sample as per Table 8, REStat version
ehs0 <- data.frame(ehs)
ehs0 <- ehs0[!(ehs0$country %in% c("SWE","GER","PRT")), ]
ehs0 <- pdata.frame(ehs0, index=c("id","year"))

## dynamic CCEs **on restricted sample**
d.cmg <- pmg(dfm, data=ehs0, model="cmg", trend=FALSE)
#summary(d.cmg) ## ok, slight diffs only from 5th decimal


###################################################
### code chunk number 82: t8.cmgt
###################################################
d.cmgt <- pmg(dfm, data=ehs0, model="cmg", trend=TRUE)
#summary(d.cmgt) ## ok, idem

## alt: CCE via pcce()
d.cmga <- pcce(dfm, ehs0, m="m", trend=F)
#summary(d.cmga)
d.cmgat <- pcce(dfm, ehs0, m="m", trend=T)
#summary(d.cmgat)


###################################################
### code chunk number 83: t8.cdtest
###################################################

## CD test

cd.d.mg <- pcdtest(d.mg)
cd.d.dmg <- pcdtest(d.dmga)
cd.d.cmg <- pcdtest(d.cmg)
cd.d.cmgt <- pcdtest(d.cmgt)


###################################################
### code chunk number 84: t8.cips
###################################################

## CIPS (PCADF) test
cips.d.mg <- cipstest(resid(d.mg), lags=1, model="cmg", type="drift")
cips.d.dmg <- cipstest(resid(d.dmga), lags=1, model="cmg", type="drift")
cips.d.cmg <- cipstest(resid(d.cmg), lags=1, model="cmg", type="drift")
cips.d.cmgt <- cipstest(resid(d.cmgt), lags=1, model="cmg", type="drift")


###################################################
### code chunk number 85: t8.rmse
###################################################

rmse.d.mg <- sqrt(mean(resid(d.mg)^2))
rmse.d.dmg <- sqrt(mean(resid(d.dmga)^2))
rmse.d.cmg <- sqrt(mean(resid(d.cmg)^2))
rmse.d.cmgt <- sqrt(mean(resid(d.cmgt)^2))



###################################################
### code chunk number 86: t8.comfactest
###################################################
## COMFAC tests:
## NB the MGs have intercept: hence allow for it in k
comfac.d.mg <- comfactest(d.mg, k=8)
comfac.d.dmg <- comfactest(d.dmga, k=8)
comfac.d.cmg <- comfactest(d.cmga)
comfac.d.cmgt <- comfactest(d.cmgat)



###################################################
### code chunk number 87: tab8d
###################################################

tab8d <- matrix(ncol=4, nrow=7*2+4)
dimnames(tab8d) <- list(c("ln Yit-1", "", "ln Lit", " ", "ln Lit-1", "  ",
                         "ln Kit", "   ", "ln Kit-1", "    ",
                         "ln Rit", "     ","ln Rit-1", "      ",
                         "COMFAC", "CD", "CIPS", "RMSE"),
                         c("MG", "CDMG", "CMG", "CMGt"))

## populate coefs
tab8d[c(1,3,5,7,9,11,13), 1] <- coef(d.mg)[1+1:7] ## because of time d.
tab8d[c(1,3,5,7,9,11,13), 2] <- coef(d.dmga)[1+1:7]
tab8d[c(1,3,5,7,9,11,13), 3] <- coef(d.cmg)[1+1:7]
tab8d[c(1,3,5,7,9,11,13), 4] <- coef(d.cmgt)[1+1:7]

## SEs
tab8d[c(2,4,6,8,10,12,14), 1] <- coeftest(d.mg)[1+1:7,3]
tab8d[c(2,4,6,8,10,12,14), 2] <- coeftest(d.dmga)[1+1:7,3]
tab8d[c(2,4,6,8,10,12,14), 3] <- coeftest(d.cmg)[1+1:7,3]
tab8d[c(2,4,6,8,10,12,14), 4] <- coeftest(d.cmgt)[1+1:7,3]


## CD test
tab8d["CD", ] <- c(cd.d.mg$p.value, cd.d.dmg$p.value,
                  cd.d.cmg$p.value, cd.d.cmgt$p.value)

## CIPS test
tab8d["CIPS", ] <- c(cips.d.mg$p.value, cips.d.dmg$p.value,
                  cips.d.cmg$p.value, cips.d.cmgt$p.value)

## COMFAC test
tab8d["COMFAC", ] <- c(comfac.d.mg$test$p.value,
                       comfac.d.dmg$test$p.value,
                       comfac.d.cmg$test$p.value,
                       comfac.d.cmgt$test$p.value)
## RMSE
tab8d["RMSE", ] <- c(rmse.d.mg, rmse.d.dmg, rmse.d.cmg,
                     rmse.d.cmgt)

xtable(tab8d, digits=3, caption="Dynamic heterogeneous models.")


###################################################
### code chunk number 88: t8.crstests.r
###################################################

## CRS tests: (must use coefs and vcovs from long-run models)
## long-run  models, use unrestricted b and vcov:

crs.d.mg <- crstest(d.mg, k=8)  # because coef has 'trend' at the end
crs.d.dmg <- crstest(d.dmga, k=8) # idem
crs.d.cmg <- crstest(d.cmg, k=8) # has XS averages
crs.d.cmgt <- crstest(d.cmgt, k=8) # has trend and XS averages



###################################################
### code chunk number 89: t8.crstests.u
###################################################

## CRS tests in papers are actually those for UNrestricted models

ucrs.d.mg <- crstest(d.mg, k=8, lrvcov=lrvcov)
ucrs.d.dmg <- crstest(d.dmga, k=8, lrvcov=lrvcov)
ucrs.d.cmg <- crstest(d.cmg, k=8, lrvcov=lrvcov)
ucrs.d.cmgt <- crstest(d.cmgt, k=8, lrvcov=lrvcov)



###################################################
### code chunk number 90: reducedw
###################################################
## make proximity matrices in land and sector for ehs0
## for use with reduced sample in CMG, CMGt
ids0 <- unique(ehs0$id)
n0 <- length(ids0)
cntw0 <- matrix(0, ncol=n0, nrow=n0)
secw0 <- matrix(0, ncol=n0, nrow=n0)

## make cntw
for(i in 1:n0) {
  ## all equal, so take the first
  icnt <- ehs0[ehs0$id==ids0[i], "country"][1]
  for(j in 1:n0) {
    jcnt <- ehs0[ehs0$id==ids0[j], "country"][1]

    if(icnt==jcnt) cntw0[i,j] <- 1
  }
}
## eliminate self-neighbours
diag(cntw0) <- 0
rownames(cntw0) <- colnames(cntw0) <- ids0

## make secw
for(i in 1:n0) {
  ## all equal, so take the first
  icnt <- ehs0[ehs0$id==ids0[i], "sector"][1]
  for(j in 1:n0) {
    jcnt <- ehs0[ehs0$id==ids0[j], "sector"][1]

    if(icnt==jcnt) secw0[i,j] <- 1
  }
}
## eliminate self-neighbours
diag(secw0) <- 0
rownames(secw0) <- colnames(secw0) <- ids0


###################################################
### code chunk number 91: t8.cdp
###################################################

## assessment of the two forms of "local" dependence
cd1.sec.d.mg <- pcdtest(d.mg, w=secw)
cd1.sec.d.dmg <- pcdtest(d.dmga, w=secw)
cd1.sec.d.cmg <- pcdtest(d.cmg, w=secw0)
cd1.sec.d.cmgt <- pcdtest(d.cmgt, w=secw0)

cd1.cnt.d.mg <- pcdtest(d.mg, w=cntw)
cd1.cnt.d.dmg <- pcdtest(d.dmga, w=cntw)
cd1.cnt.d.cmg <- pcdtest(d.cmg, w=cntw0)
cd1.cnt.d.cmgt <- pcdtest(d.cmgt, w=cntw0)


###################################################
### code chunk number 92: tab8bis
###################################################
tab8bis <- matrix(ncol=4, nrow=6)
dimnames(tab8bis) <- list(c("Global CD", "   ",
                            "CD(1), sector", "", "CD(1), country", " "),
                       c("MG", "CDMG", "CMG", "CMGt"))

tab8bis[1, ] <- c(cd.d.mg$statistic, cd.d.dmg$statistic,
                  cd.d.cmg$statistic, cd.d.cmgt$statistic)
tab8bis[2, ] <- c(cd.d.mg$p.value, cd.d.dmg$p.value,
                  cd.d.cmg$p.value, cd.d.cmgt$p.value)
tab8bis[3, ] <- c(cd1.sec.d.mg$statistic, cd1.sec.d.dmg$statistic,
                  cd1.sec.d.cmg$statistic, cd1.sec.d.cmgt$statistic)
tab8bis[4, ] <- c(cd1.sec.d.mg$p.value, cd1.sec.d.dmg$p.value,
                  cd1.sec.d.cmg$p.value, cd1.sec.d.cmgt$p.value)
tab8bis[5, ] <- c(cd1.cnt.d.mg$statistic, cd1.cnt.d.dmg$statistic,
                  cd1.cnt.d.cmg$statistic, cd1.cnt.d.cmgt$statistic)
tab8bis[6, ] <- c(cd1.cnt.d.mg$p.value, cd1.cnt.d.dmg$p.value,
                  cd1.cnt.d.cmg$p.value, cd1.cnt.d.cmgt$p.value)


###################################################
### code chunk number 93: tab8B
###################################################

tab8B <- matrix(ncol=4, nrow=3*2+1)
dimnames(tab8B) <- list(c("ln Lit", "", "ln Kit", " ", "ln Rit", "  ",
                         "CRS"),
                       c("MG", "CDMG", "CMG", "CMGt"))

## populate coefs
tab8B[c(1,3,5), 1] <- comfac.d.mg$model[1:3, 1]
tab8B[c(1,3,5), 2] <- comfac.d.dmg$model[1:3, 1]
tab8B[c(1,3,5), 3] <- comfac.d.cmg$model[1:3, 1]
tab8B[c(1,3,5), 4] <- comfac.d.cmgt$model[1:3, 1]

## SEs
tab8B[c(2,4,6), 1] <- comfac.d.mg$model[1:3, 3]
tab8B[c(2,4,6), 2] <- comfac.d.dmg$model[1:3, 3]
tab8B[c(2,4,6), 3] <- comfac.d.cmg$model[1:3, 3]
tab8B[c(2,4,6), 4] <- comfac.d.cmgt$model[1:3, 3]

## CRS test
tab8B["CRS", ] <- c(crs.d.mg$p.value, crs.d.dmg$p.value,
                    crs.d.cmg$p.value, crs.d.cmgt$p.value)

xtable(tab8B, digits=3, caption="Dynamic heterogeneous models, long run solutions imposing COMFAC restrictions (Table 8 in EHS).")


###################################################
### code chunk number 94: t8.restrictedlr
###################################################
## Unrestricted long-run coefs for all models:

# not used

#lr.d.mg <- lrcoef(d.mg, k=8)
#lr.d.dmg <- lrcoef(d.dmga, k=8)
#lr.d.cmg <- lrcoef(d.cmg, k=8)
#lr.d.cmgt <- lrcoef(d.cmgt, k=8)


###################################################
### code chunk number 95: t8.frees
###################################################
## Frees rank-based test
frees.d.mg <- pcdtest3(resid(d.mg), test="frees", exact=FALSE)
frees.d.dmg <- pcdtest3(resid(d.dmga), test="frees", exact=FALSE)
frees.d.cmg <- pcdtest3(resid(d.cmg), test="frees", exact=FALSE)
frees.d.cmgt <- pcdtest3(resid(d.cmgt), test="frees", exact=FALSE)

## relative and absolute rho:
rho.d.mg <- pcdtest(d.mg, test="rho")
rho.d.dmg <- pcdtest(d.dmga, test="rho")
rho.d.cmg <- pcdtest(d.cmg, test="rho")
rho.d.cmgt <- pcdtest(d.cmgt, test="rho")

absrho.d.mg <- pcdtest(d.mg, test="absrho")
absrho.d.dmg <- pcdtest(d.dmga, test="absrho")
absrho.d.cmg <- pcdtest(d.cmg, test="absrho")
absrho.d.cmgt <- pcdtest(d.cmgt, test="absrho")


###################################################
### code chunk number 96: t8.rw
###################################################
library(splm)
rw.sec.d.mg <- rwtest(d.mg, w=secw, rep=999)
rw.sec.d.dmg <- rwtest(d.dmga, w=secw, rep=999)
rw.sec.d.cmg <- rwtest(d.cmg, w=secw0, rep=999)
rw.sec.d.cmgt <- rwtest(d.cmgt, w=secw0, rep=999)

rw.cnt.d.mg <- rwtest(d.mg, w=cntw, rep=999)
rw.cnt.d.dmg <- rwtest(d.dmga, w=cntw, rep=999)
rw.cnt.d.cmg <- rwtest(d.cmg, w=cntw0, rep=999)
rw.cnt.d.cmgt <- rwtest(d.cmgt, w=cntw0, rep=999)


###################################################
### code chunk number 97: tab8ter
###################################################
tab8ter <- matrix(ncol=4, nrow=12)
dimnames(tab8ter) <- list(c("Global CD", "   ", "avg. rho",
                            "avg. |rho|", "Frees rank test", "    ",
                            "CD(1), sector", "", "CD(1), country", " ",
                            "RW, sector", "RW, country"),
                          c("MG", "CDMG", "CMG", "CMGt"))

tab8ter[1, ] <- c(cd.d.mg$statistic, cd.d.dmg$statistic,
                  cd.d.cmg$statistic, cd.d.cmgt$statistic)
tab8ter[2, ] <- c(cd.d.mg$p.value, cd.d.dmg$p.value,
                  cd.d.cmg$p.value, cd.d.cmgt$p.value)
tab8ter[3, ] <- c(rho.d.mg$statistic, rho.d.dmg$statistic,
                  rho.d.cmg$statistic, rho.d.cmgt$statistic)
tab8ter[4, ] <- c(absrho.d.mg$statistic, absrho.d.dmg$statistic,
                  absrho.d.cmg$statistic, absrho.d.cmgt$statistic)
tab8ter[5, ] <- c(frees.d.mg$statistic, frees.d.dmg$statistic,
                  frees.d.cmg$statistic, frees.d.cmgt$statistic)
tab8ter[6, ] <- c(frees.d.mg$p.value, frees.d.dmg$p.value,
                  frees.d.cmg$p.value, frees.d.cmgt$p.value)
tab8ter[7, ] <- c(cd1.sec.d.mg$statistic, cd1.sec.d.dmg$statistic,
                  cd1.sec.d.cmg$statistic, cd1.sec.d.cmgt$statistic)
tab8ter[8, ] <- c(cd1.sec.d.mg$p.value, cd1.sec.d.dmg$p.value,
                  cd1.sec.d.cmg$p.value, cd1.sec.d.cmgt$p.value)
tab8ter[9, ] <- c(cd1.cnt.d.mg$statistic, cd1.cnt.d.dmg$statistic,
                  cd1.cnt.d.cmg$statistic, cd1.cnt.d.cmgt$statistic)
tab8ter[10, ] <- c(cd1.cnt.d.mg$p.value, cd1.cnt.d.dmg$p.value,
                  cd1.cnt.d.cmg$p.value, cd1.cnt.d.cmgt$p.value)
tab8ter[11, ] <- c(rw.sec.d.mg$p.value, rw.sec.d.dmg$p.value,
                  rw.sec.d.cmg$p.value, rw.sec.d.cmgt$p.value)
tab8ter[12, ] <- c(rw.cnt.d.mg$p.value, rw.cnt.d.dmg$p.value,
                  rw.cnt.d.cmg$p.value, rw.cnt.d.cmgt$p.value)

xtable(tab8ter, digits=3,
       caption="Comparison of cross section dependence diagnostics for the models in Table 8. Diagnostics defined as in earlier Tables.")


