%% ML_SmokingGun_ReplicationFile3 - Reproduces Figures 2-5
% 
% Replication File for the paper 
% Monetary Policy and the Housing Market: A Structural Factor Analysis
% by Matteo Luciani (matteo.luciani@ulb.ac.be)
% ECARES Working Paper 2012-035 
% To appear on the Journal of Applied Econometrics
%

clear all; close all; clc
ML_graph_options                                                            % Default Graphs Options
TR=1;                                                                       % Data transformation
out=1;                                                                      % Elimination of Outliers
[db, label,cd] = ML_SmokingGun_Data(TR,out);                                % Upload Data
name=ML_SmokingGunNames;                                                    % Name for variables
[DB]= ML_SmokingGun_Data(0,0); TT=1963:.25:2010.75;                         % Raw Data
T=size(db,1); t0=1963+.25*TR; tt=t0:0.25:t0+(T-1)*0.25;                     % useful objects
tb=find(tt==1982.5); te=find(tt==2008); tg=find(tt==2002.25);               % Extremes to truncate the sample
db(1:tb,:)=[];   [T N]=size(db); tt=tt(tb)+.25:0.25:tt(tb)+T*0.25;          % I consider only the second half of the sample

% ___________________________ %
% Setting-up Model Parameters %
%  %
[y, My, Sy]=ML_Standardize(db);                                             % Standardize Data
yc=db-repmat(My,T,1);                                                       % Compute demeaned variables Useful for Historical Decomposition
r=7; p=1; q=4; s=21;                                                        % Parameter of the factor model
draws=1000000;                                                              % number of draws for the rotation matrix
boot=500;                                                                   % bootstrap draws
maxrot=25;                                                                  % maximum number of rotations
FP=0;                                                                       % Set wether to use the Fry Pagan modification for sign restrictions
nrest=ones(q,1)*68;                                                         % Normalization Restriction
h=4;                                                                        % Number of lags on which sign restritions are imposed
up=84; down=100-up;
for kk=1:h; rest{kk}=[[31 7 23 108:109]' -ones(5,1);68 1]; end        % Sign Restrictions - Real Effects Scenario

disp('  ______________________ ')
disp('  Sample Characteristics ')
disp('   ')
fprintf('   Transformation: '); if TR==0; disp('None'); elseif TR==1; disp('Light'); else disp('Hard'); end
fprintf('   Outliers: ');       if out==0; disp('None'); else disp('Removed'); end
fprintf('   Sample: ');         disp([ML_TimeLabel(tt(1)) '-'  ML_TimeLabel(tt(end))]);
disp(['   Static Factors: ' num2str(r)]);
disp(['   Dynamic Factors: ' num2str(q)]);
disp(['   Number of Lags in the VAR of the Factors: ' num2str(p)]);
disp(['   Number of lags on which sign restritions are imposed: ' num2str(h)]);
fprintf('   Fry & Pagan: '); if FP==0; disp('No'); else disp('Yes'); end
disp('   Variables on which sign restrictions are imposed:'); 
disp(['   Confidence band: ' num2str(up-down) '%']);
for kk=1:size(rest{1},1); disp(['    - ' name{rest{1}(kk,1)}]); end;

disp('  _______________________  ')
disp('    ')
disp('  Factor Model Estimation  ')
disp('  _______________________  ')
disp('    ')
[~,Dimp,sh,~,R,~,F,L,A,G]=ML_SmokingGun_DfmSignImp(db,cd,q,r,p,s,nrest,rest,draws,h,500,FP);    % Impulse responses whole sample
chi=(F*L').*repmat(Sy,T,1); xsi=y.*repmat(Sy,T,1)-chi;                                          % The Common and the Idiosyncratic Component
share=var(chi)./var(db);                                                                        % Percentage of variance accounted for the common component
gsr=size(Dimp,4);                                                                               % Number of good rotations
[~, xhat]=ML_decompose(Dimp, sh);                                                               % Historical Decomposition
idvar=[7 68 31 35 55 56]; nv=size(idvar,2);                                                     % Variables to be plotted


disp('  _____________________________________________  ')
disp('    ')
disp('  Counterfactual Analisys: Taylor vs. Greenspan  ')
disp('  _____________________________________________  ')
disp('    ')

a=find(tt==2002.25); b=find(tt==2006.25);                                   % Identifies the period of Taylor MP
load taylor2; taylor=taylor2; clear taylor2                                 % load the FFR suggested by Taylor (1997)
if TR==1; yy=taylor(79+1:end); else yy=ML_diff(taylor(79:end)); end         % Cut the observations prior to 1982
yyc=yy-My(68); yy01=yyc/Sy(68);                                             % compute the demean and standardize version
sh2=sh; xhat3=xhat; sh3=sh;                                                 % New shocks
mistake=repmat(yc(2:end,68)-xsi(2:end,68),1,500)-squeeze(sum(xhat(:,68,:,:),3));
disp('Exercise 1: No Greenspan Expansionary Policy')                                                         % 
clear xhat2
for ii=a-6:b-1;
    for jj=1:gsr;
        sh2(ii,1,jj)=0; 
    end;
end
disp('Exercise 2: Taylor Policy')
for ii=1:gsr;disp(gsr-ii);
    for ttx=a-1:b;
        temp1=yyc(ttx+1)-(xsi(ttx+1,68)+sum(xhat3(ttx,68,:,ii),3))-mistake(ttx,ii);
        sh3(ttx,1,ii)=sh3(ttx,1,ii)+temp1/Dimp(68,1,1,ii);                  % The new shocks necessary to replicate the JT FFR
        xhat3(:,:,:,ii)=ML_decompose(Dimp(:,:,:,ii),sh3(:,:,ii));           % the new historical decomposition
    end
end

clear xhat2 xhat3 mistake;

[~, xhat2]=ML_decompose(Dimp,sh2);                                      % The No Greenspa Historical Contribution
[~, xhat3]=ML_decompose(Dimp,sh3);                                      % The Taylor Historical Contribution


%%% _____________________ %%%
%%% Counterfactual Graphs %%%
%%%  %%%
nines=find(tt==1990);                                   % Identifies the Nineties
ombra=find((DB(:,68)-taylor) ~= 0);                     % Identifies where the taylor FFR is different from actual ffr
start=find(tt==2000);   

%% Figure 2. Monetary Policy Shocks %%
for kk=[1 3];
    axes('Parent',figure,'FontSize',12); ML_FigureSize
    if kk==1;       shplot=sh(nines-1:end,1,:); titolo='Estimated';% Greenspan Monetary Policy Shocks
    elseif kk==2;   shplot=sh2(nines-1:end,1,:);titolo='No-Greenspan';% No Greenspan Monetary Policy Shocks
    elseif kk==3;   shplot=sh3(nines-1:end,1,:);titolo='Taylor Couterfactual Exercise';% Taylor Monetary Policy Shocks
    end
    hold on
    plot(tt(nines:end),zeros(size(tt(nines:end))),'k','linewidth',.5);
    ha=area(tt(nines:end),[prctile(shplot,down,3)...
        prctile(shplot,up,3)-prctile(shplot,down,3)],'linestyle','none');   % area plot
    set(ha(1), 'FaceColor', 'none'); set(ha(2), 'FaceColor', [0.7 0.7 0.7]) % modifies the shaded area
    plot(tt(nines:end),prctile(shplot,50,3),'k','linewidth',1);           % Plotting Shocks
    title(titolo,'FontAngle','italic','fontsize',14);                  
    axis tight;hold off;
end


%% Figure 3. Federal Funds Rate %%
ii=68; axes('Parent',figure,'FontSize',10); ML_FigureSize; hold on;
storia=squeeze(xhat2(nines-1:end,ii,1,:))-squeeze(xhat(nines-1:end,ii,1,:));
dbhatu=db(nines:end,ii)+prctile(storia,up,2);   
dbhatd=db(nines:end,ii)+prctile(storia,down,2);   
dbhatm=db(nines:end,ii)+prctile(storia,50,2);   
ha=area(tt(nines:end),[dbhatu dbhatd-dbhatu],'linestyle','none');           % area plot
set(ha(1), 'FaceColor', 'none'); set(ha(2), 'FaceColor', [0.7 0.7 0.7])     % modifies the shaded area
plot(tt(nines:end),yy(nines:end),'k:','linewidth',1)                        % Taylor
plot(tt(nines:end),dbhatm,'k','linewidth',1)                                % The Conterfactual
plot(tt(nines:end),db(nines:end,ii),'k','Marker','.','linewidth',1.5)       % the variable
hold off; axis tight; 
title(name{ii},'FontAngle','italic','fontsize',14);                         % Name of the Variable

%% Figure 4. Counterfactuals Based on Structural Shocks: No Monetary Policy Shocks Exercise %%
%% Figure 5. Counterfactuals Based on Structural Shocks: Taylor Conterfactual Exercise %%
ii=[31 35 55 56];                                                                                   % Variables of interests
for kk=1:2;
    axes('Parent',figure,'FontSize',12); ML_FigureSize;
    for jj=1:4;    
        if kk==1;
            storia=squeeze(xhat2(start-1:end,ii(jj),1,:))-squeeze(xhat(start-1:end,ii(jj),1,:));    % Eliminating MP Shocks
        else
            storia=squeeze(xhat3(start-1:end,ii(jj),1,:))-squeeze(xhat(start-1:end,ii(jj),1,:));    % Taylor Scenario
        end
        dbhatu=db(start:end,ii(jj))+prctile(storia,up,2);                                           % Percentiles of the counterfactual
        dbhatd=db(start:end,ii(jj))+prctile(storia,down,2);                                         % -------------------------------
        dbhatm=db(start:end,ii(jj))+prctile(storia,50,2);                                           % -------------------------------
        subplot(2,2,jj);hold on;                                                                    % Subplotting
        plot(tt(start:end),zeros(size(tt(start:end))),'k','linewidth',.5);                          % Zero Line
        ha=area(tt(start:end),[dbhatu dbhatd-dbhatu],'linestyle','none');                           % Confidence Bands
        set(ha(1), 'FaceColor', 'none'); set(ha(2), 'FaceColor', [0.7 0.7 0.7])                     % modifies the shaded area
        plot(tt(start:end),db(start:end,ii(jj)),'k','linewidth',2);                                 % The true variable
        plot(tt(start:end),dbhatm,'k','linewidth',1);                                               % The Conterfactual
        ylabel('Growth Rate','FontAngle','italic','fontsize',8);                                    % ylabel 
        title(name{ii(jj)},'FontAngle','italic','fontsize',10); hold off; axis tight;               % Name of the Variable         
    end
end


