% ML_SmokingGun_DfmSignImp - SDFM Estimation with Sign Restrictions (Special Version)
%
% [Imp,Dimp,sh,imp,R] = DfmSignImp(X,cd,q,r,k,h,nrest,rest,ndraws,horiz)
%     Impp - Cumulate IRF
%     Dimp - IRF
%       sh - shocks
%      imp - all impulse that sitisfies the sign restrictions
%        R - Rotation Matrix
% 
%        X - Data                                                           (1)
%       cd - order of differentiation of the variables                      (2)
%        q - number of dynamic factors                                      (3)
%        r - number of static factors                                       (4)
%        k - number of lags in the VAR for the static factors               (5)
%        h - number of lags of the Impulse Responses                        (7)
%    nrest - normalization restrictions                                     (8)
%     rest - cell containing the sign restrictions                          (9)
%   ndraws - number of draws for the bootstrap algorithm                    (10)
%    horiz - horizon in which sign restrictions are imposed                 (11) optional
%       FP - if 1 uses the Fry & Pagan Modification                         (12) optional
% 

% Written by Mario Forni
% Modified and commented by Matteo Luciani (matteo.luciani@ulb.ac.be)

function [Imp,Dimp,sh,imp,R,rsh,F,L,A,G] = ML_SmokingGun_DfmSignImp(X,cd,q,r,k,h,nrest,rest,ndraws,horiz,maxrot,FP)

tic
if nargin<12; FP=0; end;
if nargin<11; maxrot=ndraws; end;

WW = diag(std(X)); WWW=repmat(diag(WW),1,q);                                % Standard deviation
[F, L, A, rsh, G] = ML_SmokingGun_efactorsML(X,q,r,k,100,3);                % ML Estimation of the DFM
BB = ML_MA(h,A,0);                                                          % MA Representation of the static factors    
for ii=1:h; B(:,:,ii)=(L*BB(:,:,ii)*G).*WWW; end;                           % MA Representation of common components    
C = CumImp(B, cd);                                                          % comulate IRFs whether it is necessary
GR = ML_GoodPolar(C, nrest, rest, ndraws, horiz,maxrot);                    % look for admissible rotations
[Imp, sh] = ML_ComputeIrf(C,GR,rsh);                                        % computes structural IRF & shocks for all admissible rotation
Dimp = ML_ComputeIrf(B,GR);                                                 % computes structural IRF & shocks for all admissible rotation


ngr=size(Imp,4);
if ngr==0;
    imp=[]; R=[];
else    
    imp=Imp; R=GR;
    if FP==1;                                                               % In case the Fry Pagan Correction is Required
        BR=ML_SignRestrictionFryPagan(Imp,rest);                            % Identify the best rotation in the sense of Fry & Pagan           
        Imp=Imp(:,:,:,BR); Dimp=Dimp(:,:,:,BR); sh = sh(:,:,BR);            % Keep Only the FP rotation
        R=GR(:,:,BR);                                                       % Rotation Matrix
    end 
end

disp([ngr toc]) 

