########################################################################################
##         Main program for Laroque-Salanie,                       #####################
##           Identifying the Response of Fertility to Financial Incentives #############
##          (Journal of Applied Econometrics)                      #####################
##################   THIS IS AN R + MPI PROGRAM:                   #####################
############### IT REQUIRES R, WITH PACKAGES RMPI AND SNOW         #####################
########################################################################################


## start with a clean slate
rm(list=ls())

## load Snow and Rmpi
library(Rmpi)
library(snow)

## include auxiliary routines
source("auxilRoutines.R")

## include routines using Rmpi/Snow
source("mpiRoutines.R")

debug <- F
if (debug) {
## avoid /dev/null redirection
setDefaultClusterOptions(outfile="")
}

## split over processes
nprocs <- mpi.universe.size()-1
nprocs <- 8

## start the processes
cl <- startsnow(nprocs)
## check all went well
print(clusterInfo(cl,c("nodename","machine")))

pstarline()
cat("\nWorking with ", nprocs," processes.\n")
pstarline()


library(stats,MASS)


########################################################################################
####         WE DEFINE VARIOUS OPTIONS                       ###########################
########################################################################################

## we use m quantiles of N(0,1) for integration
m <- 50
vqc1 <- (1:(m-1))/m
qc1 <- qnorm(vqc1)
phiqc1 <- dnorm(qc1)
epsbar <- phiqc1[1:(m-2)]-phiqc1[2:(m-1)]
epsbar <- c(-phiqc1[1],epsbar,phiqc1[m-1])*m


## select one observation in nsel, if nsel > 1 (useful for testing)
nsel <- 1

## we take initial values here
initVals <- "initialVals.RData"

## we write  results here
estfile <- "estResults.RData"

## format for results
options(digits=4,width=70)

## do we check the gradient before maximizing the likelihood?
checkgrad <- F
epsg <- 0.0001

## algorithm prints more (larger integer) to nothing (0) during maximization
plev <- 1

########################################################################################
####         WE READ THE DATA                                ###########################
########################################################################################

load("LaroqueSalanie.RData")
nobs <- NROW(df2)

## select smaller sample if nsel > 1
if (nsel > 1) {
  df2 <- df2[seq(1,nobs,nsel),]
  nobs <- NROW(df2)
}

attach(df2)

## save initial income values  (needed for simulation purposes)
r01base <- r01   ## R_01, with L=0 and F=1
r11base <- r11   ## R_11, with L=1 and F=1 so a matrix of 10 simulated values
r11ebase <- r11e ## same for employees, for whom we know the observed value

pstarline(2)
cat("We have ", nobs, " observations",  "\n") 
pstarline()



################################################################
##      WE DEFINE THE KNOTS FOR THE VARIOUS SPLINES OF AGE    ##
################################################################

## knots for the splines on  labor supply
agdls <- quantile(agd,c(.1,.25,.5,.75,.9))

## knots for the splines on fertility
agdsplp1 <- quantile(agd[parity==1 & fert==1],c(.15,.5,.85))
agdsplp2 <- quantile(agd[parity==2 & fert==1],c(.15,.5,.85))
agdsplp3 <- quantile(agd[parity==3 & fert==1],c(.15,.5,.85))

## knots for the splines on ssmf and ssfm (genders of the first two children) interacted with age
agdss <- agdsplp3


################################################################
###        SPECIFICATION                                      ##
################################################################

## number of parameters to be estimated
## for labor supply
npls <- 32
## their names
nparsls <- c("dspl10",   "dspl25",     "dspl50",     "dspl75",     "dspl90",
             "an98",   "an99",
             "dip1",   "dip3",   "dip4",   "de1a3",  "fn6",   
             "par2",   "par3",    "notmar",
             "c1","dipsup", "par3", "notmar","agd","agd2", 
             "tspl10",   "tspl25",     "tspl50",     "tspl75",     "tspl90",
             "dipsup", "par2", "par3",
             "one",    "agd",    "par3")  
## for fertility
npfert <- 85
## their names
nparsfert <- c("p1vdspl1", "p1vdspl2",  "p1vdspl3", 
               "p1v0spl1", "p1v0spl2",  "p1v0spl3",
               "p1Ctspl1", "p1Ctspl2",  "p1Ctspl3",   "p1vddips",
               "p1v0dips", 
               "p2vdspl1", "p2vdspl2",  "p2vdspl3",
               "p2v0spl1", "p2v0spl2",  "p2v0spl3", 
               "p2Ctspl1", "p2Ctspl2",  "p2Ctspl3",
               "p2vdzag1", "p2vdzag2",  "p2vdzag3", "p2vddips",  
               "p2v0zag1", "p2v0zag2",  "p2v0zag3", "p2v0dips",  
               "p3vdspl1", "p3vdspl2",  "p3vdspl3",
               "p3v0spl1", "p3v0spl2",  "p3v0spl3", 
               "p3Ctspl1", "p3Ctspl2",  "p3Ctspl3",
               "p3vdzag1", "p3vdzag2",  "p3vdzag3", "p3vddips",  
               "p3v0zag1", "p3v0zag2",  "p3v0zag3", "p3v0dips",  
               "p1notmarr", "p1dipsup", "p1agdnm", "p1agds", 
               "p2notmarr", "p2dipsup", "p2agdnm", "p2agds",
               "p2zag1",  "p2zag2",  "p2zag3",  "p2zag4",  "p2zag5", "p2zag6",
               "p3notmarr", "p3dipsup", "p3agdnm", "p3agds",
               "p3zag1",  "p3zag2",  "p3zag3",  "p3zag4",  "p3zag5", "p3zag6",
               "p3f7to10","resgen0","resgen1","resgen2",
               "ssmf1","ssmf2","ssmf3",
               "ssfm1","ssfm2","ssfm3",
               "bigc00","bigc10","bigc01","bigc11","bigc02","bigc12")
## total
nparams <- npls+npfert
## with names
npars <- vector("character", nparams)
npars[1:npls] <- nparsls
npars[(npls+1):nparams] <- nparsfert


################################################################
## labor supply conditional on fertility                      ##
################################################################
##    we define the matrices of covariates used
## vector of ones
c1 <- matrix(1,nobs)  
## age splines have 5 coeffs to be added to xdc and xtc
## D_0 and D_1
xdc <- cbind(an98,an99,dip1,dip3,dip4,de1a3,fn6,
             parity2,parity3,notmarr)              
## D_1 only
xd1 <- cbind(c1,dipsup,parity3,notmarr,agd,agd2) 
## \tau_0 and \tau_1
xtc <- cbind(dipsup,parity2,parity3)               
## \tau_1 only
xt1 <- cbind(c1,agd,parity3)                     


################################################################################
##  load routines that evaluate the likelihood and run the simulations #########
################################################################################
## first tell MPI cluster a few things:

## set up data matrix for MPI routines
setupMPIdata <- function() {
  ## returns
  cbind(xdc,xd1,xtc,xt1,agd,dipsup,
        notmarr,agdnm,agds,resgen,
        zag1,zag2,zag3,zag4,zag5,zag6,
        ssmf,ssfm,f7to10,parity1,parity2,parity3,
        r00,r01,r10e,r11e,r10,r11)
}

dataMPI0 <- setupMPIdata()

## the names of functions it needs
informNodes(cl,dlsf)
informNodes(cl,taulsf)
informNodes(cl,xthls)
informNodes(cl,bigh)
informNodes(cl,bigv)
informNodes(cl,argfert)
informNodes(cl,xthfert)
informNodes(cl,pfert)
informNodes(cl,jfunc)
informNodes(cl,qfuncpara)
informNodes(cl,testpos)
informNodes(cl,fsplval)

## the names of variables it needs
informNodes(cl,nparams)
informNodes(cl,epsbar)
informNodes(cl,npfert)
informNodes(cl,npls)
informNodes(cl,m)
informNodes(cl,agdsplp1)
informNodes(cl,agdsplp2)
informNodes(cl,agdsplp3)
informNodes(cl,agdss)
informNodes(cl,agdls)
informNodes(cl,r00)
informNodes(cl,r01)
informNodes(cl,debug)


#############################################
## read initial values for parameters      ##
#############################################

load(initVals)

pinit <- resest[[2]]

loglik <- 0
varcov <- 0



## check gradient
if (checkgrad) {
  cat("Checking gradient: \n")
  dd <- checkgradscal(loglikfun,llgrad,pinit)
}


#############################################
## maximize likelihood
#############################################
maxmod <- optim(pinit, loglikfun, llgrad, method = "BFGS",
                control=list(trace=plev,maxit=100000,
                  REPORT=1,
                  reltol=1e-9))
## check convergence: code should be 0
cat("Convergence code", maxmod$convergence, "\n")

## estimated parameters
estpars <- maxmod$par
## max log(L)
loglik <- -maxmod$value
## vector of scores
fgl <- llder(estpars,T)
g <- fgl[[2]]
## I matrix
bigi <- (t(g) %*% g)/nobs
## invert it with Cholesky
cbigi <- chol(bigi)
## variance-covariance matrix of estimates
varcov <- chol2inv(cbigi)/nobs
stderr <- sqrt(diag(varcov))
## store estimation results
resest <- data.frame(npars,estpars,stderr)
names(resest) <- c("Coeffs", "Estim", "Stderr")
cat("Value of total log(L) is ", loglik, "\n")
cat("Coefficient estimates: \n")
print(resest)


################################################################
##     SIMULATIONS OF ESTIMATED MODEL                 ##########
################################################################

############################################
#### BASE SIMULATION #######################
############################################

## set up data matrix for multicore routine
dataMPI0 <- setupMPIdata()

vpfsim <- plsfertsimpara(estpars,ee)

cat("\n\n Summary of base probabilities:\n")
print(summary(vpfsim))


printSimResults <- function(vpf) {
  cat("   Summary of change in probabilities for parity 1:\n")
  print(summary((vpf-vpfsim)[parity==1,]))
  cat("   Summary of change in probabilities for parity 2:\n")
  print(summary((vpf-vpfsim)[parity==2,]))
  cat("   Summary of change in probabilities for parity 3:\n")
  print(summary((vpf-vpfsim)[parity==3,]))   
}

##########################################
## UNCONDITIONAL CHILD CREDIT: ###########
## 150 euros/month, all parities  ########
##########################################

eurorate <- 6.55957
chcred <- .15*eurorate
r01 <- r01base+chcred
r11 <- r11base+chcred
r11e <- r11ebase+chcred

## set up data matrix for multicore routine
dataMPI0 <- setupMPIdata()

vpfsimUCC <- plsfertsimpara(estpars,ee)
cat("\n\n Unconditional child credit:\n\n")
printSimResults(vpfsimUCC)

###################################################
## EMPLOYMENT-TESTED CHILD CREDIT, SAME VALUE  ####
###################################################

r01 <- r01base+chcred
r11 <- r11base
r11e <- r11ebase

## set up data matrix for multicore routine
dataMPI0 <- setupMPIdata()

vpfsimETCC <- plsfertsimpara(estpars,ee)
cat("\n\n Employment-tested child credit:\n\n")
printSimResults(vpfsimETCC)

##################################################################
##  ADDITIONAL SIMULATION USED TO COMPUTE THE STANDARD ERRORS  ###
##      OF THE EFFECTS OF THE CHILD CREDITS ON FERTILITY #########
##    we compute the derivative of the probaility of a birth   ###
##           wrt the parameters                           ########
##################################################################

## base simulation
r01 <- r01base
r11 <- r11base
r11e <- r11ebase

## set up data matrix for multicore routine
dataMPI0 <- setupMPIdata()

gr <- T
pfsim <- qfuncpara(estpars,ee,gr)


## change incomes for marginal simulation
## try with 150 euros
eurorate <- 6.55957

d <- eurorate*0.15
r01 <- r01base+d
r11 <- r11base+d
r11e <- r11ebase+d

## set up data matrix for multicore routine
dataMPI0 <- setupMPIdata()

gr <- T
pfsimd <- qfuncpara(estpars,ee,gr)

## derivative of P(F=1) wrt (V1-V0)
dpfv <- (pfsimd[[1]]-pfsim[[1]])/d

## cross-derivative of P(F=1) wrt (V1-V0)*params
dpfvder <- (pfsimd[[2]]-pfsim[[2]])/d

## save estimation and simulation results
save(loglik,resest, varcov,
     dls0, dls1, tauls0, tauls1,
     vpfsim,vpfsimntat,vpfsimetcc, dpfv, dpfvder,agd,parity,
     file=estfile)







