
********************************************************************************
/*	 
	 Revisiting Gender Identity and Relative Income within Households –
	A cautionary tale on the potential pitfalls of density estimators.

		Daniel Kuehnle, Michael Oberfichtner, Kerstin Ostermann
						 Last update: March 2021
								
					 Dofile for the German analyses
						- analysis file
						
*******************************************************************************/

cap log close
log using $log\3_analysis.txt, text replace

clear all
use $data\2_merge.dta, replace

********************************************************************************
*Analysis preparation
********************************************************************************

gen all = 1

* generate variables that are filled in loop
gen share = .
lab var share "Share earned by the wife"
gen relfemshare=  .
lab var relfemshare "Share of couples"

*x-axis
global fs "fem_share"

* Make snapshot of data
snapshot erase _all
snapshot save, label(before_loop)

* Groups
global groups west NoEqInc_West east NoEqInc_East 

foreach group in $groups {
foreach size in 0.01 0.05 {

disp "`size' % bins, group: `group', RD density polynomial: 2"

snapshot restore 1

* Insert sample restriction here
quietly keep if `group' == 1
disp r(N_drop) "  couples dropped due to group"

local bp 0.500001
local size100 = `size'* 100 // will use this to name files
local n_bins = 1 / `size'

* generate share group variable and compute shares
quietly count if inrange($fs,0,1) // couples with 0 and 1 are set to . earlier
local __N = r(N)

quietly foreach num of numlist 1/`n_bins' {
replace share = `num' / `n_bins' - 0.5 * `size' in `num'
count if $fs > (`num' - 1) * `size' & $fs <= `num' * `size'
replace relfemshare = r(N) / `__N' in `num'
}

**"locally weighted scatterplot smoothing" on each side of femshare=1/2
lowess relfemshare share if share<`bp', gen(Yj_smooth_left) bwidth(0.5) nograph
lowess relfemshare share if share>`bp' & share<=1 , gen(Yj_smooth_right) bwidth(0.5) nograph

//Run DCdensity command
*default bin size
cap drop Xj Yj r0 fhat se_fhat
DCdensityRR $fs,  breakpoint(`bp') generate(Xj Yj r0 fhat se_fhat) nograph

*save results and default parameters
local thetad = round(r(theta), 0.001)
local sed = round(r(se), 0.001)
local tstatd = round(r(theta)/r(se), 0.001)
local binsize = round(r(binsize), 0.001)

gen dXj =Xj
gen dYj =Yj
gen dr0 = r0 
gen dfhat = fhat

* Run DCdensity command
*specified bin size
cap drop Xj Yj r0 fhat se_fhat
DCdensityRR $fs, b(`size') breakpoint(`bp') generate(Xj Yj r0 fhat se_fhat) 

*save results
local theta = round(r(theta), 0.001)
local se = round(r(se), 0.001)
local tstat = round(r(theta)/r(se), 0.001)

// Run rddensity command
sum $fs
local min_fs = r(min)
local max_fs = r(max)
cap drop RD_*
rddensity $fs, c(`bp') plot plot_range(`min_fs' `max_fs') plot_n(50 50) ///
            genvars(RD) plot_grid(es)  ///
	graph_options(xtitle("Share earned by the wife") ///
	ytitle("Density") legend(off) ///
	graphregion(col(white)icol(white)) ///
	plotregion(col(white)icol(white)))

*save results
local fqr = round(e(f_qr), 0.001)
local fq = round(e(f_qr) - e(f_ql), 0.001)
local lnfq = round(ln(e(f_qr)) - ln(e(f_ql)), 0.001)
local seq = round(e(se_q), 0.001)
local tstatq = round((e(f_qr) - e(f_ql))/e(se_q), 0.001)

local tstatq_r = round(`tstatq' * 1000, 1) / 1000

********************************************************************************
*generate graph
********************************************************************************
//Notes contain estimates displayed in Table 1
gr twoway (scatter relfemshare share, msymbol(circle_hollow) mcolor(black)) ///
(line Yj_smooth_left share if  share< `bp' , ///
lcolor(black) lwidth(medthick) lpattern(longdash))   ///
(line Yj_smooth_right  share if  share > `bp',  ///
lcolor(black) lwidth(medthick) lpattern(longdash)) , ///
xline(`bp', lcolor(black)) legend(off) ///
graphregion(col(white)icol(white)) ///
plotregion(col(white)icol(white)) ///
subtitle("`group', `size100'% bins, RDdensity polynomial 2") ///
note("DCdensity default: log difference in height (t-stat): `thetad' (`tstatd'), bin size: `binsize'" ///
	"DCdensity: log difference in height (t-stat): `theta' (`tstat')" ///
	"RDdensity: [log] difference in height (t-stat): [`lnfq'] `fq' (`tstatq_r'); N=`__N' couples")

graph export "$log\3_IEB_`group'_`size100'bin.png", replace as(png)

//generate density plots (Figure 2)
** Panel D, G, E, H (McCrary)
twoway (line fhat r0 if r0<=`bp' , lcolor(black) lpattern(solid))  ///
       (line fhat r0 if r0>`bp' , lcolor(black) lpattern(solid)) ///
	, xline(`bp') ytitle("Density") xtitle("Share earned by the wife") ///
	legend(off) ///
	graphregion(col(white)icol(white)) ///
	plotregion(col(white)icol(white))
	
graph export "$log\3_Figure2_McC_IEB_`group'_`size100'bin.png", replace as(png)
	
** Panel F and I (CJM)
* for East Germany include yscale(range(0 (1) 0.4)) ylabel(0 (1) 4) /// 
twoway (line RD_f RD_grid if RD_group == 0, lcolor(black) lpattern(solid))  ///
            (line RD_f RD_grid if RD_group == 1, lcolor(black) lpattern(solid))  ///
            , xline(`bp' ) ytitle("Density") xtitle("Share earned by the wife") ///
            legend(off) ///
            graphregion(col(white)icol(white)) ///
            plotregion(col(white)icol(white)) 
graph export "$log\3_Figure2_CJM_IEB_`group'.png", replace as(png)

} //end bin size
} //end group

log close

	
	