%%************************************************************************%
%                                                                         %
%        Forecasting Economic Activity in Data-Rich Environment           %
%                    Leroux, Kotchoni Stevanovic, 2016                    %
%                                                                         %
%*************************************************************************%
%parpool(18)
% Put the path where you saved the folder here
path = '

%%========================================================================%
%                      Access to Matlab functions                         %
%=========================================================================%
addpath(genpath('\KLS2018_files\02_functions'))



clear all
clc
%%========================================================================%
%                            Some choices                                 %
%=========================================================================%
First_date = 1960+1/12;
insize = [];         % estimation window size
lastin = 1969+12/12;        % last in-sample obs 

removeoutlier = 2;    % 1:yes 2:no

horizon = [1 2 3 4 5 6 7 8 9 10 11 12];


rollwind = 1;         %1: rollwind 0:expanding 
predictvars = {'INDPRO','PAYEMS','CPIAUCSL','CPIULFSL','GS10','S&P 500','EXUSUK','EXCAUS'};

% Model spec
pmax = 6; % y lag
kmax = 6; % nb of factors
mmax = 6; % nb of lags of factors
KK = 30; % nb of series to aim for with the softthresholding
n_T = 20;


% % Short test
% horizon =1 ;
% pmax = 1;
% kmax=1;
% mmax=1;
% lastin=2013+12/12;
% predictvars = {'INDPRO','PAYEMS'};


spec = [pmax;...
        kmax;...
        mmax;
        KK;
        n_T];



% Transformation based on SW (1) or BBE (2)
Trans_based = 1;

Density = [0]; % This is not really used. We considered density forecasts but changed our mind.




%% Load data
cd('\KLS2018_files\00_data')
dum=importdata('FRED-MD_2015m1.csv');

% Keeping some information on dates
[months, years] = strtok(dum.textdata(3:end,1),'/');
[days, years] = strtok(years,'/');
[years, ~] = strtok(years,'/');
 months = str2double(months);
 years = str2double(years);
final_month=months(end);
final_year=years(end);
first_year = years(1);
first_month = months(1);
     
start_all = first_year+first_month/12;
end_all = final_year+final_month/12;
dates = [start_all:1/12:end_all]';

        
% Variable names
names=dum.textdata(1,2:end);

% Transformation numbers
Trans_code=dum.data(1,:);

% Changing all I(2) variables to I(1) if I want to
if Trans_based == 2
    ind_6 = find(Trans_code == 6);
    Trans_code(ind_6) = 5;
end
       
        
% Rawdata
rawdata=dum.data(2:end,:);

% Stationnary panel
yt=prepare_missing(rawdata,Trans_code);

% Remove outlier if I want to
if removeoutlier == 1
    [McNg_panel_sta,n]=remove_outliers(yt);
else
    n = zeros(1,size(yt,2));
    McNg_panel_sta = yt;
end

McNg_panel = rawdata;
% Fin du script de McCracken et Ng (2014)
clearvars -except McNg_panel_sta McNg_panel dates names Trans_code horizon rollwind lastin insize...
    start_all end_all predictvars spec n Density First_date


%=========================================================================%
%                        Setting up certain objects                       %
%=========================================================================%

% Setting the time span
start_ind = find(dates==First_date);
end_ind = find(dates==end_all);
OBS = start_ind:end_ind;

dates = dates(OBS);
    
% In-sample size
if isempty(lastin) == 1
    insample = 1:insize;
    taille = insize;
else
    % Last obs of the in-sample
    last_ind = find(dates==lastin);
    insample = 1:last_ind;
    taille = size(insample,2);
end


%%========================================================================%
%                Keep observations based on the chosen time period        %
%=========================================================================%

McNg_panel = McNg_panel(OBS,:);
McNg_panel_sta = McNg_panel_sta(OBS,:);
complete = [sum(isnan(McNg_panel_sta)) == 0];

% Identifying series to forecast in the panel
pred_ind = NaN(1,size(predictvars,2));
for ii = 1:size(predictvars,2)
    pred_ind(1,ii) = find(strcmp(names,predictvars{ii})==1);
end

%%========================================================================%
%                    Matrix of series to forecast (level)                 %
%=========================================================================%
data_to_forecast = McNg_panel(:,pred_ind);
Trans_code_forecast = Trans_code(1,pred_ind);

% Check if missing obs in data_to_forecast, if so remove the lines and had
% new horizons (wont happen here)
if sum(find(isnan(data_to_forecast(end-1:end,:))))>0
    data_to_forecast = data_to_forecast(1:end-1,:);
    McNg_panel = McNg_panel(1:end-1,:);
    McNg_panel_sta = McNg_panel_sta(1:end-1,:);
    horizon = [horizon, horizon(end)+1];
end

d_to_f_trans=prepare_missing(data_to_forecast,Trans_code_forecast);

% Some useful dimensions
Dsize = size(data_to_forecast,2);
Hsize = size(horizon,2);
OOSsize = size(data_to_forecast,1)-taille;
kd = length(Density);

%=========================================================================%
%            List of models (not all in the paper)                        %
%=========================================================================%
% Random walk (equivalent to 0 when forecasting I(1), only for SP500 and exchange rates)
% Random walk with drift (equivalent to historical average when I(1), only for SP500 and exchanges rates)
% AR(p) direct
% AR(p) iterative
% ARMA(1,1) iterative
% ARMA(1,1) direct (bug)
% ARMAX(1,1) direct plus facteurs (bug)
% AR-DI, choix du nombre de facteurs et de lags (m�me nombre de lags pour
% facteurs d'un m�me lot de donn�es
% AR_DIy1, 1 lag de y impos�
% AR-DI, soft treshold
% AR-DI, hard treshold
% AR-DI tstat, 1 lag des facteurs avec choix du facteurs avec stat t
% Three pass filter
% Complete subset regression
% AR-DI (DU) avec facteurs dynamique (m�me que premier AR-DI)
% FAVAR iterative (SS)
% FAVAR direct (SD)
% Dynamic factor model (DN)
% FAVARMA (4 cases)

%=========================================================================%
%                          Only constant model                            %
%=========================================================================%
CONST_Yhat = NaN(OOSsize,Hsize,Dsize);
CONST_eps = NaN(OOSsize,Hsize,Dsize);

for y = 1:size(data_to_forecast,2)
    Y = data_to_forecast(:,y);
    tcode = Trans_code_forecast(y);
    for h = 1:Hsize       
        [junk,junk,CONST_Yhat(:,h,y),CONST_eps(:,h,y)] = meanforecast(Y,taille,horizon(h),rollwind,tcode);
    end
end


%=========================================================================%
%                              Random walk                                %
%=========================================================================%
RW_Yhat = zeros(OOSsize,Hsize,Dsize);
RW_eps = NaN(OOSsize,Hsize,Dsize);

for d = 1:Dsize
    for hh = 1:Hsize
        Y_real = Forecast_trans(data_to_forecast(:,d),horizon(hh),taille,Trans_code_forecast(d)); 
        RW_eps(:,hh,d) = Y_real(end-OOSsize+1:end,1)-RW_Yhat(:,hh,d);
    end
end

%=========================================================================%
%                         Random walk with drift                          %
%=========================================================================%
RWD_Yhat = NaN(OOSsize,Hsize,Dsize);
RWD_eps = NaN(OOSsize,Hsize,Dsize);

for d = 1:Dsize
    for hh = 1:Hsize        
        [RWD_Yhat(:,hh,d),RWD_eps(:,hh,d)] = RWD_forecast(data_to_forecast(:,d),horizon(hh),taille,rollwind,Trans_code_forecast(d));
    end
end

%==========================================================================
%                             AR(p) direct
%==========================================================================
ARD_MSPE = NaN(Hsize,Dsize);
ARD_MAPE = NaN(Hsize,Dsize);
ARD_PR2 = NaN(Hsize,Dsize);
ARD_THEIL = NaN(Hsize,Dsize);
ARD_TBIAIS = NaN(Hsize,Dsize);
ARD_TVAR = NaN(Hsize,Dsize);
ARD_TCOV = NaN(Hsize,Dsize);
ARD_SIGN = NaN(Hsize,Dsize);
ARD_Yhat = NaN(OOSsize,Hsize,Dsize);
ARD_eps = NaN(OOSsize,Hsize,Dsize);
ARD_lag = NaN(OOSsize,Hsize,Dsize);

P = spec(1);
for y = 1:size(data_to_forecast,2)
    Y = data_to_forecast(:,y);
    tcode = Trans_code_forecast(y);
    parfor h = 1:Hsize
        [ARD_MSPE(h,y),ARD_MAPE(h,y),ARD_PR2(h,y),ARD_THEIL(h,y),ARD_TBIAIS(h,y)...
            ,ARD_TVAR(h,y),ARD_TCOV(h,y),ARD_SIGN(h,y),ARD_Yhat(:,h,y),ARD_eps(:,h,y),ARD_lag(:,h,y),~,ARD_density(:,:,h,y)]...
            = ARDforecast(Y,P,horizon(h),taille,rollwind,tcode,Density,1);
    end
end

%==========================================================================
%                             AR(p) iterative
%==========================================================================
ARI_MSPE = NaN(Hsize,Dsize);
ARI_MAPE = NaN(Hsize,Dsize);
ARI_PR2 = NaN(Hsize,Dsize);
ARI_THEIL = NaN(Hsize,Dsize);
ARI_TBIAIS = NaN(Hsize,Dsize);
ARI_TVAR = NaN(Hsize,Dsize);
ARI_TCOV = NaN(Hsize,Dsize);
ARI_SIGN = NaN(Hsize,Dsize);
ARI_Yhat = NaN(OOSsize,Hsize,Dsize);
ARI_eps = NaN(OOSsize,Hsize,Dsize);
ARI_lag = NaN(OOSsize,Hsize,Dsize);


P = spec(1);
for y = 1:size(data_to_forecast,2)
    Y = data_to_forecast(:,y);
    tcode = Trans_code_forecast(y);
    for h = 1:Hsize
        [ARI_MSPE(h,y),ARI_MAPE(h,y),ARI_PR2(h,y),ARI_THEIL(h,y),ARI_TBIAIS(h,y)...
            ,ARI_TVAR(h,y),ARI_TCOV(h,y),ARI_SIGN(h,y),ARI_Yhat(:,h,y),ARI_eps(:,h,y),ARI_lag(:,h,y),~,ARI_density(:,:,h,y)]...
            = ARitforecast(Y,P,horizon(h),taille,rollwind,tcode,Density);  
    end
end



%=========================================================================%
%              Setting up some objects to use parallel loops              %
%=========================================================================%

% Y: Variables to predict
ch1=sort(repmat(1:Dsize,1,Hsize));
% H: Horizons
ch2=repmat(horizon,1,Dsize);

for y = 1:length(ch1)
    yy=ch1(y);
    to_forecast{y} = data_to_forecast(:,yy);
    t_code(y) =  Trans_code_forecast(yy);
end

%=========================================================================%
%                            ARMA(1,1) iterative                          %
%=========================================================================%

parfor j = 1:length(ch1)
    [ARMAI_MSPE(j),ARMAI_MAPE(j),ARMAI_PR2(j),ARMAI_THEIL(j),ARMAI_TBIAIS(j)...
        ,ARMAI_TVAR(j),ARMAI_TCOV(j),ARMAI_SIGN(j),ARMAI_Yhat(:,j),ARMAI_eps(:,j),~,~,ARMAI_density(:,:,j)]...
        = ARMAitforecast(to_forecast{j},1,1,ch2(j),taille,rollwind,t_code(j),Density);  
end

 
ARMAI_MSPE = reshape(ARMAI_MSPE,Hsize,Dsize);
ARMAI_MAPE = reshape(ARMAI_MAPE,Hsize,Dsize);
ARMAI_PR2 = reshape(ARMAI_PR2,Hsize,Dsize);
ARMAI_THEIL = reshape(ARMAI_THEIL,Hsize,Dsize);
ARMAI_TBIAIS = reshape(ARMAI_TBIAIS,Hsize,Dsize);
ARMAI_TVAR = reshape(ARMAI_TVAR,Hsize,Dsize);
ARMAI_TCOV = reshape(ARMAI_TCOV,Hsize,Dsize);
ARMAI_SIGN = reshape(ARMAI_SIGN,Hsize,Dsize);
ARMAI_Yhat = reshape(ARMAI_Yhat,OOSsize,Hsize,Dsize);
ARMAI_eps = reshape(ARMAI_eps,OOSsize,Hsize,Dsize);


%=========================================================================%
%                             ADL                                         %
%=========================================================================%

TB3MS = find(strcmp(names,'TB3MS')==1);
CPI = find(strcmp(names,'CPIAUCSL')==1);
CPIcore = find(strcmp(names,'CPIULFSL')==1);
INDPRO = find(strcmp(names,'INDPRO')==1);
PAYEMS = find(strcmp(names,'PAYEMS')==1);
SP500 = find(strcmp(names,'S&P 500')==1);
EXUSUK = find(strcmp(names,'EXUSUK')==1);
EXCAUS = find(strcmp(names,'EXCAUS')==1);
GS10 = find(strcmp(names,'GS10')==1);
RPI = find(strcmp(names,'RPI')==1);
HOUST = find(strcmp(names,'HOUST')==1);
CSUSHPISA = find(strcmp(names,'CSUSHPISA')==1);
PCEC96 = find(strcmp(names,'PCEC96')==1);


X2=McNg_panel(:,[TB3MS CPI CPIcore INDPRO PAYEMS GS10 SP500 EXUSUK EXCAUS CSUSHPISA]);
tcode2 = Trans_code([TB3MS CPI CPIcore INDPRO PAYEMS GS10 SP500 EXUSUK EXCAUS CSUSHPISA]);

ADL_MSPE = NaN(Hsize,Dsize);
ADL_MAPE = NaN(Hsize,Dsize);
ADL_PR2 = NaN(Hsize,Dsize);
ADL_THEIL = NaN(Hsize,Dsize);
ADL_TBIAIS = NaN(Hsize,Dsize);
ADL_TVAR = NaN(Hsize,Dsize);
ADL_TCOV = NaN(Hsize,Dsize);
ADL_SIGN = NaN(Hsize,Dsize);
ADL_Yhat = NaN(OOSsize,Hsize,Dsize);
ADL_eps = NaN(OOSsize,Hsize,Dsize);
ADL_lags = NaN(OOSsize,size(X2,2)+1,Hsize,Dsize);
ADL_density = NaN(OOSsize,kd,Hsize,Dsize);

p=spec(1);

for y = 1:size(data_to_forecast,2)

    if y == 1
        Ys = X2(:,[4,1,2]);
        tcode = tcode2([4,1,2]);
    elseif y == 2
        Ys = X2(:,[5,1,2]);
        tcode = tcode2([5,1,2]);
    elseif y == 3
        Ys = X2(:,[2,4,1]);
        tcode = tcode2([2,4,1]);
    elseif y == 4
        Ys = X2(:,[3,4,1]);
        tcode = tcode2([3,4,1]);
    elseif y == 5
        Ys = X2(:,[6,2,1,4]);
        tcode = tcode2([6,2,1,4]);
    elseif y == 6
        Ys = X2(:,[7,2,1,4]);
        tcode = tcode2([7,2,1,4]);
    elseif y == 7
        Ys = X2(:,[8,2,1,4]);
        tcode = tcode2([8,2,1,4]);
    else
        Ys = X2(:,[9,2,1,4]);
        tcode = tcode2([9,2,1,4]);
    
    end

    parfor h = 1:length(horizon)
        [ADL_MSPE(h,y),ADL_MAPE(h,y),ADL_PR2(h,y),ADL_THEIL(h,y)...
            ,ADL_TBIAIS(h,y),ADL_TVAR(h,y),ADL_TCOV(h,y),...
            ADL_SIGN(h,y),ADL_Yhat(:,h,y),ADL_eps(:,h,y),~,ADL_density(:,:,h,y)]...
            = ADLforecast(Ys,taille,p,horizon(h),rollwind,tcode,Density);
    end
end

%=========================================================================%
%            Setting up some objects for the data-rich models             %
%=========================================================================%

McNg_series_ind = 1:size(McNg_panel_sta,2);
for y = 1:length(ch1)
    yy=ch1(y);
    W1 = McNg_panel_sta;
    W1(:,pred_ind(1,yy)) = [];
    series_ind = McNg_series_ind;
    series_ind(:,pred_ind(1,yy)) = []; 
    to_forecast{y} = data_to_forecast(:,yy);
    t_code(y) =  Trans_code_forecast(yy);
    panel{y} = W1;   
    pan_ind{y} = series_ind;
end


%=========================================================================%
%                             AR-DI                                       %
%=========================================================================%
M = spec(3);
K = spec(2);
P = spec(1);

parfor j = 1:size(ch1,2)
  [ARDI_MSPE(j),ARDI_MAPE(j),ARDI_PR2(j),ARDI_THEIL(j)...
      ,ARDI_TBIAIS(j),ARDI_TVAR(j),ARDI_TCOV(j)...
      ,ARDI_SIGN(j),ARDI_Yhat(:,j),ARDI_eps(:,j)...
      ,ARDI_Flags(:,:,j),ARDI_Fnumber(:,:,j),ARDI_lags(:,j)...
      ,ARDI_nvars(:,:,j),ARDI_kept1(:,:,j),ARDI_kept2(:,:,j),ARDI_kept3(:,:,j),ARDI_density(:,:,j)]...
      = DIINDEX3(to_forecast{j},panel{j},[],[],pan_ind{j},[],[],ch2(j),1,P,M,K,taille,10000,rollwind,t_code(j),Density);
end        


ARDI_MSPE = reshape(ARDI_MSPE,Hsize,Dsize);
ARDI_MAPE = reshape(ARDI_MAPE,Hsize,Dsize);
ARDI_PR2 = reshape(ARDI_PR2,Hsize,Dsize);
ARDI_THEIL = reshape(ARDI_THEIL,Hsize,Dsize);
ARDI_TBIAIS = reshape(ARDI_TBIAIS,Hsize,Dsize);
ARDI_TVAR = reshape(ARDI_TVAR,Hsize,Dsize);
ARDI_TCOV = reshape(ARDI_TCOV,Hsize,Dsize);
ARDI_SIGN = reshape(ARDI_SIGN,Hsize,Dsize);
ARDI_Yhat = reshape(ARDI_Yhat,OOSsize,Hsize,Dsize);
ARDI_eps = reshape(ARDI_eps,OOSsize,Hsize,Dsize);
ARDI_Flags = reshape(ARDI_Flags,OOSsize,3,Hsize,Dsize);
ARDI_Fnumber = reshape(ARDI_Fnumber,OOSsize,3,Hsize,Dsize);
ARDI_nvars = reshape(ARDI_nvars,OOSsize,3,Hsize,Dsize);


%=========================================================================%
%                             AR-DI, softreshold                          %
%=========================================================================%
M = spec(3);
K = spec(2);
P = spec(1);
KK = spec(4);

parfor j = 1:size(ch1,2)
  [ARDIsoft_MSPE(j),ARDIsoft_MAPE(j),ARDIsoft_PR2(j),ARDIsoft_THEIL(j)...
      ,ARDIsoft_TBIAIS(j),ARDIsoft_TVAR(j),ARDIsoft_TCOV(j)...
      ,ARDIsoft_SIGN(j),ARDIsoft_Yhat(:,j),ARDIsoft_eps(:,j)...
      ,ARDIsoft_Flags(:,:,j),ARDIsoft_Fnumber(:,:,j),ARDIsoft_lags(:,j)...
      ,ARDIsoft_nvars(:,:,j),ARDIsoft_kept1(:,:,j),ARDIsoft_kept2(:,:,j),ARDIsoft_kept3(:,:,j),ARDIsoft_density(:,:,j)]...
      = DIINDEX3(to_forecast{j},panel{j},[],[],pan_ind{j},[],[],ch2(j),2,P,M,K,taille,KK,rollwind,t_code(j),Density);
end    

ARDIsoft_MSPE = reshape(ARDIsoft_MSPE,Hsize,Dsize);
ARDIsoft_MAPE = reshape(ARDIsoft_MAPE,Hsize,Dsize);
ARDIsoft_PR2 = reshape(ARDIsoft_PR2,Hsize,Dsize);
ARDIsoft_THEIL = reshape(ARDIsoft_THEIL,Hsize,Dsize);
ARDIsoft_TBIAIS = reshape(ARDIsoft_TBIAIS,Hsize,Dsize);
ARDIsoft_TVAR = reshape(ARDIsoft_TVAR,Hsize,Dsize);
ARDIsoft_TCOV = reshape(ARDIsoft_TCOV,Hsize,Dsize);
ARDIsoft_SIGN = reshape(ARDIsoft_SIGN,Hsize,Dsize);
ARDIsoft_Yhat = reshape(ARDIsoft_Yhat,OOSsize,Hsize,Dsize);
ARDIsoft_eps = reshape(ARDIsoft_eps,OOSsize,Hsize,Dsize);
ARDIsoft_Flags = reshape(ARDIsoft_Flags,OOSsize,3,Hsize,Dsize);
ARDIsoft_Fnumber = reshape(ARDIsoft_Fnumber,OOSsize,3,Hsize,Dsize);
ARDIsoft_nvars = reshape(ARDIsoft_nvars,OOSsize,3,Hsize,Dsize);
ARDIsoft_kept1 = reshape(ARDIsoft_kept1,OOSsize,size(McNg_panel_sta,2)-1,Hsize,Dsize);


%=========================================================================%
%                             AR-DI, hardtreshold                         %
%=========================================================================%


M = spec(3);
K = spec(2);
P = spec(1);

% 1.28 %                                 
tc=1.28;

parfor j = 1:size(ch1,2)
  [ARDIhards1_MSPE(j),ARDIhards1_MAPE(j),ARDIhards1_PR2(j)...
      ,ARDIhards1_THEIL(j),ARDIhards1_TBIAIS(j),ARDIhards1_TVAR(j)...
      ,ARDIhards1_TCOV(j),ARDIhards1_SIGN(j),ARDIhards1_Yhat(:,j)...
      ,ARDIhards1_eps(:,j),ARDIhards1_Flags(:,:,j),ARDIhards1_Fnumber(:,:,j)...
      ,ARDIhards1_lags(:,j),ARDIhards1_nvars(:,:,j),ARDIhards1_kept1(:,:,j)...
      ,ARDIhards1_kept2(:,:,j),ARDIhards1_kept3(:,:,j),ARDIhards1_density(:,:,j)]...
      = DIINDEX3hard(to_forecast{j},panel{j},[],[],pan_ind{j},[],[],ch2(j),P,M,K,taille,tc,rollwind,t_code(j),Density);
  
end    

ARDIhards1_MSPE = reshape(ARDIhards1_MSPE,Hsize,Dsize);
ARDIhards1_MAPE = reshape(ARDIhards1_MAPE,Hsize,Dsize);
ARDIhards1_PR2 = reshape(ARDIhards1_PR2,Hsize,Dsize);
ARDIhards1_THEIL = reshape(ARDIhards1_THEIL,Hsize,Dsize);
ARDIhards1_TBIAIS = reshape(ARDIhards1_TBIAIS,Hsize,Dsize);
ARDIhards1_TVAR = reshape(ARDIhards1_TVAR,Hsize,Dsize);
ARDIhards1_TCOV = reshape(ARDIhards1_TCOV,Hsize,Dsize);
ARDIhards1_SIGN = reshape(ARDIhards1_SIGN,Hsize,Dsize);
ARDIhards1_Yhat = reshape(ARDIhards1_Yhat,OOSsize,Hsize,Dsize);
ARDIhards1_eps = reshape(ARDIhards1_eps,OOSsize,Hsize,Dsize);
ARDIhards1_Flags = reshape(ARDIhards1_Flags,OOSsize,3,Hsize,Dsize);
ARDIhards1_Fnumber = reshape(ARDIhards1_Fnumber,OOSsize,6,Hsize,Dsize);
ARDIhards1_nvars = reshape(ARDIhards1_nvars,OOSsize,3,Hsize,Dsize); 
ARDIhards1_kept1 = reshape(ARDIhards1_kept1,OOSsize,size(McNg_panel_sta,2)-1,Hsize,Dsize);


% 1.65 %          
tc=1.65;
parfor j = 1:size(ch1,2)
  [ARDIhards2_MSPE(j),ARDIhards2_MAPE(j),ARDIhards2_PR2(j)...
      ,ARDIhards2_THEIL(j),ARDIhards2_TBIAIS(j),ARDIhards2_TVAR(j)...
      ,ARDIhards2_TCOV(j),ARDIhards2_SIGN(j),ARDIhards2_Yhat(:,j)...
      ,ARDIhards2_eps(:,j),ARDIhards2_Flags(:,:,j),ARDIhards2_Fnumber(:,:,j)...
      ,ARDIhards2_lags(:,j),ARDIhards2_nvars(:,:,j),ARDIhards2_kept1(:,:,j)...
      ,ARDIhards2_kept2(:,:,j),ARDIhards2_kept3(:,:,j),ARDIhards2_density(:,:,j)]...
      = DIINDEX3hard(to_forecast{j},panel{j},[],[],pan_ind{j},[],[],ch2(j),P,M,K,taille,tc,rollwind,t_code(j),Density);
end

ARDIhards2_MSPE = reshape(ARDIhards2_MSPE,Hsize,Dsize);
ARDIhards2_MAPE = reshape(ARDIhards2_MAPE,Hsize,Dsize);
ARDIhards2_PR2 = reshape(ARDIhards2_PR2,Hsize,Dsize);
ARDIhards2_THEIL = reshape(ARDIhards2_THEIL,Hsize,Dsize);
ARDIhards2_TBIAIS = reshape(ARDIhards2_TBIAIS,Hsize,Dsize);
ARDIhards2_TVAR = reshape(ARDIhards2_TVAR,Hsize,Dsize);
ARDIhards2_TCOV = reshape(ARDIhards2_TCOV,Hsize,Dsize);
ARDIhards2_SIGN = reshape(ARDIhards2_SIGN,Hsize,Dsize);
ARDIhards2_Yhat = reshape(ARDIhards2_Yhat,OOSsize,Hsize,Dsize);
ARDIhards2_eps = reshape(ARDIhards2_eps,OOSsize,Hsize,Dsize);
ARDIhards2_Flags = reshape(ARDIhards2_Flags,OOSsize,3,Hsize,Dsize);
ARDIhards2_Fnumber = reshape(ARDIhards2_Fnumber,OOSsize,6,Hsize,Dsize);
ARDIhards2_nvars = reshape(ARDIhards2_nvars,OOSsize,3,Hsize,Dsize);
ARDIhards2_kept1 = reshape(ARDIhards2_kept1,OOSsize,size(McNg_panel_sta,2)-1,Hsize,Dsize); 


%=========================================================================%
%                       AR-DI tstat 1 lag of factors                      %
%=========================================================================%
K = 10;
P = spec(1);
tc = 1.96;

parfor j = 1:size(ch1,2)
 [ARDItstat_Yhat(:,j)...
     ,ARDItstat_eps(:,j),ARDItstat_Fnumber(:,j),ARDItstat_lags(:,j),ARDItstat_density(:,:,j)]...
     = SWtstat(to_forecast{j},panel{j},[],[],ch2(j),P,K,K,K,taille,tc,rollwind,t_code(j),Density)
end        

ARDItstat_Yhat = reshape(ARDItstat_Yhat,OOSsize,Hsize,Dsize);
ARDItstat_eps = reshape(ARDItstat_eps,OOSsize,Hsize,Dsize);
ARDItstat_Fnumber = reshape(ARDItstat_Fnumber,OOSsize,Hsize,Dsize);
  

%=========================================================================%
%                             Three pass filter                           %
%=========================================================================%
proxilag = 4;
parfor j = 1:size(ch1,2)
 [TPF_MSPE(j),TPF_MAPE(j),TPF_PR2(j),TPF_THEIL(j)...
     ,TPF_TBIAIS(j),TPF_TVAR(j),TPF_TCOV(j),TPF_SIGN(j)...
     ,TPF_Yhat(:,j),TPF_eps(:,j),TPF_density(:,:,j)]...
     = tpassfilter3(to_forecast{j},panel{j},[],[],ch2(j),taille,proxilag,rollwind,t_code(j),Density)
end        

TPF_MSPE = reshape(TPF_MSPE,Hsize,Dsize);
TPF_MAPE = reshape(TPF_MAPE,Hsize,Dsize);
TPF_PR2 = reshape(TPF_PR2,Hsize,Dsize);
TPF_THEIL = reshape(TPF_THEIL,Hsize,Dsize);
TPF_TBIAIS = reshape(TPF_TBIAIS,Hsize,Dsize);
TPF_TVAR = reshape(TPF_TVAR,Hsize,Dsize);
TPF_TCOV = reshape(TPF_TCOV,Hsize,Dsize);
TPF_SIGN = reshape(TPF_SIGN,Hsize,Dsize);
TPF_Yhat = reshape(TPF_Yhat,OOSsize,Hsize,Dsize);
TPF_eps = reshape(TPF_eps,OOSsize,Hsize,Dsize);


%=========================================================================%
%                     AR-DI (DU) with dynamic factors                     %
%=========================================================================%
M = spec(3);
K = spec(2);
P = spec(1);

parfor j=1:size(ch1,2)
  [ARDIDU_MSPE(j),ARDIDU_MAPE(j),ARDIDU_PR2(j),ARDIDU_THEIL(j)...
      ,ARDIDU_TBIAIS(j),ARDIDU_TVAR(j),ARDIDU_TCOV(j),ARDIDU_SIGN(j)...
      ,ARDIDU_Yhat(:,j),ARDIDU_eps(:,j),ARDIDU_Flags(:,:,j),ARDIDU_Fnumber(:,:,j),...
      ARDIDU_lags(:,j),ARDIDU_density(:,:,j)] = DIINDEX3_DY(to_forecast{j},panel{j},[],[],ch2(j),P,M,K,taille,rollwind,t_code(j),Density);
end  

ARDIDU_MSPE = reshape(ARDIDU_MSPE,Hsize,Dsize);
ARDIDU_MAPE = reshape(ARDIDU_MAPE,Hsize,Dsize);
ARDIDU_PR2 = reshape(ARDIDU_PR2,Hsize,Dsize);
ARDIDU_THEIL = reshape(ARDIDU_THEIL,Hsize,Dsize);
ARDIDU_TBIAIS = reshape(ARDIDU_TBIAIS,Hsize,Dsize);
ARDIDU_TVAR = reshape(ARDIDU_TVAR,Hsize,Dsize);
ARDIDU_TCOV = reshape(ARDIDU_TCOV,Hsize,Dsize);
ARDIDU_SIGN = reshape(ARDIDU_SIGN,Hsize,Dsize);
ARDIDU_Yhat = reshape(ARDIDU_Yhat,OOSsize,Hsize,Dsize);
ARDIDU_eps = reshape(ARDIDU_eps,OOSsize,Hsize,Dsize);
ARDIDU_Flags = reshape(ARDIDU_Flags,OOSsize,3,Hsize,Dsize);
ARDIDU_Fnumber = reshape(ARDIDU_Fnumber,OOSsize,3,Hsize,Dsize);


%==========================================================================
%                Guided random subset regression
%==========================================================================
% Hardtreshold, K=10
KK = 1.65;
K = 10;
parfor j=1:size(ch1,2)
    [~,~,~,~,~,~,~,~,GRSRh10_Yhat(:,j),GRSRh10_eps(:,j),GRSRh10_disp(:,j)] = GRSR(panel{j},to_forecast{j},1,5000,KK,K,ch2(j),taille,rollwind,t_code(j),2,Density);
end
GRSRh10_Yhat = reshape(GRSRh10_Yhat,OOSsize,Hsize,Dsize);
GRSRh10_eps = reshape(GRSRh10_eps,OOSsize,Hsize,Dsize);

% Hardtreshold, K=20
KK = 1.65;
K = 20;
parfor j=1:size(ch1,2)
    [~,~,~,~,~,~,~,~,GRSRh20_Yhat(:,j),GRSRh20_eps(:,j),GRSRh20_disp(:,j)] = GRSR(panel{j},to_forecast{j},1,5000,KK,K,ch2(j),taille,rollwind,t_code(j),2,Density);
end
GRSRh20_Yhat = reshape(GRSRh20_Yhat,OOSsize,Hsize,Dsize);
GRSRh20_eps = reshape(GRSRh20_eps,OOSsize,Hsize,Dsize);


% Softtreshold, K=10
KK = 30;
K=10;
parfor j=1:size(ch1,2)
    [~,~,~,~,~,~,~,~,GRSRs10_Yhat(:,j),GRSRs10_eps(:,j),GRSRs10_disp(:,j)] = GRSR(panel{j},to_forecast{j},1,5000,KK,K,ch2(j),taille,rollwind,t_code(j),1,Density);
end
GRSRs10_Yhat = reshape(GRSRs10_Yhat,OOSsize,Hsize,Dsize);
GRSRs10_eps = reshape(GRSRs10_eps,OOSsize,Hsize,Dsize);

% Softtreshold, K=20
KK = 30;
K=20;
parfor j=1:size(ch1,2)
    [~,~,~,~,~,~,~,~,GRSRs20_Yhat(:,j),GRSRs20_eps(:,j),GRSRs20_disp(:,j)] = GRSR(panel{j},to_forecast{j},1,5000,KK,K,ch2(j),taille,rollwind,t_code(j),1,Density);
end
GRSRs20_Yhat = reshape(GRSRs20_Yhat,OOSsize,Hsize,Dsize);
GRSRs20_eps = reshape(GRSRs20_eps,OOSsize,Hsize,Dsize);


%==========================================================================
%                Random subset ridge regression
%==========================================================================
% K = 10
K=10;
parfor j = 1:size(ch1,2)
[~,~,~,~,~,~,~,~,RSRR10_Yhat(:,j),RSRR10_eps(:,j),~] = ridge_subset(panel{j},to_forecast{j},K,1,5000,ch2(j),taille,rollwind,t_code(j),Density);
end
RSRR10_Yhat = reshape(RSRR10_Yhat,OOSsize,Hsize,Dsize);
RSRR10_eps = reshape(RSRR10_eps,OOSsize,Hsize,Dsize);

% K = 20
K=20;
parfor j = 1:size(ch1,2)
[~,~,~,~,~,~,~,~,RSRR20_Yhat(:,j),RSRR20_eps(:,j),~] = ridge_subset(panel{j},to_forecast{j},K,1,5000,ch2(j),taille,rollwind,t_code(j),Density);
end
RSRR20_Yhat = reshape(RSRR20_Yhat,OOSsize,Hsize,Dsize);
RSRR20_eps = reshape(RSRR20_eps,OOSsize,Hsize,Dsize);

%==========================================================================
%                               Lasso
%==========================================================================
% Nombre maximal de lag de y
P = spec(1);
KK=30;

parfor j=1:size(ch1,2)
    [Lasso_Yhat(:,j),Lasso_eps(:,j)] = Direct_lasso_forc(to_forecast{j},panel{j},ch2(j),P,taille,KK,rollwind,t_code(j),Density);
end
Lasso_Yhat = reshape(Lasso_Yhat,OOSsize,Hsize,Dsize);
Lasso_eps = reshape(Lasso_eps,OOSsize,Hsize,Dsize);


%==========================================================================
%                             Complete subset regression
%==========================================================================
k_var = [1,10,20];
ksize = size(k_var,2);

ch1 = sort(repmat(1:Dsize,1,Hsize*ksize));
ch2 = repmat(sort(repmat(k_var,1,Hsize)),1,Dsize);
ch3 = repmat(horizon,1,Dsize*ksize);

% Maximum number of models to estimate 
maximum = 5000;

for y = 1:length(ch1)
    yy = ch1(y);
    qq = ch2(y);
    kk = ch3(y);

    W = McNg_panel_sta;
    W(:,pred_ind(1,yy)) = [];
        
  
    tot = factorial(size(W,2))/(factorial(kk)*factorial(size(W,2)-kk));
    if tot <= maximum
        Draws(y) = tot;
    else
        Draws(y) = maximum;
    end

    to_forecast{y} = data_to_forecast(:,yy);
    t_code(y) =  Trans_code_forecast(yy);
    pan{y} = W;    
end


% lag maximum de y
pmax = 1;

parfor j = 1:size(ch1,2) 
[SUB_MSPE(j),SUB_MAPE(j),SUB_PR2(j),SUB_THEIL(j)...
   ,SUB_TBIAIS(j),SUB_TVAR(j),SUB_TCOV(j),SUB_SIGN(j)...
   ,SUB_Yhat(:,j),SUB_eps(:,j),~]...
   = subset(pan{j},to_forecast{j},ch2(j),pmax,Draws(j),ch3(j),taille,rollwind,t_code(j),Density);
end 

SUB_MSPE = reshape(SUB_MSPE,Hsize,ksize,Dsize);
SUB_MAPE = reshape(SUB_MAPE,Hsize,ksize,Dsize);
SUB_PR2 = reshape(SUB_PR2,Hsize,ksize,Dsize);
SUB_THEIL = reshape(SUB_THEIL,Hsize,ksize,Dsize);
SUB_TBIAIS = reshape(SUB_TBIAIS,Hsize,ksize,Dsize);
SUB_TVAR = reshape(SUB_TVAR,Hsize,ksize,Dsize);
SUB_TCOV = reshape(SUB_TCOV,Hsize,ksize,Dsize);
SUB_SIGN = reshape(SUB_SIGN,Hsize,ksize,Dsize);
SUB_Yhat = reshape(SUB_Yhat,OOSsize,Hsize,ksize,Dsize);
SUB_eps = reshape(SUB_eps,OOSsize,Hsize,ksize,Dsize);


%=========================================================================%
%        Setting up some objects for the factor structure models          %
%=========================================================================%
% We can't use the same panel since there are some missing observations in
% in some series. We thus need to create a new panel without those series.
 
McNg_panel1 = McNg_panel(:,n<1);
Trans_code1 = Trans_code(n<1);
names1 = names(n<1);

[junk,panel_NaN] = find(isnan(McNg_panel1(3:end,:))==1);
panel_NaN=unique(panel_NaN);
panel_NoNaN = 1:size(McNg_panel1,2);
panel_NoNaN(panel_NaN)=[];


Trans_code_FAVAR = Trans_code1(panel_NoNaN);
names_FAVAR = names1(panel_NoNaN);
McNg_panel_FAVAR = McNg_panel1(:,panel_NoNaN);
McNg_panel_sta_FAVAR = prepare_missing(McNg_panel_FAVAR,Trans_code_FAVAR);

data_to_forecast_FAVAR = McNg_panel_FAVAR;

% Find series to forecast in the new panel
pred_ind_FAVAR = NaN(1,size(predictvars,2));
for ii = 1:size(predictvars,2)
    pred_ind_FAVAR(1,ii) = find(strcmp(names_FAVAR,predictvars{ii})==1);
end

Dsize_FAVAR = size(data_to_forecast_FAVAR);

%==========================================================================
%                             FAVAR iterative (SS)
%==========================================================================
M = spec(3);
Kmax = 15;
P = spec(1);

parfor j = 1:length(horizon)
[FAVARIT_MSPE(j,:),FAVARIT_MAPE(j,:),FAVARIT_PR2(j,:),FAVARIT_THEIL(j,:)...
    ,FAVARIT_TBIAIS(j,:),FAVARIT_TVAR(j,:),FAVARIT_TCOV(j,:),FAVARIT_SIGN(j,:)...
    ,FAVARIT_Yhat(:,:,j),FAVARIT_eps(:,:,j),FAVARIT_Flag(:,j),FAVARIT_Ilags(:,:,j)...
    ,FAVARIT_Fnumber(:,j),FAVARIT_density(:,:,:,j)]...
    = ITFAVAR(data_to_forecast_FAVAR,McNg_panel_sta_FAVAR,horizon(j),P,M,Kmax,[],taille,rollwind,Trans_code_FAVAR,Density);
end        

FAVARIT_Yhat = permute(FAVARIT_Yhat,[1,3,2]);
FAVARIT_eps = permute(FAVARIT_eps,[1,3,2]);
FAVARIT_Ilags = permute(FAVARIT_Ilags,[1,3,2]);


%==========================================================================
%                             FAVAR direct (SD)
%==========================================================================
M = spec(3);
Kmax = 15;
P = spec(1);

parfor j = 1:length(horizon)
[FAVARDI_MSPE(j,:),FAVARDI_MAPE(j,:),FAVARDI_PR2(j,:),FAVARDI_THEIL(j,:)...
    ,FAVARDI_TBIAIS(j,:),FAVARDI_TVAR(j,:),FAVARDI_TCOV(j,:),FAVARDI_SIGN(j,:)...
    ,FAVARDI_Yhat(:,:,j),FAVARDI_eps(:,:,j)...
    ,FAVARDI_Fnumber(:,j),FAVARDI_density(:,:,:,j)]...
    = DIFAVAR(data_to_forecast_FAVAR,McNg_panel_sta_FAVAR,horizon(j),P,M,Kmax,[],taille,rollwind,Trans_code_FAVAR,Density);
end    

FAVARDI_Yhat = permute(FAVARDI_Yhat,[1,3,2]);
FAVARDI_eps = permute(FAVARDI_eps,[1,3,2]);


%==========================================================================
%                       FAVARMA(p,q) FMA Bai-Ng #facteurs
%==========================================================================
M = spec(3);
Qmax = 4;
nbfac= 1;
n_t1 = 15;
ncase = 3;

warning ('off','all');

parfor j = 1:length(horizon)
[FAVARMA_FMA_MSPE(j,:),FAVARMA_FMA_MAPE(j,:),FAVARMA_FMA_PR2(j,:),FAVARMA_FMA_THEIL(j,:)...
    ,FAVARMA_FMA_TBIAIS(j,:),FAVARMA_FMA_TVAR(j,:),FAVARMA_FMA_TCOV(j,:),FAVARMA_FMA_SIGN(j,:)...
    ,FAVARMA_FMA_Yhat(:,:,j),FAVARMA_FMA_eps(:,:,j),FAVARMA_FMA_FacMAlags(:,:,j),FAVARMA_FMA_FacARlags(:,:,j)...
    ,FAVARMA_FMA_IARlags(:,:,j),FAVARMA_FMA_Fnumber(:,j),FAVARMA_FMA_density(:,:,:,j)]...
    = FAVARMA(data_to_forecast_FAVAR,McNg_panel_sta_FAVAR,horizon(j),M,Qmax,Kmax,[],taille,rollwind,Trans_code_FAVAR,n_t1,ncase,Density)
end    


FAVARMA_FMA_Yhat = permute(FAVARMA_FMA_Yhat,[1,3,2]);
FAVARMA_FMA_eps = permute(FAVARMA_FMA_eps,[1,3,2]);


%==========================================================================
%                       FAVARMA(p,q) FAR Bai-Ng #facteurs
%==========================================================================
M = spec(3);
Qmax = 4;
n_t1 = 15;
ncase = 4;

warning ('off','all');
parfor j = 1:length(horizon)
[FAVARMA_FAR_MSPE(j,:),FAVARMA_FAR_MAPE(j,:),FAVARMA_FAR_PR2(j,:),FAVARMA_FAR_THEIL(j,:)...
    ,FAVARMA_FAR_TBIAIS(j,:),FAVARMA_FAR_TVAR(j,:),FAVARMA_FAR_TCOV(j,:),FAVARMA_FAR_SIGN(j,:)...
    ,FAVARMA_FAR_Yhat(:,:,j),FAVARMA_FAR_eps(:,:,j),FAVARMA_FAR_FacMAlags(:,:,j),FAVARMA_FAR_FacARlags(:,:,j)...
    ,FAVARMA_FAR_IARlags(:,:,j),FAVARMA_FAR_Fnumber(:,j),FAVARMA_FAR_density(:,:,:,j)]...
    = FAVARMA(data_to_forecast_FAVAR,McNg_panel_sta_FAVAR,horizon(j),M,Qmax,Kmax,[],taille,rollwind,Trans_code_FAVAR,n_t1,ncase,Density)
end    

FAVARMA_FAR_Yhat = permute(FAVARMA_FAR_Yhat,[1,3,2]);
FAVARMA_FAR_eps = permute(FAVARMA_FAR_eps,[1,3,2]);


%==========================================================================
%                             Dynamic factor model (DN)
%==========================================================================
Kmax = 10;
P = spec(1);
parfor j = 1:length(horizon)
    [DFM_MSPE(j,:),DFM_MAPE(j,:),DFM_PR2(j,:),DFM_THEIL(j,:)...
        ,DFM_TBIAIS(j,:),DFM_TVAR(j,:),DFM_TCOV(j,:),DFM_SIGN(j,:)...
        ,DFM_Yhat(:,:,j),DFM_eps(:,:,j)...
        ,DFM_Fnumber(:,j)]...
        = BOING_DN(data_to_forecast_FAVAR,McNg_panel_sta_FAVAR,horizon(j),P,Kmax,[],taille,rollwind,Trans_code_FAVAR);
end     

DFM_Yhat = permute(DFM_Yhat,[1,3,2]);
DFM_eps = permute(DFM_eps,[1,3,2]);


cd('C:\Users\maxim\Desktop\KLS2018_files\Files_to_share\03_Results')
save('Results_OOS.mat')





