"""
Performs a time series API query—not used for downloading the full data, just
for context.
"""

# The necessary imports for this application
import requests
import json
import os
from time import sleep
from pathlib import Path
import numpy as np
import uuid


# Directory for downloads
articles_dir = Path(os.path.realpath("DataSources/DowJones"))
# Directory for credentials
creds_dir_full = Path(os.path.realpath("DataSourceGrab"))
# Name of file with credegntials in
cred_file_name = Path("creds_guardian.txt")
# URL of the analytics endpoint (for time series analytics endpoint calls)
url_analytics = "https://api.dowjones.com/alpha/analytics"
# Temporary fix for path issues
MY_API_KEY = open(creds_dir_full / cred_file_name).read().strip()
headers_analytics = {"content-type": "application/json", "user-key": MY_API_KEY}

# =============================================================================
# QUERY DESIGN
# =============================================================================
start_date = "1990-01-01 00:00:00"
publication_source_codes = [
    "DAIM",  # Daily Mail
    "DMIRR",  # Daily Mirror
    "GRDN",  # Guardian
    "IND",  # Independent
    "THESUN",  # The Sun
    "T",  # The Times
]
publication_source_codes = [x.lower() for x in publication_source_codes]
subject_codes = [
    #'gpir',     # Politics/intl relations
    #'gpol',     # Domestic politics
    "mcat",  # Commodity/Financial Market News
    "ccat",  # Corp/Industrial News
    "ecat",  # Economic News
]
article_type_codes = [
    "nedi",  # Editorials
    "nedc",  # Commentaries/Opinions
]
exclude_list = [
    "nrgn",  # Routine General News
]

query_analytics = {
    "query": {
        "where": "language_code = 'en' AND"
        + " publication_datetime >= '"
        + start_date
        + "' AND "
        + "(LOWER(source_code) IN ('"
        + "', '".join(publication_source_codes)
        + "') )",
        "frequency": "DAY",
        "date_field": "publication_datetime",
        "group_by_source_code": True,
        "includes": {"subject_codes": subject_codes + article_type_codes},
        "exlcudes": {"subject_codes": exclude_list},
    }
}


# =============================================================================
# Run a query
# =============================================================================
# Generate query id
query_id = str(uuid.uuid4())


# Save query information
fname = articles_dir / Path(query_id + "_query.json")
with open(fname, "w") as f:
    print("Writing to", fname)
    # re-serialize it for pretty indentation
    f.write(json.dumps(query_analytics, indent=2))
# Make request
response = requests.post(
    url_analytics, data=json.dumps(query_analytics), headers=headers_analytics
)
# Check the explain to verify the query was valid
if response.status_code != 201:
    print("ERROR: An error occurred creating an explain: " + response.text)
else:
    responsejson = response.json()
    responsetext = response.text
    # Save response to json
    fname = articles_dir / Path(query_id + "_response.json")
    with open(fname, "w") as f:
        print("Writing to", fname)
        # re-serialize it for pretty indentation
        f.write(json.dumps(responsejson, indent=2))
    # Save response to texr
    fname = articles_dir / Path(query_id + "_responsetxt.txt")
    with open(fname, "w") as f:
        print("Writing to", fname)
        # re-serialize it for pretty indentation
        f.write(json.dumps(responsetext, indent=2))

    # Show result of request:
    print(responsejson)

    explain = response.json()
    print("Explain Created. Job ID: " + explain["data"]["id"])
    state = explain["data"]["attributes"]["current_state"]

    # wait for explain job to complete
    while state != "JOB_STATE_DONE":
        sleep(2)
        self_link = explain["links"]["self"]
        response_get = requests.get(self_link, headers=headers_analytics)
        explain = response_get.json()
        state = explain["data"]["attributes"]["current_state"]
        # Save query information
        fname = articles_dir / Path(query_id + "_qdata.json")
        with open(fname, "w") as f:
            print("Writing to", fname)
            # re-serialize it for pretty indentation
            f.write(json.dumps(explain, indent=2))
