#-------- Simulate DGPs with Copula-Garch Model --------#
#-------- The Benchmark Case with GARCH error term following Normal Distribution --------#
## Author: Ke Wu
## Last Update: 05/2018

rm(list=ls())
library(copula)
library(rugarch)
setwd('.../code_sum/simulation')

#---------------------------------- Read in FF portfolio data -------------------------------------
EWSize_port <- read.delim("simDGPs/EWSize_port.txt", header=T)

Eszport<- as.matrix(EWSize_port)
rmrf = Eszport[,12]
rf = Eszport[,15]

# Compute portfolio excess returns
exEszport <- Eszport[,2:11]-rf


#------------------------------- Monte Carlo Simulation -------------------------------
## Fit copula to actual data to estimate parameters
sz5mk <- cbind(exEszport[,5], rmrf)
ecdf_sz5mk <- pobs(sz5mk)

clayton.cop <- claytonCopula(param=1, dim = 2)
normal.cop <- normalCopula(param=.6, dim = 2, dispstr = "un")

ktau <- cor(sz5mk[, 1], sz5mk[, 2], method = "kendall")
cltna.0 <- 2*ktau/(1-ktau)  ## compute initial param value via Method of Moments 
norma.0 <- sin(ktau * pi/2)

cltnafit.ml <- fitCopula(clayton.cop, ecdf_sz5mk, method="ml", start=cltna.0)
normafit.ml <- fitCopula(normal.cop, ecdf_sz5mk, method="ml", start=norma.0)

myCop.clayton <- archmCopula(family = "clayton", dim = 2, param = coef(cltnafit.ml))
myCop.normal <- normalCopula(param = coef(normafit.ml), dim = 2, dispstr = "un")


#------------- Fit GARCH(1,1) to get simulation parameters -------------#
meanModel <- list(armaOrder = c(0,0))
varModel <- list(model = "sGARCH", garchOrder = c(1,1)) # standard GARCH
garch_spec <- ugarchspec(varModel, mean.model = meanModel, distribution.model = "norm") 

sz5param <- ugarchfit(garch_spec, data = sz5mk[,1])@fit$coef
mktparam <- ugarchfit(garch_spec, data = sz5mk[,2])@fit$coef



#------------- Simulation, T=3600, No. sim=1000 -------------#
# original parameter settings: 
# Clayton theta = 4.350962 range(0, inf)
# normal rho = 0.9144078
# marginals: standard normal
# 
T <- 3600
N <- 1000

simdata.cl <- matrix(NA, nr=T, nc=N*2)
simdata.nr <- matrix(NA, nr=T, nc=N*2)
simdata.25clnr <- matrix(NA, nr=T, nc=N*2)
simdata.375clnr <- matrix(NA, nr=T, nc=N*2)
simdata.50clnr <- matrix(NA, nr=T, nc=N*2)
simdata.75clnr <- matrix(NA, nr=T, nc=N*2)

set.seed(12345)
# Uniform Random Vector generated via copulas
for(i in seq(1, N*2, by=2)){
  simdata.cl[,i:(i+1)] <- qnorm(rCopula(T, myCop.clayton), mean = 0, sd = 1)
  simdata.nr[,i:(i+1)] <- qnorm(rCopula(T, myCop.normal), mean = 0, sd = 1)
}

# Sampling from the mixture copula
# 25% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.25){
      simdata.25clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.normal), mean = 0, sd = 1)
    }else{
      simdata.25clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.clayton), mean = 0, sd = 1)
    }
  }
  #print(i)
}

# 37.5% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.375){
      simdata.375clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.normal), mean = 0, sd = 1)
    }else{
      simdata.375clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.clayton), mean = 0, sd = 1)
    }
  }
  #print(i)
}

# 50% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.5){
      simdata.50clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.normal), mean = 0, sd = 1)
    }else{
      simdata.50clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.clayton), mean = 0, sd = 1)
    }
  }
  #print(i)
}

# 75% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.75){
      simdata.75clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.normal), mean = 0, sd = 1)
    }else{
      simdata.75clnr[t,i:(i+1)] <- qnorm(rCopula(1, myCop.clayton), mean = 0, sd = 1)
    }
  }
  #print(i)
}
apply(simdata.cl[, 1:20], 2, var) # basic sanity check that var==1
apply(simdata.nr[, 1:20], 2, var) # basic sanity check that var==1
apply(simdata.25clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simdata.375clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simdata.50clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simdata.75clnr[, 1:20], 2, var) # basic sanity check that var==1


# Separate size and mkt sim returns
simsz5.cl <- simdata.cl[, seq(1, N*2, by=2)]
simmkt.cl <- simdata.cl[, seq(2, N*2, by=2)]
simsz5.nr <- simdata.nr[, seq(1, N*2, by=2)]
simmkt.nr <- simdata.nr[, seq(2, N*2, by=2)]
simsz5.25clnr <- simdata.25clnr[, seq(1, N*2, by=2)]
simmkt.25clnr <- simdata.25clnr[, seq(2, N*2, by=2)]
simsz5.375clnr <- simdata.375clnr[, seq(1, N*2, by=2)]
simmkt.375clnr <- simdata.375clnr[, seq(2, N*2, by=2)]
simsz5.50clnr <- simdata.50clnr[, seq(1, N*2, by=2)]
simmkt.50clnr <- simdata.50clnr[, seq(2, N*2, by=2)]
simsz5.75clnr <- simdata.75clnr[, seq(1, N*2, by=2)]
simmkt.75clnr <- simdata.75clnr[, seq(2, N*2, by=2)]


# Simulate GARCH(1,1) returns from Clayton
meanModel <- list(armaOrder = c(0,0))
varModel <- list(model = "sGARCH", garchOrder = c(1,1)) # standard GARCH
uspec.sz5 <- ugarchspec(varModel, mean.model = meanModel,
                        fixed.pars = ugarchfit(garch_spec, data = sz5mk[,1])@fit$coef)
uspec.mkt <- ugarchspec(varModel, mean.model = meanModel,
                        fixed.pars = ugarchfit(garch_spec, data = sz5mk[,2])@fit$coef)  


## Simulate GARCH models using the dependent innovations
## Note: ugarchpath(): simulate from a spec; ugarchsim(): simulate from a fitted object
simgarch.sz5.cl <- ugarchpath(uspec.sz5,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simsz5.cl))@path$seriesSim # passing (T, N)-matrix of *standardized* 
simgarch.mkt.cl <- ugarchpath(uspec.mkt,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simmkt.cl))@path$seriesSim 

# Simulate GARCH(1,1) returns from Normal
simgarch.sz5.nr <- ugarchpath(uspec.sz5,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simsz5.nr))@path$seriesSim # passing (T, N)-matrix of *standardized* 
simgarch.mkt.nr <- ugarchpath(uspec.mkt,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simmkt.nr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .25 normal mixture Clayton and Normal
simgarch.sz5.25clnr <- ugarchpath(uspec.sz5,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simsz5.25clnr))@path$seriesSim 
simgarch.mkt.25clnr <- ugarchpath(uspec.mkt,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simmkt.25clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .375 normal mixture Clayton and Normal
simgarch.sz5.375clnr <- ugarchpath(uspec.sz5,
                                    n.sim = T, 
                                    m.sim = N, 
                                    custom.dist = list(name = "sample", distfit = simsz5.375clnr))@path$seriesSim 
simgarch.mkt.375clnr <- ugarchpath(uspec.mkt,
                                    n.sim = T, 
                                    m.sim = N, 
                                    custom.dist = list(name = "sample", distfit = simmkt.375clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .5 normal mixture Clayton and Normal
simgarch.sz5.50clnr <- ugarchpath(uspec.sz5,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simsz5.50clnr))@path$seriesSim 
simgarch.mkt.50clnr <- ugarchpath(uspec.mkt,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simmkt.50clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .75 normal mixture Clayton and Normal
simgarch.sz5.75clnr <- ugarchpath(uspec.sz5,
                                  n.sim = T, 
                                  m.sim = N, 
                                  custom.dist = list(name = "sample", distfit = simsz5.75clnr))@path$seriesSim 
simgarch.mkt.75clnr <- ugarchpath(uspec.mkt,
                                  n.sim = T, 
                                  m.sim = N, 
                                  custom.dist = list(name = "sample", distfit = simmkt.75clnr))@path$seriesSim 


save(simgarch.sz5.25clnr, simgarch.mkt.25clnr, simgarch.sz5.375clnr, simgarch.mkt.375clnr, 
     simgarch.sz5.50clnr, simgarch.mkt.50clnr, simgarch.sz5.75clnr, simgarch.mkt.75clnr,
     simgarch.sz5.nr, simgarch.mkt.nr, simgarch.sz5.cl, simgarch.mkt.cl, 
     file="simDGPs/simCopGarch3600norm.RData") 















