#-------- Simulate DGPs with Copula-Garch Model --------#
#-------- The Alternative Case with GARCH error term following t Distribution --------#
## Author: Ke Wu
## Last Update: 05/2018

rm(list=ls())
library(copula)
library(rugarch)
library(fGarch)
setwd('.../code_sum/simulation')

#---------------------------------- Read in FF portfolio data -------------------------------------
EWSize_port <- read.delim("simDGPs/EWSize_port.txt", header=T)

Eszport<- as.matrix(EWSize_port)
rmrf = Eszport[,12]
rf = Eszport[,15]

# Compute portfolio excess returns
exEszport <- Eszport[,2:11]-rf


#------------------------------- Monte Carlo Simulation -------------------------------
## Fit copula to size portfolio data to estimate parameters
sz5mk <- cbind(exEszport[,5], rmrf)
ecdf_sz5mk <- pobs(sz5mk)

clayton.cop <- claytonCopula(param=1, dim = 2)
normal.cop <- normalCopula(param=.6, dim = 2, dispstr = "un")

ktau <- cor(sz5mk[, 1], sz5mk[, 2], method = "kendall")
cltna.0 <- 2*ktau/(1-ktau)  ## compute initial param value via Method of Moments 
norma.0 <- sin(ktau * pi/2)

cltnafit.ml <- fitCopula(clayton.cop, ecdf_sz5mk, method="ml", start=cltna.0)
normafit.ml <- fitCopula(normal.cop, ecdf_sz5mk, method="ml", start=norma.0)

myCop.clayton <- archmCopula(family = "clayton", dim = 2, param = coef(cltnafit.ml))
myCop.normal <- normalCopula(param = coef(normafit.ml), dim = 2, dispstr = "un")



#------------- Fit GARCH(1,1) to get simulation parameters -------------#
meanModel <- list(armaOrder = c(0,0))
varModel <- list(model = "sGARCH", garchOrder = c(1,1)) # standard GARCH
garch_spec <- ugarchspec(varModel, mean.model = meanModel, distribution.model = "std") # conditional innovation density (or use, e.g., "std")

sz5param <- ugarchfit(garch_spec, data = sz5mk[,1])@fit$coef
mktparam <- ugarchfit(garch_spec, data = sz5mk[,2])@fit$coef



#------------- Simulation, T=1200, No. sim=1000 -------------#
# original parameter settings: 
# Clayton theta = 4.350962 range(0, inf)
# normal rho = 0.9144078
# marginals: t with df=6.54 & 8.07
# 
T <- 1200
N <- 1000

simudata.cl <- matrix(NA, nr=T, nc=N*2)
simudata.nr <- matrix(NA, nr=T, nc=N*2)
simudata.25clnr <- matrix(NA, nr=T, nc=N*2)
simudata.375clnr <- matrix(NA, nr=T, nc=N*2)
simudata.50clnr <- matrix(NA, nr=T, nc=N*2)
simudata.75clnr <- matrix(NA, nr=T, nc=N*2)

set.seed(12345)
nu.1 <- sz5param[5] # df of the size portfolio t margins
nu.2 <- mktparam[5] # df of the market portfolio t margins

# Uniform Random Vector generated via copulas
for(i in seq(1, N*2, by=2)){
  simudata.cl[,i:(i+1)] <- rCopula(T, myCop.clayton)
  simudata.nr[,i:(i+1)] <- rCopula(T, myCop.normal) 
}


# Sampling from the mixture copula
# 25% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.25){
      simudata.25clnr[t,i:(i+1)] <- rCopula(1, myCop.normal)
    }else{
      simudata.25clnr[t,i:(i+1)] <- rCopula(1, myCop.clayton)
    }
  }
  print(i)
}

# 37.5% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.375){
      simudata.375clnr[t,i:(i+1)] <- rCopula(1, myCop.normal)
    }else{
      simudata.375clnr[t,i:(i+1)] <- rCopula(1, myCop.clayton)
    }
  }
  #print(i)
}

# 50% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.5){
      simudata.50clnr[t,i:(i+1)] <- rCopula(1, myCop.normal)
    }else{
      simudata.50clnr[t,i:(i+1)] <- rCopula(1, myCop.clayton)
    }
  }
  #print(i)
}

# 75% normal weighted mixture Clayton and normal copulas
for(i in seq(1, N*2, by=2)){
  U = runif(T) # Sample T random uniforms U
  for(t in 1:T){
    if(U[t]<.75){
      simudata.75clnr[t,i:(i+1)] <- rCopula(1, myCop.normal)
    }else{
      simudata.75clnr[t,i:(i+1)] <- rCopula(1, myCop.clayton)
    }
  }
  #print(i)
}

# Separate size and mkt simulated uniform probability
simusz5.cl <- simudata.cl[, seq(1, N*2, by=2)]
simumkt.cl <- simudata.cl[, seq(2, N*2, by=2)]
simusz5.nr <- simudata.nr[, seq(1, N*2, by=2)]
simumkt.nr <- simudata.nr[, seq(2, N*2, by=2)]
simusz5.25clnr <- simudata.25clnr[, seq(1, N*2, by=2)]
simumkt.25clnr <- simudata.25clnr[, seq(2, N*2, by=2)]
simusz5.375clnr <- simudata.375clnr[, seq(1, N*2, by=2)]
simumkt.375clnr <- simudata.375clnr[, seq(2, N*2, by=2)]
simusz5.50clnr <- simudata.50clnr[, seq(1, N*2, by=2)]
simumkt.50clnr <- simudata.50clnr[, seq(2, N*2, by=2)]
simusz5.75clnr <- simudata.75clnr[, seq(1, N*2, by=2)]
simumkt.75clnr <- simudata.75clnr[, seq(2, N*2, by=2)]

# Transform uniform probability to quantiles (returns)
simsz5.cl <- qstd(simusz5.cl, mean = 0, sd = 1, nu = nu.1) # margins must have mean 0 and variance 1 for ugarchpath
#simsz5.cl <- sqrt((nu.1-2)/nu.1) * qt(simusz5.cl, df = nu.1) 
simmkt.cl <- qstd(simumkt.cl, mean = 0, sd = 1, nu = nu.2)
simsz5.nr <- qstd(simusz5.nr, mean = 0, sd = 1, nu = nu.1)
simmkt.nr <- qstd(simumkt.nr, mean = 0, sd = 1, nu = nu.2)
simsz5.25clnr <- qstd(simusz5.25clnr, mean = 0, sd = 1, nu = nu.1)
simmkt.25clnr <- qstd(simumkt.25clnr, mean = 0, sd = 1, nu = nu.2)
simsz5.375clnr <- qstd(simusz5.375clnr, mean = 0, sd = 1, nu = nu.1)
simmkt.375clnr <- qstd(simumkt.375clnr, mean = 0, sd = 1, nu = nu.2)
simsz5.50clnr <- qstd(simusz5.50clnr, mean = 0, sd = 1, nu = nu.1)
simmkt.50clnr <- qstd(simumkt.50clnr, mean = 0, sd = 1, nu = nu.2)
simsz5.75clnr <- qstd(simusz5.75clnr, mean = 0, sd = 1, nu = nu.1)
simmkt.75clnr <- qstd(simumkt.75clnr, mean = 0, sd = 1, nu = nu.2)

apply(simsz5.cl[, 1:20], 2, var) # basic sanity check that var==1
apply(simsz5.nr[, 1:20], 2, var) # basic sanity check that var==1
apply(simsz5.25clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simsz5.375clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simsz5.50clnr[, 1:20], 2, var) # basic sanity check that var==1
apply(simsz5.75clnr[, 1:20], 2, var) # basic sanity check that var==1


# Simulate GARCH(1,1) returns from Clayton
meanModel <- list(armaOrder = c(0,0))
varModel <- list(model = "sGARCH", garchOrder = c(1,1)) # standard GARCH
uspec.sz5 <- ugarchspec(varModel, mean.model = meanModel, distribution.model = "std",
                        fixed.pars = ugarchfit(garch_spec, data = sz5mk[,1])@fit$coef) # conditional innovation density (or use, e.g., "std")
uspec.mkt <- ugarchspec(varModel, mean.model = meanModel, distribution.model = "std",
                        fixed.pars = ugarchfit(garch_spec, data = sz5mk[,2])@fit$coef)  


## Simulate GARCH models using the dependent innovations
## Note: ugarchpath(): simulate from a spec; ugarchsim(): simulate from a fitted object
simgarch.sz5.cl <- ugarchpath(uspec.sz5,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simsz5.cl))@path$seriesSim # passing (T, N)-matrix of *standardized* 
simgarch.mkt.cl <- ugarchpath(uspec.mkt,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simmkt.cl))@path$seriesSim 

# Simulate GARCH(1,1) returns from normal
simgarch.sz5.nr <- ugarchpath(uspec.sz5,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simsz5.nr))@path$seriesSim # passing (T, N)-matrix of *standardized* 
simgarch.mkt.nr <- ugarchpath(uspec.mkt,
                               n.sim = T, # simulated path length
                               m.sim = N, # number of paths to simulate
                               custom.dist = list(name = "sample", distfit = simmkt.nr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .25 normal mixture Clayton and normal
simgarch.sz5.25clnr <- ugarchpath(uspec.sz5,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simsz5.25clnr))@path$seriesSim 
simgarch.mkt.25clnr <- ugarchpath(uspec.mkt,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simmkt.25clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .375 normal mixture Clayton and normal
simgarch.sz5.375clnr <- ugarchpath(uspec.sz5,
                                    n.sim = T, 
                                    m.sim = N, 
                                    custom.dist = list(name = "sample", distfit = simsz5.375clnr))@path$seriesSim 
simgarch.mkt.375clnr <- ugarchpath(uspec.mkt,
                                    n.sim = T, 
                                    m.sim = N, 
                                    custom.dist = list(name = "sample", distfit = simmkt.375clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .5 normal mixture Clayton and normal
simgarch.sz5.50clnr <- ugarchpath(uspec.sz5,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simsz5.50clnr))@path$seriesSim 
simgarch.mkt.50clnr <- ugarchpath(uspec.mkt,
                                   n.sim = T, 
                                   m.sim = N, 
                                   custom.dist = list(name = "sample", distfit = simmkt.50clnr))@path$seriesSim 

# Simulate GARCH(1,1) returns from .75 normal mixture Clayton and normal
simgarch.sz5.75clnr <- ugarchpath(uspec.sz5,
                                 n.sim = T, 
                                 m.sim = N, 
                                 custom.dist = list(name = "sample", distfit = simsz5.75clnr))@path$seriesSim 
simgarch.mkt.75clnr <- ugarchpath(uspec.mkt,
                                 n.sim = T, 
                                 m.sim = N, 
                                 custom.dist = list(name = "sample", distfit = simmkt.75clnr))@path$seriesSim 

save(simgarch.sz5.25clnr, simgarch.mkt.25clnr, simgarch.sz5.375clnr, simgarch.mkt.375clnr, 
     simgarch.sz5.50clnr, simgarch.mkt.50clnr, simgarch.sz5.75clnr, simgarch.mkt.75clnr, 
     simgarch.sz5.nr, simgarch.mkt.nr, simgarch.sz5.cl, simgarch.mkt.cl, 
     file="simDGPs/simCopGarch1200t.RData")












