## Test equallity of exceedance dependence of different parts of return
## distribution at any given exceedance level. The original source codes
## are modified to ensure there are enough observations on tails.
## Dependence is captured by Exceedance Mutual Information rho^+_c and rho^-_c
## Jiang, Maasoumi, Pan and Wu, Journal of Applied Econometrics, 2018
## Author: Jiening Pan
## Last Update: 04/2018

npcdepKL <- function (data.x = NULL, data.y = NULL, bw.x = NULL, bw.y = NULL, bw.joint = NULL,
                    C = NULL, method = c("MI","VI"), bootstrap = NULL, boot.num = NULL, idx = NULL,
                    boot.method = c("iid","fixed","geom"), blocklen = NULL, random.seed = 24,...){
  
  ## Trap fatal errors
  
  if(is.data.frame(data.x)||is.data.frame(data.y)) stop(" you must enter two data vectors (and not data frames)")
  if(is.factor(data.x)||is.factor(data.y)) stop(" does not support factors")
  if(is.null(data.x)||is.null(data.y)) stop(" you must enter x and y data vectors")
  if(ncol(data.frame(data.x)) != 1) stop(" data must have one dimension only")
  if(length(data.x)!=length(data.y)) stop(" data vectors must be of equal length")
  if(boot.num < 9) stop(" number of bootstrap replications must be >= 9")
  
  boot.method <- match.arg(boot.method)
  method <- match.arg(method)
  
  ## Save seed prior to setting
  
  if(exists(".Random.seed", .GlobalEnv)) {
    save.seed <- get(".Random.seed", .GlobalEnv)
    exists.seed = TRUE
  } else {
    exists.seed = FALSE
  }
  
  set.seed(random.seed)
  
  ## If the variable is a time series convert to type numeric
  
  if(is.ts(data.x)) data.x <- as.numeric(data.x)
  if(is.ts(data.y)) data.y <- as.numeric(data.y)
  
  ## Remove any NAs from paired data
  
  tmp <- na.omit(data.frame(data.x,data.y))
  data.x <- tmp$data.x
  data.y <- tmp$data.y
  rm(tmp)
  
  ## Follow finance literature, standardize returns to be zero mean and unit variance.
  data.x <- (data.x-mean(data.x))/sd(data.x)
  data.y <- (data.y-mean(data.y))/sd(data.y)
  
  ## Compute and save bandwidths (save for bootstrapping if requested)
  
  if (is.null(bw.x)){bw.x <- npudensbw(~data.x)$bw}
  if (is.null(bw.y)){bw.y <- npudensbw(~data.y)$bw}
  
  data <- cbind(data.x,data.y)
  if (is.null(bw.joint)){bw.joint <- npudensbw(dat = cbind(data.x,data.y))$bw}
  
  console <<- newLineConsole()
  console <<- printPush(paste(sep="", "Constructing the Mutual Information Measure..."), console = console)
  
  if(is.null(C)) {C=0}
  srho <- KLdiv(data.x=data.x,data.y=data.y,data.original=data, C=C,
                bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
  d.srho.data <- srho[1]-srho[2]
  if (is.na(d.srho.data)|is.nan(d.srho.data)) {return(NA)}
  
  if(is.null(bootstrap)){
    bootstrap=F
  }

  if (bootstrap==T){
    boot.fun <- function(ii){return(ii)}
    
    B1 <- round(boot.num/10)
    
    resample.stat <- matrix(data=NA,nrow=boot.num,ncol=1)
    d.srho.nested <- matrix(data=NA,nrow=B1,ncol=1)
    
    if (boot.method == "iid"){
      i = 1
      while (i<=B1){
        sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = 1, sim = "fixed")$t
        sample.nested <- t(sample.nested) 
        
        data.nested <- data[sample.nested,]
        data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
        data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
        srho <- KLdiv(data.x=data.nested[,1], data.y=data.nested[,2], data.original=data,
                      C=C, bw.x=bw.x, bw.y=bw.y, bw.joint=bw.joint, method=method)
        d.srho.temp <- srho[1]-srho[2]
        if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
          d.srho.nested[i,1] <- d.srho.temp
          i = i+1
        }
      }

      sigma.srho <- sd(d.srho.nested,na.rm=T)
      if ((sigma.srho==0)|is.nan(sigma.srho)) {return(NA)}
      t.stat <- d.srho.data/sigma.srho

      B.counter = 1
      while (B.counter<=boot.num){
        console <<- printClear(console)
        console <<- printPush(paste(sep="", "Bootstrap replication ", B.counter, "/", boot.num, " of the ", idx, " sample..."), 
                              console = console)
        
        d.srho <- NA
        while (is.na(d.srho)|is.nan(d.srho)){
          boot.sample <- tsboot (tseries = 1:(dim(data)[1]), 
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = 1, sim = "fixed")$t
          boot.sample <- t(boot.sample)
          
          boot.data <- data[boot.sample,]
          boot.data[,1] <- (boot.data[,1]-mean(boot.data[,1]))/sd(boot.data[,1])
          boot.data[,2] <- (boot.data[,2]-mean(boot.data[,2]))/sd(boot.data[,2])
          
          srho <- KLdiv(data.x=boot.data[,1],data.y=boot.data[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho <- srho[1]-srho[2]
        }
        
        j = 1
        while (j<=B1) {
          sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                   statistic = boot.fun,
                                   R = 1, n.sim = dim(data)[1],
                                   l = 1, sim = "fixed")$t
          sample.nested <- t(sample.nested)
            
          data.nested <- boot.data[sample.nested,]
          data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
          data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
            
          srho <- KLdiv(data.x=data.nested[,1],data.y=data.nested[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho.temp <- srho[1]-srho[2]
          if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
            d.srho.nested[j,1] <- d.srho.temp
            j = j+1
          }

          sigma.srho <- sd(d.srho.nested,na.rm=T)
          resample.stat[B.counter,1] <- (d.srho-d.srho.data)/sigma.srho
          }
        B.counter = B.counter+1
        }
    } else if(boot.method == "fixed") {
      
      if (is.null(blocklen)) {blcoklen = 6}
      
      i = 1
      while (i<=B1){
        sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = blcoklen, sim = "fixed")$t
        sample.nested <- t(sample.nested) 
        
        data.nested <- data[sample.nested,]
        data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
        data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
        srho <- KLdiv(data.x=data.nested[,1], data.y=data.nested[,2], data.original=data,
                      C=C, bw.x=bw.x, bw.y=bw.y, bw.joint=bw.joint, method=method)
        d.srho.temp <- srho[1]-srho[2]
        if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
          d.srho.nested[i,1] <- d.srho.temp
          i = i+1
        }
      }
      
      sigma.srho <- sd(d.srho.nested,na.rm=T)
      if ((sigma.srho==0)|is.nan(sigma.srho)) {return(NA)}
      t.stat <- d.srho.data/sigma.srho

      B.counter = 1
      while (B.counter<=boot.num){
        console <<- printClear(console)
        console <<- printPush(paste(sep="", "Bootstrap replication ", B.counter, "/", boot.num, " of the ", idx, " sample..."), 
                              console = console)
        
        d.srho <- NA
        while (is.na(d.srho)|is.nan(d.srho)){
          boot.sample <- tsboot (tseries = 1:(dim(data)[1]), 
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = blcoklen, sim = "fixed")$t
          boot.sample <- t(boot.sample)
          
          boot.data <- data[boot.sample,]
          boot.data[,1] <- (boot.data[,1]-mean(boot.data[,1]))/sd(boot.data[,1])
          boot.data[,2] <- (boot.data[,2]-mean(boot.data[,2]))/sd(boot.data[,2])
          
          srho <- KLdiv(data.x=boot.data[,1],data.y=boot.data[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho <- srho[1]-srho[2]
        }
        
        j = 1
        while (j<=B1) {
          sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                   statistic = boot.fun,
                                   R = 1, n.sim = dim(data)[1],
                                   l = blcoklen, sim = "fixed")$t
          sample.nested <- t(sample.nested)
          
          data.nested <- boot.data[sample.nested,]
          data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
          data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
          
          srho <- KLdiv(data.x=data.nested[,1],data.y=data.nested[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho.temp <- srho[1]-srho[2]
          if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
            d.srho.nested[j,1] <- d.srho.temp
            j = j+1
          }
          
          sigma.srho <- sd(d.srho.nested,na.rm=T)
          resample.stat[B.counter,1] <- (d.srho-d.srho.data)/sigma.srho
          }
        B.counter = B.counter+1
      }
    }
    else {
      
      boot.blocklen <- mean(b.star(data, round=T)[,1])

      i = 1
      while (i<=B1){
        sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = boot.blocklen, sim = "geom")$t
        sample.nested <- t(sample.nested) 
        
        data.nested <- data[sample.nested,]
        data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
        data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
        srho <- KLdiv(data.x=data.nested[,1], data.y=data.nested[,2], data.original=data,
                      C=C, bw.x=bw.x, bw.y=bw.y, bw.joint=bw.joint, method=method)
        d.srho.temp <- srho[1]-srho[2]
        if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
          d.srho.nested[i,1] <- d.srho.temp
          i = i+1
        }
      }
      
      sigma.srho <- sd(d.srho.nested,na.rm=T)
      if ((sigma.srho==0)|is.nan(sigma.srho)) {return(NA)}
      t.stat <- d.srho.data/sigma.srho

      B.counter = 1
      while (B.counter<=boot.num){
        console <<- printClear(console)
        console <<- printPush(paste(sep="", "Bootstrap replication ", B.counter, "/", boot.num, " of the ", idx, " sample..."), 
                              console = console)
        
        d.srho <- NA
        while (is.na(d.srho)|is.nan(d.srho)){
          boot.sample <- tsboot (tseries = 1:(dim(data)[1]), 
                                 statistic = boot.fun,
                                 R = 1, n.sim = dim(data)[1],
                                 l = boot.blocklen, sim = "geom")$t
          boot.sample <- t(boot.sample)
          
          boot.data <- data[boot.sample,]
          boot.data[,1] <- (boot.data[,1]-mean(boot.data[,1]))/sd(boot.data[,1])
          boot.data[,2] <- (boot.data[,2]-mean(boot.data[,2]))/sd(boot.data[,2])
          
          srho <- KLdiv(data.x=boot.data[,1],data.y=boot.data[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho <- srho[1]-srho[2]
        }
        
        j = 1
        while (j<=B1) {
          sample.nested <- tsboot (tseries = 1:(dim(data)[1]),
                                   statistic = boot.fun,
                                   R = 1, n.sim = dim(data)[1],
                                   l = boot.blocklen, sim = "geom")$t
          sample.nested <- t(sample.nested)
          
          data.nested <- boot.data[sample.nested,]
          data.nested[,1] <- (data.nested[,1]-mean(data.nested[,1]))/sd(data.nested[,1])
          data.nested[,2] <- (data.nested[,2]-mean(data.nested[,2]))/sd(data.nested[,2])
          
          srho <- KLdiv(data.x=data.nested[,1],data.y=data.nested[,2],data.original=boot.data,
                        C=C,bw.x=bw.x,bw.y=bw.y,bw.joint=bw.joint,method=method)
          d.srho.temp <- srho[1]-srho[2]
          if (is.na(d.srho.temp)|is.nan(d.srho.temp)) {} else {
            d.srho.nested[j,1] <- d.srho.temp
            j = j+1
          }
          
          sigma.srho <- sd(d.srho.nested,na.rm=T)
          resample.stat[B.counter,1] <- (d.srho-d.srho.data)/sigma.srho
        }
        B.counter = B.counter+1
      }
    }
    P <- mean(ifelse(((resample.stat>=abs(t.stat))|(resample.stat<= -abs(t.stat))),1,0),na.rm = T)
    return(P)
  } else {return(srho)}
}