#-------- Exceedance Correlation tests in HTZ(2007) --------#
## R version of Andrew Patton's Matlab code
## Computes the 'exceedance correlations' discussed in
## Ang and Chen, Journal of Financial Economics, 2002.
##
## Corr[X,Y|X>v,Y>v] if v>0 and X<v,Y<v if v<0
##
## Testing the significance of the differences using the test of Hong, Tu and Zhou (2003).
## 
## INPUTS:  X, a Tx1 vector of data
##  			Y, a Tx1 vector of data
##				qc, a (k)x1 vector of quantiles or standard deviations to estimate the correl. must be increasing, and only the
##		    		"upper" part (ie, >=0.5 for q and >=0 for c)
##               indic, a scalar, equals 1 if using quantiles to determine cut-offs, =0 if using standard deviations
##
## OUTPUT:	Test statistic
## Author: Ke Wu
## Last Update: 05/27/2014

exceed_correl  <- function(X = NULL, 
                               Y = NULL,
                               qc = NULL) {
  

T <- length(X)
k <- length(qc)
  
X <- (X-mean(X))/sd(X)
Y <- (Y-mean(Y))/sd(Y)

# firstly: ripping through to make sure I have enough data in each tail
qc2 <- qc

for(jj in 1:length(qc)) {
  temp1 = which( (X<=-qc[jj]) & (Y<=-qc[jj]) )
  temp2 = which( (X>=qc[jj])& (Y>=qc[jj]) )
  
  if(length(unique(X[temp1]))<3 | length(unique(Y[temp1]))<3 
   | length(unique(X[temp2]))<3 | length(unique(Y[temp2]))<3){ # then not enough distinct data in at least one tail
    qc2 = setdiff(qc2,qc[jj])
  }
}
  
# now I am sure that each cut-off has enough data
qc2 = sort(unique(c(qc2, -qc2)))   # sorting and making sure that 0 does not appear twice    

if(0 %in% qc) {
  qc1 <- c(qc2[1:((length(qc2)+1)/2)], qc2[((length(qc2)+1)/2):length(qc2)])
}
else {
  qc1 <- qc2
}

c = qc1[(length(qc1)/2+1):length(qc1)]

  
k = length(c)
xi = matrix(NA, nrow=T, ncol=k)   #% variable used in constructing test statistic
out1 = matrix(NA, 2*k, 1)  	
rhodiff = matrix(NA, k, 1)

for(jj in 1:k) {
  temp1 = which( X<= -c[jj] & Y<= -c[jj] )
  out1[jj, 1] = cor(X[temp1], Y[temp1])
  temp2 = which( X>=c[jj] & Y>=c[jj] )   
  out1[(length(out1)+1-jj), 1] = cor(X[temp2],Y[temp2])
  rhodiff[jj] = cor(X[temp2],Y[temp2]) - cor(X[temp1],Y[temp1])      # computing the test stat
  
  x1plus = (X-mean(X[temp2]))/sd(X[temp2])
  x2plus = (Y-mean(Y[temp2]))/sd(Y[temp2])
  x1minus = (X-mean(X[temp1]))/sd(X[temp1])
  x2minus = (Y-mean(Y[temp1]))/sd(Y[temp1])
  
  xi[, jj] = T/length(temp2)*(x1plus*x2plus - cor(X[temp2], Y[temp2]))*((X>=c[jj])*(Y>=c[jj])) - 
             T/length(temp1)*(x1minus*x2minus - cor(X[temp1],Y[temp1]))*((X<=-c[jj])*(Y<=-c[jj]))
  
}

omegahat = newey_west(xi)      # computing the covariance matrix
teststat = T*t(rhodiff)%*%solve(omegahat)%*%rhodiff
pval = 1 - pchisq(teststat, df=k)
return(c(as.numeric(teststat), pval)) 

}


#-------- Newey West Covariance matrix --------#
## Newey-West estimator of the covariance matrix of data
##
## Last Update: 04/19/2014

newey_west <- function (data = NULL, 
                        lag = NULL) {


T <- dim(data)[1]
K <- dim(data)[2]

if(is.null(lag)) lag <- floor(4*((T/100)^(2/9)))  # this is the rule used by EViews

data <- data - matrix(1, T, 1)%*%colMeans(data)

B0 <- t(data)%*%data/T
for(ii in 1:lag) {
  B1 <- t(data[(1+ii):T, ])%*%data[1:(T-ii), ]/T
  B0 <- B0 + (1-ii/(lag+1))*(B1+t(B1))
}

return(B0)

}

























