% date: 20210105
% author: Simon Kwok
% purpose: to run local polynomial regression of option price on strikes

% input variables: 
% kin   current strike price (scalar)
% vk    vector of strikes (column vector, nk x 1)
% vc    vector of option prices (column vector, nk x 1)
% p     power p 
% opth  switch for bandwidth 
%           0 = use h0
%           1 = optimal local bandwidth
%           2 = optimal global bandwidth
% h0    user-specified bandwidth

% output variables:
% bp    local polynomial estimate (p+1)x1
% bp_se standard error
% h     optimal bandwidth
% sing  (for diagnostics) indicator for singularity

function [bp,bp_se,h,sing]=lpoly(kin,vk,vc,p,opth,h0)
nk = length(vk);
  
if p == 1
    hcon = 0.776;
elseif p == 2
    hcon = 0.884;
elseif p == 3
    hcon = 1.006;
end
    
% optimal bandwidth

if opth == 1 || opth == 2  % Ait-Sahalia and Duarte (2003)
    x = ones(nk,1);
    d = 0;  
    while d<p+3
        d = d + 1;
        x = [x,vk.^d];     % nk x (p+4)
    end
    alpha = (x'*x)\(x'*vc);
    res = vc - x*alpha;
    ssr = sum(res.^2);
    
    w0 = @(v) (v>mean(vk)-1.5*std(vk)).*(v<mean(vk)+1.5*std(vk));
    mp1 = @(v) factorial(p+1)*alpha(p+2) + 1/2*factorial(p+2)*alpha(p+3)*v + 1/6*factorial(p+3)*alpha(p+4)*v.^2;
    wmp1 = sum(mp1(vk).^2 .* w0(vk));

    if opth == 2        % optimal global bandwidth 
        h = hcon*(ssr*3*std(vk)/wmp1/nk)^(1/(2*p+3));    
        % see Ait-Sahalia and Duarte (2003, eq (3.23))
    
    elseif opth == 1    % optimal local bandwidth (from previous run of lpoly)
        h = h0;
    end    
    
elseif opth == 0
    h = h0;
end

if nk<10 && h<h0
    h = h0;
end

ps = 0;                         % pow for smoothing s2 hat
hs = h;                         % h for smoothing s2 hat


% compute local polynomial estimate

XX = ones(nk,1);
d=0;
nu = 1;
while d<p
    d = d+1;
    XX = [XX,(vk-kin).^d];
    nu = [nu;factorial(d)];
end
W = diag(phi((vk-kin)/h)/h);
if isnan(rcond(XX'*W*XX))
    sing=2;
elseif rcond(XX'*W*XX)<=1e-16
    sing=1;
else
    sing=0;
end
bp = nu.*((XX'*W*XX)\(XX'*W*vc));    % (p+1) x 1


% compute standard error of local polynomial estimate

S = nu.^2.*(XX'*W*XX)\(XX'*W*W*XX)/(XX'*W*XX);    % (p+1) x (p+1)
resid2 = (vc - bp(1)).^2;     % squared residuals
resid2(isnan(resid2)) = 0;
resid2(isinf(resid2)) = 0;

XX = ones(nk,1);
d=0;
nu = 1;
while d<ps
    d = d+1;
    XX = [XX,(vk-kin).^d];
    nu = [nu;factorial(d)];
end
W = diag(phi((vk-kin)/hs)/hs);
s2hat = nu.*((XX'*W*XX)\(XX'*W*resid2));    % (ps+1) x 1
bp_var = s2hat*S;
bp_se = real(sqrt(real(diag(bp_var))));

if opth == 1        % optimal local bandwidth (to be fed into next run of lpoly)
    fhat = mean(phi((vk-kin)/hs)/hs);
    h = hcon*(mean(resid2)/mp1(kin)^2/fhat/nk)^(1/(2*p+3));
    % see Fan and Gijbels (1996), eq (3.20); or Ait-Sahalia and Duarte (2003, eq (3.19)
end
end

function f=phi(x)
f=exp(-x.^2/2)/sqrt(2*pi);
end

