% date: 20210105
% author: Simon Kwok

% purposes:	
%  to compute simple and naive bounds of bubbles (see Jarrow and Kwok (2021), Proposition 2 and Corollary 1)
%  to compute put-call disparity

% input variables: 
% yr 		list of years
 
% output variables (as inputs in bub12_trade.m):
% bubout	cell array containing variables related to bubbles
% dataout	cell array containing raw data series
% setout	cell array containing environmental variables and input parameters

% input datasets:
% allopt_ddMmmyy_to_ddMmmyy_count.csv
% allopt_ddMmmyy_to_ddMmmyy.csv

% example:
% yr = {'1996','1997','1998','1999','2000','2001','2002','2003','2004','2005','2006','2007','2008','2009','2010','2011','2012','2013','2014','2015'};
% [bubout,dataout,setout]=bub12(yr);

function [bubout,dataout,setout]=bub12(yr)

filetype = 'allopt';

warning('off','MATLAB:nearlySingularMatrix')
warning('off','MATLAB:SingularMatrix')

modelname='bub12';

nyr = length(yr);
nperiod = 0;
moneycut = 0.1;

%% import data
for j=1:nyr
    
    % Import count data
    filename = strcat(filetype,'_01Jan',yr{j},'to31Dec',yr{j},'_','count','.csv');
    fd=fopen(filename);
    q=fread(fd,inf,'*uchar');
    fclose(fd);
    nlines = sum(q==10) - 1;
    source = fopen(filename);
    head = textscan(source,'%s %s',1,'delimiter',',');
    c = textscan(source,'%s %f','delimiter',',');
    nkcnt = c{2};

    % Import call and put data
    filename = strcat(filetype,'_01Jan',yr{j},'to31Dec',yr{j},'.csv');
    fd=fopen(filename);
    q=fread(fd,inf,'*uchar');
    fclose(fd);
    source = fopen(filename);
    head = textscan(source,'%s %s %s %s %s %s %s %s %s %s %s %s %s',1,'delimiter',',');

    for i=1:nlines
        t = nperiod+i;
        c = textscan(source,'%s %s %s %f %f %f %f %f %f %f %f %f %f',nkcnt(i),'delimiter',',');
        dateraw{t} = c{1}';
        cp_flag{t} = c{2}';
        exdateraw{t} = c{3}';
        tauday{t} = c{4}';
        X{t} = c{5}';
        s{t} = c{6}';
        dyma{t} = c{7}';    
        tr{t} = c{8}';
        money{t} = c{9}';
        oprice{t} = c{10}';
        volume{t} = c{11}';
        iv{t} = c{12}';
        deltachk{t} = c{13}';
    
        cp{t}=strcmp(cp_flag{t},'C');     % cp = 1 if call,   cp = 0 if put
        ncall(t) = sum(cp{t});

        tau{t} = tauday{t}/365;   % time to maturity
        sout(t) = s{t}(1);

        da(t,1)=datenum(dateraw{t}(1),'ddmmmyyyy');
    end
    fclose(source);

    nperiod = nperiod + nlines;
end

%% compute simple and naive bounds of bubbles

tic
mntau = 20;

motmc = zeros(nperiod,mntau);

pckcnt = zeros(nperiod,mntau);
pcp_mu = zeros(nperiod,mntau);
pcp_atm_mu = zeros(nperiod,mntau);
pcp_otm_mu = zeros(nperiod,mntau);
pcp_itm_mu = zeros(nperiod,mntau);
pcp_absmu = zeros(nperiod,mntau);
pcp_r = zeros(nperiod,mntau);
s_opt_mu = zeros(nperiod,mntau);

sbub_simbb_mu = zeros(nperiod,mntau);
sbub_simbb_m = zeros(nperiod,mntau);
sbub_simbb_lb = zeros(nperiod,mntau);
sbub_simbb_ub = zeros(nperiod,mntau);

sumvolp = zeros(nperiod,3,mntau);
sumvolc = zeros(nperiod,3,mntau);
sumnump = zeros(nperiod,3,mntau);
sumnumc = zeros(nperiod,3,mntau);
lp = zeros(nperiod,3,mntau);
lc = zeros(nperiod,3,mntau);
up = zeros(nperiod,3,mntau);
uc = zeros(nperiod,3,mntau);
lpall = zeros(nperiod,mntau);
lcall = zeros(nperiod,mntau);
upall = zeros(nperiod,mntau);
ucall = zeros(nperiod,mntau);
ncpk = zeros(nperiod,mntau);
ncpk_atm = zeros(nperiod,mntau);
ncpk_otm = zeros(nperiod,mntau);
ncpk_itm = zeros(nperiod,mntau);

for t=1:nperiod   
    ptaulist = unique(tau{t}(~cp{t}));
    ctaulist = unique(tau{t}(cp{t}));
    taulist = intersect(ptaulist,ctaulist);
    ntau = length(taulist);  
    
    for j=1:ntau   
        vput = ((tau{t}==taulist(j)) & ~cp{t});
        np = sum(vput);
        put = oprice{t}(vput);
        pk = X{t}(vput);
        volp = volume{t}(vput);
        
        vcall = ((tau{t}==taulist(j)) & cp{t});
        nc = sum(vcall);
        call = oprice{t}(vcall);
        ck = X{t}(vcall); 
        volc = volume{t}(vcall);
        
        % simple bounds        
        dis = exp(-tr{t}(1)*taulist(j));
        
        if pk(1) <= ck(1) && pk(end) <= ck(end) && ck(1) <= pk(end)
            lc_p = find(pk>=ck(1),1,'first');
            up_c = find(ck<=pk(end),1,'last');
            if isempty(lc_p)
                lc_p = find(pk<=ck(1),1,'last');
            end
            if isempty(up_c)
                up_c = find(ck>=pk(end),1,'first');
            end
            simbbc_ub = -(-dis*ck(1) + put(lc_p) + call(end) - call(1));
            simbbp_ub = -(-dis*pk(end) + put(end) + call(end) - call(up_c));
            simbbc_lb = -(-dis*ck(1) + put(lc_p) - put(1) - call(1));
            simbbp_lb = -(-dis*pk(end) + put(end) - put(1) - call(up_c));
            simbbc_m = dis*ck(1) - put(lc_p) + put(1) - call(end) + call(1);
            simbbp_m = dis*pk(end) - put(end) + put(1) - call(end) + call(up_c);
            pckcnt(t,j) = 1;
        elseif ck(1) <= pk(1) && ck(end) <= pk(end) && pk(1) <= ck(end)
            lp_c = find(ck>=pk(1),1,'first');
            uc_p = find(pk<=ck(end),1,'last');
            if isempty(lp_c)
                lp_c = find(ck<=pk(1),1,'last');
            end
            if isempty(uc_p)
                uc_p = find(pk>=ck(end),1,'first');
            end
            simbbc_ub = -(-dis*pk(1) + put(1) + call(end) - call(lp_c));
            simbbp_ub = -(-dis*ck(end) + put(uc_p));
            simbbc_lb = -(-dis*pk(1) - call(lp_c));
            simbbp_lb = -(-dis*ck(end) + put(uc_p) - put(1) - call(end));
            simbbc_m = dis*pk(1) - call(end) + call(lp_c);
            simbbp_m = dis*ck(end) - put(uc_p) + put(1);
            pckcnt(t,j) = 2;
       
        elseif pk(1) <= ck(1) && ck(end) <= pk(end) 
            lc_p = find(pk>=ck(1),1,'first');
            uc_p = find(pk<=ck(end),1,'last');
            if isempty(lc_p)
                lc_p = find(pk<=ck(1),1,'last');
            end            
            if isempty(uc_p)
                uc_p = find(pk>=ck(end),1,'first');
            end
            simbbc_ub = -(-dis*ck(1) + put(lc_p) + call(end) - call(1));
            simbbp_ub = -(-dis*ck(end) + put(uc_p));
            simbbc_lb = -(-dis*ck(1) + put(lc_p) - put(1) - call(1));
            simbbp_lb = -(-dis*ck(end) + put(uc_p) - put(1) - call(end));
            simbbc_m = dis*ck(1) - put(lc_p) + put(1) - call(end) + call(1);
            simbbp_m = dis*ck(end) - put(uc_p) + put(1);            
            pckcnt(t,j) = 3;
        elseif ck(1) <= pk(1) && pk(end) <= ck(end)
            lp_c = find(ck>=pk(1),1,'first');
            up_c = find(ck<=pk(end),1,'last');
            if isempty(lp_c)
                lp_c = find(ck<=pk(1),1,'last');
            end           
            if isempty(up_c)
                up_c = find(ck>=pk(end),1,'first');
            end
            simbbc_ub = -(-dis*pk(1) + put(1) + call(end) - call(lp_c));
            simbbp_ub = -(-dis*pk(end) + put(end) + call(end) - call(up_c));
            simbbc_lb = -(-dis*pk(1) - call(lp_c));
            simbbp_lb = -(-dis*pk(end) + put(end) - put(1) - call(up_c));
            simbbc_m = dis*pk(1) - call(end) + call(lp_c);
            simbbp_m = dis*pk(end) - put(end) + put(1) - call(end) + call(up_c);
            pckcnt(t,j) = 4;
        elseif pk(end) <= ck(1)           
            simbbc_ub = dis*pk(end) + call(1) - call(end) - put(end);
            simbbp_ub = simbbc_ub;
            simbbc_lb = dis*ck(1) + call(1) + put(1) - put(end);
            simbbp_lb = simbbc_lb;
            simbbc_m = dis*(ck(1)+pk(end))/2 - put(end) + put(1) - call(end) + call(1);
            simbbp_m = simbbc_m;
            pckcnt(t,j) = 5;
        elseif ck(end) <= pk(1)
            simbbc_ub = dis*ck(end) - put(1);
            simbbp_ub = simbbc_ub;
            simbbc_lb = dis*pk(1) + call(end);
            simbbp_lb = simbbc_lb;
            simbbc_m = dis*(ck(end)+pk(1))/2;
            simbbp_m = simbbc_m;
            pckcnt(t,j) = 6;
        else
            pckcnt(t,j) = 7;
        end
        
        n = nc + np;
        % weights on calls and puts: nc/n and np/n
        simbb_lb = nc/n*simbbc_lb + np/n*simbbp_lb;
        simbb_ub = nc/n*simbbc_ub + np/n*simbbp_ub;
        simbb_mu = 0.5*(simbb_ub + simbb_lb);
        simbb_m = nc/n*simbbc_m + np/n*simbbp_m;
        
        % compute bubble bounds (excluding dividends)
        motmc(t,j) = call(end);                     % naive bound                  
        
        % simple bounds
        sbub_simbb_mu(t,j) = sout(t) - simbb_mu;      
        sbub_simbb_m(t,j) = sout(t) - simbb_m; 
        sbub_simbb_lb(t,j) = sout(t) - simbb_lb;   
        sbub_simbb_ub(t,j) = sout(t) - simbb_ub;   
        
        % check put-call parity
        npk = length(pk);
        nck = length(ck);
        exdiv = 1;      % S&P 500 is ex-dividend
        moneyp = zeros(1,npk);
        moneyc = zeros(1,nck);
        
        for i=1:npk
            moneyp(i) = log(pk(i)*dis/(sout(t)*exdiv));
        end
        otmp = (moneyp<-moneycut);
        atmp = ((moneyp>=-moneycut)&(moneyp<=moneycut));
        itmp = (moneyp>moneycut);
        
        for i=1:nck
            moneyc(i) = log(ck(i)*dis/(sout(t)*exdiv));
        end
        otmc = (moneyc>moneycut);
        atmc = ((moneyc>=-moneycut)&(moneyc<=moneycut));
        itmc = (moneyc<-moneycut);
        
        aggvolp = zeros(1,3);
        aggvolc = zeros(1,3);

        aggvolp(1) = sum(volp(otmp));
        aggvolp(2) = sum(volp(atmp));
        aggvolp(3) = sum(volp(itmp));

        aggvolc(1) = sum(volc(otmc));
        aggvolc(2) = sum(volc(atmc));
        aggvolc(3) = sum(volc(itmc));
        
        aggnump(1) = sum(otmp);
        aggnump(2) = sum(atmp);
        aggnump(3) = sum(itmp);

        aggnumc(1) = sum(otmc);
        aggnumc(2) = sum(atmc);
        aggnumc(3) = sum(itmc);        
        
        lpall(t,j) = min(pk);
        upall(t,j) = max(pk);
        lcall(t,j) = min(ck);
        ucall(t,j) = max(ck);
        
        if sum(otmp)>0
            lp(t,1,j) = min(pk(otmp));
            up(t,1,j) = max(pk(otmp));
        else
            lp(t,1,j) = NaN;
            up(t,1,j) = NaN;
        end
        if sum(atmp)>0
            lp(t,2,j) = min(pk(atmp));
            up(t,2,j) = max(pk(atmp));
        else
            lp(t,2,j) = NaN;
            up(t,2,j) = NaN;
        end
        if sum(itmp)>0
            lp(t,3,j) = min(pk(itmp));
            up(t,3,j) = max(pk(itmp));
        else
            lp(t,3,j) = NaN;
            up(t,3,j) = NaN;
        end
        if sum(otmc)>0
            lc(t,1,j) = min(ck(otmc));
            uc(t,1,j) = max(ck(otmc));
        else
            lc(t,1,j) = NaN;
            uc(t,1,j) = NaN;
        end
        if sum(atmc)>0
            lc(t,2,j) = min(ck(atmc));
            uc(t,2,j) = max(ck(atmc));
        else
            lc(t,2,j) = NaN;
            uc(t,2,j) = NaN;
        end
        if sum(itmc)>0
            lc(t,3,j) = min(ck(itmc));
            uc(t,3,j) = max(ck(itmc));
        else
            lc(t,3,j) = NaN;
            uc(t,3,j) = NaN;
        end
        
        for k=1:3
            sumvolp(t,k,j) = aggvolp(k);
            sumvolc(t,k,j) = aggvolc(k);
            sumnump(t,k,j) = aggnump(k);
            sumnumc(t,k,j) = aggnumc(k);
        end
        
        for i=1:nck
            ck_p = find(pk==ck(i),1,'first');           
            if ~isempty(ck_p)
                pcp = sout(t)*exdiv + put(ck_p) - call(i) - ck(i)*dis;
                s_opt = - put(ck_p) + call(i) + ck(i)*dis;
                pcp_mu(t,j) = pcp_mu(t,j) + pcp;
                pcp_absmu(t,j) = pcp_absmu(t,j) + abs(pcp);
                s_opt_mu(t,j) = s_opt_mu(t,j) + s_opt;
                ncpk(t,j)=ncpk(t,j)+1;
            end
        end
        
        for i=1:nck
            ck_p_atm = find(((pk==ck(i))&(atmc(i)==1)),1,'first'); 
            ck_p_otm = find(((pk==ck(i))&(otmc(i)==1)),1,'first'); 
            ck_p_itm = find(((pk==ck(i))&(itmc(i)==1)),1,'first');           
            if ~isempty(ck_p_atm)
                pcp_atm = sout(t)*exdiv + put(ck_p_atm) - call(i) - ck(i)*dis;
                pcp_atm_mu(t,j) = pcp_atm_mu(t,j) + pcp_atm;
                ncpk_atm(t,j)=ncpk_atm(t,j)+1;
            end
            if ~isempty(ck_p_otm)
                pcp_otm = sout(t)*exdiv + put(ck_p_otm) - call(i) - ck(i)*dis;
                pcp_otm_mu(t,j) = pcp_otm_mu(t,j) + pcp_otm;
                ncpk_otm(t,j)=ncpk_otm(t,j)+1;
            end
            if ~isempty(ck_p_itm)
                pcp_itm = sout(t)*exdiv + put(ck_p_itm) - call(i) - ck(i)*dis;
                pcp_itm_mu(t,j) = pcp_itm_mu(t,j) + pcp_itm;
                ncpk_itm(t,j)=ncpk_itm(t,j)+1;
            end
        end
        pcp_r(t,j) = (sout(t)*exdiv);
    end

end        

toc


%% report

bubout.motmc = motmc;
bubout.sbub_simbb_mu = sbub_simbb_mu;
bubout.sbub_simbb_m = sbub_simbb_m;
bubout.sbub_simbb_lb = sbub_simbb_lb;
bubout.sbub_simbb_ub = sbub_simbb_ub;
bubout.pckcnt = pckcnt;
bubout.pcp_mu = pcp_mu;
bubout.pcp_atm_mu = pcp_atm_mu;
bubout.pcp_otm_mu = pcp_otm_mu;
bubout.pcp_itm_mu = pcp_itm_mu;
bubout.pcp_absmu = pcp_absmu;
bubout.pcp_r = pcp_r;
bubout.ncpk = ncpk;
bubout.ncpk_atm = ncpk_atm;
bubout.ncpk_otm = ncpk_otm;
bubout.ncpk_itm = ncpk_itm;

bubout.sumvolp = sumvolp;
bubout.sumvolc = sumvolc;
bubout.sumnump = sumnump;
bubout.sumnumc = sumnumc;

bubout.lp = lp;
bubout.up = up;
bubout.lc = lc;
bubout.uc = uc;
bubout.lpall = lpall;
bubout.upall = upall;
bubout.lcall = lcall;
bubout.ucall = ucall;

setout.filetype = filetype;
setout.modelname = modelname;
setout.yr = yr;
setout.nperiod = nperiod;
setout.moneycut = moneycut;

dataout.sout = sout;
dataout.oprice = oprice;
dataout.cp = cp;
dataout.X = X;
dataout.tau = tau;
dataout.tr = tr;
dataout.dyma = dyma;
dataout.da = da;

end

