# This script replicates the main findings of the US monetary
# policy application in the paper. Before we start, make sure
# that all files (including the CSV files as well as
# auxilliary_functions.R) are located within the same folder
# that is specified to be your working directory.

# Make sure that the threshtvp package is installed before you run this code!

# Source auxilliary helper files necessary to compute impulse responses
source("auxilliary_functions.R")

# Load the threshtvp package that allows to estimate our proposed mixture innovation model
require(threshtvp)
require(tis) # package necessary to include NBER recessions in plots
require(progress) # package for the progress bar used below

#Load in the US macro data and perform some transformations
Xraw <-read.csv("usdata.csv", sep=";")
sl_log <- c("consumption","investment","output","hours")
Xraw[,sl_log] <- log(Xraw[,sl_log])*100
sl.diff <- c("consumption","investment","output","hours","real.wage")
Xraw[2:nrow(Xraw),sl.diff] <- apply(Xraw[,sl.diff],2,diff)
Xraw <- as.matrix(Xraw[2:(nrow(Xraw)),])
class(Xraw) <- "numeric"

#Model specification and MCMC preliminaries
p <- 2 #lag length
nsave <- 5000 #number of saved draws
nburn <- 5000 #number of burned draws
horizon <- 20 #IRF forecast horizon
thin <- 1 #Thinning factor

cov.grid <- c(3) #Uses a weakly informative Gamma prior on the state innovation precisions (see also line 38)
c.grid <- c(-0.1/6) #Specifies the off-setting constant in the "no-movement" regime. A negative number implies that this quantity is scaled with the corresponding OLS S.D. obtained from estimating a standard regression model
d.grid <- c(0.1) #Lower bound for the Uniform prior (scaled with the s.d. of the state innovation variance in the upper regime)
e.grid <- c(1.5) #Upper bound for the Uniform prior (scaled with the s.d. of the state innovation variance in the upper regime)

#This part is only necessary if you are interested in trying out multiple specification, you can change lines 29-32 by adding additional parameters and then do a loop with run from 1 to nrow(combinations) as your loop index
combinations <- expand.grid(cov.grid,c.grid,d.grid,e.grid)
run  <- 1#as.integer(Sys.getenv("SGE_TASK_ID"))#CHGCHG
comb.slct <- as.numeric(combinations[run,])

#Parameters from the cluster
cov.list <- list(one=c(3,0.03),two=c(1.5,1),three=c(0.001,0.001))
cov.spec <- cov.list[[comb.slct[[1]]]]
kappa.spec <- comb.slct[[2]]
pi.0 <- comb.slct[[3]]
pi.1 <- comb.slct[[4]]

m <- ncol(Xraw)
Y <- Xraw
#Create lagged Y matrix
Xlag <- mlag(Y,p)
Y <- Y[(p+1):nrow(Y),,drop=FALSE]
X <- cbind(Xlag[(p+1):nrow(Xlag),],1)

#The following part estimates the TVP VAR. The estimate_tvp function estimates a multivariate time series model if Y is a T by m matrix while it estimates
#a standard univariate regression in the case that Y is a T by 1 matrix/vector
#Notice that this version of the package also allows to infer the off-setting constant used in the lower regime from the data (in the case that sim.kappa0=TRUE)
start <- Sys.time()
model_VAR <- estimate_tvp(Y,X,save=nsave,burn=nburn,p=p,sv_on = TRUE,thin = thin,priorbtheta = list(B_1=cov.spec[[1]],B_2=cov.spec[[2]],kappa0=kappa.spec),priormu=c(0,10),h0prior="stationary", grid.length = 150, thrsh.pct = pi.0,thrsh.pct.high = pi.1,TVS=TRUE,CPU=4,sim.kappa0 = FALSE,kappa0.grid = seq(1e-4,1e-2,length.out=30))
end <- Sys.time()
print(end-start)

dir.create("Results", showWarnings = FALSE)
foldername <- paste0("Results/Results_","pi0_",pi.0,"pi1_",pi.1,"tau_",kappa.spec,"cov_",cov.spec[[1]], "/",sep="")
dir.create(foldername, showWarnings = FALSE)

#Plots the traces of each innovation VC matrix per equation over time (normalized)
det.matrix <- matrix(NA,nrow(Y),ncol(Y))
for (mm in 1:ncol(Y)){
omega.t <- log(apply(model_VAR$posterior[[mm]]$Omega,c(1,3),prod))
det.omega.t <- apply(omega.t,2,mean)
det.omega.t <- det.omega.t#/max(det.omega.t)
det.omega.t <- ts(det.omega.t,start=c(1947,4),frequency = 4)
det.matrix[,mm] <- det.omega.t

pdf(paste(foldername,substr(colnames(Xraw)[[mm]],1,3),"covtrace.pdf",sep=""))
ts.plot(window(exp(det.omega.t-mean(det.omega.t)),start=c(1948,1)),type="l",ylab="",xlab="",lwd=0.1); grid(); nberShade(); lines(window(exp(det.omega.t-mean(det.omega.t)),start=c(1948,1)),lwd=1.5)
dev.off()
}


A_median <- apply(model_VAR$VAR_coeff$A_post,c(1,2,3),median)
A_low <- apply(model_VAR$VAR_coeff$A_post,c(1,2,3),quantile,0.05)
A_high <- apply(model_VAR$VAR_coeff$A_post,c(1,2,3),quantile,0.95)


#This part computes the impulse responses over time using Monte Carlo integration by taking the draws from the posterior of the mixture innovation VAR and then computing the IRFs
T <- nrow(A_median)
IRF.store <- array(NA,c(thin*nsave,T,m,horizon))
eigs.store <- array(NA,c(thin*nsave,T))
uncmean.store <- array(NA,c(thin*nsave,T,m))
pb <- progress_bar$new(format = "  [:bar] :percent in :eta", total = (thin*nsave), clear = FALSE, width= 60)

#IRF.store.std <- array(NA,c(nsave,T,m,horizon))
#Monte Carlo Integration
for (irep in 1:(thin*nsave)){
  for (tt in 1:T){
    A_draw <- model_VAR$VAR_coeff$A_post[tt,1:(m*p),,irep]
   
    comp <- get_companion(A_draw[1:(m*p),],c(m,0,p))
    eig.max <- (max(abs(Re(eigen(comp$MM)$values)))); #print(eig.max)
    eigs.store[irep,tt] <- eig.max
    uncmean.store[irep,tt,] <- (solve(diag(m*p)-comp$MM)%*%c(model_VAR$VAR_coeff$A_post[tt,(m*p+1),,irep],rep(0,((p-1)*m))))[1:m]
    SIG_draw <- model_VAR$VAR_coeff$S_post[tt,,,irep]
   
    #Create A_array (M,M,p)
    PHI_array <- array(0,c(m,m,p)); PHI_array.std <- array(0,c(m,m,p))
    for (k in 1:p){
      PHI_array[,,k] <- t(A_draw[((k-1)*m+1):(k*m),]) #CHG
      
    }
    cholSIG <- try(t(chol(SIG_draw)),silent=TRUE)
    if (is(cholSIG,"try-error")) next
    IRF.t <- impulsdtrf(B=PHI_array,smat =cholSIG,nstep=horizon)
    IRF.store[irep,tt,,] <- IRF.t[,7,]/IRF.t[7,7,1]
  }
  pb$tick()
  
}
date <- seq(from=as.Date("1947-10-01"), to=as.Date("2014-12-31"), by="3 months")

norm <- 0.01*100 #25 basis points increase in FEdFUNDS
IRF.store <- IRF.store[!apply(eigs.store,1,max)>1,,,] #Drop draws with eigenvalues exceeding 1 (dirty but includes only a tiny fraction of draws)

IRF.low <- apply(IRF.store,c(2,3,4),quantile,0.16,na.rm=TRUE)*norm
IRF.high <- apply(IRF.store,c(2,3,4),quantile,0.84,na.rm=TRUE)*norm
IRF.mean <- apply(IRF.store,c(2,3,4),median,na.rm=TRUE)*norm
IRF.std.low <- apply(IRF.store,c(2,3,4),quantile,0.05,na.rm=TRUE)*norm
IRF.std.high <- apply(IRF.store,c(2,3,4),quantile,0.95,na.rm=TRUE)*norm


#Creates a PDF File that shows the evolution of the determinant over time
det.overall <- exp(apply(det.matrix,1,sum)-mean(apply(det.matrix,1,sum)))
start.point <- 15
det.subwindow <- ts(det.overall[start.point:T],start=c(1951,2),frequency = 4)

cc <- 1
pdf(paste0(foldername,"irfs%03d.pdf"),width=16,height=10)
for (tt in start.point:T){
  layout(matrix(c(1:8,rep(9,4)),nrow=3,byrow=TRUE))
  par(mar=c(2,2,2,2),mgp=c(1.6,0.6,0))
  for (cc in 1:7){
    range0 <- range(IRF.low[,cc,],IRF.high[,cc,],0)
    matplot(cbind(IRF.low[tt,cc,],IRF.mean[tt,cc,],IRF.high[tt,cc,]),ylim=range0,type="l",col=c("blue","blue"),lty=c(1,1),main=colnames(Y)[[cc]],ylab="")
    abline(h=0,col="red")
  }
  plot.new()
  text(.5,.5,date[[tt]],cex=2)
  ts.plot(det.subwindow); nberShade()
  lines(det.subwindow,col="black")
  lines(ts(det.subwindow[1:(tt-start.point+1)],start=c(1951,2),frequency = 4),col="red")
  #abline(v=tt-start.point,col="red")
}  
dev.off()

#If one is interested in looking at the unconditional mean of the model over time
uncmean.median <- ts(apply(uncmean.store,c(2,3),median),start=c(1947,4),frequency=4)
uncmean.low <- ts(apply(uncmean.store,c(2,3),quantile,0.16),start=c(1947,4),frequency = 4)
uncmean.high <- ts(apply(uncmean.store,c(2,3),quantile,0.84),start=c(1947,4),frequency = 4)

pdf(paste0(foldername,"uncmean.pdf"),width=16,height=10)
for (jj in 1:ncol(Y)){
  par(mfrow=c(2,1))
  ts.plot(cbind(uncmean.low[,jj],uncmean.high[,jj],uncmean.median[,jj]),main=colnames(Y)[[jj]])
  abline(h=mean(Y[,jj]))
  ts.plot(ts(Y[,jj],start=c(1947,4),frequency=4),uncmean.high[,jj])
  
}
dev.off()


#Draws the (sliced) IRFs for horizons one year ahead (5), two years ahead (9), as well as three years ahead (13)
plot.hor <- c(5,9,13)
pdf(paste(foldername,"RA.pdf"),width = 35,height = 25)
par(mfrow=c(3,4))
par(mar=c(3,5,4,0.5))
for (jj in plot.hor){
  if (jj==plot.hor[[1]]) x.labels <- c("Consumption", "Investment", "Output","Inflation") else x.labels <- c("","","","")
  
  ylims <-c(min(IRF.std.low[,1,plot.hor],IRF.std.high[,1,plot.hor]),max(IRF.std.low[,1,plot.hor],IRF.std.high[,1,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,1,jj],IRF.mean[,1,jj],IRF.high[,1,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[1]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,1,jj],rev(IRF.std.high[,1,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,1,jj],rev(IRF.high[,1,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,1,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  mtext(paste("t=",jj-1), side=2, line=2.5, cex=2)
  
  
  ylims <-c(min(IRF.std.low[,2,plot.hor],IRF.std.high[,2,plot.hor]),max(IRF.std.low[,2,plot.hor],IRF.std.high[,2,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,1,jj],IRF.mean[,1,jj],IRF.high[,1,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[2]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,2,jj],rev(IRF.std.high[,2,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,2,jj],rev(IRF.high[,2,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,2,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
 # mtext(paste("t=",jj), side=2, line=2.5, cex=2)

  
  ylims <-c(min(IRF.std.low[,3,plot.hor],IRF.std.high[,3,plot.hor]),max(IRF.std.low[,3,plot.hor],IRF.std.high[,3,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,1,jj],IRF.mean[,1,jj],IRF.high[,1,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[3]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,3,jj],rev(IRF.std.high[,3,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,3,jj],rev(IRF.high[,3,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,3,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  
  
  ylims <-c(min(IRF.std.low[,5,plot.hor],IRF.std.high[,5,plot.hor]),max(IRF.std.low[,5,plot.hor],IRF.std.high[,5,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,5,jj],IRF.mean[,5,jj],IRF.high[,5,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[4]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,5,jj],rev(IRF.std.high[,5,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,5,jj],rev(IRF.high[,5,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,5,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  # mtext(paste("t=",jj), side=2, line=2.5, cex=2)
  
  
}
dev.off()



pdf(paste(foldername,"supply.pdf"),width = 35,height = 25)
par(mfrow=c(3,3))

par(mar=c(3,5,4,0.5))
for (jj in plot.hor){
  if (jj==plot.hor[[1]]) x.labels <- c("Hours worked", "Real wages","Federal funds rate") else x.labels <- c("","","")
  
  ylims <-c(min(IRF.std.low[,4,plot.hor],IRF.std.high[,4,plot.hor]),max(IRF.std.low[,4,plot.hor],IRF.std.high[,4,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,4,jj],IRF.mean[,4,jj],IRF.high[,4,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[1]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,4,jj],rev(IRF.std.high[,4,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,4,jj],rev(IRF.high[,4,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,4,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  mtext(paste("t=",jj-1), side=2, line=2.5, cex=2)
  
    
  
  ylims <-c(min(IRF.std.low[,6,plot.hor],IRF.std.high[,6,plot.hor]),max(IRF.std.low[,6,plot.hor],IRF.std.high[,6,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,6,jj],IRF.mean[,6,jj],IRF.high[,6,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[2]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,6,jj],rev(IRF.std.high[,6,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,6,jj],rev(IRF.high[,6,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,6,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  
  ylims <-c(min(IRF.std.low[,7,plot.hor],IRF.std.high[,7,plot.hor]),max(IRF.std.low[,7,plot.hor],IRF.std.high[,7,plot.hor]))
  
  matplot(ylab="",cbind(IRF.low[,4,jj],IRF.mean[,4,jj],IRF.high[,4,jj]),type="l",lty=c(0,0),xaxt="n",ylim=ylims,lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main=x.labels[[3]],cex.main=4)
  polygon(c(1:(T),rev(1:(T))),c(IRF.std.low[,7,jj],rev(IRF.std.high[,7,jj])),col="#FF9999",border=NA)#,irf_true[jj,kk,]
  polygon(c(1:(T),rev(1:(T))),c(IRF.low[,7,jj],rev(IRF.high[,7,jj])),col="#FF6666",border=NA)#,irf_true[jj,kk,]
  lines(IRF.mean[,7,jj],col="black",lty=1,lwd=3)
  abline(h=0,col="blue",lty=2,lwd=4)
  grid()
  d1 <- strftime(date, format="%Y")
  axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
  
  
}
dev.off()


#Plot the stochastic volatitilities used in the VAR
for (jj in 1:m){
  
pdf(paste(foldername,substr(colnames(Xraw)[[jj]],1,3),".pdf",sep=""))
#Plot volas
vola_median <- (exp(0.5*apply(model_VAR$posterior[[jj]]$H,2,median,na.rm=TRUE)))
vola_low <- (exp(0.5*apply(model_VAR$posterior[[jj]]$H,2,quantile,0.16,na.rm=TRUE)))
vola_high <- (exp(0.5*apply(model_VAR$posterior[[jj]]$H,2,quantile,0.84,na.rm=TRUE)))
vola_low.5 <- (exp(0.5*apply(model_VAR$posterior[[jj]]$H,2,quantile,0.05,na.rm=TRUE)))
vola_high.95 <- (exp(0.5*apply(model_VAR$posterior[[jj]]$H,2,quantile,0.95,na.rm=TRUE)))

matplot(ylab="",cbind(vola_low.5,vola_high.95),type="l",lty=c(0,0),xaxt="n",ylim=c(min(vola_low.5),max(vola_high.95)),lwd=c(2,2),cex.axis=2,col=c("red","black","red"),main="",cex.main=2)
polygon(c(1:(T),rev(1:(T))),c(vola_low.5,rev(vola_high.95)),col="#FF9999",border=NA)#,irf_true[jj,kk,]
polygon(c(1:(T),rev(1:(T))),c(vola_low,rev(vola_high)),col="#FF6666",border=NA)#,irf_true[jj,kk,]
lines(vola_median,col="black",lty=1,lwd=3)
grid()
d1 <- strftime(date, format="%Y")
axis(side=1, at=seq(1,length(date),by=1), labels=d1, cex.axis=2,tick=FALSE)
dev.off()
}

#This part plots the IRFs over different subsamples
Variables<-colnames(Xraw);Variables<-gsub(".","_",Variables,fix=TRUE)
dimnames(IRF.mean)[[1]]<-date
dimnames(IRF.mean)[[2]]<-Variables

Split<-100
Bounds.l<-apply(IRF.store[,1:Split,,],c(3:4),function(x) quantile(x,0.16,na.rm=T))
Bounds.h<-apply(IRF.store[,1:Split,,],c(3:4),function(x) quantile(x,0.84,na.rm=T))


pdf(paste0(foldername,"split_100.pdf"))
for(xx in 1:length(Variables)){
  Range<-range(cbind(Bounds.h[xx,],Bounds.l[xx,]))
  matplot(t(IRF.mean[1:Split,xx,]),ylim=Range,type="l",col=rev(heat.colors(Split)),ylab="",main=Variables[xx]) #before and after 1980
  lines(Bounds.l[xx,],lty=2,col="black");lines(Bounds.h[xx,],lty=2,col="black")
  abline(h=0);grid()
}
dev.off()


Split<-which(date=="1979-04-01")
Bounds.l<-apply(IRF.store[,1:Split,,],c(3:4),function(x) quantile(x,0.16,na.rm=T))
Bounds.h<-apply(IRF.store[,1:Split,,],c(3:4),function(x) quantile(x,0.84,na.rm=T))

Bounds.l2<-apply(IRF.store[,(Split+1):T,,],c(3:4),function(x) quantile(x,0.16,na.rm=T))
Bounds.h2<-apply(IRF.store[,(Split+1):T,,],c(3:4),function(x) quantile(x,0.84,na.rm=T))


for(xx in 1:length(Variables)){
  pdf(paste0(foldername,"pre_volcker_",Variables[xx],".pdf"))
  Range<-range(cbind(Bounds.h[xx,],Bounds.l[xx,],Bounds.l2[xx,],Bounds.h2[xx,]))
  matplot(t(IRF.mean[1:Split,xx,]),ylim=Range,type="l",col=rev(heat.colors(Split)),
          ylab="",main="",xaxt="n",cex.axis=2);axis(las=1,side=1,at=c(1,5,11,16,21),label=c(0,4,10,15,20),cex.axis=2) #before and after 1980
  lines(Bounds.l[xx,],lty=2,col="black");lines(Bounds.h[xx,],lty=2,col="black")
  abline(h=0,col="black",lwd=4,lty=1);grid()
  dev.off()
}


for(xx in 1:length(Variables)){
  pdf(paste0(foldername,"post_volcker_",Variables[xx],".pdf"))
  Range<-range(cbind(Bounds.h[xx,],Bounds.l[xx,],Bounds.l2[xx,],Bounds.h2[xx,]))
   matplot(t(IRF.mean[(Split+1):T,xx,]),ylim=Range,type="l",
          col=rev(heat.colors(length((Split+1):T))),ylab="",main="",
          xaxt="n",cex.axis=2);axis(las=1,side=1,at=c(1,5,11,16,21),label=c(0,4,10,15,20),cex.axis=2) #vor und nach 1980 #vor und nach 1980
  lines(Bounds.l2[xx,],lty=2,col="black");lines(Bounds.h2[xx,],lty=2,col="black")
  abline(h=0,col="black",lwd=4,lty=1);grid()
  dev.off()
}
