%% Policy_XVIII

% This MAIN file simulates the effect of a policy: a subsidy in tuition fees 
% or parental income. 


clear all
clear global
clc

%% GLOBALS
global dversion rversion % version
global K2 dimK2 K0 dimK0 Sc dimSc dimEc Xc dimXc Yp dimYp Tc dimTc Uc dimUc 
global beta scale nn ni no % Model
global tWAG tLAM tDTA tPHI tSCH tTHF pk % Estimates 
global np qual subvec % Policy Counterfactual

%% VERSION
pwd = 'versionXVIII'; % present working directory
dversion = 'XV';        % data version
rversion = 'XVsq';      % results version

[fDat fEst fLog fRes] = f_pnames(pwd,dversion,rversion);

%% MODEL PARAMETERS
beta        = 0.95; % Discount factor
scale       = 1e3;  % Income scaling factor
nn          = 5e4;  % number of individuals to simulate

%% POLICY PARAMETERS
qual        = 3;        % Qualification that we target (1)OLev, (2)ALev, (3)HE
subvec      = 1:7;      % type of subsidy: (1)OLev, (2)ALev, (3)HE, (4)Ab2|sub, (5)Ab0|sub, (6)Ab2|eta, (5)Ab0|eta
trgt        = 0.01;     % define percentage increase in enrollment  
np          = 4;        % number of periods for counterfactual
etaOLS      = 0.0668/1; % OLS estimate of the return to parental income (Age 16)

%% DATA
load(fDat)
[id Ab2 Ab0 Ypr Prd Sta Sch Qua Typ Wag Exp Obs Q1 Q2 Q3 ni no] = ...
    f_data(data);

%% INDICATORS
[ind1 ind2 ind3 ind4 ind5 ind6 ind7 ind8 ind9 ind10 ...
        ind11 ind13 ind14 ind15 ind16 ind17 ind19 ind20 indW] = ...
        f_identifiers(Typ,Sta,Wag);

%% MODEL
[AbMat,A2,A0,K2,K0,Sc,Yp,Xc,Tc,Uc,...
    dimEc,dimK2,dimK0,dimSc,dimXc,dimYp,dimTc,dimUc] = ...
    f_states(Ab2,Ab0,Sch,Ypr,Exp,Prd);

%% LOAD ESTIMATES
load(fEst) % Estimated Coefficients
[THETA,pk,tWAG,tLAM,tDTA,tPHI,tSCH,tTHF] = ...
    f_estimates(THETA2,dimUc);
clear -regexp mat

%% Initialize Matrices
[STres.tstmat] = f_matinitial([3 size(subvec,2)+1]); 
[STres.etamat] = f_matinitial([2 size(subvec,2)+1]); 
[STres.k2mat STres.k0mat] = f_matinitial([3 size(subvec,2)+1]); % Age 16 and 7 cognitive skills (average, std, and change/std)
[STres.SWFmat] = f_matinitial([3 size(subvec,2)+1 2]); % Social Welfare Functions (value and ratio to baseline case)
[STres.SCHmat] = f_matinitial([3 size(subvec,2)+1]); % Enrlolment matrix
[STres.Vmat] = f_matinitial([nn size(subvec,2)+1]); % Life time wealth cells

%% Baseline case
subtype=0; tut=zeros(dimSc-1,1); sub=0; eta=etaOLS; 
[Smat0,Qmat0,Tmat0,Vlife0,Ab2s0,Ab0s0]=f_psimulation2(subtype,tut,sub,eta,THETA,AbMat,Ab2,Ab0,Prd);
[STres] = f_pmatrices(STres,subtype,tut,sub,eta,Smat0,Vlife0,Ab2s0,Ab0s0,Ypr);

%% Subtypes
options=optimset('Display','iter','MaxIter',100,'MaxFunEvals',1000,'TolFun',1e-5,'TolX',1e-3);    

tic;        
for sb=subvec % type of subsidy (None,OLev, ALev, HE, Ab2, Ab0)
    target=Smat0(1,1,qual+(qual==3))+trgt; % targeted increase in enrolment 
    tut=zeros(dimSc-1,1); sub=0; eta=etaOLS; % re-initialize 
    
    if sb>0 && sb<=3 % search for min grant
        tut_i=-prctile(Ypr,75)/sb; % guess
        tut_i=fminsearch(@(tut_i) f_fminfun(target,sb,tut_i,sub,eta,THETA,AbMat,Ab2,Ab0,Prd),tut_i,options); 
        tut(sb)=tut_i; % update tuitions vector
    elseif sb>=4 && sb<=5 % search for min return to parental income
        sub=min(STres.tstmat(qual,2:4)); % find smallest Grant between OLev, ALev, HE
        tut_i=[]; % this is not used in fminsearch
        eta=eta*3; % guess
        eta=fminsearch(@(eta) f_fminfun(target,sb,tut_i,sub,eta,THETA,AbMat,Ab2,Ab0,Prd),eta,options); 
    elseif sb>=6 && sb<=7 % search for min parental income subsidy 
        sub=sb*min(STres.tstmat(qual,2:4)); % guess is proportional to smallest Grant between OLev, ALev, HE
        tut_i=[]; % this is not used in fminsearch
        sub=fminsearch(@(sub) f_fminfun(target,sb,tut_i,sub,eta,THETA,AbMat,Ab2,Ab0,Prd),sub,options);    
    end
    
    [Smat,Qmat,Tmat,Vlife,Ab2s,Ab0s]=f_psimulation2(sb,tut,sub,eta,THETA,AbMat,Ab2,Ab0,Prd);

    [STres] = f_pmatrices(STres,sb,tut,sub,eta,Smat,Vlife,Ab2s,Ab0s,Ypr);
       
end
toc;

f_log(fLog,toc,pwd,qual,trgt,beta,nn,STres); % generate log file with resutls
f_pgraphics(STres); % generate welfare graphics

save fRes STres;

