% VERSION XVIII
% This is the main file. It loads the data, defines the parameters and
% calls the likelihood minimization routine.
% Individuals choose between Education, Work and Unemployment
% where schooling is a 0,1,2,3 variable. It takes into account the distinction
% between participating in education and actually gaining the qualification
% needed to acces the following education stage.
% It is an optimal stopping problem.
% Choices are modelled between age 16-41.

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% [ This is an unbalanced panel! Individuals are kept in the survey as long as
%%     we can define their status. ]
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all
clear global
clc

global dimK2 dimK0 dimSc dimEc dimXc dimYp dimTc 
global K2 K0 Sc Yp Xc Tc 
global scale
global beta tut rho
global AbMat YprMat

global ind1 ind2 ind3 ind4 ind5 ind6 ind7 ind8 ind9 ind10
global ind11 ind12 ind13 ind14 ind15 ind16 ind17 ind18 ind19 ind20 indW
global id Ab2 Ab0 Prd Sch Qua Wag Exp Q1 Q2 Q3 Ypr
global nn np ni no
global tollerance fhundle THETAF tWAG tLAM tDTA tPHI tSCH

% Unobserved Heterogeneity
global dimUc Uc
global idh niter
global Prk 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% ROUTINE DATA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

Estimate        =1; % Estimate (1) or import estimated parameters (0)
Sequential      =1; % Estimate by Sequential Likelohood (1) or FIML (0)
Dynamic         =1; % Discount factor = 0.95 (1) or 0 (0)
Guess           =1; % Mini routine to give good guess for Rs. On (1), Off (0)
Simulation      =1; % Compute and display (1) simulated choices 
Saved           =1; % Use saved coefficients (1) as a guess
EM              =0; % If Saved (1), use previous EM iteration (1)

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% DATA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

load versionXV.mat

id=data(:,1);
Ab2=data(:,2);
Ab0=data(:,3);
Ypr=data(:,4);
Prd=data(:,5);
Sta=data(:,6);
Sch=data(:,7);
Qua=data(:,8);
Typ=data(:,9);
Wag=data(:,10);
Exp=data(:,11);
Obs=(1:size(data,1))';

clear data;

% Data coding

scale=1e3;
Ypr=Ypr/scale;
Wag=Wag/scale;

kp=30;
Ab2=MyDiscretize(Ab2,kp);
Ab0=MyDiscretize(Ab0,kp);
Ypr=MyDiscretize(Ypr,kp);

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% INDICATORS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

a=tabulate(id);
ni=sum(a(:,2)~=0); % N = number of individuals
no=length(id); % number of observations
np=max(Prd); % number of periods 

% Identify types
ind1    =find(Typ==1)   ; % W -> W
ind2    =find(Typ==2)   ; % U -> W
ind3    =find(Typ==3)   ; % U -> U
ind4    =find(Typ==4)   ; % W -> U
ind5    =find(Typ==5)   ; % S0 -> S1
ind6    =find(Typ==6)   ; % S0 -> U
ind7    =find(Typ==7)   ; % S1 -> S2 | Q1=1
ind8    =find(Typ==8)   ; % S1 -> W  | Q1=1
ind9    =find(Typ==9)   ; % S1 -> W  | Q1=0
ind10   =find(Typ==10)  ; % S1 -> U  | Q1=1
ind11   =find(Typ==11)  ; % S1 -> U  | Q1=0

ind13   =find(Typ==13)  ; % S2 -> S3 | Q2=1
ind14   =find(Typ==14)  ; % S2 -> W  | Q2=1
ind15   =find(Typ==15)  ; % S2 -> W  | Q2=0
ind16   =find(Typ==16)  ; % S2 -> U  | Q2=1
ind17   =find(Typ==17)  ; % S2 -> U  | Q2=0

ind19   =find(Typ==19)  ; % S3 -> W  | Q3=1
ind20   =find(Typ==20)  ; % S3 -> W  | Q3=1

indW=find(Sta==2&Wag~=0); %  -> W observed 
Q1=(Qua==1); % Schooling Dummies
Q2=(Qua==2); % Schooling Dummies
Q3=(Qua==3); % Schooling Dummies

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% MODEL
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

%State variables
A2=MyTabulate(Ab2);
A0=MyTabulate(Ab0);

% Grid
% K2=MyGrid(Ab2,[10  90]);    %A2(:,1); %       % skill age 16 grid (0,1,2...)
% K0=MyGrid(Ab0,[10  90]);    %A0(:,1); %       % skill age 7 grid (0,1,2...)
K2=MyGrid(Ab2,[0 10 25 33 50 66 75 90 100]);    %A2(:,1); %       % skill age 16 grid (0,1,2...)
K0=MyGrid(Ab0,[0 10 25 33 50 66 75 90 100]);    %A0(:,1); %       % skill age 7 grid (0,1,2...)
Sc=[0:max(Sch)]';                               % schooling grid
% Yp=MyGrid(Ypr,[10  90]);                % parental income grid (0,1,2...)
Yp=MyGrid(Ypr,[0 25 50 75 100]);                % parental income grid (0,1,2...)
Xc=[0:max(Exp)]';                               % experience grid    
Tc=[1:(65-16)]';                                % time grid
Uc=[0:2]';                                      % unobserved heterogeneity grid

% Dimensions
dimEc=4;              % shock
dimK2=length(K2);     % skill type age 16
dimK0=length(K0);     % skill type age 7
dimSc=length(Sc);     % schooling: No Qual, O Level, A Level, H.E.
dimXc=length(Xc);     % experience
dimYp=length(Yp);     % parental income
dimTc=length(Tc);     % time
dimUc=length(Uc);     % unobserved heterogeneity 

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% PARAMETERS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

dimT=300; % number of iterations
tol=5e-4; % tolerance in fixed point

if Dynamic == 1
    beta    = 0.95          ;   % discount factor
elseif Dynamic == 0
    beta    = 0;            ;   % discount factor
end

tut     =[0 0 0]'             ;   % Rs param: tuition fees O Levels

% OLS to find good guess for initial values in wage equation
t1=size(indW,1);
X=[ones(t1,1) Ab2(indW) Ab0(indW) Q1(indW) Q2(indW) Q3(indW)...
    Exp(indW) Exp(indW).*Q1(indW) Exp(indW).*Q2(indW) Exp(indW).*Q3(indW)];

Y=log(Wag(indW));
[bW]=ols(Y,X);

% OLS to find good guess for initial values in Rs
Y=Wag(indW);
% X=[ones(t1,1),Ypr(indW)];
X=[ones(t1,1)];
[bS]=ols(Y,X);
bS.beta=repmat(bS.beta,[dimSc-1,1]);

% Probit to find good guess for probability of success in school
indL1=find(Prd==1&Sta==1); % find individuals enrolled O Levels
indL1m=(id(indL1)~=id(indL1+1)); % check they are observed period after
indL1(indL1m==1)=[];
[bL1]=probit(Q1(indL1+1),[ones(size(indL1,1),1),Ab2(indL1),Ab0(indL1)]); % coeff in P(Q1=1|S1)

indL2=find(Prd==3&Sta==1); % find individuals enrolled A Levels
indL2m=(id(indL2)~=id(indL2+1)); % check they are observed period after
indL2(indL2m==1)=[];
[bL2]=probit(Q2(indL2+1),[ones(size(indL2,1),1),Ab2(indL2),Ab0(indL2)]); % coeff in P(Q1=1|S1)

bL.beta=[bL1.beta;bL2.beta];

% Probit to find good guess for probability of having a job offer
indD=find(Sta~=2&Obs<max(Obs)); % not working -> receive a job offer
indDm=((id(indD)~=id(indD+1))|(Sta(indD+1)==1)); % check they are observed period after
indD(indDm==1)=[];
[bD]=probit(Sta(indD+1)==2,[ones(size(indD,1),1),Ab2(indD+1),Ab0(indD+1),...
    Q1(indD+1),Q2(indD+1),Q3(indD+1),Exp(indD+1)...
    (Prd(indD+1)==2),(Prd(indD+1)==2).*Q1(indD+1) ]); %

% Probit to find good guess for probability of being fired
indP=find(Sta==2&Obs<max(Obs)); % working -> might be fired
indPm=(id(indP)~=id(indP+1)); % check they are observed period after
indP(indPm==1)=[];
[bP]=probit(Sta(indP+1)==3,[ones(size(indP,1),1),Ab2(indP+1),Ab0(indP+1),...
    Q1(indP+1),Q2(indP+1),Q3(indP+1),Exp(indP+1)]); %

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% UNOBSERVED HETEROGENEITY
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

% Cluster analysis to find good guess for initial values in wage equation
[IDX,C]=kmeans(log(Wag(indW)),dimUc);
amat=tabulate(IDX);
pk=amat(:,3)*1e-2; % Get a guess of proportions
a=[C pk]; b=sortrows(a); C=b(:,1); pk=b(:,2); % Sort in ascending order
CW=[C(1);diff(C)]; % Define as deviation from previous type

% Index individuals. Important if panel is unbalanced. 
idk=[1; cumsum(id(1:end-1)~=id(2:end))+1]; % id ordered 1,2,...,N
idh =(idk-1)*max(Prd)+Prd; % het location to be used in computing likelihood

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% INITIALIZING THETA
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

sig_0=[bW.sige^0.5]';                                       % S.D.: Rs shock, measurement error
sig_E=[repmat(1e7/scale,[3,1])];                            % S.D.: Rs taste shock
al_0=[CW;bW.beta(2:end)];                                   % Rw coeff: wage equation  
gm_0=[bS.beta;repmat(1e-1,[dimUc-1,1])];                    % Rs coeff: schooling equation  
la_0=[bL.beta;repmat(1e-1,[dimUc-1,1])];                    % Lambda coeff's
dt_0=[bD.beta;repmat(1e-1,[dimUc-1,1])];                    % Delta coeff's
ph_0=[bP.beta;repmat(1e-1,[dimUc-1,1])];                    % Phi coeff's

sigU_0=[sig_0*1e2];     sigL_0=[sig_0*1e-10];
sigU_E=[sig_E+1e6];     sigL_E=[sig_E-1e6];
alU_0=[al_0+3e1];       alL_0=[al_0-3e1];
laU_0=[la_0+3e1];       laL_0=[la_0-3e1];
dtU_0=[dt_0+3e1];       dtL_0=[dt_0-3e1];
phU_0=[ph_0+3e1];       phL_0=[ph_0-3e1];
gmU_0=[gm_0+1e3];       gmL_0=[gm_0-1e3];

ThetaWAG=[sig_0; al_0]; blWAG=[sigL_0; alL_0];  buWAG=[sigU_0; alU_0];  tWAG=length(ThetaWAG);
ThetaLAM=[la_0];        blLAM=[laL_0];          buLAM=[laU_0];          tLAM=length(ThetaLAM);
ThetaDTA=[dt_0];        blDTA=[dtL_0];          buDTA=[dtU_0];          tDTA=length(ThetaDTA);
ThetaPHI=[ph_0];        blPHI=[phL_0];          buPHI=[phU_0];          tPHI=length(ThetaPHI);
ThetaSCH=[sig_E; gm_0]; blSCH=[sigL_E; gmL_0];  buSCH=[sigU_E; gmU_0];  tSCH=length(ThetaSCH);

bl=[blWAG;blLAM;blDTA;blPHI;blSCH];
bu=[buWAG;buLAM;buDTA;buPHI;buSCH];

%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

THETAF=[ThetaWAG; ThetaLAM; ThetaDTA; ThetaPHI];

if Estimate==1 && Saved==0 && Guess==0
    % do nothing
elseif Estimate==1 && Saved==0 && Guess==1 % improve on guess
    [ThetaSCH]=f_guessSCH(ThetaSCH,pk);
    THETA=[THETAF;ThetaSCH]; 
elseif Estimate==1 && Saved==1 && EM==1 % use previous iteration estimates
    load EMresultsXVI.mat
    THETA=Res_t(1:end-dimUc-1,end);
    pk=Res_t(end-dimUc:end-1,end);
    clear Res_t;  
    ThetaWAG=THETA(1                           :tWAG);
    ThetaLAM=THETA(tWAG+1                      :tWAG+tLAM);
    ThetaDTA=THETA(tWAG+tLAM+1                 :tWAG+tLAM+tDTA);
    ThetaPHI=THETA(tWAG+tLAM+tDTA+1            :tWAG+tLAM+tDTA+tPHI);
    ThetaSCH=THETA(tWAG+tLAM+tDTA+tPHI+1       :tWAG+tLAM+tDTA+tPHI+tSCH);
    THETAF=[ThetaWAG;ThetaLAM;ThetaDTA;ThetaPHI];
elseif Estimate==1 && Saved==1 && EM==0 % use previous version estimates
    load FresultsXVsq.mat
    THETA=THETA2(1:end-dimUc);
    pk=THETA2(end-dimUc+1:end);
    clear THETA2;  
    ThetaWAG=THETA(1                           :tWAG);
    ThetaLAM=THETA(tWAG+1                      :tWAG+tLAM);
    ThetaDTA=THETA(tWAG+tLAM+1                 :tWAG+tLAM+tDTA);
    ThetaPHI=THETA(tWAG+tLAM+tDTA+1            :tWAG+tLAM+tDTA+tPHI);
    ThetaSCH=THETA(tWAG+tLAM+tDTA+tPHI+1       :tWAG+tLAM+tDTA+tPHI+tSCH);
    THETAF=[ThetaWAG;ThetaLAM;ThetaDTA;ThetaPHI];
end

%Final THETA
THETA0=THETA; % net of type probabilities
THETA=[THETA;pk];
THETAg=THETA; % save final guess vector

logL=f_logLk(THETA);
logL1=logL;

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% SIMULATION
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

nn=1e4; % number of individuals to simulate
AbMat=cell(max(Prd),1);
for i=1:max(Prd) % define Skill age 16 and Skill age 7 data distributions
     AbMat{i}=MyTabulate2(Ab2(Prd==i),Ab0(Prd==i));
end

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% MINIMIZATION ROUTINE
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

disp('PAUSED. Routine is about to start. Press CTRL+C to break now.');
pause(10)

tic;

niter=1;                            % number of max iteration in ESM algorithm
tollerance=1e-3;                    % tollerance within ESM iteration
ESMtol=1e-1;                        % tollerance of ESM algorithm

logLE0=logL1;                       % initialize log-likelihood (iteration 0)    
logLWAG=0;
logLLAM=0;
logLDTA=0;
logLPHI=0;
logLSCH=0;

 Res_t(:,1)=[THETA;logLE0*1.0e-03];  % initialize vector of results [coefs,weights,log-lik]
 
 delete EM.txt
 diary EM.txt
 datestr(now)
 [0 pk' logLE0*1.0e-03 0]
 diary off
 
 for i=1:niter
     
     t1=cputime; % define cputime of each iteration
     
     %E Step: find conditional probabilities using Bayes rule
     Prk=f_pcond(THETA);
     
     if Sequential==0 % EM algorithm
 
         %M Step 1: find argmax Likelihood
         fhundle=@f_likelihood;    [THETA0,logL]=f_mstep(THETA0,bl,bu); 
         
     elseif Sequential==1 % ESM algorithm (Arcidiacono & Jones)
       
         %M Step 1: find argmax Likelihood
         fhundle=@f_likelihoodWAG; [ThetaWAG,logLWAG]=f_mstep(ThetaWAG,blWAG,buWAG); 
         fhundle=@f_likelihoodLAM; [ThetaLAM,logLLAM]=f_mstep(ThetaLAM,blLAM,buLAM); 
         fhundle=@f_likelihoodDTA; [ThetaDTA,logLDTA]=f_mstep(ThetaDTA,blDTA,buDTA); 
         fhundle=@f_likelihoodPHI; [ThetaPHI,logLPHI]=f_mstep(ThetaPHI,blPHI,buPHI); 
 
         THETAF=[ThetaWAG;ThetaLAM;ThetaDTA;ThetaPHI]; % THETA input vector for logLSCH
 
         fhundle=@f_likelihoodSCH; [ThetaSCH,logLSCH]=f_mstep(ThetaSCH,blSCH,buSCH);
         
     end
     
     %M Step 2: (update pk)
     pk=(sum(Prk,1)/ni)'; % new pk    
 
     if Sequential==0
         THETA=[THETA0;pk];
     elseif Sequential==1
         THETA=[ThetaWAG; ThetaLAM; ThetaDTA; ThetaPHI; ThetaSCH; pk];
     end
     
     logLE1=f_logLk(THETA); % update log-lik
     
     if i>1 & (logLE0>logLE1) & (logLE0-logLE1<ESMtol)
         disp('end of iterations: convergence achieved')
         break
     elseif (logLE0<logLE1)
         disp('error: logLE0<logLE1')
         disp('pause model')
         pause        
     else
         logLE0=logLE1;
         t2=cputime;
         Res_t(:,i+1)=[THETA;logLE0*1.0e-03];
         save EMresultsXVI.mat Res_t;
         diary on
         [i pk' logLE0*1.0e-03 (t2-t1)/60]
         diary off
     end
     
 end
        
% ESM output
iter=i;
THETA2=THETA;
logLE=logLE0;

%Standard Errors
S=gradp(@f_se,THETA2);
H=(S'*S);
I=inv(H);
se=sqrt(diag(I));
zratio=THETA2./se;

aa=toc/60;     % minutes to complete routine

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% RESULTS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

diary FresultsXVIII.txt

coeff={
'sigU   =  ';
'al_01  =  ';'al_02  =  ';'al_03  =  ';
'al_K2  =  ';'al_K0  =  ';
'al_S1  =  ';'al_S2  =  ';'al_S3  =  ';
'al_X0  =  ';'al_X1  =  ';'al_X2  =  ';'al_X3  =  ';

'l1_01  =  ';
'l1_K2  =  ';'l1_K0  =  ';
'l2_00  =  ';
'l2_K2  =  ';'l2_K0  =  ';
'l1_02  =  ';'l1_03  =  ';

'dt_01  =  ';
'dt_K2  =  ';'dt_K0  =  ';
'dt_S1  =  ';'dt_S2  =  ';'dt_S3  =  ';
'dt_X1  =  ';  
'dt_01  =  ';'dt_11  =  ';
'dt_02  =  ';'dt_03  =  ';

'ph_01  =  ';
'ph_K2  =  ';'ph_K0  =  ';
'ph_S1  =  ';'ph_S2  =  ';'ph_S3  =  ';
'ph_X1  =  ';
'ph_02  =  ';'ph_03  =  ';

'sigE1  =  ';'sigE2  =  ';'sigE3  =  ';
'g1_01  =  ';%'g1_Yp  =  ';'g1_K2  =  ';'g1_K0  =  ';
'g2_00  =  ';%'g2_Yp  =  ';'g2_K2  =  ';'g2_K0  =  ';
'g3_00  =  ';%'g3_Yp  =  ';'g3_K2  =  ';'g3_K0  =  ';
'g1_02  =  ';'g1_03  =  ';

'pk_01  =  ';'pk_02  =  ';'pk_03  =  ';
};

Res=num2cell([THETAg THETA2 se zratio]);
index=num2cell((1:length(THETA2))');
results={index coeff Res};

disp('**********************************************************');
disp('**********************************************************');
datestr(now)
disp([cd]);
disp(['Estimate                     ' num2str(Estimate)]);
disp(['Sequential Likelihood        ' num2str(Sequential)]);
disp(['Dynamic                      ' num2str(Dynamic)]);
disp(['Guess                        ' num2str(Guess)]);
disp(['Simulation                   ' num2str(Simulation)]);
disp(['Saved                        ' num2str(Saved)]);
disp(['routine completed in         ' num2str(aa) ' minutes']);
disp(['tolerance:                   ' num2str(tollerance)]);
disp(['discount factor:             ' num2str(beta)]);
disp(['Log Likelihood guess:        ' num2str(logL)]);
disp(['Log Likelihood value:        ' num2str(logLE)]);
disp(['Log Likelihood WAG:          ' num2str(logLWAG)]);
disp(['Log Likelihood LAM:          ' num2str(logLLAM)]);
disp(['Log Likelihood DTA:          ' num2str(logLDTA)]);
disp(['Log Likelihood PHI:          ' num2str(logLPHI)]);
disp(['Log Likelihood SC1:          ' num2str(logLSCH)]);
disp(['number of individuals:       ' num2str(ni)]);
disp(['number of observations:      ' num2str(no)]);
disp(['simulation sample size:      ' num2str(nn)]);
disp(['number of periods:           ' num2str(np)]);
disp(['number of wage points:       ' num2str(length(indW))]);
disp(['K2 K0 tabulate:              ' num2str(kp)]);
disp(['grid points K2 Sc K0 Ec Xc : ' num2str([dimK2 dimSc dimK0 dimEc dimXc])]);
disp(['unbalance panel:             ' num2str(ni*np>no)]);
disp(['missing wages:               ' num2str(ni-length(indW))]);
disp(['wage scale:                  ' num2str(scale)]);
disp(['number of ESM iterations:    ' num2str([niter iter])]);
disp(['eigenv. smaller than one:    ' num2str(find(eig(H)<1)')])

[results{1} results{2} results{3}]

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% SIMULATION & STATISTICS
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%   

[SSmat,QSmat,TSmat,WSmat,KSmat,TSmatH,...
    QK2Smat,QK0Smat,QK2SHmat,QK0SHmat,WSHmat,WXSmat]=f_simulation(THETA2);

Smat=zeros(3,1,np);
Qmat=zeros(dimSc,1,np);
Tmat=zeros(20,1,np);
Wmat=zeros(dimSc+1,1,3);
Kmat=zeros(dimSc,1,3);

i=1;

% Status 
for y=1:np
    for z=1:3 % 3 status: 1 school, 2 work, 3 unemployment 
        Smat(z,1,y)=sum(Sta(Prd==y)==z)/sum(Prd==y);   
    end

% Qualification
    for z=0:dimSc % 4 schooling levels : 0 No Qual, 1 O Lev, 2 A Lev, 3 HE
        Qmat(z+1,1,y)=sum(Qua(Prd==y)==z)/sum(Prd==y);   
    end

% Transition
    for z=1:20 % 20 transitions 
            Tmat(z,1,y)=sum(Typ(Prd==y)==z)/sum(Prd==y);   
    end
            
    if y==8 || y==19 || y==27
        Wmat(1,1,i)=mean(Wag(Wag~=0));
        Wmat(1,2,i)=mean(Wag(Prd==y&Wag~=0));
         for z=0:dimSc-1
            Wmat(z+2,1,i)=mean(Wag(Wag~=0&Qua==z)); 
            Wmat(z+2,2,i)=mean(Wag(Prd==y&Wag~=0&Qua==z));
            Kmat(z+1,1,i)=mean(Ab2(Prd==y&Qua==z));
        end
        i=i+1;
    end
end

Smat(end+1,:,:)=sum(Smat); 
Qmat(end+1,:,:)=sum(Qmat); 
Tmat(end+1,:,:)=sum(Tmat); 

Diff=Tmat-TSmat;
Diff(end,1,:)=sum(abs(Diff(1:end-1,1,:)),1);

disp('*** Schooling Choices ***');
[repmat((1:21)',[1 1 4]) Tmat(:,:,1:4) TSmat(:,:,1:4) Diff(:,:,1:4)]

disp('*** Schooling Choices with Heterogeneity ***');
[repmat((1:21)',[1 1 4]) TSmatH(:,:,1:4) ]

disp('*** Wage fit ***');
[Wmat*scale WSmat*scale (Wmat-WSmat)*scale]

disp('*** Retuns to Experience (with X=10) ***');
[THETAg(1+dimUc+2+1)+THETAg(1+dimUc+2+dimSc+1)*10, THETA2(1+dimUc+2+1)+THETA2(1+dimSc+2+dimSc+1)*10]
[THETAg(1+dimUc+2+2)+THETAg(1+dimUc+2+dimSc+2)*10, THETA2(1+dimUc+2+2)+THETA2(1+dimSc+2+dimSc+2)*10]
[THETAg(1+dimUc+2+3)+THETAg(1+dimUc+2+dimSc+3)*10, THETA2(1+dimUc+2+3)+THETA2(1+dimSc+2+dimSc+3)*10]

diary off;

xAb2=prctile(Ab2(Prd==8),20:20:100);
j=1;
for i=1:length(xAb2)
    for k=0:max(Qua)
        if i==1
            a1=find(Ab2<=xAb2(i)&Prd==8);
            QK2mat(j,k+2)=sum(Qua(a1)==k)/length(a1);
        else
            a2=find(Ab2>xAb2(i-1)&Ab2<=xAb2(i)&Prd==8);
            QK2mat(j,k+2)=sum(Qua(a2)==k)/length(a2);
        end
    end
    QK2mat(j,1)=i;
    
    % Average Wage by Cognitive Skills (Age 16)
    if i==1
        WK2mat(j,2)=mean(Wag(Wag~=0&Ab2<=xAb2(i)));
        for k=0:max(Qua)
            WK2mat(j,2+k+1)=mean(Wag(Wag~=0&Ab2<=xAb2(i)&Qua==k));
        end
    else
        WK2mat(j,2)=mean(Wag(Wag~=0&Ab2>xAb2(i-1)&Ab2<=xAb2(i)));  
        for k=0:max(Qua)
            WK2mat(j,2+k+1)=mean(Wag(Wag~=0&Ab2>xAb2(i-1)&Ab2<=xAb2(i)&Qua==k));
        end
    end
    WK2mat(j,1)=i;
    
    j=j+1;
end

xAb0=prctile(Ab0(find(Prd==8)),[20:20:100]);
j=1;
for i=1:length(xAb0)
    for k=0:max(Qua)
        if i==1
            a1=find(Ab0<=xAb0(i)&Prd==8);
            QK0mat(j,k+2)=sum(Qua(a1)==k)/length(a1);
        else
            a2=find(Ab0>xAb0(i-1)&Ab0<=xAb0(i)&Prd==8);
            QK0mat(j,k+2)=sum(Qua(a2)==k)/length(a2);
        end
    end
    
    QK0mat(j,1)=i;
    j=j+1;
end

for i=1:dimXc
    WXmat(i,1)=i;
    for j=0:dimSc-1
        a=find(Wag~=0&Exp==i&Qua==j);
        if length(a)>0
            WXmat(i,j+2)=mean(Wag(a));
        else
            WXmat(i,j+2)=0;
        end
    end
end
      
if Sequential==0

    save FresultsXVIII.mat ...
       logLE THETA2 THETAg ...
       Smat SSmat Tmat TSmat TSmatH...
       QK2mat QK0mat QK2Smat QK0Smat QK2SHmat QK0SHmat...
       Wmat WSmat WSHmat WXmat WXSmat;
   
elseif Sequential==1

    save FresultsXVIIIsq.mat ...
       logLE THETA2 THETAg ...
       Smat SSmat Tmat TSmat TSmatH...
       QK2mat QK0mat QK2Smat QK0Smat QK2SHmat QK0SHmat...
       Wmat WSmat WSHmat WXmat WXSmat;   
end
