%%% Accompanying Matlab code for "Family Planning in a Life Cycle Model with Income Risk"
%%% by Mette Ejrns and Thomas H. Jrgensen.
%%% This script estimates the dynamic model of consumption, imperfect contraception control and abortion
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% By Thomas Jrgensen, tjo@econ.ku.dk.
%%% This version: April, 2018.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

clear all; clc
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 1. compile mex and settings
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
threads = 7;

% only re-compile if not done before
%prep.compile_mex('mex_solve',threads,0);
%prep.compile_mex('mex_simulate',threads,0);

SIM_FAC = 80; % number of simulation datasets used
LOAD    = 0;
SAVE    = 1; % 1-> results are saved in a struct
ALT     = 1;
EDUCs   = [0 1];


spec_vars = {'beta','rho','sigma_perm','sigma_trans'};


%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% 2. Optimization settings %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% i. Parameters to be estimated
EstPar      = {'add_1','add_2','add_3','marg','cost_abort','p_unplanned','cost_perm_1','cost_perm_2'};
UseMoments  = {'RelShare','Unplanned','abortion_rate','Kids1','Kids2','Kids3','IncGrowth'};

% ii. Options for particleswarm:
optionsG.do_particleswarm   = 1;
optionsG.do_patternsearch   = 0;
optionsG.NumStart           = 2; % Number of multi-starts to run
optionsG.options_swarm      = optimoptions('particleswarm','Display','iter','SwarmSize',50,'MaxIter',40*numel(EstPar),'FunctionTolerance',1.0e-4);

% iii. Options for fminsearch:
optionsG.options        = optimset('MaxIter',1000,'TolX',1.0e-6,'TolFun',1.0e-6,'Display','iter');



for EDUC  = EDUCs  % 1->High skilled
    if EDUC==0
        specs = { {spec_vars , [.97 1.5 .1*ones(1,2)]} ...
            };
    else
        specs = { {spec_vars , [.98 1.5 .1*ones(1,2)]} ...
            };
    end

    for s=1:numel(specs)
        spec_vars = specs{s}{1};
        spec_pars = specs{s}{2};

        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 3. Setup model solution and load data %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % i. Setup model parameters and state-space (depend on educational attainment)
        par             = est.setup_est(EDUC);
        par.do_alt      = ALT;
        
        if EDUC==0
            
            par.add_1=6.8648;
            par.add_2=2.6085;
            par.add_3=1.8509;
            par.marg=1.8139;
            par.cost_abort=14.7216;
            par.p_unplanned=6.3312;
            par.cost_perm_1=-4.8424;
            par.cost_perm_2=-0.1543;
            
        else
 
            par.add_1=7.2837;
            par.add_2=1.2349;
            par.add_3=1.1045;
            par.marg=1.3352;
            par.cost_abort=7.1872;
            par.p_unplanned=3.8529;
            par.cost_perm_1=-7.5799;
            par.cost_perm_2=0.2673;
            
        end
        
        % update based on specs
        for i=1:numel(spec_pars)
            par.(spec_vars{i}) = spec_pars(i);
        end
        par             = prep.construct_grids(par);
        
        par.UseMoments  = UseMoments;
        rng(par.seed) % set seed
        
        name = ['_educ' num2str(EDUC) '_beta' num2str(par.beta) '_rho' num2str(par.rho) '_sigma' num2str(par.sigma_perm)];
        
        diary off
        if SAVE==1
            diary(['logs\Alt' num2str(ALT) name '.txt'])
        end
        
        % ii. Load BHPS data used to simulate savings series.
        data = est.load_data(EDUC,par);
        
        % iii. Calculate the moments using the data and Bootstrap the covariance
        par.mom_data = est.calc_moments(data,par);
        
        % iv. Bootstrap the covariance matrix of the moments from data
        par.CoVar = est.bootstrap_cov(data,par);
        
        par.mom_data_std = sqrt(diag(par.CoVar));
        par.Omega        = par.CoVar*data.Nobs;
        par.W            = inv(diag(diag(par.Omega)));
        
        % adjust the weight put on the income moments (two last moments)
        par.W(end-1:end,end-1:end) = 10*par.W(end-1:end,end-1:end); % increase the weight
        % adjust the moments associated with the saving rate response (first two moments)
        par.W(1:2,1:2) = 10*par.W(1:2,1:2); % increase the weight
        % adjust the moments associated with the unintended births (third through fifth moments)
        par.W(3:5,3:5) = 100*par.W(3:5,3:5); % increase the weight
        
        moms = [par.mom_data,par.mom_data_std,diag(par.W)]
        
        
        % v. Draw shocks and initial values. A multiple of the sample size
        par.S       = SIM_FAC;
        par.simN    = data.Nobs*par.S;
        par.simT    = par.simT;
        data.draws  = est.draws(par,data);
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % 4. Estimate model parameters %
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        % i. initial values
        theta0 = NaN(numel(EstPar),1);
        for p=1:numel(EstPar), theta0(p) = par.(EstPar{p}); end
        optionsG.options_swarm.InitialSwarmMatrix = theta0';
        
        % ii. global search
        if LOAD==0
            [estimat,obj,all]  = est.estimate(EstPar,data,par,optionsG);
            par.est            = estimat;
            
            [obj_opt,output]= est.ObjFun(par.est,EstPar,data,par,inv(par.Omega));
            par             = output.par;
            par.EstPar      = EstPar;
            par.obj         = obj;
            par.obj_opt     = obj_opt;
            
            % iii. calculate standard errors
            [par.SE ,par.grad,par.sens] = est.SE_sens(par.est,EstPar,data,par,par.W);
            
        else
            load(['results\est' name '.mat']);
        end
        
        % iv. save estimation results
        if SAVE==1
            save(['results\est' name '.mat'],'par');
        end
        
        diary off
    end % specs
end % educational group loop

