classdef prep
    methods (Static)
        function [] = compile_mex(name,threads,intel)
            
            % a. clear existing mex-file in userpath
            dirpath = sprintf('%s/cfuncs',userpath);
            if isdir(dirpath)
                rmdir(dirpath,'s')
            end
            copyfile(sprintf('%s/cfuncs',pwd),sprintf('%s/cfuncs',userpath))

            % b. build string
            str = sprintf('mex  -largeArrayDims -outdir %s %s/cfuncs/%s.cpp -DMAXTHREADS=%d',pwd,userpath,name,threads);
            %str = sprintf('mex -largeArrayDims -outdir %s %s/cfuncs/%s.cpp -DMAXTHREADS=%d',userpath,userpath,name,threads);
            
            % flags
            if intel == 0
                str = sprintf('%s %s',str,' CXXFLAGS="$CXXFLAGS -std=c++11 -O3 -Wall -fopenmp -ffast-math"');
            else
                %str = sprintf('%s %s',str,' COMPFLAGS=''$COMPFLAGS /o3 /openmp /arch:CORE-AVX512''');
                str = sprintf('%s %s',str,' COMPFLAGS=''$COMPFLAGS /o3 /openmp ''');
            end
            
            % libgomp (for OpenMP)
            if intel == 0
                str = sprintf('%s %s',str,' C:/ProgramData/MATLAB/SupportPackages/R2018b/3P.instrset/mingw_w64.instrset/lib/gcc/x86_64-w64-mingw32/6.3.0/libgomp.a');
            end
            
            % c. evaluate
            eval(str);
            fprintf('\n');
            
        end
        
        function par = setup()
            
            % Parameters
            par.beta = 0.97;
            par.rho  = 2;
            
            par.gamma = 1;
            par.R     = 1.03;
            
            par.marg         = 0.05;
            par.add_1        = 0.1;
            par.add_2        = 0.1;
            par.add_3        = 0.1;
            par.cost_abort   = 3;
            
            % Demographic
            par.agemin = 23;
            par.agemax = 62;
            par.TR     = par.agemax-par.agemin+1;
            par.T      = par.TR + 20;
            par.max_age_pregnant = 47;
%             par.agemin = 25;
%             par.agemax = 59;
            
            % biological fecundity
            % Trussel and Wilson (1985, Table 8)
            TWage = [ 22  27  32   37   42   47]';
            TWinf = [4.6 9.1 16.6 25.4 62.2 92.9]'./100; % infertility share
            x     = [ones(numel(TWage),1), TWage];
            b     =(x'*x)\(x'*log(TWinf));
            fit   = exp(x*b);
            
            fit_fun = @(age) min( 1 , exp([ones(size(age,1),1), age ]*b) );
            par.fert_fun  = @(age) 1 - fit_fun(age);

            %fert_fun = @(Age) min(1 , max(0 , 1-interp1(TWage,fit,Age,'linear','extrap')) );
            
            par.fecundity    =  par.fert_fun((par.agemin:(par.agemin+par.T))');
            
            par.p_unplanned  = 0.017; % probability for unplanned pregnancy
            
            % Income process
            par.alpha = 1;
            par.kappa = .9; % replacement rate in retirement
            
            % income growth parameters
            par.g0    = 0;
            par.g1    = 0;
            
            par.Nshocks     = 1;%5;
            par.sigma_trans = 0.0;%0.1;
            par.sigma_perm  = 0.0;%0.1;
            
            par.cost_perm_1 = 0.0;
            par.cost_perm_2 = 0.00;
            par.cost_perm_3 = 0.0;
            par.cost_trans_1 = 0.0;
            par.cost_trans_2 = 0.00;
            par.cost_trans_3 = 0.0;
            
            par.cost_perm_1_u = 0.0;
            par.cost_perm_2_u = 0.0;
            par.cost_perm_3_u = 0.0;
            
            % welfare calculations 
            par.cost_welfare  = 0;
            
            % State variables
            par.Nk   = 2;%3+1;%1+3; % max number of kids including none
            par.Ng   = 2;%2;%3 % number of pregnancy states. 0: not pregnant, 1: pregnant, planned, 2: pregnant, unplanned
            par.Ne   = 2;%2; % effort: 0: no effort, 1: effort
            par.Nq   = 2;%2; % abortion: 0: no abortion, 1: abortion
            
            par.Np    = 40;%10; % number of grid points in the permanent income grid
            par.min_P = .05;%.1;
            par.max_P = 15;%10; % in 100,000DKK
            
            par.Nm   = 300; % Cash-on-hand grid
            par.Na   = round(1.5*par.Nm); % savings grid
            par.Na_ret = 400; % savings grid in retirement.
            par.max_m = 20; % as a share of permanent income
            par.min_m = 1.0e-5;
            
            par.credit = 0.0; % share of permanent income allowed to be borrowd (positive number)
            par.Nc_guess = 400;
            par.min_C    = 1.0e-6;
            
            par.do_pd    = 1;
            par.do_nlopt = 0;
            par.do_multistart = 0;
            par.do_egm = 1;
            
            % Simulation
            par.seed = 2018;
            par.simN = 1000;
            par.simT = par.TR;
            
            par.do_euler_error = 0;
            par.do_welfare = 0;
            
        end
        function par = construct_grids(par)
            
            % state space
            par.grid_m = prep.nonlinspace(par.min_m - par.credit,par.max_m,par.Nm,1.2);
            par.grid_a = prep.nonlinspace(par.min_m - par.credit,1.1*par.max_m,par.Na,1.2);
            par.grid_a_ret = prep.nonlinspace(par.min_m - par.credit ,par.max_m,par.Na_ret,1.2);
            par.grid_P = prep.nonlinspace(par.min_P,par.max_P,par.Np,1.2);
            par.grid_P_alpha = par.grid_P.^par.alpha;
            
            % stochastic elements
            [x,w]  = prep.gauss_hermite(par.Nshocks);
            nodes  = x*sqrt(2.0);
            
            perm   = exp(par.sigma_perm*nodes - .5*par.sigma_perm^2);
            perm_w = w*pi^(-1/2);
            
            trans   = exp(par.sigma_trans*nodes - .5*par.sigma_trans^2);
            trans_w = w*pi^(-1/2);
            
            % vectorize income shocks and weights
            [par.perm,par.trans] = ndgrid(perm,trans);
            par.perm  = par.perm(:);
            par.trans = par.trans(:);
            [perm_w,trans_w]     = ndgrid(perm_w,trans_w);
            par.weight           = perm_w(:).*trans_w(:);
            
            % income growth
            age_grid = (1:par.T) + par.agemin-1;
            par.G = exp(par.g0 + par.g1*(age_grid - par.agemin)/100);
            par.G(par.TR+1:end) = 1;
            
        end
        
        
        function x     = nonlinspace(lo,hi,n,phi)
            % recursively constructs an unequally spaced grid
            % phi > 1 -> more mass at the lower end of the grid
            % lo can be a vector (x then becomes a matrix)
            x      = NaN(n,length(lo));
            x(1,:) = lo;
            for i = 2:n
                x(i,:) = x(i-1,:) + (hi-x(i-1,:))./((n-i+1)^phi);
            end
        end
        function [x,w] = gauss_hermite(n)
            if n == 1
                xw = [0 sqrt(pi)];
            elseif n == 2
                xw = ...
                    [-7.071067811865476e-1   8.86226925452758e-1,
                    7.071067811865476e-1    8.86226925452758e-1];
            elseif n == 4
                xw = ...
                    [-1.650680123885785e0    8.13128354472452e-2,
                    -5.246476232752904e-1   8.04914090005513e-1,
                    5.246476232752904e-1    8.04914090005513e-1,
                    1.650680123885785e0     8.13128354472452e-2];
            elseif n==5
                xw = ...
                    [-2.020182870456085632929	0.01995324205904591320774
                    -0.9585724646138185071128	0.393619323152241159828
                    0                           0.945308720482941881226
                    0.9585724646138185071128	0.3936193231522411598285
                    2.020182870456085632929     0.01995324205904591320774];
                
            elseif n == 8
                xw = ...
                    [-2.930637420257244e0    1.996040722113676e-4,
                    -1.981656756695843e0    1.707798300741347e-2,
                    -1.15719371244678e0     2.078023258148919e-1,
                    -3.811869902073221e-1   6.611470125582414e-1,
                    3.811869902073221e-1    6.611470125582414e-1,
                    1.15719371244678e0      2.078023258148919e-1,
                    1.981656756695843e0     1.707798300741347e-2,
                    2.930637420257244e0     1.996040722113676e-4];
            elseif n == 10
                xw = ...
                    [-3.436159118 0.7640432855e-5,
                    -2.532731674 0.1343645746e-2,
                    -1.756683649 0.3387439445e-1,
                    -1.036610829 0.2401386110,
                    -0.3429013272 0.6108626337,
                    0.3429013272 0.6108626337,
                    1.036610829 0.2401386110,
                    1.756683649 0.3387439445e-1,
                    2.532731674 0.1343645746e-2,
                    3.436159118 0.7640432855e-5];
                error('unknown number of GaussHermite nodes');
            end
            x = xw(:,1);
            w = xw(:,2);
        end
        
        
        
    end
end