classdef est
    methods (Static)
        function par = setup_est(EDUC)
            
            par = prep.setup();
            
            par.PRINT  = 1;
            par.num_bs = 1000; % number of bootstrap-replications to estimate covariance of moments
            par.do_alt = 0;
            
            % Parameters
            par.beta = 0.97;
            if EDUC==1
                par.beta = 0.98;
            end
            par.rho  = 1.5;
            
            par.gamma = 1;
            par.R     = 1.03;
            
            par.marg         = 0.1;
            par.add_1        = 0.1;
            par.add_2        = 0.1;
            par.add_3        = 0.1;
            par.cost_abort   = .8;
            
            % Demographic
            par.agemin = 23;
            par.agemax = 62;
            par.TR     = par.agemax-par.agemin+1;
            par.T      = par.TR + 20;
            par.max_age_pregnant = 47;

            par.fecundity    =  par.fert_fun((par.agemin:(par.agemin+par.T))');
            par.p_unplanned  = 0.017; % probability for unplanned pregnancy
            
            % Income process
            par.alpha = 1;
            par.kappa = .7; % replacement rate in retirement

            % State variables
            par.Nk   = 4; % max number of kids including none
            par.Ng   = 3; % number of pregnancy states. 0: not pregnant, 1: pregnant, planned, 2: pregnant, unplanned
            par.Ne   = 2; % effort: 0: no effort, 1: effort
            par.Nq   = 2; % abortion: 0: no abortion, 1: abortion
            
            par.Np    = 50;%30; % number of grid points in the permanent income grid
            par.min_P = .05;
            par.max_P = 15;
            
            par.Nm     = 200; % Cash-on-hand grid
            par.Na     = par.Nm;%round(1.5*par.Nm); % savings grid
            par.Na_ret = 400; % savings grid in retirement.
            par.max_m  = 15; % as a share of permanent income
            par.min_m  = 1.0e-5;
            
            par.credit   = 0.0; % share of permanent income allowed to be borrowed (positive number)
            
            par.do_pd    = 1;
            par.do_egm   = 1;
            par.do_euler_error = 0;
            
            par.AgeGroup_fun= @(age) 1*(age>=25 & age<=29) ...
                       + 2*(age>=30 & age<=34) ...
                       + 3*(age>=35 & age<=39);
            par.AgeGrid     = 1:3;
            
            if EDUC==0
                par.g0 = 0.025;
                par.g1 = -0.1;
                
%                 par.g0 = 0.031;
%                 par.g1 = -.11;
            else
%                 par.g0 = 0.045;
%                 par.g1 = -.14;
                par.g0 = 0.041;0.045;
                par.g1 = -0.13;-.14;
            end

            % shocks
            par.Nshocks     = 5;
            if EDUC==0
                par.sigma_trans   = 0.1;%sqrt(0.020);   % transitory income shock variance
                par.sigma_perm    = 0.1;%sqrt(0.018);   % persistent income shock variance
            else
                par.sigma_trans   = 0.1;%sqrt(0.021);   % transitory income shock variance
                par.sigma_perm    = 0.1;%sqrt(0.022);   % persistent income shock variance
            end
            
            % income penalty of children
            if EDUC==0
                par.cost_perm_1 = -.06; % effect of 1st child on persistent income
                par.cost_perm_2 = 0; % effect of 2nd child on persistent income

            else
                par.cost_perm_1 = -.06;%-0.111; % effect of 1st child on persistent income
                par.cost_perm_2 = .0;%-0.027; % effect of 2nd child on persistent income

            end
            
            % when estimating (simpler) income process with other parameters
            par.cost_trans_1 = 0;
            par.cost_trans_2 = 0;
            par.cost_trans_3 = 0;
            par.cost_perm_3  = 0;

            par.simT = par.TR;
        end
        
        % Estimation and objective function
        function [estimat,fval,all] = estimate(EstPar,data,par,options)
            
            % initial values
            MIN     = Inf;
            est_all = [];
            
            [LB,UB] = est.set_bounds(EstPar);
            obj     = @(theta) est.ObjFun(theta,EstPar,data,par,par.W);
            
            % loop over number of random starting points
            for i=1:options.NumStart
                
                all(i).UB = UB;
                all(i).LB = LB;
               
                % 1: particle swarm
                if options.do_particleswarm == 1
                    [est_swarm,f_swarm] = particleswarm(obj,numel(UB),LB,UB,options.options_swarm);
                else
                    est_swarm = options.options_swarm.InitialSwarmMatrix(1,:);
                end
                % 2: pattern search
                if options.do_patternsearch == 1
                    [est_pat,f_pat]     = patternsearch(obj,est_swarm,[],[],[],[],LB,UB,[]);%,options.options_pat);
                else
                    est_pat = est_swarm;
                end
                % use those estimates to initialize fminsearch
                [all(i).est,all(i).fval,all(i).exitflag] = fminsearch(obj,est_pat,options.options);
                if all(i).fval<MIN
                    MIN  = all(i).fval;
                    fval = all(i).fval;
                    estimat  = all(i).est;
                end
               
                % store
                est_all = [est_all;all(i).est];
                past_est = est_all(max(1,i-5):i,:);
                
            end
           
        end
        function [Obj,output]       = ObjFun(theta,EstPar,data,par,W)
            
            % 1. Update parameter struct with the given parameter values in theta of EstPar
            par = est.update_par(theta,EstPar,par,par.PRINT);    % transform parameters to struct

                % 1.1 Update grids: income parameters etc used here
                par   = prep.construct_grids(par);

            % 2. Solve model for the given parameters in theta
            sol = mex_solve(par);
           
            % 4. Simulate data (par.S*data.Nobs observations for par.simT periods)
            sim       = est.simulate(par,sol,data.draws); 
            mom_sim   = est.calc_moments(sim,par);
            
            % 5. Calculate criteria function 
            distance  = mom_sim-par.mom_data; 
            Obj       = distance'*W*distance;
           
            if par.PRINT==1, fprintf(' ->Obj=%3.7f\n',Obj); end
           
            % 6. Store additional output in struct
            par.mom_sim     = mom_sim;
            output.mom_sim  = mom_sim;
            output.mom_data = par.mom_data;
            output.Weight   = par.W;
            output.par      = par;
            
        end
        function sim                = simulate(par,sol,draws)
            
            sim = mex_simulate(par,sol,draws);
            
            %sim.SavingRate = max( 0 , (sim.Y - sim.C)./sim.Y );
            sim.type = 'model';
            
            if par.do_alt==0
%                 sim.UnplannedBirth  = sim.b.*(1-sim.effort);
%                 sim.PlannedBirth    = sim.b.*sim.effort;
%             
%                 sim.EffortNotPregnant = (1-sim.b).*sim.effort;

            else
                % solve and simulate perfect control version
                par_alt             = par;
                par_alt.do_alt      = 0; % otherwise it will loop forever 
                par_alt.p_unplanned = 0;
                par_alt.Ng          = 2; % no unintended
                par_alt.Nq          = 1; % thus no abortion choice
                
                sol     = mex_solve(par_alt);
                sim_alt = est.simulate(par_alt,sol,draws);
                
                % calculate unintended and intended childbirths
                sim.TargetChildren  = repmat(max(sim_alt.k , [] ,2) , 1 , par_alt.simT);
                UnplannedChild      = (sim.k>sim.TargetChildren);
                sim.UnplannedBirth  = sim.b.*UnplannedChild;
                sim.PlannedBirth    = sim.b.*(1-UnplannedChild);
                
                sim.PlannedBirth_nonan    = sim.PlannedBirth;
                sim.UnplannedBirth_nonan    = sim.UnplannedBirth;
            end
         
        end
        
        function par        = update_par(theta,EstPar,par,PRINT)
            
            i_par = 1;
            for i=1:numel(EstPar)
                num_now = numel(par.(EstPar{i}));
                par.(EstPar{i}) = theta(i_par:i_par+num_now-1);
                if nargin<4 || PRINT==0
                    % Do not print
                else
                    if num_now>1
                        for j=1:num_now
                            var_now = par.(EstPar{i});
                            fprintf('%s[%d]=%2.3f, ',EstPar{i},j,var_now(j));
                        end
                    else
                        fprintf('%s=%2.4f, ',EstPar{i},par.(EstPar{i}));
                    end
                end
                i_par = i_par + num_now;
            end
        end
        function [LB,UB]    = set_bounds(EstPar)
            LB = NaN(numel(EstPar),1);
            UB = NaN(numel(EstPar),1);
            for p=1:numel(EstPar)
                if strcmp(EstPar{p},'add_1') || strcmp(EstPar{p},'add_2') || strcmp(EstPar{p},'add_3')
                    LB(p) = 0 * 100;
                    UB(p) = 0.13*100;
                elseif strcmp(EstPar{p},'cost_abort')
                    LB(p) = .05 * 100;
                    UB(p) = 1.5 * 100;
                elseif strcmp(EstPar{p},'marg') 
                    LB(p) = 0;
                    UB(p) = .1 * 100;
                elseif strcmp(EstPar{p},'p_unplanned')
                    LB(p) = 0.005 * 100;
                    UB(p) = 0.1 * 100;
                elseif strcmp(EstPar{p},'cost_perm_1')
                    LB(p) = -0.15 * 100;
                    UB(p) = 0.00; 
                elseif strcmp(EstPar{p},'cost_perm_2')
                    LB(p) = -0.10 * 100;
                    UB(p) = 0.01 *100;        
                end
            end
        end
        
        % Data and draws
        function data   = load_data(EDUC,par)
            data.age            = importdata(strcat('MatlabData\age_',num2str(EDUC),'.txt')) ;
            data.Y              = importdata(strcat('MatlabData\Income_',num2str(EDUC),'.txt')) / 10000 ;
            data.A              = importdata(strcat('MatlabData\Wealth_',num2str(EDUC),'.txt')) / 10000;
            
            data.SavingRate     = importdata(strcat('MatlabData\SavingRate_',num2str(EDUC),'.txt'));
            data.k              = importdata(strcat('MatlabData\NumKids_',num2str(EDUC),'.txt'));
            data.b              = importdata(strcat('MatlabData\newborn_',num2str(EDUC),'.txt'));
            data.e              = importdata(strcat('MatlabData\ExpChildUpdate_',num2str(EDUC),'.txt'));
            data.e(data.e<0)    = NaN; % replace -2 with NaN because they are encoded like that in STATA
            data.wave           = importdata(strcat('MatlabData\wave_out_',num2str(EDUC),'.txt'));
            
            naneffort = isnan(lagmatrix(data.e',1)');
            data.UnknownBirth    = data.b.*naneffort;
            
            data.UnplannedBirth  = data.b.*(1-lagmatrix(data.e',1)');
            data.UnplannedBirth_nonan = data.UnplannedBirth;
            data.UnplannedBirth_nonan(naneffort) = 0;
            
            data.PlannedBirth    = data.b.*lagmatrix(data.e',1)';
            data.PlannedBirth_nonan = data.PlannedBirth;
            data.PlannedBirth_nonan(naneffort) = 0;
            
            data.dlogY           = lagmatrix(log(data.Y)',-1)'-lagmatrix(log(data.Y)',1)'; 
            data.d2SavingRate    = data.SavingRate-lagmatrix(data.SavingRate',2)'; 

            data.init.Y    = importdata(strcat('MatlabData\initIncome_',num2str(EDUC),'.txt')) / 10000;
            data.init.P    = importdata(strcat('MatlabData\initIncome_',num2str(EDUC),'.txt')) / 10000;
            data.init.A    = importdata(strcat('MatlabData\initWealth_',num2str(EDUC),'.txt')) / 10000;
            data.init.k    = importdata(strcat('MatlabData\initNumKids_',num2str(EDUC),'.txt'));
            
            data.Nobs = size(data.age,1);
            data.type = 'BHPS';
            
        end
        function draws  = draws(par,data)
            draws.perm      = randn(par.simN,par.simT);
            draws.trans     = randn(par.simN,par.simT);
            draws.pregnant  = rand(par.simN,par.simT);
            
            draws.g0     = zeros(par.simN,1); % assume nobody is pregnant.. TODO: use info if a child arrives next period
            if nargin<2
                draws.A0 = zeros(par.simN,1);
                draws.P0 = ones(par.simN,1);
                draws.k0 = zeros(par.simN,1);
            else
                % draw from the data
                Num_init = size(data.init.Y,1);
                I        = randsample(Num_init,par.simN,true);
                
                %draws.A0 = data.init.A(I);
                draws.A0 = 0*data.init.A(I);
                draws.P0 = data.init.Y(I)./1.1; % remove first periods growth
                draws.k0 = min(data.init.k(I),par.Nk-1);
            end
            
        end
        
        % Moments
        function [mom,MOMS] = calc_moments(data,par)

            UseMoments = par.UseMoments;
            AgeGrid    = par.AgeGrid;
            
            % Saving rates and income around childbirth 
            [MOMS.RelShare,MOMS.IncGrowth]  = est.outcomes_around_birth(data,par);
            
            % Children age profile
            if strcmp(data.type,'BHPS')
                AgeGroups      = par.AgeGroup_fun(data.age);
                isnotNaN_kids  = isnan(data.k)==0;
                MOMS.Kids1     = est.age_profile(AgeGrid,AgeGroups(isnotNaN_kids),data.k(isnotNaN_kids)>=1);
                MOMS.Kids2     = est.age_profile(AgeGrid,AgeGroups(isnotNaN_kids),data.k(isnotNaN_kids)>=2);
                MOMS.Kids3     = est.age_profile(AgeGrid,AgeGroups(isnotNaN_kids),data.k(isnotNaN_kids)>=3);
            else
                MOMS.Kids1 = NaN(numel(AgeGrid),1);
                MOMS.Kids2 = NaN(numel(AgeGrid),1);
                MOMS.Kids3 = NaN(numel(AgeGrid),1);
                
                age_min = 25;
                for a=1:numel(AgeGrid)
                    I                 = data.age>=age_min & data.age<=(4+age_min);
                    MOMS.Kids1(a)     = nanmean( data.k(I)>=1);
                    MOMS.Kids2(a)     = nanmean( data.k(I)>=2);
                    MOMS.Kids3(a)     = nanmean( data.k(I)>=3);
                    
                    age_min = age_min + 5;
                end
            end
            
            % Share of unplanned childbirths of all births. Devided by parity
            min_age = 25;
            max_age = 39;
            I = data.age>=min_age & data.age<=max_age;
            
            B1 = (data.k<=1).*data.b + 0*data.k.*data.UnplannedBirth; % add NaNs 
            B2 = (data.k==2).*data.b + 0*data.k.*data.UnplannedBirth;
            B3 = (data.k>=3).*data.b + 0*data.k.*data.UnplannedBirth;

            MOMS.Unplanned1 = nansum(B1(I).*data.UnplannedBirth(I))/nansum(B1(I));
            MOMS.Unplanned2 = nansum(B2(I).*data.UnplannedBirth(I))/nansum(B2(I));
            MOMS.Unplanned3 = nansum(B3(I).*data.UnplannedBirth(I))/nansum(B3(I));
            
            MOMS.Unplanned1(isnan(MOMS.Unplanned1)) = 0;
            MOMS.Unplanned2(isnan(MOMS.Unplanned2)) = 0;
            MOMS.Unplanned3(isnan(MOMS.Unplanned3)) = 0;

            MOMS.Unplanned = [MOMS.Unplanned1 ; MOMS.Unplanned2 ; MOMS.Unplanned3];


            % Abortion rates, relative to conceptions
            if strcmp(data.type,'BHPS')
                abortion_rate = 8.01/100;
            else
                aborted_pregnancies                   = data.q;
                aborted_pregnancies(data.g<1)         = NaN;  % remove all non-pregnancies. Now,, the age(-group) profile is the share of pregnancies resulting in abortions
                abortion_rate                         = nanmean(aborted_pregnancies(I));
                
                abortion_rate(isnan(abortion_rate))   = 0.0; 
                
            end
            MOMS.abortion_rate           = abortion_rate;
            
            % output the chosen moments
            mom = [];
            for i=1:numel(UseMoments)
                mom = [mom;  MOMS.(UseMoments{i})];
            end
            

        end
        function [sav,inc]  = outcomes_around_birth(data,par)
            % run a regression of saving rate growth and incoem growth on changes in the
            % number of children.
            
            min_age = 25;
            max_age = 39;
            I = data.age>=min_age & data.age<=max_age;
            
            % construct wave dummies
            if strcmp(data.type,'BHPS')
                dwave = dummyvar(data.wave(I));
                dwave = dwave(:,4:end); % exclude the first 1+1+1 waves dummy: 1 because expectations are asked for the first time in wave 2. 1 because we need lagged expectations, 1 because we also include a constant and the last becuase we lead the log income growth. 
            
                add_var = data.UnknownBirth(I);
            else
                dwave = []; % if simulation, there is no wave-effects
                add_var = [];
            end
            
            % construct age-controls
            dage = [data.age(I) data.age(I).*data.age(I)];

            % Saving rate: collect variables
            X = [data.PlannedBirth_nonan(I) data.UnplannedBirth_nonan(I) ones(size(data.UnplannedBirth_nonan(I))) dwave dage add_var];
            y = data.d2SavingRate(I); % do this d2 because of the timing in the model

            b   = est.reg(y,X);
            sav = b(1:2);
            
            % Income: collect variables
            dwave = dwave(:,1:end-1); % remove the last dummy due to the longer differences
            
            dKids1 = data.b.*(data.k==1) + 0*data.k;
            dKids2 = data.b.*(data.k==2) + 0*data.k;
            X      = [dKids1(I) dKids2(I) ones(size(dKids1(I))) dwave dage];
            
            y = data.dlogY(I); % this is now changed to the 3-period change
            
            b   = est.reg(y,X);
            inc = b(1:2);
            
        end     
        function profile    = age_profile(AgeGrid,age,VAR)
            Num = numel(AgeGrid);
            profile  = NaN(Num,1);
            for i=1:Num
                profile(i)  = nanmean(VAR(age==AgeGrid(i)));
            end
        end
        
        function b          = reg(y,X)
            % check and remove NaNs
            wasnan = any(isnan([y X]),2);
            if any(wasnan)
                y(wasnan)   = [];
                X(wasnan,:) = [];
            end
            
            % check if invertable
            if rcond(X'*X)<(1.0e-20)
                b = 10000*ones(size(X,2),1);
                
            else
                % return the OLS estimates
                b = (X'*X)\(X'*y);
            end
            
        end
        function dist       = distance(data,parity)
            K = data.k;
            K(isnan(K)) = 0;
            
            B = data.b;
            B(isnan(B)) = 0;
            
            Bp  =  (B==1 & K==parity);
            
            agep          = max(data.age.*Bp,[],2);
            agep(agep==0) = NaN;
            
            dist = data.age - agep;
        end
        
        function bsCov      = bootstrap_cov(data,par)
            
            N           = data.Nobs;
            num_mom     = numel(par.mom_data);
            bsMoments   = NaN(num_mom , par.num_bs);
            
            vars = {'A','Y','b','e','wave','age','SavingRate','k','UnplannedBirth','PlannedBirth','UnknownBirth','UnplannedBirth_nonan','PlannedBirth_nonan','dlogY','d2SavingRate'}; 
            
            for bs_i=1:par.num_bs

                % Draw random sample from the data. With replacement
                [~,id] = datasample(ones(N,1),N);
                id     = id';
                
                % Draw the same households used to calculate moments from data
                bs_data.type = data.type;
                for v=1:numel(vars)
                    bs_data.(vars{v}) = data.(vars{v})(id,:);
                end
                
                bsMoments(:,bs_i) = est.calc_moments(bs_data,par);
            end
            
            bsCov = cov(bsMoments');

            % adjust some elements of the covariance matrix if external (constant) sources are used
            %VAR.abortion_rate = .1*ones(3,1)/N;%[0.00000049 ; 0.00000051 ; 0.00000209];
            %VAR.abortion_rate = .2*ones(1,1)/N;%[0.00000049 ; 0.00000051 ; 0.00000209];
            VAR.abortion_rate = 1*ones(1,1)/N;%[0.00000049 ; 0.00000051 ; 0.00000209];
            j=1;
            while j<=size(bsCov,1)
                if bsCov(j,j)<1.0e-20
                    variance = VAR.abortion_rate;
                    num      = numel(variance);
                    bsCov(j:j+num-1,j:j+num-1) = diag(variance);
                else
                    num = 1;
                end
                j = j+num;
                
            end

        end
        
        function [SE,grad,sens] = SE_sens(theta,EstPar,data,par,W)
            
            % Calculate numerical gradient for all parameters
            num_mom  = size(par.CoVar,1);
            num_par  = numel(theta);
            h    = 1.0e-4;
            grad = NaN(num_mom,num_par);
            for i=1:numel(theta)
                var_now      = zeros(size(theta));
                var_now(i)   = 1;
                
                h_now = h*abs(theta(i));

                [~,output] = est.ObjFun(theta+h_now*var_now,EstPar,data,par,W);
                forward    = - output.mom_sim; % take the negative because it is (data-sim)
                [~,output] = est.ObjFun(theta-h_now*var_now,EstPar,data,par,W);
                backward   = - output.mom_sim;
                
                grad(:,i)   = (forward-backward)./(2*h_now); 
            end
            
            
            % calculate objects re-used below
            GW       = grad'*W;
            GWG      = GW*grad;
            S        = par.Omega;
            
            % Calculate asymptotic variance and (adjusted) standard errors
            Avar    = (GWG\(GW*S*GW'))/GWG;
            SE      = sqrt( diag((1+1/par.S)*Avar)./data.Nobs );
            
            % 4. calculate sensitivity measure (NumPar x NumMom) matrix
            sens.M1       = -GWG\GW; 
            std_moms      = sqrt(diag(S));  
            sens.M1e      = sens.M1.*repmat(std_moms',num_par,1);
            
            % 5. alternatives from Honor, Jrgensen and de Paula
            GSi  = grad'/S;
            GSiG = GSi*grad;
            
            sens.M2 = NaN(numel(theta),num_mom);
            sens.M3 = NaN(numel(theta),num_mom);
            sens.M4 = NaN(numel(theta),num_mom);
            sens.M5 = NaN(numel(theta),num_mom);
            
            sens.M2e = NaN(numel(theta),num_mom);
            sens.M3e = NaN(numel(theta),num_mom);
            sens.M4e = NaN(numel(theta),num_mom);
            
            for k = 1:num_mom
                % pick out the kk'th element: Okk
                O      = zeros(num_mom);
                O(k,k) = 1;
                
                M2kk     = (GSiG\(GSi*O*GSi'))/GSiG;          % NumPar-by-NumPar
                M3kk     = (Avar/GWG)*(GW*O*GW')*(GWG\Avar);  % NumPar-by-NumPar
                M4kk     =  - GWG\(grad'*O*grad)*Avar ...
                            + GWG\(grad'*O*S*W*grad)/GWG ...
                            + GWG\(grad'*W*S*O*grad)/GWG ...
                            - Avar*(grad'*O*grad)/GWG;  % NumPar-by-NumPar
                
                sens.M2(:,k)  = diag(M2kk); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                sens.M3(:,k)  = diag(M3kk); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                sens.M4(:,k)  = diag(M4kk); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                
                sens.M2e(:,k)  = diag(M2kk)./diag(Avar) * S(k,k); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                sens.M3e(:,k)  = diag(M3kk)./diag(Avar) * S(k,k); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                sens.M4e(:,k)  = diag(M4kk)./diag(Avar) * W(k,k); % store only the diagonal: the effect on the variance of a given parameter from a slight change in the variance of the kth moment
                
                
                % remove the kth moment from the weight matrix and
                % calculate the asymptotic variance without this moment
                W_now      = W;
                W_now(k,:) = 0;
                W_now(:,k) = 0;
                
                GW_now   = grad'*W_now;
                GWG_now  = GW_now*grad;
                Avar_now = (GWG_now\(GW_now*S*GW_now'))/GWG_now;
                
                sens.M5(:,k)  = diag(Avar_now) - diag(Avar);
                sens.M5e(:,k) = sens.M5(:,k)./diag(Avar); % update these
            end
            
            
        end

        % costs of risks
        function Obj = obj_cost(theta,data,par,value_base)
            % 1. Update parameter struct with the given parameter values in theta of EstPar
            par = est.update_par(theta,{'cost_welfare'},par,par.PRINT);    % transform parameters to struct

                % 1.1 Update grids: income parameters etc used here
                par   = prep.construct_grids(par);

            % 2. Solve model for the given parameters in theta
            sol = mex_solve(par);
           
            % 4. Simulate data (par.S*data.Nobs observations for par.simT periods)
            sim       = est.simulate(par,sol,data.draws); % TODO: construct this
            value_now = mean(sim.welfare);
            
            % 5. Calculate criteria function 
            Obj       = (value_base-value_now)^2;
           
            if par.PRINT==1, fprintf(' ->Obj=%3.7f\n',Obj); end
        end

    end
end