//////////////////////////
// 1. external includes //
//////////////////////////

#include <cmath>
#include <cstdio>
#include <cstdlib>
#include <cstdarg>
#include <string>
#include <omp.h>
#include "mex.h"
#include "matrix.h"


//////////////////////////
// 2. define statements //
//////////////////////////

#define MAX(X,Y) ((X)>(Y)?(X):(Y))
#define MIN(X,Y) ((X)<(Y)?(X):(Y))
#define BOUND(X,A,B) MIN(MAX(X,A),B)
#define THREADS MAXTHREADS


//////////////////////////
// 3. internal includes //
//////////////////////////

// a. generic
#include "HighResTimer_class.hpp" // timer class
//#include "includes\assert.cpp"             // assert() function
#include "logs.cpp"               // log:: functions
//#include "includes\linear_interp.cpp"      // linear_interp:: functions
#include "index.cpp"              // index:: functions
#include "misc.cpp"              // functions to interact with mex

// b. basic
#include "par_struct.cpp"  // define par_struct + setup/destroy functions
#include "sol_struct.cpp"  // define sol_struct + setup/destroy functions
#include "util.cpp"   // transformation and utility functions

// c. solve
#include "interpolate.cpp"     // setup and interpolant gatewys


////////////////
// 4. gateway //
////////////////


void mexFunction(int nlhs, mxArray *plhs[],
                 int nrhs, const mxArray *prhs[])
{
     
    //HighResTimer timer_all, timer;
    //timer_all.StartTimer();

        //logs::solve(0,"Solving model.\n"); // reset log file
        

    //////////////////
    // 5. setup par //
    //////////////////
   
    par_struct* par = new par_struct;
    par::setup(par,plhs,prhs,2);

     auto P = par->sim_P;
     auto Y = par->sim_Y;
     auto M = par->sim_M;
     auto C = par->sim_C;
     auto A = par->sim_A;
     auto e = par->sim_e;
     auto q = par->sim_q;
     auto g = par->sim_g;
     auto k = par->sim_k;
     auto b = par->sim_b;
     auto age = par->sim_age;

     auto dlogY  = par->sim_dlogY;
     //auto effort = par->sim_effort;
     auto d2SavingRate = par->sim_d2SavingRate;
     
    auto welfare = par->sim_welfare;

    /////////////////
    // 6. parallel //
    /////////////////

    #pragma omp parallel num_threads(THREADS)
    {
        #pragma omp for
        for (size_t i = 0; i < par->simN; i++){
            

            if(par->do_welfare==1){
                welfare[i] = 0.0;
            }

            for(size_t t=0;t < par->simT; t++){ // TODO: ensure some timing thing here
                size_t it   = i + t*par->simN;
                size_t it_1 = i + (t-1)*par->simN;
                
                // Last period variables
                double lag_P,lag_A;
                size_t lag_g,lag_k,lag_q;
                if(t==0){ // assume that all is simulated from agemin
                    lag_P = par->P0[i];
                    lag_A = par->A0[i];

                    lag_g = 0;
                    lag_q = 0;
                    lag_k = (size_t) par->k0[i];

                    g[it] = (size_t) par->g0[i];

                    b[it] = 0;
                    k[it] = lag_k + b[it];

                } else {

                    // lagged wealth and income
                    lag_P = P[it_1];
                    lag_A = A[it_1];

                    lag_g = (size_t) g[it_1];
                    lag_k = (size_t) k[it_1];
                    lag_q = (size_t) q[it_1];

                    // determine if a child is born this period and thus the number of children
                    if(lag_g>0 && lag_q==0){
                        b[it] = 1;
                    } else {
                        b[it] = 0;
                    }
                    k[it] = lag_k + b[it];

                    // determine pregnancy status today: depends on choice last period but also stochastic
                    double p_pregnant = util::prob_pregnant((size_t) e[it_1],(size_t) k[it],t-1,par);
                    if(p_pregnant>par->draws_pregnant[it_1] && par->Ng>1){
                    
                        if((size_t) e[it_1] < 1){
                            g[it] = (double) par->Ng-1; 
                        } else {
                            g[it] = 1;
                        }
                    
                    } else {
                        g[it] = 0;
                    }
                    
                }
           
                // Resources:
                double perm  = exp(-.5*par->sigma_perm*par->sigma_perm   + par->sigma_perm*par->draws_perm[it]);
                double trans = exp(-.5*par->sigma_trans*par->sigma_trans + par->sigma_trans*par->draws_trans[it]);

                double cost_trans = util::cost_trans(lag_k,(size_t)k[it],(size_t)g[it],par); 
                double cost_perm  = util::cost_perm(lag_k,(size_t)k[it],(size_t)g[it],par); 

                P[it] = cost_perm * par->G[t] * pow(lag_P,par->alpha) * perm;
                Y[it] = (1.0-par->cost_welfare)*cost_trans* P[it]     * trans;
                M[it] = par->R*lag_A + Y[it];

                // Determine optimal choices
                double V_opt = -LARGE_NUMBER; 
                double e_opt=0.0,q_opt=0.0; 
                for(size_t ie=0; ie<par->Ne; ie++) {// loop through discrete choices available
                    for(size_t iq=0; iq<par->Nq; iq++) {
                        if(((size_t)g[it])<2 && iq>0){         continue;} // not pregnant or intended pregnancy so cannot have an abortion
                        if(((size_t)k[it])==par->Nk-1 && ie>0){continue;} // have maximum number of children so cannot get pregnant. Since there is no effort cost households would be indifferent between trying and not
                        
                        size_t d  = index::discrete(ie,iq,par->Nq);
                        size_t id = index::d5(d,(size_t)g[it],(size_t)k[it],0,0,par->Ng,par->Nk,par->Np,par->Nm);
                        
                        double V_now = interp::vfi(&(par->Vd[t][id]), par->grid_P,par->grid_m, P[it], M[it]/P[it],par->Np,par->Nm,1,1,par);
                        
                        if(V_now>V_opt){
                            V_opt = V_now;
                            e_opt = ie;
                            q_opt = iq;
                        }
                        
                    }
                }

                // store the optimal choice
                e[it] = e_opt;
                q[it] = q_opt;

                // consumption choice: just interpolate the overall solution since the maximum has been taken
                size_t iC = index::d4((size_t)g[it],(size_t)k[it],0,0,par->Nk,par->Np,par->Nm); 
                C[it]     = interp::vfi(&(par->C[t][iC]), par->grid_P,par->grid_m, P[it], M[it]/P[it],par->Np,par->Nm,1,1,par);

                // end-of-period wealth and age
                A[it]   = M[it] - C[it];
                age[it] = (double) par->agemin + t;


                if(t>1){
                    size_t it_2 = i + (t-2)*par->simN;

                    double SavingRate   = MAX(0.0 , (Y[it]-C[it])/Y[it]); 
                    double SavingRate_2 = MAX(0.0 , (Y[it_2]-C[it_2])/Y[it_2]);
                    d2SavingRate[it]    = SavingRate - SavingRate_2;
                } else {
                    //effort[it] = mxGetNaN();
                    d2SavingRate[it]    = mxGetNaN();
                }


                if(par->do_welfare==1){

                    //welfare[i] += pow(par->beta , t )*util::U(C[it], (size_t) q[it], (size_t) k[it],t,par);
                    welfare[i] +=                     util::U(C[it], (size_t) q[it], (size_t) k[it],t,par);

                }


                
            } // time

            dlogY[ i ]                              = mxGetNaN();
            dlogY[ i + (par->simT-1)*par->simN ]    = mxGetNaN();
            for(size_t t=1;t < (par->simT-1); t++){ 
                    dlogY[i + t*par->simN]          = log(Y[ i + (t+1)*par->simN ]) - log(Y[i + (t-1)*par->simN]);
            }


        } // individual
    } // pragma



    /////////////////
    // 7. clean up //
    /////////////////
    
    par::destroy(par,2);
    delete par;

} // mex gateway


