%--------------------------------------------------------------------------
%  ------------------------ Replication code for ------------------------  
%      --- "Trade Openness and Growth: A Network-Based Approach" --- 
%  -------- Georg Duernecker, Moritz Meyer, Fernando Vega-Redondo --------
%  ------------------ Journal of Applied Econometrics ------------------ 
%--------------------------------------------------------------------------
%                       This version: May 2022
%                      Written for: Matlab R2020a
%--------------------------------------------------------------------------
% This script loads the UN Comtrade data and computes the centrality index
% -------------------------------------------------------------------------
clear
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Select commodity code
commodity_code  = 100;      % 100: aggregate trade flows
% -------------------------------------------------------------------------
% Select year/s
YEAR    = 1962:2016;
n_y     = numel(YEAR);
% -------------------------------------------------------------------------
% Country codes
% -------------------------------------------------------------------------
country_code_all = [4     8     12    20    24    28    31    32    36    40    44    48    50    51    52    56    58    60    64  ...
                    68    70    72    76    84    90    96    100   104   108   112   116   120   124   132   140   144   148   152 ...
                    156   170   174   178   180   188   191   192   196   200   203   204   208   212   214   218   222   226   230 ...
                    231   232   233   242   246   251   258   262   266   268   270   276   278   280   288   292   296   300   304 ...
                    308   320   324   328   332   340   344   348   352   356   360   364   368   372   376   381   384   388   392 ...
                    398   400   404   408   410   414   417   418   422   426   428   430   434   440   442   446   450   454   458 ...
                    459   462   466   470   478   480   484   496   498   499   504   508   512   516   524   528   530   533   540 ...
                    554   558   562   566   579   583   584   585   586   588   590   591   592   598   600   604   608   616   620 ...
                    624   634   642   643   646   658   659   662   670   674   682   686   688   690   694   699   702   703   704 ...
                    705   706   710   711   716   720   724   736   740   748   752   757   760   762   764   768   776   780   784 ...
                    788   792   795   798   800   804   807   810   818   826   834   841   842   850   854   858   860   862   882 ...
                    886   887   890   891   894  ];
% -------------------------------------------------------------------------
% Number of observations
n_obs_all = size(country_code_all, 2);
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Initialize matrix
CENTRALITY	= zeros(n_obs_all, n_y);
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
% Start loop over all years
% -------------------------------------------------------------------------
for t = 1:n_y
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Select current year
    year    = YEAR(t);
% -------------------------------------------------------------------------
    
% -------------------------------------------------------------------------
% Find last digit of year
    year_category = year - floor(year/10)*10;
% -------------------------------------------------------------------------
    
% -------------------------------------------------------------------------
% Load data
% -------------------------------------------------------------------------
% Column 1: Year
% Column 2: Reporter Code
% Column 3: Partner Code
% Column 4: Trade Flow Code / 2:Export, 1:Import
% Column 5: Commodity Code / see above
% Column 6: Value, in current USD
% -------------------------------------------------------------------------
    data = csvread(['trade_data_code_',num2str(commodity_code),'_year_',num2str(year_category),'.csv'],1,0);
% -------------------------------------------------------------------------
    
% -------------------------------------------------------------------------
% Exclude observations for other years 
    data( data(:,1)~=year, :) = [];
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Select all countries in the sample who (i) report to export to at least 
% one other country or (ii) are trading partner of at least one importing country.
% -------------------------------------------------------------------------
% (i) Exporter:
    exporter = unique( data( data(:,4)==2, 2 ) );
% -------------------------------------------------------------------------
% (ii) Importer:
    importer = unique( data( data(:,4)==1, 3 ) );
% -------------------------------------------------------------------------
% Intersection of (i) and (ii)
    sample_countries = unique( [exporter; importer] );
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Exclude country/year pairs for which trade flows are too sparse
% -------------------------------------------------------------------------
    if ( year >= 1962 )&&( year < 1970 )    % Exclude BELIZE in 1962-69
        sample_countries( sample_countries == 84 ) = [];    end        

    if ( year >= 1962 )&&( year < 1970 )    % Exclude MACAO in 1962-69
        sample_countries( sample_countries == 446 ) = [];    end        

    if ( year == 1967 )                     % Exclude UAE in 1967
        sample_countries( sample_countries == 784 ) = [];    end        
    
    if ( year == 1969 )                     % Exclude SAINT LUCIA in 1969
        sample_countries( sample_countries == 662 ) = [];    end        
    
    if ( year == 1993 )                     % Exclude ERITREA in 1993
        sample_countries( sample_countries == 232 ) = [];    end        
    
    if ( year == 2005 )                     % Exclude SERBIA in 2005
        sample_countries( sample_countries == 688 ) = [];    end        
    
    if ( year == 2009 )||( year == 2013 )||( year == 2015 )   % Exclude IRAQ in 2009, 2013 and 2015
        sample_countries( sample_countries == 368 ) = [];    end        
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Number of countries for the current year
    n_obs_sample = size( sample_countries, 1 );
% -------------------------------------------------------------------------
% Matrix of bilateral trade flows
    Matrix_trade = zeros(n_obs_all);
% -------------------------------------------------------------------------
% Index: position of country in list
    index = 1:n_obs_all;
% -------------------------------------------------------------------------
    country_code = country_code_all;
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
    for i = 1:n_obs_sample
% -------------------------------------------------------------------------
% Select country with observed trade flow
        country_i = sample_countries(i);
% -------------------------------------------------------------------------
% Find position in country list
        pos_country_i = find( country_i == country_code );
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
        if isempty( pos_country_i ) == 0 % exporting country is in list
% -------------------------------------------------------------------------
            % All exports of country i
            exports_i = data( data(:,2)==country_i&data(:,4)==2, :);
% -------------------------------------------------------------------------
            % Number of export destinations
            n_obs_exports_i = size( exports_i, 1);
% -------------------------------------------------------------------------
            
% -------------------------------------------------------------------------
            if n_obs_exports_i ~= 0     % exports are reported for i
% -------------------------------------------------------------------------
                for j = 1:n_obs_exports_i % 
% -------------------------------------------------------------------------
                    country_j = exports_i(j,3);
% -------------------------------------------------------------------------
                    % Find position of export destination in country list
                    pos_country_j = find( country_j == country_code );
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
                    if isempty( pos_country_j ) == 0 % country is in list
% -------------------------------------------------------------------------
                        % Value of exports
                        exports_i_j = exports_i(j,6);
% -------------------------------------------------------------------------
                        % Insert value into matrix of bilateral trade flows
                        Matrix_trade(pos_country_i, pos_country_j) = exports_i_j;
% -------------------------------------------------------------------------
                    end
% -------------------------------------------------------------------------
                end
% -------------------------------------------------------------------------
            else    % no exports are recorded for country i / use imports
% -------------------------------------------------------------------------
                % All imports shipped from country i
                imports_i = data( data(:,3)==country_i&data(:,4)==1, :);
% -------------------------------------------------------------------------
                % Number of importing destinations
                n_obs_imports_i = size( imports_i, 1);
% -------------------------------------------------------------------------
                
% -------------------------------------------------------------------------
                if n_obs_imports_i ~= 0     % imports are reported for i
% -------------------------------------------------------------------------
                    for j = 1:n_obs_imports_i % 
% -------------------------------------------------------------------------
                        country_j = imports_i(j,2);
% -------------------------------------------------------------------------
                        % Find position of importer in country list
                        pos_country_j = find( country_j == country_code );
% -------------------------------------------------------------------------
                        if isempty( pos_country_j ) == 0 % country is in list
% -------------------------------------------------------------------------
                            % Value of imports
                            imports_i_j = imports_i(j,6);
% -------------------------------------------------------------------------
                            % Insert value into matrix of bilateral trade flows
                            Matrix_trade(pos_country_i, pos_country_j) = imports_i_j;
% -------------------------------------------------------------------------
                        end
% -------------------------------------------------------------------------
                    end
% -------------------------------------------------------------------------
                end
% -------------------------------------------------------------------------
            end
% -------------------------------------------------------------------------
        end
% -------------------------------------------------------------------------
    end
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Exclude non-exporting and non-importing countries
% -------------------------------------------------------------------------
% Sum of country exports
    sum_exports = sum(Matrix_trade,2);
% -------------------------------------------------------------------------
% Delete row and column of non-exporting country
% -------------------------------------------------------------------------
    Matrix_trade(sum_exports==0,:)  = [];
    Matrix_trade(:,sum_exports==0)  = [];
    country_code(sum_exports==0)    = [];
    index(sum_exports==0)           = [];
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Sum of country imports
    sum_imports = sum(Matrix_trade,1);
% -------------------------------------------------------------------------
% Delete row and column of non-importing country
% -------------------------------------------------------------------------
    Matrix_trade(sum_imports==0,:)  = [];
    Matrix_trade(:,sum_imports==0)  = [];
    country_code(sum_imports==0)    = [];
    index(sum_imports==0)           = [];
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
    n_obs = size(Matrix_trade, 1);  % number of countries in the network
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
% Load GDP data
% -------------------------------------------------------------------------
% The matrix "data_GDP" has the following elements:
% Column 1: Country code
% Column 2: Year
% Column 3: GDP in current USD
% -------------------------------------------------------------------------
    data_GDP = csvread('GDP_DATA_WORLDBANK.csv',1,2);
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Select GDP observations of current year
    data_GDP(data_GDP(:,2)~= year, :) = [];
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
    GDP = zeros(n_obs,1);
% -------------------------------------------------------------------------
    for i = 1:n_obs
% -------------------------------------------------------------------------
        country_i   = country_code(i);                          % Country code
        GDP(i)      = data_GDP( data_GDP(:,1)==country_i, 3);	% Find GDP for country
    end
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Exclude countries with EXP/GDP < 0.1 percent
% -------------------------------------------------------------------------
    EXP         = sum(Matrix_trade, 2); % Total exports of each country
    exclude_i   = EXP./GDP < 0.001;     % Export to GDP ratio
% -------------------------------------------------------------------------
    
% -------------------------------------------------------------------------
% Delete row and column of countries with EXP/GDP < 0.1 percent
    Matrix_trade(exclude_i==1,:)    = [];
    Matrix_trade(:,exclude_i==1)    = [];
    country_code(exclude_i==1)      = [];
    index(exclude_i==1)             = [];
    GDP(exclude_i==1)               = [];
% -------------------------------------------------------------------------
    n_obs = size(Matrix_trade, 1);  % Number of countries in the network
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
    disp(['year: ', num2str(year), ', # of countries: ', num2str(size(country_code,2) ) ] )
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Compute country i's share in total (world GDP)
    beta_i = GDP./sum(GDP);
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Compute the adjustment factor theta
    EXP         = sum(Matrix_trade, 2);             % Total exports of each country
    theta_i = min( EXP./( (1-beta_i).*GDP ), 1);    % Adjustment factor
% -------------------------------------------------------------------------
    IMP         = sum(Matrix_trade, 1);             % Total exports of each country
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Normalize the export flows of each country by their aggregate exports
    Matrix_trade_normalized = Matrix_trade./( EXP*ones(1,n_obs) );
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Adjacency matrix A (row-stochastic)
    Matrix_A = Matrix_trade_normalized.*( theta_i*ones(1, n_obs) ) + diag(1-theta_i);
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% "computephi" computes the expected number of steps between i and j
    phi_ij = computephi(Matrix_A);
% -------------------------------------------------------------------------
    phi_ij = phi_ij + eye(n_obs);
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
% Re-scaled GDP weight
    weight  = beta_i*ones(1,n_obs);
    weight  = weight.*( ones(n_obs) - eye(n_obs) );
    weight  = weight./( ones(n_obs,1)*sum( weight ) );
% -------------------------------------------------------------------------
    
% -------------------------------------------------------------------------
%   \delta^{\log (\varphi_{ji}(t) )}
% -------------------------------------------------------------------------
    f = 0.93.^( log(phi_ij) );
% -------------------------------------------------------------------------
    CENTRALITY(index,t) = (  sum( f.*weight ) )';
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
end
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
% Some countries change the country code
% -------------------------------------------------------------------------
% 58  1962-98 Belgium-Luxembourg
% 56  1999-   Belgium
% -------------------------------------------------------------------------
% 230 1962-92 Fmr Ethiopia
% 231 1993-   Ethiopia
% -------------------------------------------------------------------------
% 280 1962-90 Former Fed. Rep. of Germany
% 276 1991-   Germany
% -------------------------------------------------------------------------
% 356 1962-74 India, excl. Sikkim
% 699 1975-   India
% -------------------------------------------------------------------------
% 459 1962-63 Peninsula Malaysia
% 458 1964-   Malaysia
% -------------------------------------------------------------------------
% 588 1962-71 East and West Pakistan
% 586 1972-   Pakistan
% -------------------------------------------------------------------------
% 590 1962-77 Fmr Panama, excl.Canal Zone
% 591 1978-   Panama
% -------------------------------------------------------------------------
% 711 1962-99 South African Customs Union
% 710 2000-   South Africa
% -------------------------------------------------------------------------
% 841 1962-80 USA
% 842 1981-   USA
% -------------------------------------------------------------------------
% 886 1962-90 Fmr Arab Rep. of Yemen
% 887 1991    Yemen
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
code_old = [58    230   280   356   459   588   590   711   841   886  ];
code_new = [56    231   276   699   458   586   591   710   842   887  ];
code_yr  = [1998  1992  1990  1974  1963  1971  1977  1999  1980  1990 ];
% -------------------------------------------------------------------------
% Number of countries who change country code
n_change = size(code_yr, 2);
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
for i = 1:n_change
% -------------------------------------------------------------------------
% Find and connect the old and the new series
% -------------------------------------------------------------------------
    index_old = find( country_code_all==code_old(i) );
    index_new = find( country_code_all==code_new(i) );
    yr_change = find( YEAR==code_yr(i) );
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
% Connect the two series and drop old series
% -------------------------------------------------------------------------
    CENTRALITY(index_new,1:yr_change)           = CENTRALITY(index_old,1:yr_change);
% -------------------------------------------------------------------------
    CENTRALITY(index_old,:,:)       = [];
% -------------------------------------------------------------------------
    country_code_all(index_old)     = [];
% -------------------------------------------------------------------------
end
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
CENTRALITY(CENTRALITY == 0) = NaN;
% -------------------------------------------------------------------------


% -------------------------------------------------------------------------
% Export data for dataset
% -------------------------------------------------------------------------
DATASET_ = input('Prepare data for dataset? 1 = yes  ');
% -------------------------------------------------------------------------
if DATASET_ == 1
% -------------------------------------------------------------------------
% Time horizon
% -------------------------------------------------------------------------
    i_yrs    = find(YEAR==1962):find(YEAR==2016);
% -------------------------------------------------------------------------
% Country codes
% -------------------------------------------------------------------------
    country_code_DATA = [4	8	12	20	24	28	31	32	36	40	44	48	50	51	52	56	60	64	68	70	72	76	84	90	96	100	104	108	112	116	120	124	132	140	144	148	152	156	170	174	178	180	188	191	192	196	200	203	204	208	212	214	218	222	226	231	232	233	242	246	251	258	262	266	268	270	276	278	288	292	296	300	304	308	320	324	328	332	340	344	348	352	360	364	368	372	376	381	384	388	392	398	400	404	408	410	414	417	418	422	426	428	430	434	440	442	446	450	454	458	462	466	470	478	480	484	496	498	499	504	508	512	516	524	528	530	533	540	554	558	562	566	579	583	584	585	586	591	592	598	600	604	608	616	620	624	634	642	643	646	658	659	662	670	674	682	686	688	690	694	699	702	703	704	705	706	710	716	720	724	736	740	748	752	757	760	762	764	768	776	780	784	788	792	795	798	800	804	807	810	818	826	834	842	850	854	858	860	862	882	887	890	891	894];
    n_obs_DATA        = size(country_code_DATA, 2);
% -------------------------------------------------------------------------
    DATASET = [];
% -------------------------------------------------------------------------
    for i = 1:n_obs_DATA
% -------------------------------------------------------------------------
% Structure of the dataset:
% Column 1: Country code
% Column 2: Year
% Column 3: Centrality index
% -------------------------------------------------------------------------
        DATASET = [        DATASET;
                        [   country_code_DATA(i)*ones(size(i_yrs'))  YEAR(i_yrs)'     CENTRALITY( country_code_DATA(i) == country_code_all, i_yrs)' ] ];
% -------------------------------------------------------------------------
    end
% -------------------------------------------------------------------------
end
% -------------------------------------------------------------------------



% -------------------------------------------------------------------------
% This function computes the matrix "phi_ij". It uses the matrix of
% relative export flows as input. The element in the i-th row and j-th
% column of "phi_ij" gives the expected number of steps between i and j
% -------------------------------------------------------------------------
function [phi_ij] = computephi(Matrix_A)
% -------------------------------------------------------------------------
n_j = size(Matrix_A, 1);    % Number of countries in the set
e   = ones(n_j-1,1);        % Construct a vector of size n_j-1 containing 1s
ee  = eye(n_j-1);           % Construct an identity matrix of dimension (n_j-1)x(n_j-1)
% -------------------------------------------------------------------------
phi_ij = zeros(n_j);        % Initialize the matrix
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
for j = 1:n_j               % Go through all countries j = 1, ..., n_j
% -------------------------------------------------------------------------
    auxA        = Matrix_A;
% -------------------------------------------------------------------------
    auxA(j,:)   = [];     % Remove the j-th row of matrix AA
    auxA(:,j)   = [];     % Remove the j-th column of matrix AA
% -------------------------------------------------------------------------

% -------------------------------------------------------------------------
    aux0        = inv( ee - auxA)*e;    % Do the inversion and 
% -------------------------------------------------------------------------
    phi_ij([1:j-1],j) = aux0(1:j-1);    % ... fill the resulting elements into matrix phi_ij
                                        % Notice that at the j-th position
                                        % there is a 0
% -------------------------------------------------------------------------
    phi_ij([j+1:n_j],j) = aux0(j:end);
% -------------------------------------------------------------------------
end
% -------------------------------------------------------------------------
end
% -------------------------------------------------------------------------










