%--------------------------------------------------------------------------
% Script to compute solution of the consumption model using VFI 
%
% This code loops over a 100 point grid of values of the parameter GAMMA 
% in the consumpion model described in: "Likelihood Evaluation of DSGE Models 
% with Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%--------------------------------------------------------------------------

clear; clc;

%-------------------------------
% INITIALIZE PARAMETERS
%-------------------------------
R    = 1.05;
BETA = 0.945;
RHO  = 0.90;
STD_U= 0.0100;
M = 1;
% GAMMAC = 1;


% load the vector of gamma parameters from [0.1 - 4.5]
GAMMAVEC = dlmread('GAMMAVEC.txt');
%==========================================================================
%                   Particle Filter + Global Solution
%==========================================================================


%---------------------------------
% Options for VFI Solution
%---------------------------------

% Options for optimization
options = optimset('MaxFunEvals',100000,'MaxIter',1000000,'TolFun',1e-6,...
    'TolX',1e-10,'Display','none');

% Grid points for debt
nb=200;
% Income grid
nz=15;
% Width of normal distribution for discrete approximation
widthz=3.2;

% Get Tauchen Approximation
[ P, logz, ~, ~, ~ ] = markovappr(RHO,STD_U,widthz,nz) ;
Z=exp(logz);

% Bounds for discretized grid
bmin=0.75*M;
bmax=1*Z(end) ;

% Construct grid for debt/assets
B=linspace(bmin,bmax,nb);


%---------------------------------
% Loop over parameters
%---------------------------------
GAMMAVECTOR = GAMMAVEC;
for param_number = 1:numel(GAMMAVEC)
    GAMMAC = GAMMAVECTOR(param_number);
     fprintf('\n Solving for GAMMAC = %4.4f \n',GAMMAC);
     [Bdec{param_number}, Cdec{param_number}] = solve_VFI(GAMMAC,R,BETA,M,P,B,Z,nz,nb,options);
     fprintf('\n Model solved for param_number = %i, GAMMAC = %4.4f',param_number,GAMMAC);
end

save('PF_BASELINE_GRID_GAMMA1_01_45.mat', 'Bdec', 'Cdec', 'GAMMAVEC','R','BETA','RHO','STD_U','M','B','Z','P');
