%--------------------------------------------------------------------------
% Script to compute likelihood using OccBin Inversion Filter
%
% This code loops over 100 samples of model simulated data from the DGP
% described in the paper: "Likelihood Evaluation of DSGE Models with
% Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%--------------------------------------------------------------------------

clear all
warning off

% restoredefaultpath
setpathdynare4
warning off

global oo00_  M00_ M10_  M01_  M11_ M_

global params_labels params

global cof cof10 cof01 cof11 ...
    Jbarmat Jbarmat10 Jbarmat01 Jbarmat11 ...
    Dbarmat10 Dbarmat01 Dbarmat11 ...
    decrulea decruleb

global filtered_errs_switch filtered_errs_init model_temp
global datavec irep xstory fstory

irep=1; datavec=[]; xstory=[]; fstory=[];

% 0: uses csolve; 1: csolve with gradient; 2: fsolve
solver=0;

set(0,'DefaultLineLineWidth',2)
randn('seed',1);
format compact


%-----------------------------------
% DECLARE PARAMETERS
%-----------------------------------
% TRUE PARAMETER
TRUEGAMMAC = 1; 

% FIXED PARAMTERS
R    = 1.05;
BETA = 0.945;
RHO  = 0.9;
STD_U= 0.0100;
M = 1;

% Initialize GAMMA
GAMMAC = TRUEGAMMAC;

save PARAM_EXTRA_BABY.mat R BETA RHO STD_U GAMMAC M STD_U GAMMAC

%-----------------------------------
% PARAMETER GRID
%-----------------------------------
GAMMAVEC=dlmread('GAMMAVEC.txt');

%-----------------------------------
% LOAD DATA
%-----------------------------------
simC=dlmread('simC.txt');


%-----------------------------------
% Create scripts to speed up filtering
%-----------------------------------

modnam_00 = 'borrcon00'; % base model (constraint 1 and 2 below don't bind)
modnam_10 = 'borrcon10'; % first constraint is true
modnam_01 = 'borrcon00'; % second constraint is true
modnam_11 = 'borrcon00'; % both constraints bind
constraint1 = 'lb<-lb_ss';
constraint_relax1 = 'b>bnot';
constraint2 = 'lb<-1000000';
constraint_relax2 = 'lb>-1000000';
curb_retrench =0;
maxiter = 20;
obs_list = char('c');


call_pre_estimation_script
%%
% Loop over 100 data samples: 0-99, to keep similar notation from PF codes.
for jobID=0:99;

    iii = jobID+1;

    fprintf('******************************************** \n')
    fprintf('*  SOLVING FOR DATASET %i                  * \n',iii);
    fprintf('******************************************** \n')

        %-------------------------------
        % Load data and Declare observables
        %-------------------------------
        simuse = simC(end-499:end,iii);
        obs = (simuse - (2-R));        
       
        tstar=1;
        sampsize = length(obs); 
        ntrain = 0;
        
        out = NaN(100,3);
    tic
    
    % Loop over parameters in the grid: gamma = [0.1-4.5]
    for param_number=1:100;
        %----------------------------------------------------------------------
        % Invoke calibrated parameters and load estimated one via GMM if needed
        %----------------------------------------------------------------------
        
        GAMMAC = GAMMAVEC(param_number);
        
        %save PARAM_EXTRA_CALIBRATED R BETA RHO STD_U GAMMAC M
        save PARAM_EXTRA_BABY.mat R BETA RHO STD_U GAMMAC M STD_U GAMMAC
        
        %------------------------------------------------------------------
        % Keep this for compatibility with ESTOBIN routines
        
        % Indicator variable set 1 if priors are used or not
        IPRIOR=0;
        
        params_matrix = {  ...
            'GAMMAC '    GAMMAC  0.01000 	10.0000     1     'NORMAL_PDF'  3 1
            'STD_U '     0.010000  0.0001  	 0.9999     1     'INV_GAMMA_PDF' 0.01 1
            } ;
        
        save params_matrix params_matrix
        
        err_list = char('eps_u');
        H0 = 1; %diag(cell2mat(params_matrix(:,5)).^2) ;
        
        params_labels = params_matrix(:,1);
        params0 =   cell2mat(params_matrix(:,2));
        params_lo = cell2mat(params_matrix(:,3));
        params_hi = cell2mat(params_matrix(:,4));
        params_mean = cell2mat(params_matrix(:,7));
        params_std = cell2mat(params_matrix(:,8));
        
        dist_names = params_matrix(:,6);
        codes = dist_names2codes(dist_names);
        [p6 p7] = get_dist_inputs(codes,params_mean,params_std);
        
        
        for i=1:numel(params_labels)
            evalc([ cell2mat(params_labels(i)) '= params0(' num2str(i) ')']) ;
        end
        
        method = 'initial_check';
        
        fprintf('Filtering for gammac= %4.4f \n', GAMMAC);

        %------------------------------------------------------------------
        
        
        %------------------------------------------------------------------
        % Compute likelihood using OccBin Inversion Filter
        %------------------------------------------------------------------
        
        params1=params0;
        
        [posterior, filtered_errs, like, prior, resids ]=...
            posterior1(params1,params_labels,params_lo,params_hi,...
            modnam_00,modnam_10,modnam_01,modnam_11,...
            constraint1_difference, constraint2_difference,...
            constraint_relax1_difference, constraint_relax2_difference,...
            err_list,obs_list,obs,ntrain,codes, p6, p7,IPRIOR,solver);
        fprintf('\n ****************************************** ');
        fprintf('\n Param number: %i, GAMMAC: %4.4f, logL: %4.4f \n', param_number, M00_.params(end), -like);
        fprintf(' ****************************************** \n');
        out(param_number,:) = [param_number, GAMMAC, -like];

    end
    timeiter = toc;
    fprintf('Elapsed time dataset %i = %4.4f \n',iii,timeiter);
    
    dlmwrite(['MC100GAM1_' num2str(iii-1) '_OCCINV_sample_' num2str(sampsize) '.txt'],out,'precision','%4.6f')

end

