function run_Filter_VFIINV_500(jobID)
%==========================================================================
% Function to compute likelihood using Inversion Filter + VFI Solution
%
% This code loops over 100 samples of model simulated data from the DGP
% described in the paper: "Likelihood Evaluation of DSGE Models with
% Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Input: jobID = Sample path to be evaluated. Type: string. Values: 0-99
% Output:   text file = ['MC100GAM1_' jobId '_VFIINV' '_sample' num2str(sampsize) '.txt']
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%
%==========================================================================



jobNum = str2double(jobID);

%-------------------------------
% FIXED PARAMETERS
%-------------------------------
R      = 1.05;
BETA   = 0.945;
RHO    = 0.9;
STD_U  = 0.0100;
M      = 1;

numobs = 500;

% LOAD DATA
simC = dlmread('simC.txt');
simB = dlmread('simB.txt'); 
simZ = dlmread('simZ.txt'); 

statestart = length(simC)-numobs;

% SELECT SAMPLE PATH AND ADJUST SAMPLE SIZE
obs = simC(:,jobNum+1);
obs = obs(end-numobs+1:end,:);

obsB = simB(:,jobNum+1);
obsZ = simZ(:,jobNum+1);

% Load VFI solution for 100 values of gamma between [0.1, 4.5]

% load grid for GAMMA
GAMMAVEC = dlmread('GAMMAVEC.txt');

% load stored VFI solution
% See instructions on how to create VFI solution in
% \ParticleFilter\Readme.txt.

load PF_VFI_GRID_GAMMA1_01_45


% Loop over the grid for gamma
for ii = 1:length(GAMMAVEC)

    % Clean matrices
    likivec = zeros(length(obs),1);
    epsivec = zeros(length(obs),1);
    jacobvec = zeros(length(obs),1);
    
    % Set parameter value
    GAMMAC = GAMMAVEC(ii);
    
    % Get solution for GAMMAC
    Bdec_use = Bdec{ii};

    %---------------------------------
    % INVERSION FILTER
    %---------------------------------
    fprintf('\n Evaluating for param_number = %i, GAMMAC = %4.4f',ii,GAMMAC);
    
    %Initialize at truth
    b_0 = obsB(statestart);
    z_0 = obsZ(statestart);
    
    for i = 1:length(obs)
        y_in = obs(i);
        B_in = b_0;
        Z_in = z_0;
        
        [dedy,epsivec(i)] = get_derivative(M,B,Z,Bdec_use,R,RHO,y_in,B_in,Z_in,10^-6);
        
        likivec(i) = -1/2*log(2*pi) -1/2*log(STD_U^2) - 1/2*epsivec(i)^2/STD_U^2 + log(abs(dedy));
        
        jacobvec(i) = log(abs(dedy));
        
        %Propogate states forward
        ZZ = exp(RHO*log(Z_in) + epsivec(i));
        BB = lininterpn(Z,B,Bdec_use,ZZ,B_in);
        BB = min(BB,M*ZZ);
        
        b_0 = BB;
        z_0 = ZZ;
    end
    
    out(ii,:) = [ii, GAMMAC, sum(likivec)];

end

dlmwrite(['MC100GAM1_' jobID '_VFIINV' '_sample' num2str(sampsize) '.txt'],out,'precision','%4.6f')

end




