%--------------------------------------------------------------------------
%
% This code solves the model with value function iteration using the  
% parameter calibration of the DGP and produces global decision rules used
% in "Figure 1" of the paper: "Likelihood Evaluation of DSGE Models with 
% Occassionally Binding Constraints".  
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%--------------------------------------------------------------------------


%-------------------------------------------
% Housekeeping
%-------------------------------------------

% clear all
warning off


set(0,'DefaultLineLineWidth',2)
randn('seed',1);
format compact

%----------------------------------------------------------------------
% Model Parameters
%----------------------------------------------------------------------

R      = 1.05;
BETA   = 0.945;
RHO    = 0.9;
STD_U  = 0.0100;
M      = 1;
GAMMAC = 1;

%---------------------------------------------
% Options for IRF
%---------------------------------------------

nperiods = 40;

% Set sequence for IRF
sequence(1:nperiods)=0;
sequence(2)  =  2*STD_U;
sequence(21) = -2*STD_U;

% If simulating a path uncomment the following line:
% sequence = zscore(randn(nperiods,1))*STD_U;

% Map shocks sequence
err_vals = sequence;

% Create integration nodes
[n_nodes,epsi_nodes,weight_nodes] = GH_Quadrature(3,1,[STD_U^2]);

% Initialize at the steady state
initss.Zlag = 1;
initss.Blag = M*initss.Zlag;

%---------------------------------------------
% Solve Global Solution
%---------------------------------------------


% Options for optimization
options = optimset('MaxFunEvals',100000,'MaxIter',1000000,'TolFun',1e-6,...
  'TolX',1e-10,'Display','none');

% Grid points
gridpoints.nb=200;
gridpoints.ny=15;

% SIMPLE MODEL OF CONSUMPTION AND SAVING UNDER UNCERTAINTY WITH BOR CONS.
rng default

% Grid points for debt
nb=gridpoints.nb;

% Income grid
nz=gridpoints.ny;
widthz=3.2;

% Call script that implements global solution
vincent_go_continuum



%%   // simulate observations
ngrid=100;
gridy = STD_U*linspace(-3,3,100);

for jj=1:ngrid
    err_vals = gridy(jj);
outsim = simulate_global2(err_vals,initss,M,B,P,Z,Bdec,Cdec,R,GAMMAC,BETA,RHO,STD_U,n_nodes,epsi_nodes,weight_nodes);

DR_GLOBAL.B(jj) = outsim(1);
DR_GLOBAL.C(jj) = outsim(3);
DR_GLOBAL.EC(jj) = outsim(4);
DR_GLOBAL.LB(jj) = outsim(5);
DR_GLOBAL.Z(jj) = outsim(7);

end
%%
figure(1);clf;
subplot(221)
plot(DR_GLOBAL.Z,DR_GLOBAL.B,'-');
title('Borrowing');

subplot(222)
plot(DR_GLOBAL.Z,DR_GLOBAL.C,'-'); 
title('Consumption');

subplot(223)
plot(DR_GLOBAL.Z,DR_GLOBAL.LB,'-'); 
title('Multiplier');

subplot(224)
plot(gridy/STD_U,DR_GLOBAL.Z,'-');

save DR_GLOBAL.mat DR_GLOBAL