%--------------------------------------------------------------------------
%
% This code reproduces "Figure 1: Policy Functions of the 
% Consumption - Savings Model with Occasionally Binding Constraints" 
% described in the  paper: "Likelihood Evaluation of DSGE Models with 
% Occassionally Binding Constraints" 
% 
% Pablo Cuba-Borda, Luca Guerrieri, Matteo Iacoviello and Molin Zhong
% Federal Reserve Board. Washington, D.C. 
%
% Created: 11/01/2017.
% Last modified: 04/23/2019.
%--------------------------------------------------------------------------

clear all
warning off

%-------------------------------------------
% Housekeeping
%-------------------------------------------


% restoredefaultpath
setpathdynare4
warning off

global oo00_  M00_ M10_  


% 0: uses csolve; 1: csolve with gradient; 2: fsolve
solver=0;

set(0,'DefaultLineLineWidth',2)
randn('seed',1);
format compact
obs_list = char('c');
err_list = char('eps_u');


%---------------------------------------
% Solve Model at True Parameter
%---------------------------------------
modnam = 'borrcon00';
modnamstar = 'borrcon10';

% TRUE PARAMETERS
R    = 1.05;
BETA = 0.945;
RHO  = 0.9;
STD_U= 0.010;
M = 1;
GAMMAC = 1; 

save PARAM_EXTRA_BABY R BETA RHO STD_U GAMMAC

 paramvec_ = [R BETA RHO STD_U M GAMMAC];

%--------------------------------------------------------------------------
% DEFINE CONSTRAINTS
%
% see notes 1 and 2 in README. Note that 
% lb_ss is the steady-state value of the multiplier lb (see borrcon_steadystate.m).
% lb_ss is calculated automatically as additional auxiliary parameter
% around line 49 of solve_one_constraint
%--------------------------------------------------------------------------
constraint = 'lb<-lb_ss';
constraint_relax ='b>bnot';

%--------------------------------------------------------------------------
% DEFINE SHOCKS 
%--------------------------------------------------------------------------

irfshock =char('eps_u');      % label for innovation for IRFs
nperiods = 40;
sequence = zscore(randn(nperiods,1))*STD_U;

sequence(1:nperiods)=0;


% 
% %==========================================================================
% % ALTERNATIVE IMPLEMENTATION OF SIMULATION CODE
% %==========================================================================
% 
% %--------------------------------------------------------------------------
% % CREATE OBJECTS FOR OCCBIN SOLUTION
% %--------------------------------------------------------------------------
% 
modnam_00 = 'borrcon00'; % base model (constraint doesn't bind)
modnam_10 = 'borrcon10'; % first constraint is true

constraint1 = 'lb<-lb_ss';
constraint_relax1 = 'b>bnot';
curb_retrench =0;
maxiter = 20;

% Create global variables
solve_one_constraints_firstcall(modnam_00,modnam_10);

% Store steady state
zdatass = oo00_.dr.ys;

% Create constraints
[constraint1_difference, iendo1]= process_constraint_with_tokens(constraint1,'_difference',M00_.endo_names,0);
[constraint_relax1_difference, iendo2]= process_constraint_with_tokens(constraint_relax1,'_difference',M00_.endo_names,0);
%%
% Compute OccBin Solution Matrices
[nvars,ys_,endog_,exog_,params, decrulea,decruleb,cof,...
    Jbarmat,cof10,Jbarmat10,Dbarmat10] = get_occbin_solution(modnam_00,modnam_10,solver,paramvec_);
%%
% Compute OccBin Related Objects
[~, i1, ~]=intersect(M00_.endo_names,obs_list,'rows');
[~, i2, ~]=intersect(M00_.exo_names,err_list,'rows');
    
% Current observation (keep for compatibility)
current_obs = [];

%% Initial Vector: Steady State
init_val_old  = zdatass;
err_vals = sequence';


% PLOT DECISION RULE
ngrid=100;
gridy = STD_U*linspace(-3,3,100);




for jj=1:ngrid
    err_vals = gridy(jj);

[zdata, zdataconcatenated, ys_, init_out, error_flag, Ecurrent ] = solve_one_constraints_nextcall(...
    constraint1_difference,constraint_relax1_difference,...
    err_vals',err_list,nperiods,curb_retrench,maxiter,init_val_old);


DR.B(jj) = zdataconcatenated(1,1) + zdatass(1);
DR.C(jj) = zdataconcatenated(1,3) + zdatass(3);
DR.LB(jj) = (zdataconcatenated(1,5) + zdatass(5));
DR.EC(jj) = (zdataconcatenated(1,4) + zdatass(4));
DR.Z(jj) = zdataconcatenated(1,7) + zdatass(7);


DR_LINEAR.B(jj) = zdata(1,1) + zdatass(1);
DR_LINEAR.C(jj) = zdata(1,3) + zdatass(3);
DR_LINEAR.EC(jj) = zdata(1,4) + zdatass(4);
DR_LINEAR.LB(jj) =(zdata(1,5) + zdatass(5));
DR_LINEAR.Z(jj) = zdata(1,7) + zdatass(7);

end


% Load decision rule from global solution
% Run Figure3\run_decisionrule_for_figure1.m to reproduce the global
% solution used to produce the global decision rules.
load DR_GLOBAL.mat


% PLOT FIGURE
figure(1);clf;
set(figure(1),'PaperType','usletter','PaperOrientation','Landscape','PaperPosition',[0.1 0.1 11 8.5]);
subplot(221)
plot(DR.Z,DR_GLOBAL.B,'-','LineWidth',3); hold on
plot(DR.Z,DR.B,'-','LineWidth',3);
plot(DR_LINEAR.Z,DR_LINEAR.B,'--','LineWidth',3); xlim([0.97, 1.03]);
legend('VFI','OccBin','Linear','Location','NW'); legend boxoff
title('Borrowing'); xlabel('Income');

subplot(222)
plot(DR.Z,DR_GLOBAL.C,'-','LineWidth',3);  hold on
plot(DR.Z,DR.C,'-','LineWidth',3);
plot(DR_LINEAR.Z,DR_LINEAR.C,'--','LineWidth',3); xlim([0.97, 1.03]);
legend('VFI','OccBin','Linear','Location','NW'); legend boxoff

title('Consumption');xlabel('Income');


subplot(223)
plot(DR.Z,DR_GLOBAL.LB,'-','LineWidth',3);  hold on
plot(DR.Z,DR.LB,'-','LineWidth',3); 
plot(DR_LINEAR.Z,DR_LINEAR.LB,'--','LineWidth',3);xlim([0.97, 1.03]);
legend('VFI','OccBin','Linear'); legend boxoff

title('Multiplier'); xlabel('Income');


subplot(224)
plot(DR.Z,DR_GLOBAL.EC,'-','LineWidth',3); hold on
plot(DR.Z,DR.EC,'-','LineWidth',3); 
plot(DR.Z,DR_LINEAR.EC,'--','LineWidth',3); xlim([0.97, 1.03]);
legend('VFI','OccBin','Linear','Location','NW'); legend boxoff

title('Expected Consumption');xlabel('Income');


 print(1,'-dpdf','-r300','Figure1.pdf');
