%% ABG Rep -- produce figures for paper and appendix
clear all 
figflag = 2.12 ;
rng(2)                                  % Seeds random number generator

%% Options for figures
set(0,'defaultAxesFontName', 'Times New Roman')             % Set font for axes
set(0,'defaultTextFontName', 'Times New Roman')             % Set font for text
set(0,'defaultLegendFontName', 'Times New Roman')           % Set font for text
set(0,'defaultAxesFontSize', 16)                            % Set font size for axes
set(0,'defaultTextFontSize', 16)                            % Set font size for text
set(0,'defaultLegendFontSize', 16)                          % Set font size for text

%% List of figures for paper
% 1.01 = plot of data -- 1 qtr growth (from GDPC1) and NCFI
% 1.02 = QR lines 1973Q2 - 2015Q4 -- 1 qtr growth (from GDPC1)
% 1.03 = QR lines 1973Q2 - 2020Q4 -- 1 qtr growth (from GDPC1) 
% 1.04 = Unconditional Margins for Output Growth both samples -- 1 qtr growth (from GDPC1)
% 1.05 = Copula QR lines Pseudo data 1973Q2 - 2020Q4 -- 1 qtr growth (from GDPC1)
% 1.06 = Non-linear Copula QR lines 1973Q2 - 2020Q4 -- 1 qtr growth (from GDPC1)
% 1.07 = Non-linear Copula QR lines 1973Q2 - 2015Q4 -- 1 qtr growth (from GDPC1)

%% List of figures for appendix
% 2.01 = Comparison of GDPC1 growth series and ABG's series
% 2.02 = QR lines 1973Q2 - 2020Q4 -- ABG's (1dp) measure of 1 qtr growth 
% 2.03 = Simulation - 50th quantile
% 2.04 = Simulation -- 33rd quantile
% 2.05 = Narrow Replication -- QR coefficients and ABG CIs, ABG's sample
% 2.06 = Narrow Replication -- QR coefficients and ABG CIs, extended sample
% 2.07 = Narrow Replication -- distribution of GDP over time, extended sample
% 2.08 = Narrow Replication -- distribution of GDP during the pandemic
% 2.09 = Narrow Replication -- Entropy, extended sample
% 2.10 = Narrow Replication -- Longrise / Shortfall, extended sample
% 2.11 = Narrow Replication -- OOS Quantiles, extended sample
% 2.12 = Narrow Replication -- OOS Entropy, extended sample
% 2.13 = QR lines Pseudo data ABG's sample -- 1 qtr growth (from GDPC1)
% 2.14 = Broad Replication -- QR coefficients and ABG CIs, ABG's sample -- Pseudo data
% 2.15 = Broad Replication -- QR coefficients and ABG CIs, extended sample -- Pseudo data
% 2.16 = Copula Swarm, pseudo margins, extended sample
% 2.17 = Copula Swarm, observed margins, extended sample
% 2.18 = QR lines, 1974Q1 - 2015Q4 -- 4 qtr growth
% 2.19 = QR lines, 1974Q1 - 2021Q3 -- 4 qtr growth
% 2.20 = Copula QR lines Pseudo data 1974Q1 - 2015Q4 -- 4 qtr growth
% 2.21 = Copula QR lines Pseudo data 1974Q1 - 2021Q3 -- 4 qtr growth
% 2.22 = Non-linear Copula QR lines 1974Q1 - 2021Q3 -- 4 qtr growth

%% Paper Figure 1: Plots

if figflag == 1.01 ;
    
    % Plot Output Growth and NCFI 1973Q1 - 2021Q3

    load ABGRep         % Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details
    fig1 = figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    left_color = [0 0 0];
    right_color = [0 0 0];
    set(fig1,'defaultAxesColorOrder',[left_color; right_color]);
    yyaxis left 
    plot(QTR(2:end),g1GDP(2:end),'b-','linewidth',1)
    xlim([datetime(1972,01,01) datetime(2021,10,01)])
    ylabel('One-quarter Economic Growth')
    ylim([-35 35])
    yyaxis right
    plot(QTR(2:end),NFCI(1:end-1),'r-','linewidth',1)
    ylabel('Lagged NFCI')
    ylim([-2 5])
    legend('One-quarter Economic Growth','Lagged NFCI','location','sw','NumColumns',2)
    grid on 
    box on

end ;

%% Paper Figure 2: QR lines -- ABG sample (1 qtr growth from GDPC1), observed data

if figflag == 1.02 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 

    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;          % Data to 2015Q4 only
    n = n(1:end-num_ext) ;          % Data to 2015Q4 only
    gm = zscore(g) ;                % Standardize output growth - ABG sample
    nm = zscore(n) ;                % Standardize NCFI - ABG sample

    % Load estimation results: 1 qtr growth on ABG's sample
    load EST_1qtr_abg               
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    xlabel('Lagged NFCI')
    ylabel('One-quarter Economic Growth')
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    grid on 
    box on
end ;

%% Paper Figure 3: QR lines -- Extended sample (1 qtr growth from GDPC1), observed data

if figflag == 1.03 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth - extended sample
    nm = zscore(n) ;                % Standardize NCFI - extended sample

    % Load estimation results: 1 qtr growth on Extended sample for panels (c) and (d)
    load EST_1qtr_ext

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    grid on 
    box on
end ;

%% Paper Figure 4: QR lines -- Extended sample (1 qtr growth from GDPC1), pseudo data

  if figflag == 1.04 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    T = length(g) ;                 % number obs
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data output - extended sample
    nps = norminv(nr,0,1) ;         % pseudo data NFCI - extended sample
    clear T gr nr

    % Load estimation results: 1 qtr growth on Extended sample for panels (c) and (d)

    load EST_1qtr_ext
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nps,pPs,'LineWidth',2)
    hold
    scatter(nps,gps, 25, 'o','b','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-3 3 -3 3])
    grid on 
    box on
    
  end ;

%% Paper Figure 5: Non-linear Copula QR lines -- extended sample (1 qtr growth from GDPC1), observed data

if figflag == 1.05 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth - extended sample
    nm = zscore(n) ;                % Standardize NCFI - extended sample
  
    % Load estimation results: 1 qtr growth on Extended sample for panel (b)
    load EST_1qtr_ext
    Z = [ nm pCOP ] ;               % Stack NCFI data and Cop QR lines
    ZZ = sortrows(Z,1) ;            % Sort by NCFI before plot

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(ZZ(:,1),ZZ(:,2:8),'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    ylabel('Output Growth')
    xlabel('Lagged NFCI')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    grid on 
    box on
  
 end ;

%% Paper Figure 6: Unconditional Marginal Distributions

if figflag == 1.06 ;
    
    load ABGRep                                     % Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details
    g1y1 = g1GDP ;
    gm1 = zscore(g1y1) ;                            % Standardized output growth

    num_ext = 23 ;                                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g1y0 = g1GDP(1:end-num_ext) ;                   % Remove obs post-ABG sample
    gm0 = zscore(g1y0) ;                            % Standardized output growth

    xt0 = linspace(min(gm0)-2,max(gm0)+2,100) ;     % Range to fit marginal to 1973-2015
    xt1 = linspace(min(gm1)-2,max(gm1)+2,100) ;     % Ditto 1973-2021
    
    % Marginal 1973-2015
    [ssvft, ~ ] = ssvkernel(gm0,xt0) ;
    nssvft = ssvft/sum(ssvft) ;
    ssvcdf  = cumsum(nssvft) ;
    [cdf_gm0, ~ ] = unique(ssvcdf) ;
    clear ssvft nssvft ssvcdf

    % Marginal 1973-2021
    [ssvft, ~ ] = ssvkernel(gm1,xt1) ;
    nssvft = ssvft/sum(ssvft) ;
    ssvcdf  = cumsum(nssvft) ;
    [cdf_gm1, ~ ] = unique(ssvcdf) ;
    clear ssvft nssvft ssvcdf
    
    swarm01 = rand(15000,1) ;                               % swarm on 0,1 
    swarm01 = swarm01(5001:15000) ;                         % drop 5000 burn-in
    
    ms0 = interp1(cdf_gm0, xt0, swarm01, 'spline') ;        % Swarm to standardized space via 1973-2015 inverse marginal
    ms1 = interp1(cdf_gm1, xt1, swarm01, 'spline') ;        % Swarm to standardized space via 1973-2020 inverse marginal
    
    m0 = std(g1y0)*ms0 + mean(g1y0) ;                       % Swarm to observed space 1973-2015
    m1 = std(g1y1)*ms1 + mean(g1y1) ;                       % Swarm to observed space 1973-2020
   
    [ ksm0, x0 ] = ksdensity(m0) ;                          % Marginal to plot 1973-2015 using ksdensity
    [ ksm1, x1 ] = ksdensity(m1) ;                          % Marginal to plot 1973-2020 using ksdensity

    fig2 = figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(x0,ksm0,'k-','linewidth',1.5) 
    hold 
    plot(x1,ksm1,'r-','linewidth',1.5)
    legend('1973-2015','1973-2021','location','ne')

end ;

%% Paper Figure 7: Non-linear Copula QR lines -- ABG sample (1 qtr growth from GDPC1), observed data

if figflag == 1.07 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;          % Data to 2015Q4 only
    n = n(1:end-num_ext) ;          % Data to 2015Q4 only
    gm = zscore(g) ;                % Standardize output growth - ABG sample
    nm = zscore(n) ;                % Standardize NCFI - ABG sample
  
    % Load estimation results: 1 qtr growth on ABG sample
    load EST_1qtr_abg
    Z = [ nm pCOP ] ;
    ZZ = sortrows(Z,1) ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(ZZ(:,1),ZZ(:,2:8),'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    grid on 
    box on
  
 end ;

%% Appendix Figure 1: Comparison of GDPC1 growth series and ABG's series

if figflag == 2.01 ;
    
    % Compare GDP growth series
    
    load ABGRep
    gGDPround = round(g1GDP,1) ;                    % GDPC1 growth series rounded to 1 dp
    gGDPdiff = ABGgGDP - gGDPround ;                % Difference between this and ABG's measure
    
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    yyaxis left 
    plot(QTR,g1GDP,'b-','linewidth',1)
    xlim([datetime(1972,01,01) datetime(2021,10,01)])
    ylabel('One-quarter Economic Growth')
    yyaxis right
    bar(QTR,gGDPdiff)
    ylabel('Difference')
    ylim([-1 1])

end ;

%% Appendix Figure 2: QR lines extended sample 1973Q2 - 2021Q3 -- ABG's (1dp) measure of 1 qtr growth 

if figflag == 2.02 ;
    
    % ABG's 1dp growth rate -- extended sample, standardized data
    
    load ABGRep
    g = ABGgGDP(2:end) ;            % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI

    load EST_1dp_ext               % Load estimation results: 1 qtr growth on extended sample 
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    grid on 
    box on
end ;

%% Appendix Figure 3: Simulation -- 50th quantile

if figflag == 2.03 ;
    
    load MCSim                                       % Load simulation output

    [kCON0, xCON0] = ksdensity(ocCON50(:,1)) ;       % Conventional QR - no outliers
    [kCOP0, xCOP0] = ksdensity(ocCOP50(:,1)) ;       % Copula QR - no outliers
    [kCON1, xCON1] = ksdensity(ocCON50(:,2)) ;       % Conventional QR - outliers
    [kCOP1, xCOP1] = ksdensity(ocCOP50(:,2)) ;       % Copula QR - outliers

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact')
    nexttile
    plot(xCON0,kCON0,'r-','linewidth',2)
    hold
    plot(xCOP0,kCOP0,'b-','linewidth',2)
    plot(xCON1,kCON1,'r:','linewidth',2)
    plot(xCOP1,kCOP1,'b:','linewidth',2)
    legend('Conventional QR - no outliers','Copula QR - no outliers','Conventional QR - outliers','Copula QR - outliers','location','NW')
end ;

%% Appendix Figure 4: Simulation -- 33rd quantile

if figflag == 2.04 ;
    
    load MCSim                                       % Load simulation output

    [kCON0, xCON0] = ksdensity(ocCON33(:,1)) ;       % Conventional QR - no outliers
    [kCOP0, xCOP0] = ksdensity(ocCOP33(:,1)) ;       % Copula QR - no outliers
    [kCON1, xCON1] = ksdensity(ocCON33(:,2)) ;       % Conventional QR - outliers
    [kCOP1, xCOP1] = ksdensity(ocCOP33(:,2)) ;       % Copula QR - outliers

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact')
    nexttile
    plot(xCON0,kCON0,'r-','linewidth',2)
    hold
    plot(xCOP0,kCOP0,'b-','linewidth',2)
    plot(xCON1,kCON1,'r:','linewidth',2)
    plot(xCOP1,kCOP1,'b:','linewidth',2)
    legend('Conventional QR - no outliers','Copula QR - no outliers','Conventional QR - outliers','Copula QR - outliers','location','NW')
end ;

%% Appendix Figure 5: Narrow Replication -- QR coefficients and ABG CIs, ABG's sample

  if figflag == 2.05 ;
    
    % Load Output
    load ABGBands_abg_obs
    p025 = prctile(bBQ',2.5) ;
    p050 = prctile(bBQ',5) ;
    p160 = prctile(bBQ',16) ;
    p840 = prctile(bBQ',84) ;
    p950 = prctile(bBQ',95) ;
    p975 = prctile(bBQ',97.5) ;
    B2 = zeros(19,1) + B2 ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(QQ,BQ,'r-','Linewidth',2)
    hold
    plot(QQ,B2,'b-','Linewidth',2)
    plot(QQ,p975,'r--','Linewidth',2) 
    plot(QQ,p025,'r--','Linewidth',2) 
    plot(QQ,p950,'r:','Linewidth',2) 
    plot(QQ,p050,'r:','Linewidth',2) 
    plot(QQ,p160,'r-.','Linewidth',2) 
    plot(QQ,p840,'r-.','Linewidth',2) 
    line([0 1],[0 0],'linestyle','-','color','k','linewidth',1)
    ylim([-0.8 0.1])
    legend('In-Sample Fit','OLS','location','SE','NumColumns',1)

  end ;
%% Appendix Figure 6: Narrow Replication -- QR coefficients and ABG CIs, extended sample

  if figflag == 2.06 ;
    
    % Load Output
    load ABGBands_ext_obs
    p025 = prctile(bBQ',2.5) ;
    p050 = prctile(bBQ',5) ;
    p160 = prctile(bBQ',16) ;
    p840 = prctile(bBQ',84) ;
    p950 = prctile(bBQ',95) ;
    p975 = prctile(bBQ',97.5) ;
    B2 = zeros(19,1) + B2 ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(QQ,BQ,'r-','Linewidth',2)
    hold
    plot(QQ,B2,'b-','Linewidth',2)
    plot(QQ,p975,'r--','Linewidth',2) 
    plot(QQ,p025,'r--','Linewidth',2) 
    plot(QQ,p950,'r:','Linewidth',2) 
    plot(QQ,p050,'r:','Linewidth',2) 
    plot(QQ,p160,'r-.','Linewidth',2) 
    plot(QQ,p840,'r-.','Linewidth',2) 
    line([0 1],[0 0],'linestyle','-','color','k','linewidth',1)
    ylim([-0.8 0.1])
    legend('In-Sample Fit','OLS','location','SE','NumColumns',1)

  end ;
  
%% Appendix Figure 7: Narrow Replication -- Distribution of GDP growth over time, extended sample (ABG Fig 1)

if figflag == 2.07 ;
    
    % Load Output
    load ResMatch_H1
    
     qtr = (1973:0.25:2021.5)' ;                    % Dates
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    meshc(qtr, ResMatch.YY', ResMatch.PST')
    view(85, 25)
    xlim([1973 2022])
    ylim([-5 3])
    zlim([0 10])
end ;
%% Appendix Figure 8: Narrow Replication -- Distribution of GDP growth during pandemic

if figflag == 2.08 ;
    
    % Load Output
    load ResMatch_H1
    
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    yyaxis left 
    plot(ResMatch.YY', ResMatch.PST(189,:)','b:','Linewidth',2)
    hold
    plot(ResMatch.YY', ResMatch.PST(190,:)','b--','Linewidth',2)
    plot(ResMatch.YY', ResMatch.PST(191,:)','b-','Linewidth',2)
    plot(ResMatch.YY', ResMatch.PST(193,:)','g-','Linewidth',2)
    plot(ResMatch.YY', ResMatch.PST(194,:)','g--','Linewidth',2)
    plot(ResMatch.YY', ResMatch.PST(195,:)','g:','Linewidth',2)
    xlim([-5 5])
    ylim([0 1.5])
    yyaxis right
    plot(ResMatch.YY', ResMatch.PST(192,:)')
    xlim([-5 5])
    ylim([0 10])
    legend('2020:1','2020:2','2020:3','2021:1','2021:2','2021:3','2020:4','location','NW','NumColumns',2)
end ;




%% Appendix Figure 9: Narrow Replication -- Entropy

  if figflag == 2.09 ;
    
    % Load Output
    load ResMatch_H1
    
    qtr = (1973:0.25:2021.5)' ;                    % Dates
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(qtr,ResMatch.LeftEntropyST,'b-','Linewidth',2)
    hold
    plot(qtr,ResMatch.RightEntropyST,'r--','Linewidth',2) 
    xlim([1973 2022])
    legend('Downside','Upside','location','NW','NumColumns',1)

  end ;

%% Appendix Figure 10: Narrow Replication -- Longrise / Shortfall

  if figflag == 2.10 ;
    
    % Load Output
    load ResMatch_H1
    
    qtr = (1973:0.25:2021.5)' ;                    % Dates
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(qtr,ResMatch.EsfST,'b-','Linewidth',2)
    hold
    plot(qtr,ResMatch.EljST,'r--','Linewidth',2) 
    xlim([1973 2022])
    ylim([-5 5])
    legend('Shortfall','Longrise','location','NW','NumColumns',1)

  end ;

%% Appendix Figure 11: Narrow Replication -- Out of Sample Quantiles

  if figflag == 2.11 ;
  
    % Load Output
    load ResOOS_H1
    
    qtr = (1973:0.25:2021.5)' ;                    % Dates
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(qtr,YQ_IS(:,10),'b-','Linewidth',2) 
    hold
    plot(qtr,YQ_OOS(:,10), 'r--','Linewidth',2)
    plot(qtr,YQ_IS(:,1), 'b-','Linewidth',1)
    plot(qtr,YQ_IS(:,19), 'b-','Linewidth',1)
    plot(qtr,YQ_OOS(:,1), 'r--','Linewidth',1)
    plot(qtr,YQ_OOS(:,19), 'r--','Linewidth',1)
    xlim([1973 2022])
    ylim([-5 3])
    legend('In Sample','Out of Sample','location','NW','NumColumns',1)

  end ;

%% Appendix Figure 12: Narrow Replication -- Downside entropy out of sample

  if figflag == 2.12 ;
    
    % Load Output
    load ResOOS_H1
    
    qtr = (1973:0.25:2021.5)' ;                    % Dates
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(qtr,LeftEntropy_IS,'b-','Linewidth',2)
    hold
    plot(qtr,LeftEntropy_OOS,'r--','Linewidth',2) 
    xlim([1973 2022])
    legend('In Sample','Out of Sample','location','N','NumColumns',1)

  end ;

%% Appendix Figure 13: Broad Replication QR lines, ABG sample (1 qtr growth from GDPC1), pseudo data

  if figflag == 2.13 ;
    
    % Load Data -- Standardize and transform
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    
    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;          % Data to 2015Q4 only
    n = n(1:end-num_ext) ;          % Data to 2015Q4 only
    T = length(g) ;
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data output - ABG sample
    nps = norminv(nr,0,1) ;         % pseudo data NFCI - ABG sample
    clear T gr nr

    % Load estimation results: 1 qtr growth on ABG's sample
 
    load EST_1qtr_abg
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nps,pPs,'LineWidth',2)
    hold
    scatter(nps,gps, 25, 'o','b','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-3 3 -3 3])
    grid on 
    box on
    
  end ;

 
%% Appendix Figure 14: Board Replication -- QR coefficients and ABG CIs, pseudo data, ABG's sample

  if figflag == 2.14 ;
    
    % Load Output
    load ABGBands_abg_ps
    p025 = prctile(bBQ',2.5) ;
    p050 = prctile(bBQ',5) ;
    p160 = prctile(bBQ',16) ;
    p840 = prctile(bBQ',84) ;
    p950 = prctile(bBQ',95) ;
    p975 = prctile(bBQ',97.5) ;
    B2 = zeros(19,1) + B2 ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(QQ,BQ,'r-','Linewidth',2)
    hold
    plot(QQ,B2,'b-','Linewidth',2)
    plot(QQ,p975,'r--','Linewidth',2) 
    plot(QQ,p025,'r--','Linewidth',2) 
    plot(QQ,p950,'r:','Linewidth',2) 
    plot(QQ,p050,'r:','Linewidth',2) 
    plot(QQ,p160,'r-.','Linewidth',2) 
    plot(QQ,p840,'r-.','Linewidth',2) 
    line([0 1],[0 0],'linestyle','-','color','k','linewidth',1)
    ylim([-0.8 0.1])
    legend('In-Sample Fit','OLS','location','SE','NumColumns',1)

  end ;
%% Appendix Figure 15: Broad Replication -- QR coefficients and ABG CIs, pseudo data, Extended sample

  if figflag == 2.15 ;
    
    % Load Output
    load ABGBands_ext_ps
    p025 = prctile(bBQ',2.5) ;
    p050 = prctile(bBQ',5) ;
    p160 = prctile(bBQ',16) ;
    p840 = prctile(bBQ',84) ;
    p950 = prctile(bBQ',95) ;
    p975 = prctile(bBQ',97.5) ;
    B2 = zeros(19,1) + B2 ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile    
    plot(QQ,BQ,'r-','Linewidth',2)
    hold
    plot(QQ,B2,'b-','Linewidth',2)
    plot(QQ,p975,'r--','Linewidth',2) 
    plot(QQ,p025,'r--','Linewidth',2) 
    plot(QQ,p950,'r:','Linewidth',2) 
    plot(QQ,p050,'r:','Linewidth',2) 
    plot(QQ,p160,'r-.','Linewidth',2) 
    plot(QQ,p840,'r-.','Linewidth',2) 
    line([0 1],[0 0],'linestyle','-','color','k','linewidth',1)
    ylim([-0.8 0.1])
    legend('In-Sample Fit','OLS','location','SE','NumColumns',1)

  end ;

%% Appendix Figure 16: Board Replication -- Copula Swarm in Pseudo Space, Extended Sample

if figflag == 2.16 ;
    
    % Load Data 
    
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    T = length(g) ;
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data - output
    nps = norminv(nr,0,1) ;         % pseudo data NFCI

    load CopSim                     % Load Output from 1 qtr growth on extended sample

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact')
    nexttile
    scatter_kde(sw_nps,sw_gps,'filled', 'MarkerSize', 25);
    hold
    scatter(nps,gps,25,'o','b','filled') 
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-4 4 -6 6])
    grid on 
    box on
    colormap(summer)
    cb = colorbar();

end ;

%% Appendix Figure 17: Broad Replication -- Copula Swarm in Observed Space, Extended Sample

if figflag == 2.17 ;
    
    % Load Data 
    
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI

    load CopSim                     % Load Output from 1 qtr growth on extended sample

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact')
    nexttile
    scatter_kde(sw_nm,sw_gm,'filled', 'MarkerSize', 25);
    hold
    scatter(nm,gm,25,'o','r','filled') 
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    grid on 
    box on
    colormap(summer) 
    cb = colorbar();

end ;

%% Appendix Figure 18: QR lines -- ABG sample (4 qtr growth from GDPC1), observed data

if figflag == 2.18 ;
    
    % ABG QR -- their sample 4 qtr growth, standardized data
    
    load ABGRep
    g = g4GDP(5:end) ;              % output growth 
    n = NFCI(1:end-4) ;             % (lagged) NFCI 
    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;          % Data to 2015Q4 only
    n = n(1:end-num_ext) ;          % Data to 2015Q4 only
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI

    load EST_4qtr_abg               % Load estimation results: 1 qtr growth on ABG's sample
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    axis([-2 5 -7 3])
    ylabel('Four-quarter Economic Growth')
    xlabel('Lagged NFCI')
    grid on 
    box on

end ;

%% Appendix Figure 19: QR lines -- Extended  sample (4 qtr growth from GDPC1), observed data

if figflag == 2.19 ;
    
    load ABGRep
    g = g4GDP(5:end) ;              % output growth 
    n = NFCI(1:end-4) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI

    load EST_4qtr_ext               % Load estimation results: 1 qtr growth on extended sample
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('Four-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-2 5 -7 3])
    grid on 
    box on

end ;

%% Appendix Figure 20: QR lines -- ABG  sample (4 qtr growth from GDPC1), pseudo data

if figflag == 2.20 ;
    
    % QR Lines Pseudo Data -- extended sample 4 qtr growth
    
    load ABGRep
    g = g4GDP(5:end) ;              % output growth 
    n = NFCI(1:end-4) ;             % (lagged) NFCI 
    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    g = g(1:end-num_ext) ;          % Data to 2015Q4 only
    n = n(1:end-num_ext) ;          % Data to 2015Q4 only
    T = length(g) ;
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data - output
    nps = norminv(nr,0,1) ;         % pseudo data NFCI

    load EST_4qtr_abg               % Load estimation results: 1 qtr growth on extended sample
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nps,pPs,'LineWidth',2)
    hold
    scatter(nps,gps, 25, 'o','b','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('Four-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-3 3 -3 3])
    grid on 
    box on

end ; 

%% Appendix Figure 21: QR lines -- Extended  sample (4 qtr growth from GDPC1), pseudo data

if figflag == 2.21 ;
    
    % QR Lines Pseudo Data -- extended sample 4 qtr growth
    
    load ABGRep
    g = g4GDP(5:end) ;              % output growth 
    n = NFCI(1:end-4) ;             % (lagged) NFCI 
    T = length(g) ;
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data - output
    nps = norminv(nr,0,1) ;         % pseudo data NFCI

    load EST_4qtr_ext               % Load estimation results: 1 qtr growth on extended sample
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(nps,pPs,'LineWidth',2)
    hold
    scatter(nps,gps, 25, 'o','b','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('Four-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-3 3 -3 3])
    grid on 
    box on

end ;
%% Appendix Figure 22: Copula QR lines -- Extended  sample (4 qtr growth from GDPC1), observed data

if figflag == 2.22 ;
    
    % Het Cop lines -- extended sample 4 qtr growth
    
    load ABGRep
    g = g4GDP(5:end) ;              % output growth 
    n = NFCI(1:end-4) ;             % (lagged) NFCI 
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI

    load EST_4qtr_ext   % Load Output from 1 qtr growth on ABG's sample
    Z = [ nm pCOP ] ;
    ZZ = sortrows(Z,1) ;
    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(ZZ(:,1),ZZ(:,2:8),'LineWidth',2)
    hold
    scatter(nm,gm, 25, 'o','r','filled')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('Four-quarter Economic Growth')
    xlabel('Lagged NFCI')
    axis([-2 5 -7 3])
    grid on 
    box on

 end ;
 

 %%%%%%%%%%%%%%%%%% Appendix Figure 9: Copula QR lines -- Dependence ABG sample, Marginal Extended Sample (1 qtr growth from GDPC1)

if figflag == 3.01     ;
    
    num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
    
    % Load Data 
    load ABGRep
    g = g1GDP(2:end) ;              % output growth 
    n = NFCI(1:end-1) ;             % (lagged) NFCI 
    
    % Pseudo data
    T = length(g) ;
    gr = tiedrank(g)/(T+1) ;        % ranks - output growth   
    nr = tiedrank(n)/(T+1) ;        % ranks lagged NFCI
    gps = norminv(gr,0,1) ;         % pseudo data output - ABG sample
    nps = norminv(nr,0,1) ;         % pseudo data NFCI - ABG sample

    % Standardize and estimate output marginal on extended sample
    gm = zscore(g) ;                % Standardize output growth
    nm = zscore(n) ;                % Standardize NCFI
    gmtics = linspace(min(gm), max(gm), 100) ;
    [ssvft, ~ ] = ssvkernel(gm,gmtics) ;
    nssvft = ssvft/sum(ssvft) ;
    ssvcdf  = cumsum(nssvft) ;
    [cdf_gm, ~ ] = unique(ssvcdf) ;
      
    % Load estimation results: 1 qtr growth on ABG sample & take Pseudo QR lines to observed space
    load EST_1qtr_abg
    pCOP = []
    for q = 1:7 ;
        norm_pPs = normcdf(pPs(:,q),mean(gps),std(gps)) ;                   % QR lines to 0,1 space
        pCOPq = interp1(cdf_gm, gmtics, norm_pPs, 'spline') ;               % To standardized space via inverse marginal
        pCOP = [ pCOP pCOPq ] ;                                             % Store for current quantile
        clear norm_pPs pCOPq
    end ;
    clear q cdf_gm gmtics
    Z = [ nm(1:end-num_ext) pCOP ] ;
    ZZ = sortrows(Z,1) ;

    figure('position', [100, 100, 700, 500]) ;
    tiledlayout(1,1,'TileSpacing', 'compact','Padding','compact')
    nexttile
    plot(ZZ(:,1),ZZ(:,2:8),'LineWidth',2)
    hold
    scatter(nm(1:end-num_ext),gm(1:end-num_ext), 25, 'o','r','filled')
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    legend('5^{th}','10^{th}','33^{rd}','50^{th}','67^{th}','90^{th}','95^{th}','location','SW','NumColumns',2)
    ylabel('One-quarter Economic Growth')
    xlabel('Lagged NFCI')
    %axis([-2 5 -8 8])
    axis([-2 5 -5 2])
    grid on 
    box on
  
 end ;
 
