%% ABG replication -- Estimation
% (1) Estimates ABG's QR on observed data amd produces QR lines
% (2) Estimates QR on pseudo data and produces QR line
% (3) Produce copula QR lines using ECDF fitted marginal
% (4) Writes output to file to produce tables / figures

clear all

%% Pick various cases / parameters

clear all 
f_flag = 0 ;                    % Set to 1 to produce figures and 0 to supress figures
g_flag = 3 ;                  	% Set to 1 = ABG's series (1dp), 2 = 1 qtr growth from GDPC1, 3 = 4 qtr growth from GDPC1
s_flag = 1 ;                  	% Set to 0 for ABG's sample (ends 2015Q4); 1 for extended (ends 2021Q3)

%% Load Data 1973Q1 to 2021Q3 - see ABGRep_Data.m for details

load ABGRep

% Choose output growth series
if g_flag == 1 ;                % ABGs output growth data    
    g = ABGgGDP(2:end) ;        % output growth      
    gl = ABGgGDP(1:end-1) ;     % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates, start 1973Q2
elseif g_flag == 2 ;            % 1-qtr output growth data
    g = g1GDP(2:end) ;          % output growth
    gl = g1GDP(1:end-1) ;       % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates, start 1973Q2
elseif g_flag == 3 ;            % 4-qtr output growth data
    g = g4GDP(5:end) ;          % output growth
    gl = g4GDP(1:end-4) ;       % lagged
    n = NFCI(1:end-4) ;         % (lagged) NFCI 
    tt = QTR(5:end) ;           % dates, start 1974Q1
end ;

% If using ABG sample remove observations after 2015Q4
num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
if s_flag == 0 ;
    g = g(1:end-num_ext) ;
    gl = gl(1:end-num_ext) ;
    n = n(1:end-num_ext) ;
    tt = tt(1:end-num_ext) ;
end ;
T = length(tt);                 % sample size

clear ABGgGDP g1GDP g4GDP NFCI QTR

%% Standardized, rank and generate pseudo data as standard normal & fit marginal to output growth

% Standardized observed data for standard QR
gm = zscore(g) ;                % output growth
nm = zscore(n) ;                % lagged NFCI
glm = zscore(gl) ;              % lagged output growth

% Ranks
gr = tiedrank(g)/(T+1) ;        % output growth
nr = tiedrank(n)/(T+1) ;        % lagged NFCI
glr = tiedrank(gl)/(T+1) ;      % lagged output growth

% Standardized normal pseudo data
gps = norminv(gr,0,1) ;         % output growth
nps = norminv(nr,0,1) ;         % lagged NFCI
glps = norminv(glr,0,1) ;       % lagged output growth

clear gr nr glr

% Fit Marginal to output growth
gmtics = linspace(min(gm), max(gm), 100) ;
[ssvft, ~ ] = ssvkernel(gm,gmtics) ;
nssvft = ssvft/sum(ssvft) ;
ssvcdf  = cumsum(nssvft) ;
[cdf_gm, ~ ] = unique(ssvcdf) ;
if f_flag == 1 ;
    figure('position', [100, 100, 650, 650]) ;
    plot(gmtics,nssvft,'linewidth',2) 
end ;
clear ssvft nssvft ssvcdf

%% Std QR on (standardized) obs data, rep ABG with two RHS variables in regs

yObs = gm ;                     % dependent variable
xObs = [ ones(T,1) nm glm ] ;   % rhs variables

% q01 etc are vectors of parameters, qObs stores all vectors
q05Obs = rq(xObs, yObs, 0.05) ;
q10Obs = rq(xObs, yObs, 0.10) ;
q33Obs = rq(xObs, yObs, 0.33) ;
q50Obs = rq(xObs, yObs, 0.50) ; 
q67Obs = rq(xObs, yObs, 0.67) ; 
q90Obs = rq(xObs, yObs, 0.90) ;
q95Obs = rq(xObs, yObs, 0.95) ;
qObs = [ q05Obs q10Obs q33Obs q50Obs q67Obs q90Obs q95Obs ] ;

% QR lines on observed data, ie from parameters of QR to implied values of target variable using 
% expected value of glG (equals zero) to proxy third variable, pObs stores all QR lines
p05Obs= xObs(:,1:2)*q05Obs(1:2) ;
p10Obs = xObs(:,1:2)*q10Obs(1:2) ;
p33Obs = xObs(:,1:2)*q33Obs(1:2) ; 
p50Obs = xObs(:,1:2)*q50Obs(1:2) ;
p67Obs = xObs(:,1:2)*q67Obs(1:2) ;
p90Obs = xObs(:,1:2)*q90Obs(1:2) ;
p95Obs = xObs(:,1:2)*q95Obs(1:2) ;
pObs = [ p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs ] ;

% Plot figure
if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nm,pObs,'LineWidth',2)
    hold
    scatter(nm,gm, 50, 'o','r','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Observed Data')
    grid on 
    box on
end ;

clear yObs xObs q05Obs q10Obs q33Obs q50Obs q67Obs q90Obs q95Obs p05Obs p10Obs p33Obs p50Obs p67Obs p90Obs p95Obs

%% QR on pseudo data, with two RHS variables

yPs = gps ;                     % depenedent variable
xPs = [ ones(T,1) nps glps ] ;  % rhs variables

% q01 etc are vectors of parameters, qPs stores all vectors
q05Ps = rq(xPs, yPs, 0.05) ;  
q10Ps = rq(xPs, yPs, 0.10) ;       
q33Ps = rq(xPs, yPs, 0.33) ;
q50Ps = rq(xPs, yPs, 0.50) ; 
q67Ps = rq(xPs, yPs, 0.67) ; 
q90Ps = rq(xPs, yPs, 0.90) ;
q95Ps = rq(xPs, yPs, 0.95) ;  
qPs = [ q05Ps q10Ps q33Ps q50Ps q67Ps q90Ps q95Ps ] ;

% QR lines for pseudo data, ie from parameters of QR to implied values of target variable using
% expected value of glG equals zero to proxy third variable, pPs stores all QR lines
p05Ps = xPs(:,1:2)*q05Ps(1:2) ;
p10Ps = xPs(:,1:2)*q10Ps(1:2) ;
p33Ps = xPs(:,1:2)*q33Ps(1:2) ; 
p50Ps = xPs(:,1:2)*q50Ps(1:2) ; 
p67Ps = xPs(:,1:2)*q67Ps(1:2) ;
p90Ps = xPs(:,1:2)*q90Ps(1:2) ;
p95Ps = xPs(:,1:2)*q95Ps(1:2) ;
pPs = [ p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps ] ;

% Plot figure
if f_flag == 1 ;
    figure('position', [100, 100, 850, 550]) ;
    plot(nps,pPs,'LineWidth',2)
    hold
    scatter(nps,gps, 50, 'o','r','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('QR Pseudo Data')
    grid on 
    box on
end ;

clear yPs xPs q05Ps q10Ps q33Ps q67Ps q90Ps q95Ps p05Ps p10Ps p33Ps p50Ps p67Ps p90Ps p95Ps

%% Take QR lines back to observed space using inverse marginal

pCOP = [] ;
for q = 1:7 ;                                                           % Do loop through quantiles
    norm_pPs = normcdf(pPs(:,q),mean(gps),std(gps)) ;                   % QR lines to 0,1 space
    pCOPq = interp1(cdf_gm, gmtics, norm_pPs, 'spline') ;               % To standardized space via inverse ECDF marginal
    pCOP = [ pCOP pCOPq ] ;                                             % Store for current quantile
    clear norm_pPs pCOPq
end ;
clear q cdf_gm gmtics

% Plot figure
if f_flag == 1 ;
    Z = [ nm pCOP ] ;                                                   % Need to sort by fincon before plot
    ZZ = sortrows(Z,1) ;                                                % Sort pCOP by fincon
    figure('position', [100, 100, 850, 550]) ;
    plot(ZZ(:,1),ZZ(:,2:8),'LineWidth',2)
    hold
    scatter(nm,gm, 50, 'o','r','filled')
    legend('5th','10th','33rd','50th','67th','90th','95th','location','SW')
    title('Copula QR')
    grid on 
    box on
    clear Z ZZ 
end ;

%% Save objects to files to produce figures / tables

if g_flag == 1 && s_flag == 0 ;                     % ABG's 1 dp growth measure & ABG's sample
    save('EST_1dp_abg.mat','qObs','pObs','qPs','pPs','pCOP')
elseif g_flag == 1 && s_flag == 1 ;                 % ABG's 1 dp growth measure & Extended sample 
    save('EST_1dp_ext.mat','qObs','pObs','qPs','pPs','pCOP')
elseif g_flag == 2 && s_flag == 0 ;                 % 1 qtr growth measure & ABG's sample
    save('EST_1qtr_abg.mat','qObs','pObs','qPs','pPs','pCOP')
elseif g_flag == 2 && s_flag == 1 ;                 % 1 qtr growth measure & Extended sample 
    save('EST_1qtr_ext.mat','qObs','pObs','qPs','pPs','pCOP')
elseif g_flag == 3 && s_flag == 0 ;                 % 4 qtr growth measure & ABG's sample
    save('EST_4qtr_abg.mat','qObs','pObs','qPs','pPs','pCOP')
elseif g_flag == 3 && s_flag == 1 ;                 % 4 qtr growth measure & Extended sample 
    save('EST_4qtr_ext.mat','qObs','pObs','qPs','pPs','pCOP')
end ;

