%% Block Bootstrap for QR coefficients confidence intervals

%% Pick various cases / parameters
clear all 
tic
g_flag = 3 ;                            % Set to 1 = ABG's series (1dp), 2 = 1 qtr growth from GDPC1, 3 = 4 qtr growth from GDPC1
s_flag = 1 ;                            % Set to 0 for ABG's sample (ends 2015Q4); 1 for extended (ends 2021Q3)
Blength = 6 ;                           % Block length for bootstrap
nsam = 10000 ;                          % Number of bootstrap samples
rng(2)                                  % Seeds random number generator

%% Load Data 1973Q1 to 2020Q4 - see ABGRep_Data.m for details

load ABGRep

% Choose output growth series
if g_flag == 1 ;                % ABGs output growth data    
    g = ABGgGDP(2:end) ;        % output growth      
    gl = ABGgGDP(1:end-1) ;     % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates for plots, start 1973Q2
elseif g_flag == 2 ;            % 1-qtr output growth data
    g = g1GDP(2:end) ;          % output growth
    gl = g1GDP(1:end-1) ;       % lagged
    n = NFCI(1:end-1) ;         % (lagged) NFCI 
    tt = QTR(2:end) ;           % dates for plots, start 1973Q2
elseif g_flag == 3 ;            % 4-qtr output growth data
    g = g4GDP(5:end) ;          % output growth
    gl = g4GDP(1:end-4) ;       % lagged
    n = NFCI(1:end-4) ;         % (lagged) NFCI 
    tt = QTR(5:end) ;           % dates for plots, start 1973Q2
end ;

% If using ABG sample remove observations after 2015Q4
num_ext = 23 ;                  % Data extended to 2021Q3, 23 observations past end of ABG sample
if s_flag == 0 ;
    g = g(1:end-num_ext) ;
    gl = gl(1:end-num_ext) ;
    n = n(1:end-num_ext) ;
    tt = tt(1:end-num_ext) ;
end ;

T = length(tt);                 % sample size

%% Create overlapping bootstrap blocks

% Matrices to store blocks of data - each column to contain a block of Blength observations
gBB = NaN(Blength,T-Blength+1) ;        % output growth 
nBB = NaN(Blength,T-Blength+1) ;        % lagged ncfi 
glBB = NaN(Blength,T-Blength+1) ;       % lagged output growth

% Create (T - Blength) overlapping blocks
for t = 1:T-Blength+1 ;
    gBB(:,t) = g(t:t+Blength-1) ;       % output growth
    nBB(:,t) = n(t:t+Blength-1) ;       % lagged ncfi
    glBB(:,t) = gl(t:t+Blength-1) ;     % lagged output growth
end ;

numB = ceil(T/Blength) ;                % number of blocks reqd for bootstrap sample of length T 


%% Draw bootstrap samples and estimate QR specs

% Store bootstrap estimates here
qOBSbs = NaN(3,7,nsam) ;                % Observed
qPSbs = NaN(3,7,nsam) ;                 % Pseudo

for j = 1:nsam ;
	draw_j = ceil(length(gBB)*rand(numB,1)) ;       % vector of locations for start of blocks for sample j
    gj = [] ;                                       % store bootstrap sample for output here
    nj = [] ;                                       % store bootstrap sample for ncfi here
    glj = [] ;                                      % store bootstrap sample for lagged output here
    for b = 1:numB ;                                % begin drawing blocks for bootstrap sample
        gj = [ gj ; gBB(:,draw_j(b)) ] ;            % add next block for output growth
        nj = [ nj ; nBB(:,draw_j(b)) ] ;            % add next block for ncgi growth
        glj = [ glj ; glBB(:,draw_j(b)) ] ;         % add next block for lagged output growth
    end ;
    
    % Shorten series to match sample data (removes part of last block)
    gj = gj(1:T) ;                                  % output growth
    nj = nj(1:T) ;                                  % lagged nfci
    glj = glj(1:T) ;                                % lagged output groth
    
    % Standardize
    gmj = zscore(gj) ;                              % output growth
    nmj = zscore(nj) ;                              % nfci
    glmj = zscore(glj) ;                            % lagged output growth

    % Ranks
    grj = tiedrank(gmj)/(T+1) ;                     % output growth   
    nrj = tiedrank(nmj)/(T+1) ;                     % nfci
    glrj = tiedrank(glmj)/(T+1) ;                   % lagged output growth

    % Pass through inverse normal to get pseudo data
    gpsj = norminv(grj,0,1) ;                       % output growth
    npsj = norminv(nrj,0,1) ;                       % nfci
    glpsj = norminv(glrj,0,1) ;                     % lagged output growth

    % Std QR on obs data
    yOBSj = gmj ;                                   % dependent variable
    xOBSj = [ ones(T,1) nmj glmj ] ;                % rhs variables
    q05OBSj = rq(xOBSj, yOBSj, 0.05) ;
    q10OBSj = rq(xOBSj, yOBSj, 0.10) ;
    q33OBSj = rq(xOBSj, yOBSj, 0.33) ;
    q50OBSj = rq(xOBSj, yOBSj, 0.50) ; 
    q67OBSj = rq(xOBSj, yOBSj, 0.67) ; 
    q90OBSj = rq(xOBSj, yOBSj, 0.90) ;
    q95OBSj = rq(xOBSj, yOBSj, 0.95) ;
    % Store estimates for bootstrap sample j
    qOBSbs(:,:,j) = [ q05OBSj q10OBSj q33OBSj q50OBSj q67OBSj q90OBSj q95OBSj ] ;
    clear yOBSj xOBSj q05OBSj q10OBSj q33OBSj q50OBSj q67OBSj q90OBSj q95OBSj 

    % QR on pseudo data
    yPSj = gpsj ;                                   % dependent variable
    xPSj = [ ones(T,1) npsj glpsj ] ;               % rhs variables
    q05PSj = rq(xPSj, yPSj, 0.05) ;  
    q10PSj = rq(xPSj, yPSj, 0.10) ;       
    q33PSj = rq(xPSj, yPSj, 0.33) ;
    q50PSj = rq(xPSj, yPSj, 0.50) ; 
    q67PSj = rq(xPSj, yPSj, 0.67) ; 
    q90PSj = rq(xPSj, yPSj, 0.90) ;
    q95PSj = rq(xPSj, yPSj, 0.95) ;  
    % Store estimates for bootstrap sample j
    qPSbs(:,:,j) = [ q05PSj q10PSj q33PSj q50PSj q67PSj q90PSj q95PSj ] ;
    clear yPSj xPSj q05PSj q10PSj q33PSj q50PSj q67PSj q90PSj q95PSj 
     
end ;
toc

% Caclulate percentiles from BB samples for each quantile

OBS_BB = NaN(7,5,3) ;
PS_BB = NaN(7,5,3) ;

for q = 1:7 ;
    
    % OBS QR, parameters for current quantile
    OBS1q = reshape(qOBSbs(1,q,:),nsam,1) ;         % intercept
    OBS2q = reshape(qOBSbs(2,q,:),nsam,1) ;         % lagged nfci
    OBS3q = reshape(qOBSbs(3,q,:),nsam,1) ;         % lagged output growth
    
    % QR on pseudo data, parameters for current quantile
    PS1q = reshape(qPSbs(1,q,:),nsam,1) ;           % intercept
    PS2q = reshape(qPSbs(2,q,:),nsam,1) ;           % lagged nfci
    PS3q = reshape(qPSbs(3,q,:),nsam,1) ;           % lagged output growth

    % Percentiles for OBS's QR - intercept
    OBS_BB(q,1,1) = prctile(OBS1q,5) ;
    OBS_BB(q,2,1) = prctile(OBS1q,33) ;
    OBS_BB(q,3,1) = prctile(OBS1q,50) ;
    OBS_BB(q,4,1) = prctile(OBS1q,67) ;
    OBS_BB(q,5,1) = prctile(OBS1q,95) ;

    % Percentiles for OBS's QR - nfci
    OBS_BB(q,1,2) = prctile(OBS2q,5) ;
    OBS_BB(q,2,2) = prctile(OBS2q,33) ;
    OBS_BB(q,3,2) = prctile(OBS2q,50) ;
    OBS_BB(q,4,2) = prctile(OBS2q,67) ;
    OBS_BB(q,5,2) = prctile(OBS2q,95) ;
    
    % Percentiles for OBS's QR - lagged output
    OBS_BB(q,1,3) = prctile(OBS3q,5) ;
    OBS_BB(q,2,3) = prctile(OBS3q,33) ;
    OBS_BB(q,3,3) = prctile(OBS3q,50) ;
    OBS_BB(q,4,3) = prctile(OBS3q,67) ;
    OBS_BB(q,5,3) = prctile(OBS3q,95) ;

    % Percentiles for QR on pseudo data - intercept
    PS_BB(q,1,1) = prctile(PS1q,5) ;
    PS_BB(q,2,1) = prctile(PS1q,33) ;
    PS_BB(q,3,1) = prctile(PS1q,50) ;
    PS_BB(q,4,1) = prctile(PS1q,67) ;
    PS_BB(q,5,1) = prctile(PS1q,95) ;

    % Percentiles for QR on pseudo data - nfci
    PS_BB(q,1,2) = prctile(PS2q,5) ;
    PS_BB(q,2,2) = prctile(PS2q,33) ;
    PS_BB(q,3,2) = prctile(PS2q,50) ;
    PS_BB(q,4,2) = prctile(PS2q,67) ;
    PS_BB(q,5,2) = prctile(PS2q,95) ;

    % Percentiles for QR on pseudo data - lagged output
    PS_BB(q,1,3) = prctile(PS3q,5) ;
    PS_BB(q,2,3) = prctile(PS3q,33) ;
    PS_BB(q,3,3) = prctile(PS3q,50) ;
    PS_BB(q,4,3) = prctile(PS3q,67) ;
    PS_BB(q,5,3) = prctile(PS3q,95) ;
    
end ;

%% Save objects to files to produce tables

if g_flag == 1 && s_flag == 0                           % ABG's 1 dp growth measure & ABG's sample
    save('BB_1dp_abg.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
elseif g_flag == 1 && s_flag == 1 ;                     % ABG's 1 dp growth measure & Extended sample 
    save('BB_1dp_ext.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
elseif g_flag == 2 && s_flag == 0 ;                     % 1 qtr growth measure & ABG's sample
    save('BB_1qtr_abg.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
elseif g_flag == 2 && s_flag == 1 ;                     % 1 qtr growth measure & Extended sample 
    save('BB_1qtr_ext.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
elseif g_flag == 3 && s_flag == 0 ;                     % 4 qtr growth measure & ABG's sample
    save('BB_4qtr_abg.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
elseif g_flag == 3 && s_flag == 1 ;                     % 4 qtr growth measure & Extended sample 
    save('BB_4qtr_ext.mat','OBS_BB','PS_BB','qOBSbs','qPSbs')
end ;

