
/*************************************************************
	Nordhaus-type testing of serial correlation of revisions
	(Fixed event instead of fixed horizon)
	"Raiders of the Lost High Frequency Forecasts: New Data and Evidence on the Efficiency of the Fed's Forecasting"
	
	Andrew C. Chang, Board of Governors of the Federal Reserve System, a.christopher.chang@gmail.com
	ORCID: 0000-0002-9769-789X
    April 6, 2022

	1) Create fixed-event revisions of the forecasts
	1a) OPTIONAL: FORCE MINIMUM 1 BP REVISION BETWEEN OBSERVATIONS
	2) Create lagged revisions, indicators for forecast distance, 1-7+ days, and interactions
	3) Baseline MZ, One Coefficient on Lagged Revision
	4) Primary equation, One Coefficient on Lagged Revision x I(Tau)
	5) Baseline MZ, Seven Coefficients on lagged revision
	6) Primary equation, Seven Coefficients on Lagged Revision, Unweighted 
	7) Primary equation, Seven Coefficients on Lagged Revision, Weighted
	8) POST TABLES TO LATEX FOR PRIMARY EQUATION, ONE COEFFICIENT  
	
*************************************************************/

clear all
set more off
capture log close

global Data "$Work\Data"

log using ${Work}/results/logs/final/regs_final_public.log, replace

version 13.0

import delimited "$Data/public/hf_final_allobservations_public.csv"

******************************************************* 
* 1) Create fixed-event revisions of the forecasts
******************************************************* 

generate panelvar = quarterly(projqtr, "YQ") if variable == "Core PCE Inflation" //The panelvar is going to be the combination of variable-target quarter
replace panelvar = quarterly(projqtr, "YQ") + 200 if variable == "GDP"   //the +200 is just an arbitrary number so that the panelvar for GDP and inflation don't overlap
generate date_numeric = date(date,"DMY",2020)

sort variable projqtr date_numeric
generate time = _n
xtset panelvar time

* Create fixed event versions of forecast revisions
generate rgdp_rev_fixed_window = d.rgdp_fore
generate pcepilfe_rev_fixed_window = d.pcepilfe_fore
generate fore_dist_fixed_window = d.date_numeric  //For computing the number of days between forecasts in the fixed event framework


/*
**********************************************************************************************************************
** 1a) OPTIONAL: FORCE MINIMUM 1 BP REVISION BETWEEN OBSERVATIONS TO WEED OUT THIN TRADING (Different from Fixed Horizon Version) 
** Results are similar with or without this restriction
**********************************************************************************************************************

* Drop all observations of < 1 bp revision from previous forecast
foreach var in rgdp pcepilfe {

   preserve
         
   if "`var'"=="rgdp" {
      keep if variable=="GDP"
   }
   else {
      keep if variable=="Core PCE Inflation"
   }
 
   * Only takes two iterations to complete; run through obs twice
   //forvalues i = 1/2 {
      *Due to machine precision issues, in the second clause it is < 0.00999 not < 0.01
      drop if !missing(`var'_rev_fixed_window) & abs(`var'_rev_fixed_window)<0.00999   //No if clause for fore_dist > 0 here since all fore_dist_fixed_window is either missing or > 0
     // replace `var'_rev_fixed_window = d.`var'_fore
	 // replace `var'_rev_fixed_window = `var'_fore-`var'_fore[_n+1] 
      
   //}
      
   tempfile new`var'
   save `new`var''
      
   restore

}

use `newrgdp', clear
append using `newpcepilfe'

//Reset the time variable without gaps
sort variable projqtr date_numeric
replace time = _n
sort panelvar time
*/

******************************************************* 
* 2) Create lagged revisions, indicators for forecast distance, 1-7+ days, and interactions
******************************************************* 

generate l1rgdp_rev_fixed_window = L.rgdp_rev_fixed_window 
generate l1pcepilfe_rev_fixed_window = L.pcepilfe_rev_fixed_window

forvalues i == 1/6 {
	generate fore_dist_fixed_window_`i' = 0
	replace fore_dist_fixed_window_`i' = 1 if fore_dist_fixed_window == `i'
}
//Group all of the forecast distance >= 7 days into the same bucket
generate fore_dist_fixed_window_7 = 0
replace fore_dist_fixed_window_7 = 1 if fore_dist_fixed_window >= 7

forvalues i == 1/7 {

	foreach var in rgdp pcepilfe {

		generate l1`var'_rev_fixed_window_fd`i' = l1`var'_rev_fixed_window*fore_dist_fixed_window_`i'  
	
	}

}

foreach var in rgdp pcepilfe {

     local upp = upper("`var'")

	*********************************       
	* 3) Baseline MZ, One Coefficient on Lagged Revision
	*********************************
	
	/* Fixed horizon, unweighted regression */
	reg `var'_rev_fixed_window l1`var'_rev_fixed_window, vce(hc3)
	eststo `var'_unwgt_mz
	
	/* Fixed horizon, weighted regression */
	reg `var'_rev_fixed_window l1`var'_rev_fixed_window [pweight=fore_dist_fixed_window]  
	eststo `var'_wgt_mz
	
	*************************************    
	* 4) Primary equation, One Coefficient on Lagged Revision x I(Tau)
	*************************************
	
	/* Fixed horizon, unweighted regression with weeks01 */
	xi: reg `var'_rev_fixed_window i.weeks01*l1`var'_rev_fixed_window, vce(hc3) 

	
		if "`var'"=="rgdp" {
			test _cons + _Iweeks01_1 = 0
			eret2 scalar pVal_ConsPlusWeeks01 = r(p)
			test l1rgdp_rev_fixed_window + _IweeXl1rgd_1 = 0
			eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
		else {
			test _cons + _Iweeks01_1 = 0
			eret2 scalar pVal_ConsPlusWeeks01 = r(p)
			test l1pcepilfe_rev_fixed_window + _IweeXl1pce_1 = 0
			eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
	eststo `var'_unwgt_mz_w
	
	/* Fixed horizon, weighted regression with weeks01 */
	xi: reg `var'_rev_fixed_window i.weeks01*l1`var'_rev_fixed_window [pweight=fore_dist_fixed_window]
	
		if "`var'"=="rgdp" {
			 test _cons + _Iweeks01_1 = 0
			 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
			 test l1rgdp_rev_fixed_window + _IweeXl1rgd_1 = 0
			 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
		else {
			 test _cons + _Iweeks01_1 = 0
			 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
			 test l1pcepilfe_rev_fixed_window + _IweeXl1pce_1 = 0
			 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
	eststo `var'_wgt_mz_w
	

	*********************************       
	* 5) Baseline MZ, Seven Coefficients on lagged revision (one for each day from 1-6, then 7+ days between revisions)
	*********************************
	
	/* Fixed horizon, unweighted regression */
	reg `var'_rev_fixed_window l1`var'_rev_fixed_window_fd*, vce(hc3)
	eststo `var'_unwgt_mz_fd7
	
	* F test that all revision coefficients are zero
	test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = l1`var'_rev_fixed_window_fd4 = ///
	l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = l1`var'_rev_fixed_window_fd7 = 0
	
	/* Fixed horizon, weighted regression */
	reg `var'_rev_fixed_window l1`var'_rev_fixed_window_fd* [pweight=fore_dist_fixed_window]
	eststo `var'_wgt_mz_fd7
	
	* F test that all revision coefficients are zero
	test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = l1`var'_rev_fixed_window_fd4 = ///
	l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = l1`var'_rev_fixed_window_fd7 = 0
	
	
	****************************************************    
	* 6) Primary equation, Seven Coefficients on Lagged Revision, Unweighted 
	****************************************************
	
	/* Fixed horizon, unweighted regression with weeks01 */
	xi: reg `var'_rev_fixed_window i.weeks01*l1`var'_rev_fixed_window_fd1 i.weeks01*l1`var'_rev_fixed_window_fd2 ///
	i.weeks01*l1`var'_rev_fixed_window_fd3 i.weeks01*l1`var'_rev_fixed_window_fd4 ///
	i.weeks01*l1`var'_rev_fixed_window_fd5 i.weeks01*l1`var'_rev_fixed_window_fd6 ///
	i.weeks01*l1`var'_rev_fixed_window_fd7, vce(hc3)  
	
	if "`var'"=="rgdp" {
		//Test revisions > 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = ///
		l1`var'_rev_fixed_window_fd4 = l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = ///
		l1`var'_rev_fixed_window_fd7 = 0
	
		//Test difference in revisions <= 2 weeks from FOMC from those > 2 weeks from FOMC
		test _IweeXl1rgd_1 = _IweeXl1rgda1 = _IweeXl1rgdb1 = _IweeXl1rgdc1 = _IweeXl1rgdd1 = ///
		_IweeXl1rgde1 = _IweeXl1rgdf1 = 0
		
		//Test revisions <= 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 + _IweeXl1rgd_1 = l1`var'_rev_fixed_window_fd2 + _IweeXl1rgda1 = l1`var'_rev_fixed_window_fd3 + _IweeXl1rgdb1 = ///
		l1`var'_rev_fixed_window_fd4 + _IweeXl1rgdc1 = l1`var'_rev_fixed_window_fd5 + _IweeXl1rgdd1 = l1`var'_rev_fixed_window_fd6 + _IweeXl1rgde1 = ///
		l1`var'_rev_fixed_window_fd7 + _IweeXl1rgdf1 = 0
		
	}
	else {
		//Test revisions > 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = ///
		l1`var'_rev_fixed_window_fd4 = l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = ///
		l1`var'_rev_fixed_window_fd7 = 0
		
		//Test difference in revisions <= 2 weeks from FOMC from those > 2 weeks from FOMC
		test _IweeXl1pce_1 = _IweeXl1pcea1 = _IweeXl1pceb1 = _IweeXl1pcec1 = _IweeXl1pced1 = ///
		_IweeXl1pcee1 = _IweeXl1pcef1 = 0
		
		//Test revisions <= 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 + _IweeXl1pce_1 = l1`var'_rev_fixed_window_fd2 + _IweeXl1pcea1 = l1`var'_rev_fixed_window_fd3 + _IweeXl1pceb1 = ///
		l1`var'_rev_fixed_window_fd4 + _IweeXl1pcec1 = l1`var'_rev_fixed_window_fd5 + _IweeXl1pced1 = l1`var'_rev_fixed_window_fd6 + _IweeXl1pcee1 = ///
		l1`var'_rev_fixed_window_fd7 + _IweeXl1pcef1 = 0
		
	}
	
	****************************************************    
	* 7) Primary equation, Seven Coefficients on Lagged Revision, Weighted
	****************************************************
	
	/* Fixed horizon, weighted regression with weeks01 */
	xi: reg `var'_rev_fixed_window i.weeks01*l1`var'_rev_fixed_window_fd1 i.weeks01*l1`var'_rev_fixed_window_fd2 ///
	i.weeks01*l1`var'_rev_fixed_window_fd3 i.weeks01*l1`var'_rev_fixed_window_fd4 ///
	i.weeks01*l1`var'_rev_fixed_window_fd5 i.weeks01*l1`var'_rev_fixed_window_fd6 ///
	i.weeks01*l1`var'_rev_fixed_window_fd7 [pweight=fore_dist_fixed_window]
	
	if "`var'"=="rgdp" {
		//Test revisions > 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = ///
		l1`var'_rev_fixed_window_fd4 = l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = ///
		l1`var'_rev_fixed_window_fd7 = 0
	
		//Test difference in revisions <= 2 weeks from FOMC from those > 2 weeks from FOMC
		test _IweeXl1rgd_1 = _IweeXl1rgda1 = _IweeXl1rgdb1 = _IweeXl1rgdc1 = _IweeXl1rgdd1 = ///
		_IweeXl1rgde1 = _IweeXl1rgdf1 = 0
		
		//Test revisions <= 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 + _IweeXl1rgd_1 = l1`var'_rev_fixed_window_fd2 + _IweeXl1rgda1 = l1`var'_rev_fixed_window_fd3 + _IweeXl1rgdb1 = ///
		l1`var'_rev_fixed_window_fd4 + _IweeXl1rgdc1 = l1`var'_rev_fixed_window_fd5 + _IweeXl1rgdd1 = l1`var'_rev_fixed_window_fd6 + _IweeXl1rgde1 = ///
		l1`var'_rev_fixed_window_fd7 + _IweeXl1rgdf1 = 0
		
	}
	else {
		//Test revisions > 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 = l1`var'_rev_fixed_window_fd2 = l1`var'_rev_fixed_window_fd3 = ///
		l1`var'_rev_fixed_window_fd4 = l1`var'_rev_fixed_window_fd5 = l1`var'_rev_fixed_window_fd6 = ///
		l1`var'_rev_fixed_window_fd7 = 0
		
		//Test difference in revisions <= 2 weeks from FOMC from those > 2 weeks from FOMC
		test _IweeXl1pce_1 = _IweeXl1pcea1 = _IweeXl1pceb1 = _IweeXl1pcec1 = _IweeXl1pced1 = ///
		_IweeXl1pcee1 = _IweeXl1pcef1 = 0
		
		//Test revisions <= 2 weeks from FOMC
		test l1`var'_rev_fixed_window_fd1 + _IweeXl1pce_1 = l1`var'_rev_fixed_window_fd2 + _IweeXl1pcea1 = l1`var'_rev_fixed_window_fd3 + _IweeXl1pceb1 = ///
		l1`var'_rev_fixed_window_fd4 + _IweeXl1pcec1 = l1`var'_rev_fixed_window_fd5 + _IweeXl1pced1 = l1`var'_rev_fixed_window_fd6 + _IweeXl1pcee1 = ///
		l1`var'_rev_fixed_window_fd7 + _IweeXl1pcef1 = 0
		
	}
	
	
	
	************************************************************************
	* 8) POST TABLES TO LATEX FOR PRIMARY EQUATION, ONE COEFFICIENT           *
	************************************************************************
	
	 if "`var'"=="rgdp" {
		* Columns are baseline, baselineweighted,  weeks01xrevison unweighted, weeks01xrevision weighted
		* MZ WITH WEEKS01 (notation I(Tau)) OUTPUT*
		esttab  `var'_unwgt_mz `var'_wgt_mz `var'_unwgt_mz_w `var'_wgt_mz_w ///
		using ${Work}/results/latex/nonpap_regs/fixed_event/`var'_mz_w_public_fixed_event.tex, ///
		se ar2(2) replace scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H_{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H_{0}:\$\ Lagged Revision +" "placeholderrow2 \$I(\tau) \times\$Lagged Revision = 0") sfmt(2) ///
		title(""\label{}) ///
		mlabels("Unweighted" "Weighted" "Unweighted" "Weighted", lhs("")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		l1`var'_rev_fixed_window "Lagged Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXl1rgd_1 "\$I(\tau) \times\$Lagged Revision" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		fragment ///
		substitute(\_ _) ///
		
	 }
	
	else {
		esttab  `var'_unwgt_mz `var'_wgt_mz `var'_unwgt_mz_w `var'_wgt_mz_w ///
		using ${Work}/results/latex/nonpap_regs/fixed_event/`var'_mz_w_public_fixed_event.tex, ///
		se ar2(2) replace scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H_{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H_{0}:\$\ Lagged Revision +" "placeholderrow2 \$I(\tau) \times\$Lagged Revision = 0") sfmt(2) ///
		title(""\label{}) ///
		mlabels("Unweighted" "Weighted" "Unweighted" "Weighted", lhs("")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		l1`var'_rev_fixed_window "Lagged Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXl1pce_1 "\$I(\tau) \times\$Lagged Revision" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		fragment ///
		substitute(\_ _) ///
		
	}
	
}

	
log close

