/*************************************************************
   Plot Figures for "Raiders of the Lost High-Frequency Forecasts"
	
	Andrew C. Chang, October 13, 2021
	Board of Governors of the Federal Reserve System, a.christopher.chang@gmail.com
	ORCID: 0000-0002-9769-789X

	1) Six panel staff forecast (Figure 1)
	2) Forecast errors for current quarter (not in paper)
	3) RMSE by type of forecast (inline text comments in appendix)
	4) Plots of RMSE By Week to Start of Quarter (Figure 2)
	5) Plot of Macroeconomic News Index (Figure 4)
	6) Plots of RMSE By Day to Start of Quarter (Robustness)
	
*************************************************************/

clear all
set more off

global Charts "$Work/Figures"
//use "$Work/Data/public/hf_final_allobservations_public.dta"
import delimited "$Data/public/hf_final_allobservations_public.csv"

cd "$Work/Data/"

version 13.0

preserve

generate panelvar = 0
replace panelvar = horizon if variable == "Core PCE Inflation"
replace panelvar = horizon + 4 if variable == "GDP"
//Convert string date into stata format date
generate temp = date(date, "DMY")
format temp %td
drop date
generate date = temp
format date %td
drop temp

drop obsdate 
xtset panelvar date
sort date variable horizon

graph set window fontface "Times New Roman"
graph set window fontfaceserif "Times New Roman"
graph set eps fontface "Times New Roman"
graph set eps fontfaceserif "Times New Roman"
graph set ps fontface "Times New Roman"
graph set ps fontfaceserif "Times New Roman"


**************
*** 1) Six panel staff forecast (Figure 1)
**************

* Add NBER dummy
gen nber = 0
replace nber=1 if tin(01mar2001, 31oct2001)
replace nber=1 if tin(01dec2007, 31may2009)
gen zero =0


format date %tdCCYY

gen a = 10*nber
gen b = -10*nber
//GDP panels
*Dates for xlabel are January 1 of 2000, 2002, 2005, 2009, 2012
twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (line zero date, lc(black) lwidth(thin)) ///
	   (scatter rgdp_fore date if source !="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter rgdp_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==-1, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Real GDP Growth, 1-qtr Backcasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large))
graph export "../Figures/rgdp_backcasts.pdf", replace fontface(Times)

twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (line zero date, lc(black) lwidth(thin)) ///
	   (scatter rgdp_fore date if source !="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter rgdp_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Real GDP Growth, Current-Quarter Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/rgdp_nowcasts.pdf", replace fontface(Times)

twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (line zero date, lc(black) lwidth(thin)) ///
	   (scatter rgdp_fore date if source !="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter rgdp_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==1, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Real GDP Growth, 1-qtr Ahead Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/rgdp_forecasts.pdf", replace fontface(Times)

//Plot GB only for GDP forecasts (not in Figure 1)

twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (line zero date, lc(black) lwidth(thin)) ///
	   (scatter rgdp_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Real GDP Growth, Current-Quarter Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/rgdp_nowcasts_gbonly.pdf", replace fontface(Times)


drop a b 
gen a = 4*nber

//Inflation panels
twoway (rarea a zero date, msym(i) bcol(gs12)) ///
	   (scatter pcepilfe_fore date if source!="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter pcepilfe_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==-1, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Core PCE Inflation, 1-qtr Backcasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/pcepilfe_backcasts.pdf", replace fontface(Times)

twoway (rarea a zero date, msym(i) bcol(gs12)) ///
	   (scatter pcepilfe_fore date if source!="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter pcepilfe_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Core PCE Inflation, Current-Quarter Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/pcepilfe_nowcasts.pdf", replace fontface(Times)

twoway (rarea a zero date, msym(i) bcol(gs12)) ///
	   (scatter pcepilfe_fore date if source!="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter pcepilfe_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==1, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Core PCE Inflation, 1-qtr Ahead Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/pcepilfe_forecasts.pdf", replace fontface(Times)

//Plot GB only for inflation forecasts (not in Figure 1)
twoway (rarea a zero date, msym(i) bcol(gs12)) ///
	   (scatter pcepilfe_fore date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("%," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Core PCE Inflation, Current-Quarter Forecasts", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) 
graph export "../Figures/pcepilfe_nowcasts_gbonly.pdf", replace fontface(Times)



drop a


gen a = 5*nber
gen b = -10*nber

format date %tdCCYY

**************
*** 2) Forecast errors for current quarter (Figure 2)
**************

*Get percentiles for current quarter gdp errors
qui summarize date if variable == "GDP" & horizon == 0
local gdpmindate = r(min)
local gdpmaxdate = r(max)
local rightmargin = r(max) + 100
local rightmarginmedian = r(max) + 125
			
qui summarize rgdp_err if variable == "GDP" & horizon == 0
local pmin = round(r(min),0.1)
local pmax = round(r(max),0.1)

qui _pctile rgdp_err if horizon == 0, p(15,50,85)
local p15 = round(r(r1),0.1)
local p50 = round(r(r2),0.1)
local p85 = round(r(r3),0.1)

drop a b
gen a = `pmax'*nber
gen b = `pmin'*nber

twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (scatter rgdp_err date if source!="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter rgdp_err date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("p.p.," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Real GDP Growth Current-Quarter Forecast Errors", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(`pmin' `p15' `p50' `p85' `pmax',angle(0) nogextend labsize(large) glwidth(thin) glcolor(gs9))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) ///
	   text(`p85' `rightmargin' "85th Percentile", place(e) color(black) nobox) ///
	   text(`p50' `rightmarginmedian' "Median", place(e) color(black) nobox) ///
	   text(`p15' `rightmargin' "15th Percentile", place(e) color(black) nobox) ///
	   xsize(9) ysize(5.5) graphr(margin(l r+25) ilpattern(blank) lcolor(white)) 
			
graph export "../Figures/rgdp_error_nowcasts.pdf", replace fontface(Times)




*Get summary stats for inflation

qui summarize date if variable == "Core PCE Inflation" & horizon == 0
local mindate = r(min)
local maxdate = r(max)
local rightmargin = r(max) + 100
local rightmarginmedian = r(max) + 125

qui summarize pcepilfe_err if variable == "Core PCE Inflation" & horizon == 0
local pmin = round(r(min),0.1)
local pmax = round(r(max),0.1)

qui _pctile pcepilfe_err if horizon == 0, p(15,50,85)
local p15 = round(r(r1),0.1)
local p50 = round(r(r2),0.1)
local p85 = round(r(r3),0.1)

drop a b
gen a = `pmax'*nber
gen b = `pmin'*nber
			
twoway (rarea a b date, msym(i) bcol(gs12)) ///
	   (scatter pcepilfe_err date if source!="Tealbook", mc(black) msize(vsmall)) ///
	   (scatter pcepilfe_err date if source=="Tealbook", msymbol(triangle) mc(green) msize(vsmall)) if horizon==0, ///
	   leg(off) xti("Year", size(large)) yti("p.p.," "a.r.", orientation(horizontal) justification(left) size(large)) ///
	   ti("Core PCE Inflation Current-Quarter Forecast Errors", color(black)) ///
	   plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	   ylabel(`pmin' `p15' `p50' `p85' `pmax',angle(0) nogextend labsize(large) glwidth(thin) glcolor(gs9))  ///
	   xlabel(14610 15341 16437 17898 18993,labsize(large)) ///
	   text(`p85' `rightmargin' "85th Percentile", place(e) color(black) nobox) ///
	   text(`p50' `rightmarginmedian' "Median", place(e) color(black) nobox) ///
	   text(`p15' `rightmargin' "15th Percentile", place(e) color(black) nobox) ///
	   xsize(9) ysize(5.5) graphr(margin(l r+25) ilpattern(blank) lcolor(white)) 
graph export "../Figures/pcepilfe_error_nowcasts.pdf", replace fontface(Times)

restore 

//Convert string date into stata format date
generate temp = date(date, "DMY")
format temp %td
drop date
generate date = temp
format date %td
drop temp
format date %tdMonth_dd,_CCYY
/*
local gdpmindate 14997
local gdpmaxdate 18984
*/

**************
*** 3) RMSE by type of forecast (inline text comments in appendix)
**************

preserve

drop obsdate 
sort date variable horizon
drop fore_dist fomc year dateqtr prevfomc from_end weeks* zscore* fomcqtr

*Encode string variable for GDP/Inflation as numeric
encode variable, gen(var)
drop variable
rename var variable

*Encode source as "communicated" = 1 vs "not communcated" = 0
generate source_com = 1
replace source_com = 0 if source == "Weekly Ruth" | source == "GDP Killsheet" | source == "Prices Killsheet"


*Stack forecasts for GDP and inflation into single vector
replace rgdp_fore = pcepilfe_fore if rgdp_fore==.              //rgdp_fore will become vector with all forecasts
replace rgdp_err = pcepilfe_err if rgdp_err ==.
drop pcepilfe_fore pcepilfe_err
rename rgdp_fore forecast 
rename rgdp_err error

bysort horizon source_com variable: egen mse = mean(error^2)
gen rmse = sqrt(mse)
bysort horizon source_com variable: egen count = count(error)

*Tabulate current-quarter GDP forecast RMSE by "communicated" (1.71) vs "not communicated" (1.58)
tabulate source_com rmse if variable == 2 & horizon == 0
*Tabulate current-quarter Inflation forecast RMSE by "communicated" (0.49) vs "not communicated" (0.50)
tabulate source_com rmse if variable == 1 & horizon == 0

*RMSE for Greenbook/Tealbook forecasts
keep if source == "Tealbook"
bysort horizon variable: egen mse_gb = mean(error^2) if source == "Tealbook"
generate rmse_gb = sqrt(mse_gb)

*RMSE current-quarter GDP forecasts forom Greenbooks (1.66)
tabulate rmse_gb if variable == 2 & horizon == 0 & source == "Tealbook"
*RMSE current-quarter Inflation forecasts from Greenbooks (0.49)
tabulate rmse_gb if variable == 1 & horizon == 0 & source == "Tealbook"

tabulate rmse_gb if variable == 2 & horizon == 1 & source == "Tealbook"

restore

*******
* 4) Plots of RMSE By Week to Start of Quarter (Figure 3)
*******

preserve

drop obsdate 
sort date variable horizon
drop fore_dist fomc year dateqtr prevfomc from_end weeks* zscore* fomcqtr

*Encode string variable for GDP/Inflation as numeric
encode variable, gen(var)
drop variable
rename var variable

*Stack forecasts for GDP and inflation into single vector
replace rgdp_fore = pcepilfe_fore if rgdp_fore==.              //rgdp_fore will become vector with all forecasts
replace rgdp_err = pcepilfe_err if rgdp_err ==.
drop pcepilfe_fore pcepilfe_err
rename rgdp_fore forecast 
rename rgdp_err error


* RMSE by week
sort variable date horizon


//next block of code is to convert string projqtr into Stata format date
generate temp = quarterly(projqtr, "YQ")
drop projqtr
generate projqtr = temp
drop temp

sort variable date projqtr
* Find first date of quarter being forecasted
gen firstday = dofq(projqtr)
format firstday %td

gen week = floor((date- firstday)/7) //week zero is the first week of the quarter

bysort week variable: egen mse = mean(error^2)
gen rmse = sqrt(mse)
bysort week variable: egen count = count(error)

order week*, first
sort variable week

collapse count rmse, by(week variable)  //keep one observation of rmse and count for each week and variable (i.e. drop duplicates)
sort variable week


gen a = 0
gen b = 2.5 if week>=0 & week <=12  //b and c are height of bars that will denote current quarter
gen c = 1 if week>=0 & week <=12    

twoway (rarea c a week, bc(gs12)) (scatter rmse week, msize(small) mcol(black)) ///
    if variable==1 & week >=-25, ///
	xti("Week from Start of Quarter", size(large)) yti("RMSE" "p.p., a.r.", orientation(horizontal) justification(left) size(large)) ///
	ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	xlabel(,labsize(large)) ///
	plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	subti("Core PCE Inflation", size(large)) ysc(r(0 1)) ylab(0(.2)1) xmtick(##4) ///
	text(0.95 17 "BEA Release", place(e) color(black) nobox) ///
	xline(16, lc(black) lp(dash)) leg(off)
graph export "$Charts/hf_pcepi.pdf", replace fontface(Times)

twoway (rarea b a week, bc(gs12)) (scatter rmse week, msize(small) mcol(black)) ///
	if variable==2 & week>=-25, ///
	xti("Week from Start of Quarter", size(large)) yti("RMSE" "p.p., a.r.", orientation(horizontal) justification(left) size(large)) ///
	ylabel(,angle(0) nogextend nogrid labsize(large)) ///
	xlabel(,labsize(large)) ///
	plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	subti("Real GDP Growth", size(large)) xmtick(##4)  ///
	text(2.45 17 "BEA Release", place(e) color(black) nobox) ///
	xline(16, lc(black) lp(dash)) leg(off) 
graph export "$Charts/hf_rgdp.pdf", replace fontface(Times)


restore

*******
* 5) Plot of Macroeconomic News Index
*******

preserve
generate panelvar = 0
replace panelvar = horizon if variable == "Core PCE Inflation"
replace panelvar = horizon + 4 if variable == "GDP"

drop obsdate 
xtset panelvar date

* Add NBER dummy
gen nber = 0
replace nber=1 if tin(01mar2001, 31oct2001)
replace nber=1 if tin(01dec2007, 31may2009)
gen zero =0

format date %tdCCYY

bysort date: generate duplicate = cond(_N==1,0,_n)   //==0 for the first observation for that date, otherwise ==_n for that date
drop if duplicate >1
tsset date

gen a = nber*40
gen b = -nber*40

//Date is formatted in Stata Format, days since Jan 1, 1960
//To check, e.g. use display date("1-01-2001","MDY")
//Dates are January 1st of 2001, 2005, 2009, 2011

twoway (rarea a b date, msym(i) bcol(gs12)) ///
	(line zscorexreturn date, lc(blue)) ///
	(line zero date, lc(black) lwidth(thin)), ///
	xti("Year", justification(left) size(medium)) yti("Index", orientation(horizontal) justification(left) size(medium)) ///
	ylabel(,angle(0) nogextend nogrid labsize(medium)) ///
	xlabel(14976 16437 17898 18628,labsize(medium)) ///
	plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	leg(off) 
graph export "$Charts/news.pdf", replace fontface(Times)
restore

*******
* 6) Plots of RMSE By Day to Start of Quarter (Robustness)
*******

preserve

drop obsdate 
sort date variable horizon
drop fore_dist fomc year dateqtr prevfomc from_end weeks* zscore* fomcqtr

*Encode string variable for GDP/Inflation as numeric
encode variable, gen(var)
drop variable
rename var variable

*Stack forecasts for GDP and inflation into single vector
replace rgdp_fore = pcepilfe_fore if rgdp_fore==.              //rgdp_fore will become vector with all forecasts
replace rgdp_err = pcepilfe_err if rgdp_err ==.
drop pcepilfe_fore pcepilfe_err
rename rgdp_fore forecast 
rename rgdp_err error


* RMSE by week
sort variable date horizon


//next block of code is to convert string projqtr into Stata format date
generate temp = quarterly(projqtr, "YQ")
drop projqtr
generate projqtr = temp
drop temp

sort variable date projqtr
* Find first date of quarter being forecasted
gen firstday = dofq(projqtr)
format firstday %td

gen day = floor((date- firstday)) //Day zero is the first day of the quarter

bysort day variable: egen mse = mean(error^2)
gen rmse = sqrt(mse)
bysort day variable: egen count = count(error)

order day, first
sort variable day

collapse count rmse, by(day variable)  //keep one observation of rmse and count for each week and variable (i.e. drop duplicates)
sort variable day


gen a = 0
gen b = 4.0 if day>=0 & day <=90  //b and c are height of bars that will denote current quarter
gen c = 1.4 if day>=0 & day <=90    

twoway (rarea c a day, bc(gs12)) (scatter rmse day, msize(small) mcol(black)) ///
    if variable==1 & day >=-90, ///
	xti("Days from Start of Quarter", size(large)) yti("RMSE" "p.p., a.r.", orientation(horizontal) justification(left) size(large)) ///
	ylabel(,angle(0) nogextend nogrid labsize(large))  ///
	xlabel(,labsize(large)) ///
	plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	subti("Core PCE Inflation", size(large)) ysc(r(0 1)) ylab(0(.2)1.4) xmtick(##4) ///
	text(1.35 120 "BEA Release", place(e) color(black) nobox) ///
	xline(118, lc(black) lp(dash)) leg(off)
graph export "$Charts/hf_pcepi_daily.pdf", replace fontface(Times)

twoway (rarea b a day, bc(gs12)) (scatter rmse day, msize(small) mcol(black)) ///
	if variable==2 & day>=-90, ///
	xti("Days from Start of Quarter", size(large)) yti("RMSE" "p.p., a.r.", orientation(horizontal) justification(left) size(large)) ///
	ylabel(,angle(0) nogextend nogrid labsize(large)) ///
	xlabel(,labsize(large)) ///
	plotregion(fcolor(white)) graphregion(fcolor(white)) ///
	subti("Real GDP Growth", size(large)) xmtick(##4)  ///
	text(3.95 120 "BEA Release", place(e) color(black) nobox) ///
	xline(118, lc(black) lp(dash)) leg(off) 
graph export "$Charts/hf_rgdp_daily.pdf", replace fontface(Times)


restore

clear all

exit

