  /*************************************************************
	Asymmetric Loss Function Estimation (Robustness)
	"Raiders of the Lost High Frequency Forecasts: New Data and Evidence on the Efficiency of the Fed's Forecasting"
	
	Andrew C. Chang, Board of Governors of the Federal Reserve System, a.christopher.chang@gmail.com
	ORCID: 0000-0002-9769-789X
    April 6, 2022

	0) Example program for using MLE on a logit log likelihood
	0a) Example program for using quadratic loss in a regression
	1) Quadquad loss in a regression program
	2) FORCE MINIMUM 1 BP REVISION BETWEEN OBSERVATIONS TO WEED OUT THIN TRADING
	3) PRE-ANALYSIS PLAN: EQUATION 1, BASELINEMZ
	4) PRE-ANALYSIS PLAN: EQUATION 2, PRIMARY EQUATION
	5) PRE-ANALYSIS PLAN: EQUATION 3, Alternate Equation, w/ z-Score
	6) POST TABLES TO LATEX (SEPARATE TABLE FOR EACH VARIABLE AND EQUATION)
	
*************************************************************/

  
/*
************************************************************
* 0) Example program for using MLE on a logit log likelihood
************************************************************
program define mylogit
	  args lnf Xb
	  quietly replace `lnf' = -ln(1+exp(-`Xb')) if $ML_y1==1
	  quietly replace `lnf' = -`Xb' - ln(1+exp(-`Xb')) if $ML_y1==0
end


sysuse auto

ml model lf mylogit (foreign=mpg weight)
ml maximize
*/

/*
************************************************************
* 0a) Example program for using quadratic loss in a regression
************************************************************
program define quadraticloss
	  args lnf Xb
	  quietly replace `lnf' =  -($ML_y1 - `Xb')^2 //-ln(1+exp(-`Xb')) if $ML_y1==1
	  //quietly replace `lnf' = -`Xb' - ln(1+exp(-`Xb')) if $ML_y1==0
end

sysuse auto
ml model lf quadraticloss (foreign=mpg weight) [pweight=turn], vce(robust)
ml maximize

regress foreign mpg weight [pweight=turn], vce(robust)
*/

clear all

set more off
capture log close

global Data "$Work\Data"

log using ${Work}/results/logs/final/regs_final_public_asymmetric.log, replace

version 13.0


************************************************************
* 1) Quadquad loss in a regression program
************************************************************
	
program define quadquadloss
	  args lnf Xb
	  quietly replace `lnf' =  -$alpha*($ML_y1 - `Xb')^2 if $ML_y1 > 0  //if statement is for when the forecast error (LHS variable) is positive (underprediction, forecast less than actual)
	  quietly replace `lnf' =  -(1-$alpha)*($ML_y1 - `Xb')^2 if $ML_y1 <= 0 
end
  
	  
**********************************************************************
** COMPUTE HIGH-FREQUENCY REGRESSIONS, STORE OUTPUT TO LATEX TABLES **
**********************************************************************

//use "$Data/public/hf_final_allobservations_public.dta", clear
import delimited "$Data/public/hf_final_allobservations_public.csv", clear

*******************************************************************************
** 2) FORCE MINIMUM 1 BP REVISION BETWEEN OBSERVATIONS TO WEED OUT THIN TRADING **
*******************************************************************************

* Drop all observations of < 1 bp revision from previous forecast
foreach var in rgdp pcepilfe {

   preserve
         
   if "`var'"=="rgdp" {
      keep if variable=="GDP"
   }
   else {
      keep if variable=="Core PCE Inflation"
   }
 
   * Only takes two iterations to complete; run through obs twice
   forvalues i = 1/2 {
      *Due to machine precision issues, in the second clause it is < 0.00999 not < 0.01
      drop if !missing(`var'_rev) & abs(`var'_rev)<0.00999 & fore_dist!=0
      replace `var'_rev = `var'_fore-`var'_fore[_n+1] if fore_dist>0
      
   }
      
   tempfile new`var'
   save `new`var''
      
   restore

}

use `newrgdp', clear
append using `newpcepilfe'

generate panelvar = 0
replace panelvar = horizon if variable == "Core PCE Inflation"
replace panelvar = horizon + 4 if variable == "GDP"
//Convert string date into stata format date
generate temp = date(date, "DMY")
format temp %td
drop date
generate date = temp
format date %td
drop temp

xtset panelvar date

* Shape parameter of quadquad loss function
local alpha = 0.2  //using alpha = 0.35, results are qualitatively similar, but bias terms are off
					//alpha = 0.65 is more similar to alpha = 0.5

while `alpha' <= 0.9 {

	global alpha = `alpha'  //For input into quad quad loss funciton
	local alphafile = round(`alpha'*100)  //For file naming without a decimal
	//display `alphafile'

	foreach var in rgdp pcepilfe {
			
		 disp "var: `var'"
		 disp "horizon: `i'"
		 local upp = upper("`var'")

		 * Run loop through horizons -1 to 2
		 
		 forvalues i==-1/2 {

			disp "Horizon: `i'"
		 
			local j = abs(`i')

		*********************************    
		* 3) PRE-ANALYSIS PLAN: EQUATION 1 *    
		* Baseline MZ                   *
		*********************************
		
		ml model lf quadquadloss  (`var'_err=`var'_rev) [pweight=fore_dist] if horizon==`i' & fore_dist>0 
	    ml maximize
		*reg `var'_err `var'_rev [pweight=fore_dist] if horizon==`i' & fore_dist>0   
		*estat durbinalt, force robust lags(1/12)
		
		if `i'<0 {
			eststo `var'neg1_wgt_mz
			}
			else {
				eststo `var'`i'_wgt_mz
			}

		*********************************    
		* 4) PRE-ANALYSIS PLAN: EQUATION 2 *    
		* Primary Equation              *
		*********************************
		
		xi: ml model lf quadquadloss  (`var'_err=i.weeks01*`var'_rev) [pweight=fore_dist] if horizon==`i' & fore_dist>0 
	    ml maximize
		
		*xi: reg `var'_err i.weeks01*`var'_rev [pweight=fore_dist] if horizon==`i' & fore_dist>0
		*estat durbinalt, force robust lags(1/12)
		
		if "`var'"=="rgdp" {
		 test _cons + _Iweeks01_1 = 0
		 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
		 test rgdp_rev + _IweeXrgdp__1 = 0
		 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
		else {
		 test _cons + _Iweeks01_1 = 0
		 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
		 test pcepilfe_rev + _IweeXpcepi_1 = 0
		 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		}
		
			
		if `i'<0 {
			eststo `var'neg1_wgt_mz_w
			}
			else {
				eststo `var'`i'_wgt_mz_w
			}
		
		**********************************    
		* 5) PRE-ANALYSIS PLAN: EQUATION 3  *    
		* Alternate Equation, w/ z-Score *
		**********************************
			
		xi: ml model lf quadquadloss  (`var'_err=i.weeks01*`var'_rev i.weeks01|zscorexreturn) [pweight=fore_dist] if horizon==`i' & fore_dist>0 
	    ml maximize
		
		if "`var'"=="rgdp" {
		 test _cons + _Iweeks01_1 = 0
		 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
		 test rgdp_rev + _IweeXrgdp__1 = 0
		 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		 test zscorexreturn + _IweeXzscor_1 = 0
		 eret2 scalar pVal_zScorePlusWeeks01 = r(p)	 
		}
		else {
		 test _cons + _Iweeks01_1 = 0
		 eret2 scalar pVal_ConsPlusWeeks01 = r(p)
		 test pcepilfe_rev + _IweeXpcepi_1 = 0
		 eret2 scalar pVal_RevPlusWeeks01 = r(p)
		 test zscorexreturn + _IweeXzscor_1 = 0
		 eret2 scalar pVal_zScorePlusWeeks01 = r(p)	 
		}	 
		 
		if `i'<0 {
			eststo `var'neg1_wgt_mz_wr
			}
			else {
				eststo `var'`i'_wgt_mz_wr
			}
			
		}
		
		************************************************************************
		* 6) POST TABLES TO LATEX (SEPARATE TABLE FOR EACH VARIABLE AND EQUATION) *
		************************************************************************
		
		if "`var'"=="rgdp" {
		
		* BASELINE MZ OUTPUT *
		
		esttab `var'neg1_wgt_mz `var'0_wgt_mz `var'1_wgt_mz `var'2_wgt_mz ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace ///
		title(""\label{TableRGDPMZasymmetricloss}) /// 
		mlabels(, none) ///
		alignment(c) ///
		nonumbers ///
		cons ///
		collabels(none) ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_cons "Constant") /// 
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow")  ///
		posthead("") ///
		fragment ///
		eqlabels (none) ///

		* MZ WITH WEEKS01 (notation I(Tau)) OUTPUT*
		esttab `var'neg1_wgt_mz_w `var'0_wgt_mz_w `var'1_wgt_mz_w `var'2_wgt_mz_w ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_w_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H_{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H_{0}:\$\ Revision + \$I(\tau) \times\$Revision = 0") sfmt(2) ///
		title(""\label{TableRGDPMZWeeks01asymmetricloss}) ///
		mlabels("-1" "0" "1" "2", lhs("Forecast Horizon")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXrgdp__1 "\$I(\tau) \times\$Revision" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		fragment ///
		substitute(\_ _) ///
		eqlabels (none) ///

		
		* MZ WITH WEEKS01 & ZSCOREXRETURN OUTPUT*
		
		esttab `var'neg1_wgt_mz_wr `var'0_wgt_mz_wr `var'1_wgt_mz_wr `var'2_wgt_mz_wr ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_wr_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace /// 
		title(""\label{TableRGDPMZWeeks01asymmetricloss}) ///
		mlabels("-1" "0" "1" "2", lhs("Forecast Horizon")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXrgdp__1 "\$I(\tau) \times\$Revision" ///
		zscorexreturn "\$news@\tau\$" ///
		_IweeXzscor_1 "\$I(\tau) \times news@\tau\$" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H@{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H@{0}:\$\ Revision + \$I(\tau) \times\$Revision = 0" "pVal_zScorePlusWeeks01 \$H@{0}:\$\ \$news@\tau\$  + \$I(\tau) \times news@\tau\$ = 0") sfmt(2) ///
		substitute(@ _) ///
		fragment ///
		eqlabels (none) ///

		}
		
		else {

		* BASELINE MZ OUTPUT * 
		esttab `var'neg1_wgt_mz `var'0_wgt_mz `var'1_wgt_mz ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace ///
		title(""\label{TablePcepilfeMZasymmetricloss}) /// 
		mlabels(,none) ///
		alignment(c) ///
		nonumbers ///
		cons ///
		collabels(none) ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_cons "Constant") /// 
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow")  ///
		posthead("") ///
		fragment ///
		eqlabels (none) ///
		
		* For Core PCE Inflation, We Are Only Looking at Horizons from t=-1 to t=+1

		
		* MZ WITH WEEKS01 (notation I(Tau)) OUTPUT*
		esttab `var'neg1_wgt_mz_w `var'0_wgt_mz_w `var'1_wgt_mz_w ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_w_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H_{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H_{0}:\$\ Revision + \$I(\tau) \times\$Revision = 0") sfmt(2) ///
		title(""\label{TablePcepilfeMZWeeks01asymmetricloss}) ///
		mlabels("-1" "0" "1" "2", lhs("Forecast Horizon")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXpcepi_1 "\$I(\tau) \times\$Revision" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		fragment ///
		substitute(\_ _) ///
		eqlabels (none) ///
		

		* MZ WITH WEEKS01 & ZSCOREXRETURN OUTPUT *
		
		esttab `var'neg1_wgt_mz_wr `var'0_wgt_mz_wr `var'1_wgt_mz_wr ///
		using ${Work}/results/latex/nonpap_regs/asymmetricloss/`var'_mz_wr_public_asymmetricloss_`alphafile'.tex, ///
		se ar2(2) replace /// 
		title(""\label{TablePcepilfeMZWeeks01Zscoreasymmetricloss}) ///
		mlabels("-1" "0" "1", lhs("Forecast Horizon")) ///
		cons ///
		collabels(none) ///
		nonumbers ///
		coeflabels( ///
		`var'_rev "Revision" ///
		_Iweeks01_1 "\$I(\tau)\$" ///
		_IweeXpcepi_1 "\$I(\tau) \times\$Revision" ///
		zscorexreturn "\$news@\tau\$" ///
		_IweeXzscor_1 "\$I(\tau) \times news@\tau\$" ///
		_cons "Constant") ///
		nostar ///
		cells("b(fmt(2))" "se(fmt(2) par)" "placeholderrow") ///
		scalars("placeholderrow p-values: ""pVal_ConsPlusWeeks01 \$H@{0}:\$\ Constant + \$I(\tau)\$ = 0" "pVal_RevPlusWeeks01 \$H@{0}:\$\ Revision + \$I(\tau) \times\$Revision = 0" "pVal_zScorePlusWeeks01 \$H@{0}:\$\ \$news@\tau\$  + \$I(\tau) \times news@\tau\$ = 0") sfmt(2) ///
		substitute(@ _) ///
		fragment ///
		eqlabels (none) ///
		
		}	
	
	}
	
	local alpha = `alpha' + 0.15
	
}
	
	
