/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Program for reading in forecasts to compute accuracy measures reported in the paper's Tables 3-6 and corresponding
appendix tables, along with Figure 1.

Note re measures:  The program includes expected shortfall (ES) for the purpose of computing the VaR-ES score
for the average forecast. The VaR-ES score is denoted below with "FZG".  Results are produced for both
the paper's 10% tail and the 5% tail covered in the appendix.

Note re dating and samples:  Forecasts are dated by the forecast outcome date (current quarter, given nowcast setup)
*/

comp usesmpl = 0    ;* 0 for small dataset and 1985-2019 smpl, 1 for medium dataset and 2000-2019 smpl, 2 for large dataset and 2007-2019 smpl
comp dropcovid = 1  ;* 0 to eval forecasts through last avail obs in 2020, 1 to stop eval in 2019:Q4

env nowshowgraphs
if dropcovid==1
{
if usesmpl==2
 gsave(format=pdf) "comparisons_0719_*.pdf"
else if usesmpl==1
 gsave(format=pdf) "comparisons_0019_*.pdf"
else
 gsave(format=pdf) "comparisons_8519_*.pdf"
}
else if dropcovid==0
{
if usesmpl==2
 gsave(format=pdf) "comparisons_0720_*.pdf"
else if usesmpl==1
 gsave(format=pdf) "comparisons_0020_*.pdf"
else
 gsave(format=pdf) "comparisons_8520_*.pdf"
}
endif

comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp stvint = 1985:1       ;* stvint is first vintage, which corresponds to the start of forecasting
comp endvint = 2021:1      ;* last quarter of data vintages considered
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp endsmpl = endvint-useactual

comp stpt = 1985:1   ;* starting point of forecasting

all endvint
smpl stsmpl endvint

*** read in procedures for computing HAC-robust t-statistics of loss differentials and Christoffersen's (1998) test of conditional coverage
sou(noecho) ../procedures/ttestQS.src
sou(noecho) ../procedures/Christoffersentest.src
comp use2sided = 0                    ;* 0 for 1-sided tests, 1 for 2-sided tests

******************** actual GDP used as actuals
** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close
set gdp = actualGDP

**** NBER dates, quarterly
clear(length=endvint,zeroes) cycle
set cycle 1948:4 1949:4 = 1.0
set cycle 1953:2 1954:2 = 1.0
set cycle 1957:3 1958:2 = 1.0
set cycle 1960:2 1961:1 = 1.0
set cycle 1969:4 1970:4 = 1.0
set cycle 1973:4 1975:1 = 1.0
set cycle 1980:1 1980:3 = 1.0
set cycle 1981:3 1982:4 = 1.0
set cycle 1990:3 1991:1 = 1.0
set cycle 2001:1 2001:4 = 1.0
set cycle 2007:4 2009:2 = 1.0
set cycle 2019:4 2020:2 = 1.0

******************** dimensioning for reading in results
if usesmpl==2
{
comp [vec[int]] svindicator = ||1,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0,1,0,0||

comp [vec[str]] methlabel = ||"AR-SV",$
"base M:  BMF-SV",$
"base M:  BQR",$
"base M:  PQR",$
"base M-F:  BMF-SV",$
"base M-F:  BQR",$
"base M-F:  PQR",$
"base M + small weekly:  BMF-SV",$
"base M + small weekly:  BQR",$
"base M + small weekly:  PQR",$
"base M-F + small weekly:  BMF-SV",$
"base M-F + small weekly:  BQR",$
"base M-F + small weekly:  PQR",$
"base M + large weekly:  BMF-SV",$
"base M + large weekly:  BQR",$
"base M + large weekly:  PQR",$
"base M-F + large weekly:  BMF-SV",$
"base M-F + large weekly:  BQR",$
"base M-F + large weekly:  PQR"||

comp [vec[str]] filestouse = ||"out.forecastruns/AR_SV",$
"out.forecastruns/basemacro_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_empiricalpctiles",$
"out.forecastruns/basemacro_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_largewkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_largewkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_largewkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_largewkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_largewkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_largewkly_empiricalpctiles"||
}
else if usesmpl==1
{
comp [vec[int]] svindicator = ||1,1,0,0,1,0,0,1,0,0,1,0,0||

comp [vec[str]] methlabel = ||"AR-SV",$
"base M:  BMF-SV",$
"base M:  BQR",$
"base M:  PQR",$
"base M-F:  BMF-SV",$
"base M-F:  BQR",$
"base M-F:  PQR",$
"base M + small weekly:  BMF-SV",$
"base M + small weekly:  BQR",$
"base M + small weekly:  PQR",$
"base M-F + small weekly:  BMF-SV",$
"base M-F + small weekly:  BQR",$
"base M-F + small weekly:  PQR"||

comp [vec[str]] filestouse = ||"out.forecastruns/AR_SV",$
"out.forecastruns/basemacro_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_empiricalpctiles",$
"out.forecastruns/basemacro_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_smallwkly_empiricalpctiles"||
}
else
{
comp [vec[int]] svindicator = ||1,1,0,0,1,0,0||

comp [vec[str]] methlabel = ||"AR-SV",$
"base M:  BMF-SV",$
"base M:  BQR",$
"base M:  PQR",$
"base M-F:  BMF-SV",$
"base M-F:  BQR",$
"base M-F:  PQR"||

comp [vec[str]] filestouse = ||"out.forecastruns/AR_SV",$
"out.forecastruns/basemacro_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_M_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_M_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_empiricalpctiles"||
}
endif
comp nmeth = %rows(filestouse)
if %rows(filestouse)<>nmeth
 dis "mismatch in files listed versus count of labels specified"
endif

******************** setting up average forecasts
******************** note:  the setup allows for multiple average forecasts, although we use only one (average of base M-F forecasts)
******************** 
open data comboweights.xlsx
data(org=col,left=3,for=xlsx) 1 nmeth basemf
close data

make avgwts 1 nmeth
# basemf
comp [vec[str]] avgmethlabel = ||"avg.\ base M-F"||

comp navg = %cols(avgwts)       ;* number of average forecasts considered
comp totmeth = nmeth + navg

comp [vec] sumw = %sumc(avgwts)  ;* should be %sumr, but it us %sumc applied here that works right
ewise avgwts(i,j) = avgwts(i,j)/sumw(j)
dis ###.#### avgwts

comp [vec[str]] methlabel = methlabel+avgmethlabel

******************** setting up storage
comp totweek = 15

*** what we use to read in, method by method
dec vec[ser] fcseries(totweek) pct05series(totweek) pct10series(totweek) pct90series(totweek) pct95series(totweek) qs05series(totweek) qs10series(totweek) $ 
 fzg05series(totweek) fzg10series(totweek) es05series(totweek) es10series(totweek) crpsseries(totweek) qwcrpsseries(totweek)
do mm = 1,totweek 
 labels fcseries(mm)
 # 'fcseries_m'+%string(mm)
 labels pct05series(mm)
 # 'pct05_m'+%string(mm)
 labels pct10series(mm)
 # 'pct10_m'+%string(mm)
 labels pct90series(mm)
 # 'pct90_m'+%string(mm)
 labels pct95series(mm)
 # 'pct95_m'+%string(mm)
 labels qs05series(mm)
 # 'qs05_m'+%string(mm)
 labels qs10series(mm)
 # 'qs10_m'+%string(mm)
 labels fzg05series(mm)
 # 'fzg05_m'+%string(mm)
 labels fzg10series(mm)
 # 'fzg10_m'+%string(mm)
 labels es05series(mm)
 # 'es05_m'+%string(mm)
 labels es10series(mm)
 # 'es10_m'+%string(mm)
 labels crpsseries(mm)
 # 'crps_m'+%string(mm)
 labels qwcrpsseries(mm)
 # 'qwcrps_m'+%string(mm)
end do mm
clear fcseries pct05series pct10series pct90series pct95series qs05series qs10series fzg05series fzg10series es05series es10series crpsseries qwcrpsseries pct05

*** what we use to store, covering all methods
*** note that, in the hit05 and hit10 series, we define as empirical rate less nominal, rather than just the empirical rate 
dec rec[ser] sqerror(totmeth,totweek) hit05(totmeth,totweek) hit10(totmeth,totweek) pointfc(totmeth,totweek) q05fc(totmeth,totweek) q10fc(totmeth,totweek) q90fc(totmeth,totweek) q95fc(totmeth,totweek) $
 qs05(totmeth,totweek) qs10(totmeth,totweek) fzg05(totmeth,totweek) fzg10(totmeth,totweek) es05(totmeth,totweek) es10(totmeth,totweek) crps(totmeth,totweek) qwcrps(totmeth,totweek)

*** stuff for FZG score computation of average forecast, using quantiles of source base MF forecasts
comp fzgacoef = 4.0   ;* coef used in G_2 function of FZG score
comp fzgbcoef = 6.0   ;* intercept added to FZG score

******************** reading in forecasts from BMF, BQR, and PQR methods for each variable set
do nn = 1,%rows(filestouse)
 comp [str] filename = filestouse(nn)+".xls"
 dis filename
 open data &filename
 data(org=col,for=xls) / fcseries pct05series pct10series pct90series pct95series qs05series qs10series fzg05series fzg10series es05series es10series crpsseries qwcrpsseries
 close data

 do mm = 1,totweek
  set pointfc(nn,mm) = fcseries(mm){0}
  set q05fc(nn,mm) = pct05series(mm){0} 
  set q10fc(nn,mm) = pct10series(mm){0} 
  set q90fc(nn,mm) = pct90series(mm){0} 
  set q95fc(nn,mm) = pct95series(mm){0} 
  set qs05(nn,mm) = qs05series(mm){0} 
  set qs10(nn,mm) = qs10series(mm){0}
  set fzg05(nn,mm) = fzg05series(mm){0} 
  set fzg10(nn,mm) = fzg10series(mm){0}
  set es05(nn,mm) = es05series(mm){0} 
  set es10(nn,mm) = es10series(mm){0}
  set crps(nn,mm) = crpsseries(mm){0}
  set qwcrps(nn,mm) = qwcrpsseries(mm){0}
  *
  set sqerror(nn,mm) = (gdp - fcseries(mm){0})^2.
  set hit05(nn,mm) = %if(gdp{0}<=pct05series(mm){0},1.0,0.0) - 0.05
  set hit10(nn,mm) = %if(gdp{0}<=pct10series(mm){0},1.0,0.0) - 0.10
 end do mm

end do nn

** figure out common start point of sample
do i = 1,nmeth
 inquire(series=q05fc(i,1)) tempstpt tempendpt
 comp stpt = %imax(stpt,tempstpt)
 comp endsmpl = %imin(endsmpl,tempendpt)
end do i
dis %datelabel(stpt) %datelabel(endsmpl)

******************** forming average forecasts:  point and quantile
******************** note:  For point and quantile forecasts and ES, we take a simple average of the forecasts.  The average 
******************** quantile and ES forecasts are then used to compute the FZG score.
******************** for the CRPS and qwCRPS results, we compute them from the linear pooled combination density from BMF, BQR, and PQR
********************   (these are computed separately by the averageforecast.prg and read in here)
** first compute combos of point and quantile fcs
do ii = 1,navg
 do mm = 1,totweek
  clear(zeros) pointfc(nmeth+ii,mm) q05fc(nmeth+ii,mm) q10fc(nmeth+ii,mm) q90fc(nmeth+ii,mm) q95fc(nmeth+ii,mm) es05(nmeth+ii,mm) es10(nmeth+ii,mm) 
  do nn = 1,nmeth
   set pointfc(nmeth+ii,mm) = pointfc(nmeth+ii,mm){0} + avgwts(nn,ii)*pointfc(nn,mm){0}
   set q05fc(nmeth+ii,mm) = q05fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q05fc(nn,mm){0} 
   set q10fc(nmeth+ii,mm) = q10fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q10fc(nn,mm){0} 
   set q90fc(nmeth+ii,mm) = q90fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q90fc(nn,mm){0} 
   set q95fc(nmeth+ii,mm) = q95fc(nmeth+ii,mm){0} + avgwts(nn,ii)*q95fc(nn,mm){0} 
   set es05(nmeth+ii,mm) = es05(nmeth+ii,mm){0} + avgwts(nn,ii)*es05(nn,mm){0} 
   set es10(nmeth+ii,mm) = es10(nmeth+ii,mm){0} + avgwts(nn,ii)*es10(nn,mm){0} 
  end do nn
 end do mm
end do ii

** now compute squared errors, hit rates, quantile scores, and FZG scores for combo fcs
do ii = 1,navg
 do mm = 1,totweek
  set sqerror(nmeth+ii,mm) = (gdp - pointfc(nmeth+ii,mm){0})^2.
  set hit05(nmeth+ii,mm) = %if(gdp{0}<=q05fc(nmeth+ii,mm){0},1.0,0.0) - 0.05
  set hit10(nmeth+ii,mm) = %if(gdp{0}<=q10fc(nmeth+ii,mm){0},1.0,0.0) - 0.10
  *
  comp useq = 0.05
  set qs05(nmeth+ii,mm) = (GDP{0}-q05fc(nmeth+ii,mm){0})*(useq-%if(GDP{0}<=q05fc(nmeth+ii,mm){0},1.,0.))
  set adjhitrate = hit05(nmeth+ii,mm){0}+useq
  set fzg05(nmeth+ii,mm) = fzgbcoef + (adjhitrate{0}-useq)*(q05fc(nmeth+ii,mm){0}-gdp{0}) + $
    (1./useq)*exp(es05(nmeth+ii,mm)/fzgacoef)*(adjhitrate{0}*(q05fc(nmeth+ii,mm){0}-gdp{0})+useq*(es05(nmeth+ii,mm)-q05fc(nmeth+ii,mm){0}-fzgacoef))
  *
  comp useq = 0.10
  set qs10(nmeth+ii,mm) = (GDP{0}-q10fc(nmeth+ii,mm){0})*(useq-%if(GDP{0}<=q10fc(nmeth+ii,mm){0},1.,0.))
  set adjhitrate = hit10(nmeth+ii,mm){0}+useq
  set fzg10(nmeth+ii,mm) = fzgbcoef + (adjhitrate{0}-useq)*(q10fc(nmeth+ii,mm){0}-gdp{0}) + $
    (1./useq)*exp(es10(nmeth+ii,mm)/fzgacoef)*(adjhitrate{0}*(q10fc(nmeth+ii,mm){0}-gdp{0})+useq*(es10(nmeth+ii,mm)-q10fc(nmeth+ii,mm){0}-fzgacoef))
 end do mm
end do ii

** now read in CRPS for average forecast:  these results come from linear pooling of densities for BMF, BQR, PQR
comp nn = nmeth+1
comp [str] filename = "out.fcprocessingfromdraws/averageforecast_empiricaldensity_SV_baseMF.xlsx"
open data &filename
data(org=col,for=xlsx) / crpsseries qwcrpsseries
close data
do mm = 1,totweek
 set crps(nn,mm) = crpsseries(mm){0}
 set qwcrps(nn,mm) = qwcrpsseries(mm){0}
end do mm

******************** computing statistics
if dropcovid==1
 comp endsmpl = %imin(2019:4,endsmpl)  ;* shortening sample to drop pandemic period
endif
comp [rec[int]] evalranges = ||stpt,endsmpl|| 
comp nsamples = %rows(evalranges)
comp basepos = 1  ;* which method will be the baseline, which is AR-SV here
dis methlabel(basepos)

dec vec[rec] rmseres(nsamples) cover05(nsamples) cover10(nsamples) qscore05res(nsamples) qscore10res(nsamples) $
 msetstat(nsamples) cover05tstat(nsamples) qscore05tstat(nsamples) cover10tstat(nsamples) qscore10tstat(nsamples)  $
 msepvalue(nsamples) cover05pvalue(nsamples) qscore05pvalue(nsamples) cover10pvalue(nsamples) qscore10pvalue(nsamples)  $
 msepindic(nsamples) cover05pindic(nsamples) qscore05pindic(nsamples) cover10pindic(nsamples) qscore10pindic(nsamples) $
 rawrmseres(nsamples) rawqscore05res(nsamples) rawqscore10res(nsamples) $
 mseboth(nsamples) cover05both(nsamples) cover10both(nsamples) qscore05both(nsamples) qscore10both(nsamples) $
 fzg05res(nsamples) fzg10res(nsamples) fzg05tstat(nsamples) fzg10tstat(nsamples) fzg05pvalue(nsamples) fzg10pvalue(nsamples) $
 fzg05pindic(nsamples) fzg10pindic(nsamples) fzg05both(nsamples) fzg10both(nsamples) rawfzg05res(nsamples) rawfzg10res(nsamples) $
 condcov1(nsamples) condcov2(nsamples) condcov3(nsamples) christ10pindic(nsamples) christ10both(nsamples) $
 crpsres(nsamples) crpststat(nsamples) crpspvalue(nsamples) crpspindic(nsamples) rawcrpsres(nsamples) crpsboth(nsamples) $
 qwcrpsres(nsamples) qwcrpststat(nsamples) qwcrpspvalue(nsamples) qwcrpspindic(nsamples) rawqwcrpsres(nsamples) qwcrpsboth(nsamples) 
 
do i = 1,nsamples
 dim  rmseres(i)(totmeth,totweek) cover05(i)(totmeth,totweek) cover10(i)(totmeth,totweek) qscore05res(i)(totmeth,totweek) qscore10res(i)(totmeth,totweek) $
 msetstat(i)(totmeth,totweek) cover05tstat(i)(totmeth,totweek) qscore05tstat(i)(totmeth,totweek) cover10tstat(i)(totmeth,totweek) qscore10tstat(i)(totmeth,totweek)  $
 msepvalue(i)(totmeth,totweek) cover05pvalue(i)(totmeth,totweek) qscore05pvalue(i)(totmeth,totweek) cover10pvalue(i)(totmeth,totweek) qscore10pvalue(i)(totmeth,totweek)  $
 msepindic(i)(totmeth,totweek) cover05pindic(i)(totmeth,totweek) qscore05pindic(i)(totmeth,totweek) cover10pindic(i)(totmeth,totweek) qscore10pindic(i)(totmeth,totweek) $
 fzg05res(i)(totmeth,totweek) fzg10res(i)(totmeth,totweek) fzg05tstat(i)(totmeth,totweek) fzg10tstat(i)(totmeth,totweek) fzg05pvalue(i)(totmeth,totweek) fzg10pvalue(i)(totmeth,totweek) $
 fzg05pindic(i)(totmeth,totweek) fzg10pindic(i)(totmeth,totweek) condcov1(i)(totmeth,totweek) condcov2(i)(totmeth,totweek) condcov3(i)(totmeth,totweek) christ10pindic(i)(totmeth,totweek) $
 crpsres(i)(totmeth,totweek) crpststat(i)(totmeth,totweek) crpspvalue(i)(totmeth,totweek) crpspindic(i)(totmeth,totweek) rawcrpsres(i)(totmeth,totweek)  $
 qwcrpsres(i)(totmeth,totweek) qwcrpststat(i)(totmeth,totweek) qwcrpspvalue(i)(totmeth,totweek) qwcrpspindic(i)(totmeth,totweek) rawqwcrpsres(i)(totmeth,totweek)  
end do i
dec rec both(totmeth,2*totweek)  ;* for storing relative performance measures and p-value indicators together in one table

do nn = 1,totmeth
 do mm = 1,totweek
  do n = 1,nsamples
    if nn==1
     dis 'forecast origins, for sample ' n +':  ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
    *
    dis "nn = " methlabel(nn)
    dis "week = " mm
    
    sstats(mean) evalranges(n,1) evalranges(n,2) sqerror(nn,mm){0}>>mse
    comp rmseres(n)(nn,mm) = mse^0.5
    sstats(mean) evalranges(n,1) evalranges(n,2) hit05(nn,mm){0}>>cover05(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) hit10(nn,mm){0}>>cover10(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qs05(nn,mm){0}>>qscore05res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qs10(nn,mm){0}>>qscore10res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) fzg05(nn,mm){0}>>fzg05res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) fzg10(nn,mm){0}>>fzg10res(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) crps(nn,mm){0}>>crpsres(n)(nn,mm)
    sstats(mean) evalranges(n,1) evalranges(n,2) qwcrps(nn,mm){0}>>qwcrpsres(n)(nn,mm)
    *
    comp teststat = ttestQS(hit05(nn,mm),evalranges(n,1),evalranges(n,2))
    comp cover05tstat(n)(nn,mm) = teststat
    comp cover05pvalue(n)(nn,mm) = %ztest(teststat)   ;*  forcing 2-sided test of coverage %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
    comp teststat = ttestQS(hit10(nn,mm),evalranges(n,1),evalranges(n,2))
    comp cover10tstat(n)(nn,mm) = teststat
    comp cover10pvalue(n)(nn,mm) = %ztest(teststat)   ;*  forcing 2-sided test of coverage %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
    * Christoffersen tests, just p-values, 10% coverage rate
    set hitrate evalranges(n,1) evalranges(n,2) = hit10(nn,mm){0} + 0.10
    @Christoffersentest hitrate .10 evalranges(n,1) evalranges(n,2) crtestres crpvalueres
    comp condcov1(n)(nn,mm)=crpvalueres(1), condcov2(n)(nn,mm)=crpvalueres(2), condcov3(n)(nn,mm)=crpvalueres(3)
    *
    if nn<>basepos
     {
      set(scratch) lossdiff = sqerror(basepos,mm){0} - sqerror(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp msetstat(n)(nn,mm) = teststat
      comp msepvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = qs05(basepos,mm){0} - qs05(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp qscore05tstat(n)(nn,mm) = teststat
      comp qscore05pvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = qs10(basepos,mm){0} - qs10(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp qscore10tstat(n)(nn,mm) = teststat
      comp qscore10pvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = fzg05(basepos,mm){0} - fzg05(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp fzg05tstat(n)(nn,mm) = teststat
      comp fzg05pvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = fzg10(basepos,mm){0} - fzg10(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp fzg10tstat(n)(nn,mm) = teststat
      comp fzg10pvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = crps(basepos,mm){0} - crps(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp crpststat(n)(nn,mm) = teststat
      comp crpspvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
      *
      set(scratch) lossdiff = qwcrps(basepos,mm){0} - qwcrps(nn,mm){0}
      comp teststat = ttestQS(lossdiff,evalranges(n,1),evalranges(n,2))
      comp qwcrpststat(n)(nn,mm) = teststat
      comp qwcrpspvalue(n)(nn,mm) = %if(use2sided==1,%ztest(teststat),(1.-%cdf(teststat)))
     }
  end do n
 end do hh
end do nn

do n = 1,nsamples
 ** create copies to store raw rather than relative results
 comp rawrmseres(n) = rmseres(n)
 comp rawqscore05res(n) = qscore05res(n)
 comp rawqscore10res(n) = qscore10res(n) 
 comp rawfzg05res(n) = fzg05res(n)
 comp rawfzg10res(n) = fzg10res(n) 
 comp rawcrpsres(n) = crpsres(n)
 comp rawqwcrpsres(n) = qwcrpsres(n)

 ** adjust raw coverage tables to show empirical rates and not gaps relative to 0.05 nominal and .10 nominal
 ewise cover05(n)(i,j) = cover05(n)(i,j) + 0.05
 ewise cover10(n)(i,j) = cover10(n)(i,j) + 0.10

 ** now convert to relative results
 ewise rmseres(n)(i,j) = %if(i==basepos,rmseres(n)(i,j),rmseres(n)(i,j)/rmseres(n)(basepos,j))
 ewise qscore05res(n)(i,j) = %if(i==basepos,qscore05res(n)(i,j),qscore05res(n)(i,j)/qscore05res(n)(basepos,j))
 ewise qscore10res(n)(i,j) = %if(i==basepos,qscore10res(n)(i,j),qscore10res(n)(i,j)/qscore10res(n)(basepos,j))
 ewise fzg05res(n)(i,j) = %if(i==basepos,fzg05res(n)(i,j),fzg05res(n)(basepos,j)-fzg05res(n)(i,j))
 ewise fzg10res(n)(i,j) = %if(i==basepos,fzg10res(n)(i,j),fzg10res(n)(basepos,j)-fzg10res(n)(i,j))
 ewise crpsres(n)(i,j) = %if(i==basepos,crpsres(n)(i,j),crpsres(n)(i,j)/crpsres(n)(basepos,j))
 ewise qwcrpsres(n)(i,j) = %if(i==basepos,qwcrpsres(n)(i,j),qwcrpsres(n)(i,j)/qwcrpsres(n)(basepos,j))

 ** construct p-value indicators
 comp [rec] tempmat = msepvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp msepindic(n)  = tempmat

 comp [rec] tempmat = cover05pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp cover05pindic(n)  = tempmat

 comp [rec] tempmat = cover10pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp cover10pindic(n)  = tempmat

 comp [rec] tempmat = condcov3(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp christ10pindic(n)  = tempmat
 
 comp [rec] tempmat = qscore05pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp qscore05pindic(n)  = tempmat

 comp [rec] tempmat = qscore10pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp qscore10pindic(n)  = tempmat
 
 comp [rec] tempmat = fzg05pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp fzg05pindic(n)  = tempmat

 comp [rec] tempmat = fzg10pvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp fzg10pindic(n)  = tempmat
 
 comp [rec] tempmat = crpspvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp crpspindic(n)  = tempmat
 
 comp [rec] tempmat = qwcrpspvalue(n)
 ewise tempmat(i,j) = %if(tempmat(i,j)<=.01,-90,%if(tempmat(i,j)<=.05,-80,%if(tempmat(i,j)<=.10,-70,-20)))
 comp qwcrpspindic(n)  = tempmat

 *** create merged tables, with relative stats and p-value indicators
 comp [rec] statsmat = rmseres(n)
 comp [rec] pindicmat = msepindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp mseboth(n) = both

 comp [rec] statsmat = cover05(n) 
 comp [rec] pindicmat = cover05pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp cover05both(n) = both
 
 comp [rec] statsmat = cover10(n)
 comp [rec] pindicmat = cover10pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp cover10both(n) = both
 
 comp [rec] statsmat = condcov3(n)
 comp [rec] pindicmat = christ10pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp christ10both(n) = both
 
 comp [rec] statsmat = qscore05res(n)
 comp [rec] pindicmat = qscore05pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp qscore05both(n) = both
 
 comp [rec] statsmat = qscore10res(n)
 comp [rec] pindicmat = qscore10pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp qscore10both(n) = both
 
 comp [rec] statsmat = fzg05res(n)
 comp [rec] pindicmat = fzg05pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp fzg05both(n) = both
 
 comp [rec] statsmat = fzg10res(n)
 comp [rec] pindicmat = fzg10pindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp fzg10both(n) = both
 
 comp [rec] statsmat = crpsres(n)
 comp [rec] pindicmat = crpspindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp crpsboth(n) = both
 
 comp [rec] statsmat = qwcrpsres(n)
 comp [rec] pindicmat = qwcrpspindic(n)
 do k = 1,%cols(statsmat)
  comp %psubmat(both,1,k*2-1,%xcol(statsmat,k))
  comp %psubmat(both,1,k*2,%xcol(pindicmat,k))
 end do k
 comp qwcrpsboth(n) = both

end do n

******************** display results:  raw results (not relatives)

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqscore10res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawfzg10res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover10(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   CRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawcrpsres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   qwCRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqwcrpsres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawqscore05res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawfzg05res(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover05(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   RMSEs     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(rawrmseres(n),i)
  dis methlabel(i) @56 ###.### resultsvec
 end do i
end do n

******************** display results:  combined table of relative stats and p-value indicators (-90 for 1%, -80 for 5%, -70 for 10%)

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   CRPS     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(crpsboth(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qscore10both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(fzg10both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   p-values of Christoffersen condit coverage tests     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(christ10both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   qwcrps     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qwcrpsboth(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   10% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover10both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% quantile scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(qscore05both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% FZG scores     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(fzg05both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   5% coverage rates     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(cover05both(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

do n = 1,nsamples
 if n==1
  {
   dis ''
   dis '**************   RMSEs     ****************'
  }
 dis ''
 dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))
 dis ''
 do i = 1,totmeth
  comp [vec] resultsvec = %xrow(mseboth(n),i)
  dis methlabel(i) @46 ###.## resultsvec
 end do i
end do n

******************** create charts of relative results
******************** 

*********** setup stuff
smpl 1 totweek
comp n = 1 ;* will be full sample defined in evaluation block
dis 'sample: ' %datelabel(evalranges(n,1))  %datelabel(evalranges(n,2))

*********** charts of relative performance across weeks of base M variable set and BMF-SV model (benchmark used in this chart and text)
dec vec[ser] chartseries(3)
comp chartpos = 2
dis methlabel(chartpos)

** levels of CRPS, QS, VaR-ES by week
set chartseries(1) = rawfzg05res(n)(chartpos,t)
set chartseries(2) = rawcrpsres(n)(chartpos,t)
set chartseries(3) = rawqscore05res(n)(chartpos,t)
comp header = "Base M: BMF-SV accuracy by week"
graph(nodates,header=header,key=below,klab=||"10% FZG (left)", "CRPS (right scale)", "10% QS (right)"||,overlay=line,ovcount=2) 3
# chartseries(1)
# chartseries(3)
# chartseries(2)

** CRPS, QS, VaR-ES by week normalized so week 1=1 (chart used in paper)
set chartseries(1) = rawcrpsres(n)(chartpos,t)/rawCRPSres(n)(chartpos,1)
set chartseries(2) = rawqscore05res(n)(chartpos,t)/rawqscore05res(n)(chartpos,1)
set chartseries(3) = rawfzg05res(n)(chartpos,t)/rawfzg05res(n)(chartpos,1)

comp header = "Base M: BMF-SV accuracy by week (normalized to week 1)"
graph(nodates,header=header,key=below,klab=||"CRPS", "10% QS", "10% VaR-ES score"||) 3
# chartseries(1)
# chartseries(2)
# chartseries(3)

print(nodates) /  chartseries
