/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Program for reading in forecasts to construct charts for paper's Figure 3.
As indicated in the paper, this chart for 2020 uses results from the BMF-SVO specification
in lieu of the BMF-SV model, in order to mitigate the influence of COVID's extreme volatility.
*/

env nowshowgraphs
gsave(format=pdf) "fcchartsOOS_usingSVOnotSV_*.pdf"    ;* root of file name to which charts are written, in pdf format

comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp stvint = 1985:1       ;* stvint is first vintage, which corresponds to the start of forecasting
comp endvint = 2021:1      ;* last quarter of data vintages considered
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp endsmpl = endvint-useactual

comp stpt = 1985:1       ;* starting point for model estimation
comp svostart = 2019:1   ;* BMF-SVO forecasts do not start until 2019:Q1

all endvint
smpl stsmpl endvint

******************** actual GDP used as actuals
** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close
set gdp = actualGDP

******************** dimensioning for reading in results
comp [vec[int]] SVindicator = ||1,0,0,1,0,0,1,0,0||  ;* 1 for models with SV, 0 otherwise -- used to overwrite SV results at end of sample with SVO

comp [vec[str]] methlabel = ||$
"base M-F:  BMF-SV",$
"base M-F:  BQR",$
"base M-F:  PQR",$
"base M-F + small weekly:  BMF-SV",$
"base M-F + small weekly:  BQR",$
"base M-F + small weekly:  PQR",$
"base M-F + large weekly:  BMF-SV",$
"base M-F + large weekly:  BQR",$
"base M-F + large weekly:  PQR"||

comp [vec[str]] filestouse = ||$
"out.forecastruns/basemacro_basefinance_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_smallwkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_smallwkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_smallwkly_empiricalpctiles",$
"out.forecastruns/basemacro_basefinance_largewkly_SV",$
"out.fcprocessingfromdraws/fcresults_BQR_MF_largewkly_empiricalpctiles",$
"out.fcprocessingfromdraws/fcresults_PQR_MF_largewkly_empiricalpctiles"||

comp nmeth = %rows(filestouse)

******************** setting up storage
comp totweek = 15

*** what we use to read in, method by method
dec vec[ser] fcseries(totweek) pct10series(totweek) pct90series(totweek) 
do mm = 1,totweek 
 labels fcseries(mm)
 # 'fcseries_m'+%string(mm)
 labels pct10series(mm)
 # 'pct10_m'+%string(mm)
 labels pct90series(mm)
 # 'pct90_m'+%string(mm)
end do mm
clear fcseries pct10series pct90series 

*** what we use to store, covering all methods
dec rec[ser] pointfc(nmeth,totweek) q10fc(nmeth,totweek) q90fc(nmeth,totweek) 

******************** Note that we first read in a full history of BMF-SV forecasts and then read in SVO forecasts for
******************** 2019-20 and overwrite the SV forecasts for these years
do nn = 1,%rows(filestouse)
 comp [str] filename = filestouse(nn)+".xls"
 dis filename
 open data &filename
 data(org=col,for=xls) / fcseries pct10series pct90series
 close data
 if svindicator(nn)==1
  {
   comp [str] filename = filestouse(nn)+"O.xls"
   dis filename
   open data &filename
   data(org=col,for=xls) / fcseries pct10series pct90series
   close data
  }
 do mm = 1,totweek
  set pointfc(nn,mm) = fcseries(mm){0}
  set q10fc(nn,mm) = pct10series(mm){0} 
  set q90fc(nn,mm) = pct90series(mm){0} 
 end do mm
end do nn

**** form avg fcs for base M-F case by brute force
dec vec[ser] avgpointfc(totweek) avgquantfc(totweek)
clear(zeros) avgpointfc avgquantfc
do mm = 1,totweek
 set avgpointfc(mm) = (pointfc(1,mm){0}+pointfc(2,mm){0}+pointfc(3,mm){0})/3.
 set avgquantfc(mm) = (q10fc(1,mm){0}+q10fc(2,mm){0}+q10fc(3,mm){0})/3.
end do mm

******************** QUANTILE AND POINT FORECASTS, 1 VARIABLE SET, with model compared to average fc in each chart
******************** for 2020
comp ndata=3, nmodel=3  ;* 3 variable sets (base M-F, M-F+small, M-F+large) and 3 models (BMF-SVO, BQR, PQR) -- but omitting M-F+small from charts
dec rec[ser] fc2020(ndata,nmodel) ptfc2020(ndata,nmodel)
comp nquarters = 4
comp nper = nquarters*totweek
clear(length=nper) fc2020 ptfc2020 outcome fc2020_avg ptfc2020_avg
dec vec[str] xlabel(nper)

comp [vec[str]] modellabel = ||"BMF-SV","BQR","PQR"||
comp [vec[str]] datalabel = ||"base M-F: point","avg. point fc.","base M-F: 10%","avg. 10% fc."||  ;* ,"base M-F + small weekly"
comp datalabel = datalabel+||"actual GDP"||

*** store quantile forecasts and actuals in series to be used in charts
do i = 1,ndata
 do j = 1,nmodel
  do k = 1,nquarters
   do ww = 1,totweek
    comp fc2020(i,j)((k-1)*totweek+ww) = q10fc((i-1)*nmodel+j,ww)(2019:4+k)
    comp ptfc2020(i,j)((k-1)*totweek+ww) = pointfc((i-1)*nmodel+j,ww)(2019:4+k)
    comp outcome((k-1)*totweek+ww) = actualgdp(2019:4+k)
    if i==1.and.j==1
     {
      comp fc2020_avg((k-1)*totweek+ww) = avgquantfc(ww)(2019:4+k)
      comp ptfc2020_avg((k-1)*totweek+ww) = avgpointfc(ww)(2019:4+k)
     }
   end do ww
  end do k
 end do j
end do i

*** create time labels to use on x axis
do k = 1,nquarters
 do ww = 1,totweek,4
  comp xlabel((k-1)*totweek+ww) = "Q"+k+":"+ww
 end do ww
end do k

*** produce charts
comp forcemax=40., forcemin=-80.

smpl 1 nper
do i = 1,nmodel
 comp header = "Point and 10 percent quantile forecasts for 2020 from " + modellabel(i)
 graph(nodates,header=header,key=below,klab=datalabel,overlay=dots,ovsamescale,ovcount=1,xlabel=xlabel,max=forcemax,min=forcemin) 5
 # ptfc2020(1,i) / 2
 # ptfc2020_avg / 8
 # fc2020(1,i) / 4
 # fc2020_avg / 6
 # outcome / 1
 print(nodates) / ptfc2020(1,i) ptfc2020_avg fc2020(1,i) fc2020_avg outcome 
end do i


