/*
AUTHOR:   Todd Clark, Federal Reserve Bank of Cleveland, todd.clark@researchfed.org
Code associated with Andrea Carriero, Todd Clark and Massimiliano Marcellino, 
"Nowcasting Tail Risks to Economic Activity at a Weekly Frequency"

Program for estimating the nowcasting model indicated below.  In this SV/SVO case,
the program writes to an Excel file all the forecast metrics.  The program also writes all draws of forecasts
for 19 quantiles from 0.05 to 0.95.
*/

********************* OOS estimation and forecasting:  simple SV-regression with blocking
********************* base macro indicators + smaller set of other spending indicators

dis %dateandtime()
dis %ratsversion()
comp overallstart = %cputime()

/*
note:  To change variables included:
1. change the numerical settings for nmlyvar, nwklyvar at the top
2. change the series listed in the EQV command.  Last nwklyvar entries are those for weekly variables.  These are duplicated (same order)
  as monthly variables, appended to the list of monthly variables preceding the list of weekly variables
3. change the vectors of timing indicators, mlytiming and wklytiming
4. Add/change data reads as need be (references to sheets, e.g.) to read in relevant add-ons
*/

*********************
********************* BASIC SETUP & PARAMETER ENTRY
*********************

** key dimensions used below
comp nmlyvar = 5                                ;* number of purely monthly predictor variables to be considered
comp nboth = 2                                  ;* number of predictor variables that are both monthly and weekly to be considered
comp nwklyvar = 2                               ;* weekly-only predictors
comp totxvar = nmlyvar+2*nboth+nwklyvar         ;* 2*nwkly b/c, for wkly variables, we use full month readings preceding the weekly readings on the month

comp ndraws =  5000	     ;*total number of Gibbs draws
comp burndraws = 5000	 ;*burn-in draws for Gibbs sampler (discarded)
comp skipint = 5         ;*

/*
comp ndraws =  200	     ;*total number of Gibbs draws
comp burndraws = 50	 ;*burn-in draws for Gibbs sampler (discarded)
comp skipint = 1         ;*
*/

env nowshowgraphs
gsave(format=pdf) "basemacro_smallwkly_SV_*.pdf"
comp modeldesc = "BMF-SV model:  base macro + small weekly"
comp filelabel = "fcdraws/SV_M_smallwkly/"

************ note on dating convention: forecast dates refer to forecast origin, rather than forecast endpoint
comp useactual = 1          ;* adjustment for obs lost of last vintage relative to eval sample end date (used to set this at 2)
comp styr = 1947
cal(q) styr:1
comp stsmpl = styr:1	;*earliest period with data

comp stvint = 2000:1       ;* stvint is first vintage, which corresponds to the start of forecasting
comp endvint = 2021:1      ;* last quarter of data vintages considered
comp endsmpl = endvint - useactual
comp endsmpl19 = 2019:4
comp lastmo = 2             ;* last month's vintage 
comp hardstpt = 1987:1      ;* common start point for estimation based on qly-mly data for full sample
comp nv = (%year(endvint)-%year(stvint))*12+lastmo  ;* this will be the number of vintages available

all totxvar+nv endvint
smpl stsmpl endvint

comp seedval = 128*%month(endsmpl)*%year(endsmpl)
seed seedval
dis seedval

*********************
********************* DATA READ AND SETUP that can be done up front (before loop)
*********************
** declare variables to be used:  ordering of monthly only, mly both, wkly both, wkly only
eqv 1 to totxvar+nv
employ ism ipt rsafs hs claimsmly cclaimsmly claimswkly cclaimswkly steelwkly utilwkly $
gdp00m1 gdp00m2 gdp00m3 gdp00m4 gdp00m5 gdp00m6 gdp00m7 gdp00m8 gdp00m9 gdp00m10 gdp00m11 gdp00m12 gdp01m1 gdp01m2 gdp01m3 gdp01m4 $
gdp01m5 gdp01m6 gdp01m7 gdp01m8 gdp01m9 gdp01m10 gdp01m11 gdp01m12 gdp02m1 gdp02m2 gdp02m3 gdp02m4 gdp02m5 gdp02m6 gdp02m7 gdp02m8 $
gdp02m9 gdp02m10 gdp02m11 gdp02m12 gdp03m1 gdp03m2 gdp03m3 gdp03m4 gdp03m5 gdp03m6 gdp03m7 gdp03m8 gdp03m9 gdp03m10 gdp03m11 gdp03m12 $
gdp04m1 gdp04m2 gdp04m3 gdp04m4 gdp04m5 gdp04m6 gdp04m7 gdp04m8 gdp04m9 gdp04m10 gdp04m11 gdp04m12 gdp05m1 gdp05m2 gdp05m3 gdp05m4 $
gdp05m5 gdp05m6 gdp05m7 gdp05m8 gdp05m9 gdp05m10 gdp05m11 gdp05m12 gdp06m1 gdp06m2 gdp06m3 gdp06m4 gdp06m5 gdp06m6 gdp06m7 gdp06m8 $
gdp06m9 gdp06m10 gdp06m11 gdp06m12 gdp07m1 gdp07m2 gdp07m3 gdp07m4 gdp07m5 gdp07m6 gdp07m7 gdp07m8 gdp07m9 gdp07m10 gdp07m11 gdp07m12 $
gdp08m1 gdp08m2 gdp08m3 gdp08m4 gdp08m5 gdp08m6 gdp08m7 gdp08m8 gdp08m9 gdp08m10 gdp08m11 gdp08m12 gdp09m1 gdp09m2 gdp09m3 gdp09m4 $
gdp09m5 gdp09m6 gdp09m7 gdp09m8 gdp09m9 gdp09m10 gdp09m11 gdp09m12 gdp10m1 gdp10m2 gdp10m3 gdp10m4 gdp10m5 gdp10m6 gdp10m7 gdp10m8 $
gdp10m9 gdp10m10 gdp10m11 gdp10m12 gdp11m1 gdp11m2 gdp11m3 gdp11m4 gdp11m5 gdp11m6 gdp11m7 gdp11m8 gdp11m9 gdp11m10 gdp11m11 gdp11m12 $
gdp12m1 gdp12m2 gdp12m3 gdp12m4 gdp12m5 gdp12m6 gdp12m7 gdp12m8 gdp12m9 gdp12m10 gdp12m11 gdp12m12 gdp13m1 gdp13m2 gdp13m3 gdp13m4 $
gdp13m5 gdp13m6 gdp13m7 gdp13m8 gdp13m9 gdp13m10 gdp13m11 gdp13m12 gdp14m1 gdp14m2 gdp14m3 gdp14m4 gdp14m5 gdp14m6 gdp14m7 gdp14m8 $
gdp14m9 gdp14m10 gdp14m11 gdp14m12 gdp15m1 gdp15m2 gdp15m3 gdp15m4 gdp15m5 gdp15m6 gdp15m7 gdp15m8 gdp15m9 gdp15m10 gdp15m11 $
gdp15m12 gdp16m1 gdp16m2 gdp16m3 gdp16m4 gdp16m5 gdp16m6 gdp16m7 gdp16m8 gdp16m9 gdp16m10 gdp16m11 gdp16m12 gdp17m1 gdp17m2 $
gdp17m3 gdp17m4 gdp17m5 gdp17m6 gdp17m7 gdp17m8 gdp17m9 gdp17m10 gdp17m11 gdp17m12 gdp18m1 gdp18m2 gdp18m3 gdp18m4 gdp18m5 gdp18m6 $
gdp18m7 gdp18m8 gdp18m9 gdp18m10 gdp18m11 gdp18m12 gdp19m1 gdp19m2 gdp19m3 gdp19m4 gdp19m5 gdp19m6 gdp19m7 gdp19m8 gdp19m9 gdp19m10 $
gdp19m11 gdp19m12 gdp20m1 gdp20m2 gdp20m3  gdp20m4 gdp20m5 gdp20m6 gdp20m7 gdp20m8 gdp20m9 gdp20m10 gdp20m11 gdp20m12 gdp21m1 gdp21m2

** define timing indicators used below
** mlytiming vector applies to monthly-only series; "both" category series at the (prior) monthly frequency are treated as available in week 1 
comp [vec[int]] mlytiming = ||1,1,3,2,3||  ;* indicates what week of the month each true monthly freq variable becomes available
** wklytiming vector applies to series of "both" category at weekly frequency and purely weekly series
comp [vec[int]] wklytiming = ||2,3,2,2|| ;* indicates what week of the month each true weekly freq variable becomes available for week 1
comp nomitavg = 2  ;* count of last in list of weekly series to omit from averaging

comp nmo = 3                     ;* number of months in quarter -- actual data
comp npredmo = 4                 ;* number of months at which we form forecasts: months 1-3 of quarter t plus month 1 of quarter t+1
comp nweek = 4                   ;* number of weeks used in month
comp totweek = npredmo*nweek-1     ;* total number of predictions.  -1 because we don't form a forecast for a 16th week, by which time gdp(t+1) becomes available

** stuff for right-hand side variables:  rec of series for storage and assigning labels, to make regression displays more transparent
**  holding monthly indicators sampled at month 1, month 2, etc. (qly frequency), and the same for weekly indicators
comp totmlyvar=nmlyvar+nboth, totwklyser=nwklyvar+nboth
dec rec[ser] mlyser(totmlyvar,nmo) wklyser(totwklyser,totweek)

** note:  this assignment of series to arrays of series used in setting up models
**  is automated; it works off of the dimensions indicated above and the variables listed in the EQV command
do i = 1,totmlyvar
 do mm = 1,nmo
  labels mlyser(i,mm)
  # %l(i)+'_m'+%string(mm)
 end do mm
end do i
do i = 1,totwklyser
 do mm = 1,totweek
  labels wklyser(i,mm)
  # %l(totmlyvar+i)+'_w'+%string(mm)
 end do mm
end do i

** just reading in GDP and forming growth rate series:  real time data for estimation and forecasting
open data ../data/realtimeGDP.allmonths.xlsx
data(format=xlsx,org=col,sheet="forweekly") / totxvar+1 to totxvar+nv
close
do i = 1,nv
  set(scratch) totxvar+i = 400.*log((totxvar+i){0}/(totxvar+i){1});* GDP growth
end do i

** read in GDP estimates used as actuals in evaluation of forecasts:  these are 2nd avail in the quarterly RTDSM
open data ../data/GDPactuals.secondrelease.xls
data(format=xls,org=col) / actualGDP
close

*********************
********************* setting SOME (not all) stuff for prior and model setup
*********************
comp nylag = 1     ;* numbers of lags of GDP included
comp arlag = 4     ;* lags of each variable included in the autoregressions used to compute variances that enter the prior

** shrinkage hyperparameters for coefficient vector:  element 1: overall shrinkage, 2: cross-variable shrinkage, 3:  rate of lag decay/shrinkage
**      (where lower number means more shrinkage)
comp [vec] shrinkage = ||0.2, 0.2, 1.0||

dec vec priormean
dec symm priorvar

** stuff for volatility
comp varSigma0 = 4.0
comp muPhi = 0.035
comp priordfPhi = 5

comp meanSigma0 = 0.0  ;* just initialize the prior mean on period 0 volatility at 0

********************************
******************************** stuff for estimation and forecast storage
********************************
sou(noecho) ../procedures/univariate.SV.src      ;* procedure for using Gibbs sampling to estimate univariate equation, with Normal-diffuse IG prior
sou(noecho) ../procedures/fcmoments.src

comp nquant = 19
dec vec percentiles(nquant)
ewise percentiles(i) = float(i)/(nquant+1)

dec vec[ser] fcseries(totweek) pct05series(totweek) pct10series(totweek) pct90series(totweek) pct95series(totweek) crpsseries(totweek) qwcrpsseries(totweek) $
 qs05series(totweek) qs10series(totweek) es05series(totweek) es10series(totweek) fzg05series(totweek) fzg10series(totweek)

do mm = 1,totweek                   ;* create series labels to facilitate later reading results from Excel file created below
 labels fcseries(mm)
 # 'fcseries_m'+%string(mm)
 labels pct05series(mm)
 # 'pct05_m'+%string(mm)
 labels pct10series(mm)
 # 'pct10_m'+%string(mm)
 labels pct90series(mm)
 # 'pct90_m'+%string(mm)
 labels pct95series(mm)
 # 'pct95_m'+%string(mm)
 labels crpsseries(mm)
 # 'crps_m'+%string(mm)
 labels qwcrpsseries(mm)
 # 'qwcrps_m'+%string(mm)
 labels qs05series(mm)
 # 'qs05_m'+%string(mm)
 labels qs10series(mm)
 # 'qs10_m'+%string(mm)
 labels es05series(mm)
 # 'es05_m'+%string(mm)
 labels es10series(mm)
 # 'es10_m'+%string(mm)
 labels fzg05series(mm)
 # 'fzg05_m'+%string(mm)
 labels fzg10series(mm)
 # 'fzg10_m'+%string(mm)
end do mm
clear fcseries pct05series pct10series pct90series pct95series crpsseries qwcrpsseries qs05series qs10series es05series es10series fzg05series fzg10series

**** NBER dates, quarterly
clear(length=endvint,zeroes) cycle
set cycle 1948:4 1949:4 = 1.0
set cycle 1953:2 1954:2 = 1.0
set cycle 1957:3 1958:2 = 1.0
set cycle 1960:2 1961:1 = 1.0
set cycle 1969:4 1970:4 = 1.0
set cycle 1973:4 1975:1 = 1.0
set cycle 1980:1 1980:3 = 1.0
set cycle 1981:3 1982:4 = 1.0
set cycle 1990:3 1991:1 = 1.0
set cycle 2001:1 2001:4 = 1.0
set cycle 2007:4 2009:2 = 1.0
set cycle 2019:4 2020:2 = 1.0

********************************
******************************** read in and set up weekly data, which are not real time
********************************
smpl stsmpl endvint
comp wkeststpt = stsmpl
comp wkestendpt = endvint

** with setting up weekly data, we have to separately treat weeks 1-12 of quarter t, and weeks 13-15, which come from quarter t+1
open data ../data/weeklydata.notrealtime.xlsx
do mm = 1,totweek
 clear(length=endsmpl) totmlyvar+1 to totxvar
 if mm<=nmo*nweek
  comp mmm=mm, per=0
 else
   comp mmm=mm-nmo*nweek, per=-1
 *** reading data
 data(format=xlsx,org=col,sheet="intrates",sel=mmm) / tswkly cswkly sp500wkly
 data(format=xlsx,org=col,sheet="other",sel=mmm) / steelwkly utilwkly loadswkly fuelwkly rbookwkly
 data(format=xlsx,org=col,sheet="nfci",sel=mmm) / nfciwkly 
 data(format=xlsx,org=col,sheet="claims",sel=mmm) / claimswkly cclaimswkly 
 *** storing data 
 dofor i = totmlyvar+1 to totxvar
  set(scratch) wklyser(i-totmlyvar,mm) = i{per}
  inquire(series=i) tempstpt tempendpt
  comp wkeststpt = %imax(wkeststpt,tempstpt)      
  comp wkestendpt = %imin(wkestendpt,tempendpt)    
 end do i
end do mm
close data

smpl wkeststpt wkestendpt

** now, within the month, for weeks k = 2 through 4, make week k the average of months 1 through k for that month
** this makes SP500 an average of 4-week growth rates:  avg w1-w2 for w2, avg w1-w3 for w3, and avg w1-w4 for w4
do mm = 1,npredmo
 do nn = 2,nweek
  comp tt = (mm-1)*nweek+nn
  if tt>totweek
   break
  dofor i = totmlyvar+1 to totxvar-nomitavg
   set(scratch) wklyser(i-totmlyvar,tt) = wklyser(i-totmlyvar,tt){0} + wklyser(i-totmlyvar,tt-1){0}
  end do i
 end do nn
end do mm
do mm = 1,npredmo
 do nn = 2,nweek
  comp tt = (mm-1)*nweek+nn
  if tt>totweek
   break
  dofor i = totmlyvar+1 to totxvar-nomitavg
   set(scratch) wklyser(i-totmlyvar,tt) = (1./nn)*wklyser(i-totmlyvar,tt){0}  ;* now just divide to make the proper average
  end do i
 end do nn
end do mm

dis %datelabel(wkeststpt) %datelabel(wkestendpt)
*smpl eststpt estendpt

********************************
******************************** now loop over time to read monthly predictors, determine data avail, set up model, and forecast
********************************

do time = stvint,endvint-1  ;* stvint+12 ;* 
 dis ''
 dis '*******************************************************'
 dis '******************************************************* forecast date = ' %datelabel(time)
 dis '*******************************************************'
 dis ''
 comp quarter = %month(time)
 comp originst = %cputime()

 do ww = 1,totweek
  smpl stsmpl time
  comp eststpt = stsmpl
  comp estendpt = time
  ****
  comp lastmoposs = ((ww-1)/4)
  comp nn = ww - lastmoposs*4
  comp yy = %year(time)
  comp month = lastmoposs+1+(quarter-1)*3
  comp mml = 1+(ww-1)/4     ;* month number in quarter (1-3 in quarter and 4 in next quarter)
  comp nn = ww - (mml-1)*4  ;* week number in month (1-4)
  if ww<=3
   comp horz = 2
  else
   comp horz = 1

  comp filename = "../data/rtwklydata.RATS/rtwklydata."+%string(yy)+"qtr"+quarter+"wk"+ww+".rat"

  dis ""
  dis "****************************** totweek, month, week = " ww mml nn
  dis 'filename = ' filename
  *dis 'last mo possible, month = ' lastmoposs month
  *dis "end date = " %datelabel(time)

  open data &filename
  do mm = 1,nmo
   clear 1 to totmlyvar
   *** reading data
   data(format=rats,sel=mm) / 1 to totmlyvar
   dofor i = 1 to totmlyvar
    set(scratch) mlyser(i,mm) = i{0}
    inquire(series=i) tempstpt tempendpt
    comp eststpt = %imax(eststpt,tempstpt)      
    comp estendpt = %imin(estendpt,tempendpt)    
   end do i
  end do mm
  close data
  *dis %datelabel(eststpt) %datelabel(estendpt)
  
  *print eststpt endpt mlyser

  ***** GDP data
  comp gdpmo = ((ww)/4)+1+(quarter-1)*3
  comp gdpyr = yy
  if quarter==4.and.gdpmo>12
   comp gdpmo=gdpmo-12, gdpyr=gdpyr+1
  
  *dis 'GDP month = ' gdpmo
  dis 'GDP label = ' ('gdp'+%right(%string(gdpyr),2)+"m"+%string(gdpmo))
  set(scratch) gdp = %s('gdp'+%right(%string(gdpyr),2)+"m"+%string(gdpmo)){0}

  ******************************** now figure out start and end date for estimation based on avail of mly, wkly, and GDP
  comp eststpt = %imax(wkeststpt,eststpt)      
  comp estendpt = %imin(estendpt,estendpt)    
  inquire(series=gdp) tempstpt tempendpt
  comp eststpt = %imax(eststpt,tempstpt)      
  comp estendpt = %imin(estendpt,tempendpt)  
  comp eststpt = %imax(eststpt,hardstpt)      
  dis "estimation sample: " %datelabel(eststpt) %datelabel(estendpt)  
  
  ** check that it ends when it should
  if ww<=3.and.estendpt.ne.(time-2)
   dis "weeks 1-3 case: short at end of sample"
  else if ww>=4.and.estendpt.ne.(time-1)
   dis "weeks 4-15 case: short at end of sample"
   
  if eststpt<>hardstpt
   dis "different start date for estimation"

  ******************************** now build up model
    
  *** build up model:  lags of GDP growth
  comp [vec[int]] reglist = ||constant||
  if nylag>0
   {
    if mml==1.and.nn<=3
     enter(varying) reglist
     # reglist gdp{2 to 2+nylag-1}
    else
     enter(varying) reglist
     # reglist gdp{1 to 1+nylag-1}
   }
  
  *** build up model:  lags of monthly (only) variables
  do i = 1,nmlyvar
   if mml==1
    {
     if mlytiming(i)==1
      enter(varying) reglist
      # reglist mlyser(i,1){1} mlyser(i,2){1} mlyser(i,3){1}
     else 
      {
       if nn<mlytiming(i)
        enter(varying) reglist
        # reglist mlyser(i,1){1} mlyser(i,2){1}
       else
        enter(varying) reglist
        # reglist mlyser(i,1){1} mlyser(i,2){1} mlyser(i,3){1}
      }
    }  ;* end if mml==1
   else
    {
     do j = 1,mml-1
      if j==(mml-1).and.nn<mlytiming(i)
       next
      enter(varying) reglist
      # reglist mlyser(i,j){0}
     end do j
    } ;* end of case of mml>1
  end do i  
  
  *** build up model:  monthly averages of weekly variables as available
  do i = 1,nboth
   if mml==1.and.nn<3
    enter(varying) reglist
    # reglist mlyser(nmlyvar+i,1){1} mlyser(nmlyvar+i,2){1} mlyser(nmlyvar+i,3){1}
   else
    {
     do j = 1,mml-1
       enter(varying) reglist
       # reglist mlyser(nmlyvar+i,j){0}
     end do j
    } ;* end of dealing with month averages of weekly variables
  end do i  
  
  *** build up model:  weekly variables as available
  do i = 1,totwklyser
   if mml==1.and.nn<wklytiming(i)
     next
   if i>nboth.or.nn>=wklytiming(i)
     enter(varying) reglist
     # reglist wklyser(i,(mml-1)*nweek+nn-wklytiming(i)+1){0}
  end do i  

  smpl eststpt estendpt
  *** set up GDP equation for the purpose of configuring some dimensions
  equation baseline gdp
  # reglist
  comp ncoef = %eqnsize(baseline)
  
  *** now build the prior
  linreg(noprint) gdp
  # constant gdp{1 to arlag}
  comp gdpvar = %seesq

  comp priormean = %zeros(ncoef,1)
  comp priorvar = 0.01*%identity(ncoef)   ;* initialize prior variance to 0.01 times identity matrix
  comp priorvar(1,1) = 1000.^2.*gdpvar    ;* loose prior on intercept
  * lags of GDP growth
  if nylag>0
   {
    do l = 1,nylag
     comp priorvar(l+1,l+1) = (shrinkage(1)/float(l)^shrinkage(3))^2.  
    end do l
   }
  * other predictors  
  comp count = nylag+1
  do ii = count+1,ncoef
    set xseries = %eqnxvector(baseline,t)(ii)
    linreg(noprint) xseries
    # constant xseries{1 to arlag}           ;* but we don't make any adjustment for forecast horizon
    comp priorvar(ii,ii) = (gdpvar/%seesq)*(shrinkage(2)*shrinkage(1))^2.
  end do ii

  *** estimate and forecast and store results
  comp starttime = %cputime()
  @univariateSV(doforecast,prmean) gdp eststpt estendpt horz ndraws burndraws skipint priormean priorvar meanSigma0 varSigma0 muPhi priordfPhi coefRes PhiRes SigmaRes ForecastRes 
  # reglist
  dis 'run time in minutes for this single model = ' (%cputime()-starttime)/60.
  *
  comp outfilename = filelabel+"draws."+%string(yy)+"qtr"+quarter+"wk"+ww+".csv"
  open copy &outfilename
  copy(org=row,for=cdf,nodates) time time forecastres
  close copy
  
  @FCMOMENTS ForecastRes actualGDP time 1 percentiles meanres pctileres crps qwcrps qscoreres esres fzgscoreres
  comp fcseries(ww)(time) = meanres
  comp crpsseries(ww)(time) = crps
  comp qwcrpsseries(ww)(time) = qwcrps
  comp pct05series(ww)(time) = pctileres(1)
  comp pct10series(ww)(time) = pctileres(2)
  comp pct90series(ww)(time) = pctileres(nquant-1)
  comp pct95series(ww)(time) = pctileres(nquant)
  comp qs05series(ww)(time) = qscoreres(1)
  comp qs10series(ww)(time) = qscoreres(2)
  comp es05series(ww)(time) = esres(1)
  comp es10series(ww)(time) = esres(2)
  comp fzg05series(ww)(time) = fzgscoreres(1)
  comp fzg10series(ww)(time) = fzgscoreres(2)

 end do ww

 dis 'total time for this forecast origin = ' (%cputime()-originst)/60.
 
end do time

dis %dateandtime()
dis 'total estimation run time in minutes = ' (%cputime()-overallstart)/60.

********************************
******************************** tabulate and display results:  full sample
********************************
smpl stvint endsmpl19
dis 'nobs = ' (endsmpl19-stvint+1)

do mm = 1,totweek
 dis ''
 dis '************************************** forecasts using ' mm ' weeks of data'
 dis ''
 set fcerror = actualgdp - fcseries(mm){0}
 set fcerrorsq = fcerror^2.

 stats(noprint) fcerror      ;* don't need results from this, but it provides a useful check/verification of the sample we're using at each horizon
 if %nobs<>(endsmpl19-stvint+1)
  dis 'missing data at month ' mm

 sstats(mean) / fcerror>>meanerror
 sstats(mean) / fcerrorsq>>mse

 sstats(mean) / crpsseries(mm)>>avgcrps
 sstats(mean) / qwcrpsseries(mm)>>avgqwcrps

 sstats(mean) / (actualgdp{0}<=pct05series(mm){0})>>covrate05
 sstats(mean) / (actualgdp{0}<=pct10series(mm){0})>>covrate10
 
 sstats(mean) / qs05series(mm)>>avgqs05
 sstats(mean) / qs10series(mm)>>avgqs10
 
 dis 'mean error, rmse = ' @45 #####.### meanerror (mse^0.5)
 dis 'CRPS = ' @45 #####.### avgcrps
 dis 'qwCRPS = ' @45 #####.### avgqwcrps
 dis '5% coverage rate = ' @45 #####.### covrate05
 dis '10% coverage rate = ' @45 #####.### covrate10
 dis '5% quantile score = ' @45 #####.### avgqs05
 dis '10% quantile score = ' @45 #####.### avgqs10
 
 *** FZG score results
 sstats(mean) / fzg05series(mm)>>avgfzg05
 sstats(mean) / fzg10series(mm)>>avgfzg10
 dis '5% FZG score = ' @45 #####.### avgfzg05
 dis '10% FZG score = ' @45 #####.### avgfzg10
 
end do mm

********************************
******************************** write time series of forecast, forecast error, and pred. likelihoods/scores to Excel file
********************************
comp filename = %unitfnroot("input")+".xls"

open copy &filename
copy(for=xls,dates,org=col) stvint endsmpl fcseries pct05series pct10series qs05series qs10series es05series es10series fzg05series fzg10series pct90series pct95series crpsseries qwcrpsseries
close copy

********************************
******************************** charting actual growth with upper and lower tails
********************************
smpl stvint endsmpl19

comp vf = 2
comp hf = 2
comp perpage = 4

comp [vec[string]] mykey = ||'actual','mean','10th-%ile','90th-%ile'||

do mm = 1,totweek,perpage
 grparm(bold) header 14
 spgraph(vfields=vf,hfields=hf,header=modeldesc)
 grparm(bold) header 22
 do j = mm,%imin(mm+perpage-1,totweek)
  comp header = "Forecasts using " + %string(j) + " weeks of data"
  graph(header=header,dates,key=below,klab=mykey,shading=cycle) 4
  # actualGDP / 1
  # fcseries(j) / 2
  # pct10series(j) / 4
  # pct90series(j) / 4
 end do j
 spgraph(done)
end do mm
