%% What is this file doing???
%
%
%  This is the main file that computes the density and point forecasts
%  evaluated in the paper.
%
%  We take in 9 yields: 1mo, 3mo, 1yr, 2yr, 3yr, 4yr, 5yr, 7yr, & 10yr
%  from years 1985 to 2018.
%
%  The data file is named updateddata_full.xlsx, previous data,
%  is DataBliss.xls.
%
%  Selecting model parameters are largely at the beginning of the code.
%
%  Model 4 is the main, JSZ-VAR-CSV.
%
% 
%  Next, select Tstart = 120 and Tlast = 408 - 12 + 1 where each value is
%  one month.  Allowing for the Zero-Lower-Bound, this takes a couple 
%  months to output.  If doing, run two or three MATLAB to fasten up pace.
%
%  Figures 1, 2, 3 are produced via Verbose = 1.  Figures 4, 5
%  produced via V2020_main_evaluation_replicatesJESubmission.m.
%
%  Table 1 is produced by manually setting lam_value to
%  What you want.  If unspecified, integrated out.
%  Tables 2 and 3 are produced in
%  V2020_evaluation_replicatesJESubmission.m.
%  A handy way to access results is within the resu struct.
%
%  Table 1a is produced by doing this
%  ModelSeq = 4
%  and then, Tlast = 409 - 12 + 1 and then, Tstart = Tlast
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%      NOTE: We use different greek letters in this code than in the paper submission.
%
%
%                code | paper
%             -------------------
%                     |
%                teta |  theta Equation (5)
%                     |
%                vol  |  lambda
%                     |
%                psi  |  PHI VAR Coefficeints in Equation (8)
%                     |
%              lambda |  gamma Tightness Parameter bottom of page 8
%                     |
%               sigma |  V Constant Volatility in Equation (9)
%                     |
%            voladraw |  LAMBDA top of page 8
%                     |
%               loght |  log(lambda_t) Equation (10)
%                     |
%
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%% Initial operations 
clear all; clc; close all; %#ok<CLALL>

%%% Declare global variables first

global rinfQ Model JSZ CSV N k nlags NoF maturity dt data sahead verbose ...
    Tend T Tstar lambdas seedSeq teta_prior options  SEED crit DNSvar ...
    ModelSeq PARALLEL ROLL save_ sdlam K0P_cP_ K1P_cP_ cP_ W_ BOLS;

rng('default'); rng(1234589);
addpath csminSIMS 
addpath distrib
addpath gibbs
addpath jsz_library/general     % various folders with functions
addpath jsz_library/jsz
addpath Results3
addpath util

%% Options for optimization (initial condition)
crit=1e-3; %numerical differences CAN ARISE in different processors using very small crit.
options = optimset('Display','final','TolFun',1e-5,...
    'MaxFunEvals',50000,'MaxIter',50000); %fminsearch options

%% Save the densities on hard drive?
save_=1;

%% Show graphs?
verbose=1;% verbose
extra_graphs =1; % show graphs not in paper only when verbose == 1

%% Rolling or recursive estimation?
ROLL=0; % 0 means recursive, positive a rolling window of dimension ROLL

%% SEEDS (also identifies univocallly the estimation setup)

seedSeq=555; %%% 111 flat prior, 240000 draws, 10000 burn and 50 or 100 skip
             %%% 555 diffuse prior, 60000, 2500, 25 skip footnote 12

%% Full sample or recursive excercise?             
PARALLEL=1; %% 1=full sample. We do recursive exercise for forecasting
 
%% Model
ModelSeq=4;
lam_value = inf; % What value for lambda do you want?
 
%% Prior variance of tightness hyperparameter
sdlam=0.25; % Bottom of page 14.
 
%% load data 
for selectdata=1

%updated data including 1m and 4y yields
data00 = xlsread('LiuWudata.xlsx');
dataname = 'LiuWudata'; % for saving purpose
maturity = data00(1,2:end)'; % list of maturity
datamat = data00(2:end, 2:end); % the data
calendar = data00(2:end, 1); % the calendar

end % of selectdata = 1
 
%% Distribute computations among workers

for distribute=1
if PARALLEL~=1
    disp('Optimal splitting not updated, program will not run'); break;
    %{
    % optimal splitting a bit different depending on which data are used
        if seedSeq==999 || seedSeq==777
        Tstart_mat=Tstart:ceil((Tlast-Tstart)/PARALLEL):Tlast;
        Tlast_mat =ceil((Tlast-Tstart)/PARALLEL)-1+Tstart_mat;
        Tlast_mat(end)=Tlast;
        elseif seedSeq==555 || seedSeq==333 || seedSeq==444
        Tstart_mat =[   120   137   154   171   188   205   220   235   250   265   280];
        Tlast_mat = [   136   153   170   187   204   219   234   249   264   279   288];
        elseif seedSeq==111
        Tstart_mat =[   120   139   158   177   196   215   231   246   256   267   278];
        Tlast_mat = [   138   157   176   195   214   230   245   256   266   277   288];
        end
    Tstart=Tstart_mat(:,labindex)';  
    Tlast=Tlast_mat(:,labindex)'; 
    %}
end
if PARALLEL==1 % stop in 2017 when doing the full sample
    
    Tlast = 409-12+1;
    Tstart = Tlast;
    
    calendar=[calendar;calendar(end)+1/12]; %#ok<AGROW>
end
Tfinal=Tlast; % just needed to save reults
end
 
%% Model
for Model=ModelSeq
 
%% MCMC options
blocks=3; % either 1 or 3 blocks for theta
nchain=4; % we use four Markov Chains
nrep0=120000/2;       burn0=5000/2;       step0=50/2;  % Footnote 12.
%  = 60000 / 4 = 15000 draws     = 2500        = 25
 
%% CODE STARTS HERE
 
%% Main pointers
nlags=3; % number of lags in VAR
NoF=3; % number of factors
dt = 1/12; %this is the time unit
sahead=12;% steps ahead 
 
%% Appy model options
if Model==4;  CSV=1; JSZ=1;   DNSvar=1;  tightness=0.2;   end %% JSZ-VAR-CSV 
%% TIME LOOP 
for Tend=Tstart:Tlast % go through all times
 
%% Initialize lambda to 1 (its prior value)

lambdas = 1; % Bottom of page 14.

disp('lambda='); disp(lambdas); resu.lambda=lambdas; 
pause(2);
 
%% Pointers
if Tend<=rows(datamat); data_tplus1=datamat(Tend,:); end
resu.Model=Model;
resu.T=Tend-1;   
resu.MCsetup=[nrep0 burn0 step0];
resu.ROLL=ROLL;
 
if ROLL==0
data=datamat(1:Tend-1,:);
elseif ROLL>0
data=datamat(Tend-ROLL:Tend-1,:);    
end
 
N=cols(data); %Number yields
T=rows(data); %Number months
np=10+N; % 10 = 3 eigenvalues + 1 kinfQ + 6 elements of Sigma_P useful in theta
Tstar=lambdas*T; % Page 8 bottom
k = 1+N*nlags; T0=nlags; %Equation. (14) length X
rinfQ=mean(datamat(:,N)); %Footnote of Appendix Page 2.  K^Q_inf = -lambda^Q_1 * rinfQ
 
%% prepare Data and Unrestricted VAR moments
YYall     = data; XXall     = ones(T-T0, k);
i =1;  while i<=nlags
   XXall(:,1+(i-1)*N+1:1+i*N) = data(T0-i+1:T-i,:);
i=i+1; end
Y     = YYall(T0+1:T,:); X     = XXall;

YY=Y'*Y;
XY=X'*Y;
XX=X'*X;
iXX=XX\eye(k);

BOLS=XX\XY;                             % See Del Negro Schorfheide (2004) Eq (22)
SIGMA=(YY-XY'*iXX*XY)/(T-k); % See Del Negro Schorfheide (2004) Eq (23)
 
%% Priors
for priors=1
    %% Structural coefs, JSZ
    % this is the prior for the homoskedastic model.
    if JSZ==1
    teta_prior=[];
 
    % concentrate out kappas by estimating them via OLS
    % kappas are K^P_0P and K^P_1P
    % data_in=data;
    data_in=Y;
    W_ = pcacov(cov(data_in));      % See page four.
    W_ = W_(:,1:NoF)';  % size = N*J
    cP_ = data_in*W_'; % size = T*N
    [K1PplusI_, K0P_cP_, ScP] = regressVAR(cP_); % Running regression on kappas.  See bottom of page 5.  This computes the maximum likelihood estimates of the 3 variables.
    K1P_cP_ = K1PplusI_ - eye(NoF);     
    
    % FLAT PRIOR ON ALL (works well)

   size_sigmaP_prior = 3 + 1 + 6 + N;
   teta_prior(1: size_sigmaP_prior,:)= repmat([0 0 0], size_sigmaP_prior,1);
   
  
    % Informative prior
        
            % Priors on Kappas see page 15.
            teta_prior(1:3,:)=    [ -0.002 0.001 3 ;   %dlamQ1
                                    -0.020 0.01 3 ;    %dlamQ2
                                    -0.200 0.1 3 ];    %dlamQ3 
 
            teta_prior(4,:)=      [-mean(data_in(:,end))*teta_prior(1)...
                                    mean(data_in(:,end))*teta_prior(1) 0 ];         %kinf    
                
            % Prior on Sigma P
            L=chol(ScP,'Lower');
            teta_prior(5:10,:)=  [ L(1,1) abs(L(1,1)/2) 3 ; %F11
                                   L(2,1) abs(L(2,1)/2) 3 ; %F22
                                   L(2,2) abs(L(2,2)/2) 3 ; %F33
                                   L(3,1) abs(L(3,1)/2) 3 ; %F11
                                   L(3,2) abs(L(3,2)/2) 3 ; %F22
                                   L(3,3) abs(L(3,3)/2) 3 ]; %F33;
                         
    resu.priors=teta_prior;   
    end
 
    %% variance equation Eq.(10) Initialization page 15
    muPhi = 0.05;        priordfPhi = 3;     %      sd of phi_2
    alfa_pmean=0;        alfa_pstdv=0.025;   %      sd of phi_0
    delta_pmean=0.95;    delta_pstdv=0.025;  %      sd of phi_1
    
    if SEED==444
    %variance equation
    muPhi = 0.01;        priordfPhi = 10;
    alfa_pmean=0;        alfa_pstdv=0.025;
    delta_pmean=0.95;    delta_pstdv=0.0025;   
    end
    
    priorV=diag([alfa_pstdv delta_pstdv].^2);
    priorH=priorV\eye(2);
    priorB=[alfa_pmean; delta_pmean];
 
end
 
%% Starting point
for initialpoint=1
    
    rng('default'); rng(123456789); %set the seed for maximization
 
    %% compute posterior mode under homoskedasticity
    if JSZ==1

        % JSZ likelihood (with or without Hessian)
        disp('optimizing only_likelihood')
        if T==Tlast-1
            computehessianJSZ=1;
        else
            computehessianJSZ=0;
        end
        [f_JSZ,teta_JSZ,teta_JSZsd]=MaxLikJSZ(data,computehessianJSZ);
        resu.fJSZ=f_JSZ; resu.tetaJSZ=teta_JSZ;
        resu.tetaJSZsd=teta_JSZsd;


        % Ensure there are not values implying almost 0 measurement error
        teta_JSZ(3 + 1 + 6 + 1: 3 + 1 + 6 + N) = max(teta_JSZ(3 + 1 + 6 + 1: 3+ 1 + 6 + N), 1e-6);

        % posterior mode
        disp('optimizing posterior')
        % posterior mode: SIMS routine: works better, but has random numbers 
        % and if you run many Matlabs at the same time this will imply
        % different results 
        [f,teta_mode,ghat,Hhat,itct,fcount,retcode] = ...
            csminwel('posterior',teta_JSZ,0.00001*eye(np),[],crit,100,Y,YY,XX,XY); 
        resu.post_mode=f;
        resu.teta_mode=teta_mode;

        % Hessian at mode (compute here in case no CSV in the model,   
        % otherwise will be computed after rescaling happens conditional on
        % ML stochastic volatility estimate
        disp('computing Hessian at mode')
    end
 
    %% initial draws for VAR coefficients 
    if JSZ==1
    %set them equal to the posterior means for the JSZ prior (as implied by  
    %theta being at the mode)
        if ~isinf(lambdas)
            if lambdas~=0;     [GYY,GXY,GXX,retcode] = Moments(resu.teta_mode); 
            elseif lambdas==0;  GYY=0; GXY=0; GXX=0;
            end
            MYY = YY + Tstar*GYY;   MXY = XY + Tstar*GXY;   
            MXX = XX + Tstar*GXX;    iMXX = MXX\eye(k);
            %VAR coefs at mode estimates
            PSI_draw= (iMXX*MXY)';                                     %Eq. 22 of Del Negro and Schorfheide (2004)
            sig_tilda=(MYY-MXY'*iMXX*MXY)./(T + Tstar); %Eq. 23 of Del Negro and Schorfheide (2004) 
            sig_draw=sig_tilda;
        elseif isinf(lambdas)
            [GYY,GXY,GXX,retcode] = Moments(resu.teta_mode);
            MYY = GYY;  MXY = GXY;  
            MXX = GXX;    iMXX = MXX\eye(k);
            %VAR coefs at mode estimates
           PSI_draw= (iMXX*MXY)';                                                                                                      %  Eq. (22) of Del Negro and Schorfheide (2004)
            sig_star=(MYY-MXY'*iMXX*MXY); % sig_star is the limit of  sig_tilda for lambda->inf         Eq (23) of Del Negro and Schorfheide (2004)
            sig_draw=sig_star;   
        end  
    end
   
    %% ML estimation of SV model
    if CSV==1
    residuals= Y - X*PSI_draw'; % these are not the orthogonal residuals.
    residuals_ort=residuals/chol(sig_draw); % these are the structural residuals
    disp('optimizing SV')
    [f,volamax,ghat,Hhat,itct,fcount,~]=csminwel('likvolAR',[0 0.95 0.01],0.001,[],crit,100,...
                log(residuals_ort'.^2));       
    [outp,x_tm1t, P_tm1t, x_tt, P_tt, K_t, llks] =  likvolAR(volamax,log(residuals_ort'.^2));
    volaML=sqrt(exp(x_tt)');
    elseif CSV==0
    x_tt=zeros(1,T-T0);
    volaML=sqrt(exp(x_tt)');
    volamax=[0 0 0];
    end
 
    %% form initial volatility draw Phi = ( phi_0, phi_1, phi_2) => (alfa_draw, delta_draw, Phidraw)
    loght_draw=[zeros(1,T0) x_tt]';    
    ht_draw=exp(loght_draw);
    alfa_draw=volamax(1);  
    delta_draw=volamax(2);    
    Phidraw=sqrt(volamax(3)^2);  
 
    %% compute (again) posterior mode but this time using rescaled data
    % this will provide reasonable posterior hessian
    if JSZ==1 && CSV==1
        pause(2); clc; pause(1);
        
        % rescale the data
        vola=sqrt(ht_draw(T0+1:T));
        Xstar=diag(1./vola)*X;
        Ystar=diag(1./vola)*Y;
        YY=Ystar'*Ystar;
        XX=Xstar'*Xstar;
        XY=Xstar'*Ystar;
        iXX=inv(XX);
 
        % update posterior mode
        disp('optimizing posterior again')
        [f,teta_mode,ghat,Hhat,itct,fcount,retcode] = ...
            csminwel('posterior',teta_mode,0.00001*eye(np),[],crit,100,Ystar,YY,XX,XY);
        resu.post_mode=f; 
        resu.teta_mode=teta_mode;
        
        % Hessian at mode
        hessian = hessi('posterior_hessian',teta_mode,0,Ystar,YY,XX,XY);
        [u,s,v] = svd(hessian);
        for i = 1:rows(s); s(i,i) = 1/s(i,i); end
        fac= u*sqrt(s);
        
        pm.H=hessian; 
        pm.fac=fac;
        
        % initial draw of teta
        teta_draw=teta_mode;
        post_draw=f;
        
    end
 
end
 
%% MCMC Loop 
tic
 
for SEED=seedSeq; rng(SEED); resu.SEED=SEED; %loop over chains set the seed for MCMC
pause(2); clc; pause(1);
    
for create=1 %create matrices to store repetitions
 
        nrep=nrep0/nchain;
        burn=burn0;
        step=step0;
        ndraws=(nrep-burn)/step; 
 
        prior_0=zeros(ndraws,nchain); %prior on theta
        mliks_0=zeros(ndraws,nchain); %likelihood (conditional on theta, h)
        posts_0=zeros(ndraws,nchain); %posterior (conditional on theta, h)
        pdctd_0=zeros(ndraws,nchain); %posterior (conditional on theta, h)
        datadens_0=zeros(ndraws,nchain);
        VolaPrior_0=zeros(ndraws,nchain); %prior on H
        
        rejs=zeros(np,nrep,nchain);
        rejs_lam=zeros(nrep,nchain);
        summary_rej=zeros(nchain,3);
        summary_acc=zeros(nchain,3);
        
        tetas_0=zeros(np,ndraws,nchain);                        % theta
        betas_0=zeros(N*k,ndraws,nchain);                     % PHI Equation (8)
        sigmas_0=zeros(N^2,ndraws,nchain);                  % Sigma in IW prior
        VolVar_0=zeros(3,ndraws,nchain);                       % phi Equation (10)
        volatility_0=zeros(T-T0,ndraws,nchain);              % lambda
        VOLATILITIES_0=zeros(T-T0,N,ndraws,nchain);  % LAMBDA
        LAM_0=zeros(ndraws,nchain);                              % gamma
        
        Sigma_t=zeros(N,N,T,nchain);
        Sigma_fix=zeros(N,N,T,nchain);
        variadraw=zeros(T,N,nchain);
        voladraw=zeros(T,N,nchain);
 
        ht_draw_recmean=zeros(T,1,nchain); 
        variadraw_recmean=zeros(T,N,nchain); 
        voladraw_recmean=zeros(T,N,nchain);
        
        lam=zeros(ndraws,nchain);
end 
 
for chain=1:nchain % loop through chains

for rep=1:nrep
    %% start a new chain 
    if rep==1
        % theta
        if JSZ==1; teta_draw=teta_mode;        post_draw=f; end
        % beta and sigma will be implied by theta
        % form initial volatility draw
        loght_draw=[zeros(1,T0) x_tt]'; ht_draw=exp(loght_draw);
        % vol parameters (also these are not needed but update anyway)
        alfa_draw=volamax(1); delta_draw=volamax(2); Phidraw=volamax(3);  
    end
 
    %% display informations
    for information=1
    if  mod(rep,step*10)==0
        info.Model=Model;              info.T=Tend-1; 
        info.Lag=nlags;
        info.ROLL=ROLL;
        info.lambda=lambdas;           info.Seed=SEED;   
        info.totalchain=nchain;        info.chain=chain;
        info.setup=[nrep0 burn0 step0];
        info.chaindraws=ndraws;        info.totaldraws=ndraws*nchain;
        info.completion=rep/nrep;
        %info.posterior=post_draw;   
        %info.predictive=mean(pdctd_0(1:max(1,(rep-burn)/step),chain)); 
        info.rejs=(mean(rejs([1 5 11],1:rep,chain),2)'); info.accs=1- info.rejs;
        summary_rej(chain,:)=info.rejs; summary_acc(chain,:)=info.accs;
        info.rejs_lam=mean(rejs_lam(1:rep,:));
        display(info);  
        toc;
    end
    end
  
    %% Compute the GLS transformation
    % conditional on the current draw of teta and CSV
    % note that if CSV=0 then vola=1 all the time
    vola=sqrt(ht_draw(T0+1:T));
    Xstar=diag(1./vola)*X;    Ystar=diag(1./vola)*Y;
    YY=Ystar'*Ystar; XX=Xstar'*Xstar; XY=Xstar'*Ystar; iXX=inv(XX);   
    
    %%       STEP 1:      Draw lambda (actually gamma). Random walk Metropolis-Hastings.
    for draw_lambda=1
        if 0 == 0
            if rep==1 
                lam_draw=1; lambdadens_draw=post_draw; 
                cc_lambda=0.35;  
            end
 
            valid=0; 
            while valid==0
            lam_can=lam_draw+cc_lambda*randn; %only retain positive values    <=  random walk
            if lam_can>(k+N)/120 && lam_can<10 % restriction from page 14
                valid=1; end
            end

            if JSZ==1
            lambdadens_draw=priodens(lam_draw,1,sdlam,3)+...
                mlikelihood_condTight(teta_draw,Ystar,YY,XX,XY,lam_draw);
            lambdadens_cand=priodens(lam_can ,1,sdlam,3)+...
                mlikelihood_condTight(teta_draw,Ystar,YY,XX,XY,lam_can);
            end

            if  rand < min([1 ; exp(lambdadens_cand - lambdadens_draw)])    %       <= metropolis hastings
                rejs_lam(rep,chain)=0; 
                lam_draw=lam_can; lambdadens_draw=lambdadens_cand;
            else
                rejs_lam(rep,chain)=1;
            end

            lambdas=lam_draw;
            Tstar=lambdas*T;
            lam(rep,chain)=lambdas;
            %figure(999); plot(lam);
        else
            disp('Please enter valid value for tightness lambda');
            'error';
        end
    end
        
    %%     STEP 2:    Draw structural coefficients teta (theta).  Multiple block Metropolis-Hastings algorithm.
    % JSZ PRIOR
    for draw_teta=1
 
    if JSZ==1
        
        % Scaling factors to get 20-40% acceptance rates and update each 100 draws
        if rep==1
        cc_b1=  0.96;    cc_b2=0.45;   cc_b3=1.1;
        end 
 
        % Update the (conditional posterior) at the current new draw
        % DO NOT DELETE THIS!!! IT IS ESSENTIAL
        [~,   post_draw, prio_draw, mlik_draw]=posterior(teta_draw,Ystar,YY,XX,XY); 
 
            startingblock=1;       % <= multiple block Metropolis-Hastings algorithm
             
                for block_param=startingblock:3 % bottom of page A5
                    % select the block to draw 
                    if block_param==1; p1=1; p2=4;    ccc=cc_b1; end % lambda^q
                    if block_param==2; p1=5; p2=10;   ccc=cc_b2; end % k_inf^Q
                    if block_param==3; p1=11;p2=10+N; ccc=cc_b3; end % Sigma_P and Sigma_y
                   
                                     
                    ij=p1:p2; 
 
                    % draw a valid proposal
                    teta_cand=teta_draw; 
                    shock=ccc*pm.fac*randn(np,1);               % <= randomness shock
                    teta_cand(ij)=teta_draw(ij)+shock(ij);      % <= theta candidate = theta draw plus random walk.
 
                    % evaluate the proposal & decide
                    [obj, post_cand, prior_cand, mlik_cand]=posterior(teta_cand,Ystar,YY,XX,XY);
                    if rand < min([1 ; exp(post_cand - post_draw)])      % Metropolis-Hastings Acceptance.                                
                    teta_draw=teta_cand; prio_draw=prior_cand; post_draw=post_cand; mlik_draw=mlik_cand; 
                    else
                        rejs(ij,rep,chain)=1;
                    end
 
                end
 
        if max(teta_draw(1:3))>0; stop; end
        
        %store draws
        if rep>burn && mod(rep-burn,step)==0
        prior_0((rep-burn)/step,chain)=prio_draw; 
        posts_0((rep-burn)/step,chain)=post_draw; 
        mliks_0((rep-burn)/step,chain)=mlik_draw;
        tetas_0(:,(rep-burn)/step,chain)=teta_draw; 
        %tetas_0(11:10+N,(rep-burn)/step,chain)=sqrt(teta_draw(11:10+N).^2); %store the sqrt of ^2 for error variances
        end 
    end % of if JSZ==1 condition
    end % for teta_draw = 1
    
    %% Update moments for new draws of VAR coefs
    for priorVAR=1
    if JSZ==1
        [B0,B1,Omega0,Omega1,S0,S1,v0,v1]=create_prior(teta_draw,YY,XX,XY);
 
        if  ~isinf(lambdas) %% && lambdas>0; 
            %posterior moments to use in next step
            psi_pm  = vec(B1); S =S1;    
            DoF= v1;  S_prior =S0;  
            
        elseif isinf(lambdas)
            %estimates
            psi_pm  = vec(B1); S =S1;  DoF= v1;
            
            %assign the value of the draws (VAR coeffs draw will not be performed!)
            sig_draw = S;
            psi_draw = psi_pm;
            PSI_draw = reshape(psi_draw,k,N)';
        end
    end
    end
    
    %% store conditional data density using matricvariate t
    for ML=1
    % note it needs the prior moments so this is conditional on both lambda
    % and theta!
    % this under JSZ should be equal to mlikelihood so =posts_0 (prior~0)!
    if rep>burn && mod(rep-burn,step)==0
    if ~isinf(lambdas) 
    datadens_0((rep-burn)/step,chain)=0;%...
        %mvtLogpdf(Ystar,Xstar*B0,inv(eye(T-T0)+Xstar*Omega0*Xstar'), S_prior, v0);
    elseif isinf(lambdas) 
    datadens_0((rep-burn)/step,chain)=...
        MatricNpdf(Ystar,Xstar*B0,eye(T-T0)+Xstar*Omega0*Xstar', S);    
    end
    end
    end
    
    %% store predictive density using matricvariate t
    for PL=1
    if Tend<=rows(datamat)
    % note it needs the prior moments so this is conditional on both lambda
    % and theta! Yt1 is computed using a draw from ht
    if rep>burn && mod(rep-burn,step)==0
    % note alfa_draw, delta_draw, Phidraw are zero if CSV==0; (line below
    % just double checks
    Xt1=[1 Ystar(end,:) Xstar(end,2:end-N)];
    Yt1=data_tplus1/sqrt(exp(alfa_draw+delta_draw*loght_draw(T)+0*sqrt(Phidraw)*randn));  
        if ~isinf(lambdas) 
        pdctd_0((rep-burn)/step,chain)=mvtLogpdf(Yt1,Xt1*B1,inv(1+Xt1*Omega1*Xt1'), S, DoF);
        elseif isinf(lambdas) %ML approaches JSZ likelihood, what is
    %     predictive density? I guess a matricvariate normal with variannce 
        pdctd_0((rep-burn)/step,chain)=MatricNpdf(Yt1,Xt1*B1,1+Xt1*Omega1*Xt1', S);
        end
    end
    end
    end
    
    %% draw sigma and PSI.  Sigma = V.  PSI = PHI
    for drawVARparam=1
    if ~(JSZ==1 && isinf(lambdas)) %run if NOT JSZ with infinite lambda
    
        %% draw sigma ~IW
        pp = 10; 
        while pp>0
        isig_draw = draw_W(inv(S),DoF);  % IW from page A6 Step 1b.  
                                         % S = S_tilda(theta), DoF = degrees of freedom = (gamma+1)T - k.
                                          %Write in create_prior that v1 = round(T = v0)
        sig_draw=inv(isig_draw);  % Eq. (15) sig_draw is IW with scale matrix inv(H)=S. 
        % check positive definiteness
        [cfs,pp] =chol(sig_draw); 
        % vc     = kron(sig_draw,iMXX);
        % [cf,pp] =chol(vc); 
        end
 
        %% draw VAR coefs.  This is PHI.
        cfx=chol(Omega1)';
        stationary1 = 0; 
        while stationary1==0  
        %% note the equivalence of the following decompositions
        % qq=randn(k,N);
        % [vec((chol(iMXX))'*qq*chol(sig_draw)) (vec(qq)'*chol(kron(sig_draw,iMXX)))' chol(kron(sig_draw,iMXX))'*vec(qq)]                        
 
        psi_draw= psi_pm+vec(cfx*randn(k,N)*cfs);
        PSI_draw= reshape(psi_draw,k,N)';
        % Check for stationarity            
        cs =  [PSI_draw(:,(1+1):k) ; [eye(N*(nlags-1)),zeros(N*(nlags-1),N)]];  
        if  max(abs(eig(cs))) < 1; stationary1 = 1;  end
        end
              
    end 
    end
    
    %% Draw Common Stochastic Volatility.  This is LAMBDA.
    for drawVola=1
    if CSV==1
        
    % these are the structural residuals 
    residuals_ort=(Y - X*PSI_draw')/(chol(sig_draw)); % w_t in Equation (A41).
    ystar4mat(1,T0+1:T)=sum(residuals_ort.^2,2)'; 
    
    % this simply checks is the same as Todds code with factorization
    % get A and S implied by the draw of Sigma
    %     [invA,S]=CSdec(sig_draw);
    %     if abs((sum(sum((invA*S*invA')./sig_draw))/(N^2))-1)>1e-10; stop; end;
    %     A=inv(invA);
    %     % feed these in Todds formula to double check 
    %     ystar2mat=zeros(1,T);
    %     for vtime=1:rows(Y);
    %     ystar2mat(T0+vtime)= (Y(vtime,:) - X(vtime,:)*PSI_draw')*(A'/S*A)*...
    %         (Y(vtime,:) - X(vtime,:)*PSI_draw')';
    %     end;
    %     if abs(sum(ystar4mat(T0+1:T)-ystar2mat(T0+1:T)))>1e-10; stop; end;
    
    Valid=0;
    while Valid==0
        
            % Period T0+1
            loght_draw(T0+1) = 0;
            if SEED==444; loght_draw(T0+1) = 0.406; end
            ht_draw(T0+1)=exp(loght_draw(T0+1));
            
            % Periods T0+2:T 
            for vtime=T0+2:T  
                           
                if vtime==T %%end of sample
                 draw_mean = alfa_draw+delta_draw*loght_draw(vtime-1);
                 draw_sd   = sqrt(Phidraw);  
                else
                 sumd=loght_draw(vtime-1)+loght_draw(vtime+1);
                 draw_mean = (alfa_draw*(1-delta_draw)+delta_draw*sumd)/(1+delta_draw^2); % Paragraph is after Equation A (44).
                 draw_sd   = sqrt(Phidraw/(1+delta_draw^2)); % Paragraph is after Equation A (44).
                end
 
                % draw proposal 
                loght_cand = draw_mean + draw_sd*randn;
                ht_cand = exp(loght_cand);
 
                % compute acceptance probability as exp(log of terms)
                % Metropolis Hastings algorithm
                can=(ystar4mat(1,vtime)/ht_cand          + N*loght_cand); % Equation A (46)
                old=(ystar4mat(1,vtime)/ht_draw(vtime)   + N*loght_draw(vtime)); % Equation A (46)
                acceptprob = min(exp(-.5*can +.5*old),1); % Equation A (45)
                if rand<acceptprob
                loght_draw(vtime) = loght_cand; ht_draw(vtime)= ht_cand;
                end
                
                
            end  
 
    % check the volatility draw produces reasonable theta at present draw
   if JSZ==1
    Xstar=diag(1./sqrt(ht_draw(T0+1:T)))*X;    Ystar=diag(1./sqrt(ht_draw(T0+1:T)))*Y;
    YY=Ystar'*Ystar; XX=Xstar'*Xstar; XY=Xstar'*Ystar; iXX=inv(XX);
    [~,   post__, ~, ~]=posterior(teta_draw,Ystar,YY,XX,XY); 
     if post__>1000; Valid=1; info.post__=post__; end     
   end
   end
   
    end
        
    for gg=1+T0:T
    %Sigma(:,:,gg) = ht_draw(gg)*(inv(Amatdraw)*Smatdraw*(inv(Amatdraw))');  % Sigma = inv(A)*H*inv(A)' 
    Sigma_fix(:,:,gg)= sig_draw;
    Sigma_t(:,:,gg,chain)  = ht_draw(gg)*sig_draw;
    variadraw(gg,:,chain)  = diag(Sigma_t(:,:,gg,chain))';
    voladraw(gg,:)   = sqrt(variadraw(gg,:));
    end
 
    % compute and plot recursive mean of volatilities for the graph
    ht_draw_recmean(:,:,chain) =(ht_draw_recmean(:,:,chain)*(rep-1)+ht_draw)/rep;
    variadraw_recmean(:,:,chain) = (variadraw_recmean(:,:,chain)*(rep-1)+variadraw(:,:,chain))/rep;
    voladraw_recmean(:,:,chain) = (voladraw_recmean(:,:,chain)*(rep-1)+voladraw(:,:,chain))/rep; 
       
    % plot graphs
    
    if verbose==1 && rep>burn && mod(rep,step*10)==0 % plot every 250 draws
    figure(1)
    subplot(2,3,1);    plot(ht_draw(T0+1:T)); title('draw of logvol')
    subplot(2,3,2);    plot(sqrt(ht_draw_recmean(T0+1:T,:,chain))); title('recmean CSV')
    subplot(2,3,3);    plot(variadraw_recmean(T0+1:T,:,chain)); title('recmean variances')
    subplot(2,3,4);    plot(100*voladraw_recmean(T0+1:T,:,chain)); title('recmean volatilities')
    subplot(2,3,5);    plot(100*[Y Ystar]); title('yields')
    subplot(2,3,6);    plot(posts_0(1:(rep-burn)/step,chain)); title('posterior')
    figure(2) 
    for jj=1: 20;     subplot(5,4,jj);    plot(tetas_0(1:rows(tetas_0),jj,chain));     end
    end
    end
    
    %% Draw Phi, alpha, delta
    for drawPhi=1
    if CSV==1
        
    % draw phi
    % degrees of freedom for IW prior on Phi = var(innovations to log volat)
    postdfPhi = priordfPhi + T-(T0+1)+1;  
    deltainnov = loght_draw(T0+2:T)-alfa_draw-delta_draw*loght_draw(T0+1:T-1); 
    Phipost = muPhi*priordfPhi +  deltainnov'*deltainnov; %* scale term, not mean
    Phidraw = Phipost/chi2rnd(postdfPhi);
    
    %Draw alfa and delta
    Z=[ones(T-T0,1) loght_draw(T0:end-1)];
    ZZ=Z'*Z; Zy=Z'*loght_draw(T0+1:end);
    %ZZ\Zy OLS
    postH=priorH+ZZ./Phidraw;
    postV=postH\eye(2);
    postB=postH\(priorH*priorB+Zy./Phidraw);
    Bdraw=postB+chol(postV)*randn(2,1);
    alfa_draw= Bdraw(1);
    delta_draw=Bdraw(2);
    
    end
    end
    %% store the draws (thetas have been stored in draw_teta)
    for storing=1
    if rep>burn && mod(rep-burn,step)==0
    betas_0(:,(rep-burn)/step,chain)=psi_draw;
    sigmas_0(:,(rep-burn)/step,chain)=vec(sig_draw);
    VolVar_0(:,(rep-burn)/step,chain)=[alfa_draw; delta_draw; Phidraw];
    volatility_0(:,(rep-burn)/step,chain)=sqrt(ht_draw(T0+1:T)); %% common stocastic volatility_0 
    VOLATILITIES_0(:,:,(rep-burn)/step,chain)=voladraw(T0+1:T,:,chain); %% individual volatilities
    LAM_0((rep-burn)/step,chain)=lam_draw; 
    end
    end

end % of loop within one chain

end % of loop over chains
 
%% Convergence
for check_convergence=1
indeces=find(tril(sig_draw));
 
%PSRF
resu.psrf_teta=psrf(permute(tetas_0,[2 1 3]));
resu.psrf_beta=psrf(permute(betas_0,[2 1 3]));
resu.psrf_sigma=psrf(permute(sigmas_0(indeces,:,:),[2 1 3]));
resu.psrf_VolVar=psrf(permute(VolVar_0,[2 1 3]));
resu.psrf_volatility=psrf(permute(volatility_0(2:end,:,:),[2 1 3]));
resu.psrf_LAM=psrf(LAM_0);
resu.psrf_VOLATILITIES=psrf(permute(reshape(VOLATILITIES_0,...
    size(VOLATILITIES_0,1)*size(VOLATILITIES_0,2),...
    size(VOLATILITIES_0,3),size(VOLATILITIES_0,4)),[2 1 3])); 
 
%IF
resu.if_tetas=[]; resu.if_betas=[]; resu.if_sigmas=[];
resu.if_VolVar=[]; resu.if_volatility=[]; resu.if_LAM=[];
resu.if_VOLATILITIES=[];
for i=1:nchain
resu.if_tetas=[resu.if_tetas ; ineff(squeeze(tetas_0(:,:,i))')];
resu.if_betas=[resu.if_betas ; ineff(squeeze(betas_0(:,:,i))')];
resu.if_sigmas=[resu.if_sigmas ; ineff(squeeze(sigmas_0(indeces,:,i))')];
resu.if_VolVar=[resu.if_VolVar ; ineff(squeeze(VolVar_0(:,:,i))')];
resu.if_volatility=[resu.if_volatility ; ineff(squeeze(volatility_0(2:end,:,i))')];
resu.if_LAM=[resu.if_LAM ; ineff(squeeze(LAM_0(:,i)))'];
for ii=1:N
resu.if_VOLATILITIES=[resu.if_VOLATILITIES ; ineff(squeeze(VOLATILITIES_0(2:end,ii,:,i))')];
end
end
 
end

%% Now put together all the chains
for merge=1
% check this carefully
 
    % ordina una chain dopo l'altra
    % order one chain after another
    % transpose only for compatibility with previous version
    tetas=(reshape(tetas_0,np,ndraws*nchain))';
    betas=(reshape(betas_0,N*k,ndraws*nchain))';
    sigmas=(reshape(sigmas_0,N^2,ndraws*nchain))';
    VolVar=(reshape(VolVar_0,3,ndraws*nchain))';
    volatility=reshape(volatility_0,T-T0,ndraws*nchain);   
    VOLATILITIES=reshape(VOLATILITIES_0,T-T0,N,ndraws*nchain);
    LAM=(reshape(LAM_0,1,ndraws*nchain))';
    
    draws=rows(betas); 
    posts=vec(posts_0)';
    
    mliks=vec(mliks_0)';
    resu.pred_means=log(mean(exp(pdctd_0)));
    pdctd=vec(pdctd_0);
    datadens=vec(datadens_0)';
    
end
 
%% Graphs
for graphs=1
if verbose==1
    
% Create legend maturities that round to nice length
temp_values  = ones(N,1);
for i = 1:N
    if i == 1
        temp_values(i) = str2double(sprintf('%.3f',maturity(i)));
    else
        temp_values(i) = maturity(i);
    end
end

show_values = string(temp_values) + '-yrs';

%% Figure 1
%figure 1 in paper

figure(11);
ax = axes;

for i = 1:N
    plot(calendar(1:end - 1), 100*datamat(1:end, i))
    hold on
end
%title('Yields from 1985 to 2019')
legend(show_values(1:N))
set(ax, 'Xtick', 1985:5:2020, 'Ylim', [0, 12]);
ytickformat(ax, 'percentage');

saveas(gcf, 'Figure 1.png')


%% Figure 2
%figure 2 in paper

figure(31)
plot(calendar(T0+1:Tend-1), mean(volatility')' ,'b','LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1), median(volatility')','r','LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1), quantile(volatility',0.975)',':r','LineWidth',1); 
hold on
plot(calendar(T0+1:Tend-1), quantile(volatility',0.025)',':r','LineWidth',1); 
hold on
%plot(calendar(T0+1:Tend-1),volatility,'y','LineWidth',1); 
%hold on
plot(calendar(T0+1:Tend-1),mean(volatility')','b','LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1),median(volatility')','r','LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1), quantile(volatility',0.975)',':r','LineWidth',1); 
hold on
plot(calendar(T0+1:Tend-1), quantile(volatility',0.025)',':r','LineWidth',1); 
hold on
legend('Posterior mean','Posterior median','95% bounds')
saveas(gcf, 'Figure 2.png')

%% Figure 3
% Figure 3 in paper

figure(10)
ax = axes;
plot(calendar(T0+1:Tend-1),100*median(VOLATILITIES,3))
legend(show_values(1:N))
set(ax, 'Xtick', 1985:5:2020, 'Ylim', [0,1.5]);
ytickformat(ax, 'percentage');
saveas(gcf, 'Figure 3.png')

%% Additional Figures


if extra_graphs == 1
% figure of volatilities with credible bounds
figure(4)
for i=1:N
    subplot(5,2,i)
    plot(calendar(T0+1:Tend - 1), 100*quantile(squeeze(VOLATILITIES(:,i,:))',0.15),'r--','LineWidth',1)
    hold on
    plot(calendar(T0+1:Tend-1),100*mean(VOLATILITIES(:,i,:),3),'k-','LineWidth',1)
    hold on
    plot(calendar(T0+1:Tend - 1),100*quantile(squeeze(VOLATILITIES(:,i,:))',0.85),'r--','LineWidth',1)
    title([num2str(maturity(i)) '-yrs yield' ' Number of lags: ' num2str(nlags) ', T = ' num2str(Tend - 1)])
    if i==N; legend('Posterior 15% quantile','Posterior mean','Posterior 85% quantile'); end 
end
 
% figure of CSV vs Maximum likelihood estimate
figure(3)
plot(calendar(T0+1:Tend-1),[volaML mean(volatility')' median(volatility')'],'LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1),volatility,'y','LineWidth',1); 
hold on
plot(calendar(T0+1:Tend-1),[volaML mean(volatility')'],'LineWidth',2); %#ok<UDIM>
hold on
plot(calendar(T0+1:Tend-1),median(volatility')','r','LineWidth',2); %#ok<UDIM>
hold on
legend('ML','Posterior mean','Posterior median')
title(['CSV vs. Maximum Likelihood Estimate ' ' Number of lags: ' num2str(nlags) ', T = ' num2str(Tend - 1)])
 
% make a plot of the tightness posterior
figure(789); 
histogram(LAM,30);


end % If extra_graph ==1 
end % If verbose == 1
end % For Graphs = 1
 
%% results
for results=1
    
% rejections
resu.rejections=summary_rej;
resu.acceptance=summary_acc;
 
% resu.tetas=tetas;
resu.teta=mean(tetas);resu.tetasd=std(tetas);
% resu.betas=betas;
resu.beta=mean(betas);resu.betasd=std(betas);
% resu.sigmas=sigmas;
resu.sigma=mean(sigmas); resu.sigmasd=std(sigmas);
% volatility
resu.volatility=mean(volatility,2)'; resu.volatilitysd=std(volatility'); %#ok<UDIM>
% resu.VolVars=VolVar;
resu.VolVar=mean(VolVar); resu.VolVarsd=std(VolVar); 
% resu.VolVars=VolVar;
resu.LAM=mean(LAM); resu.LAMsd=std(LAM); 
resu.LAMall=LAM;
end
 
%% Show results and summary convergence tables
for showres=1
display(resu);
IF_=[];

% Create IF groups
if_tetas=[median(resu.if_tetas(:)) ...
   std(resu.if_tetas(:)) ...
   quantile(resu.if_tetas(:),0.05) ...
   quantile(resu.if_tetas(:),0.95)...
   min(resu.if_tetas(:)) ...
   max(resu.if_tetas(:)) ];

if_betas=[median(resu.if_betas(:)) ...
    std(resu.if_betas(:)) ...
    quantile(resu.if_betas(:),0.05) ...
    quantile(resu.if_betas(:),0.95) ...
    min(resu.if_betas(:)) ...
    max(resu.if_betas(:)) ];

if_sigmas=[median(resu.if_sigmas(:)) ...
    std(resu.if_sigmas(:)) ...
    quantile(resu.if_sigmas(:),0.05) ...
    quantile(resu.if_sigmas(:),0.95) ...
    min(resu.if_sigmas(:)) ...
    max(resu.if_sigmas(:)) ];

if_VolVar=[median(resu.if_VolVar(:)) ...
    std(resu.if_VolVar(:)) ...
    quantile(resu.if_VolVar(:),0.05) ...
    quantile(resu.if_VolVar(:),0.95) ...
    min(resu.if_VolVar(:)) ...
    max(resu.if_VolVar(:)) ];

if_vola=[median(resu.if_volatility(:)) ...
    std(resu.if_volatility(:)) ...
    quantile(resu.if_volatility(:),0.05) ...
    quantile(resu.if_volatility(:),0.95) ...
    min(resu.if_volatility(:)) ...
    max(resu.if_volatility(:)) ];

if_LAM=[median(resu.if_LAM(:)) ...
    std(resu.if_LAM(:)) ...
    quantile(resu.if_LAM(:),0.05) ...
    quantile(resu.if_LAM(:),0.95) ...
    min(resu.if_LAM(:)) ...
    max(resu.if_LAM(:)) ];

if_VOLATILITIES=[mean(resu.if_VOLATILITIES(:)) ...
    std(resu.if_VOLATILITIES(:)) ...
    quantile(resu.if_VOLATILITIES(:),0.05) ...
    quantile(resu.if_VOLATILITIES(:),0.95) ...
    min(resu.if_VOLATILITIES(:)) ...
    max(resu.if_VOLATILITIES(:)) ];
% Create PSRF groups
psrf_tetas=[median(resu.psrf_teta(:)) ...
    std(resu.psrf_teta(:)) ...
    quantile(resu.psrf_teta(:),0.05) ...
    quantile(resu.psrf_teta(:),0.95) ...
    min(resu.psrf_teta(:)) ...
    max(resu.psrf_teta(:)) ];

psrf_betas=[median(resu.psrf_beta(:)) ...
    std(resu.psrf_beta(:)) ...
    quantile(resu.psrf_beta(:),0.05) ...
    quantile(resu.psrf_beta(:),0.95) ...
    min(resu.psrf_beta(:)) ...
    max(resu.psrf_beta(:)) ];

psrf_sigma=[median(resu.psrf_sigma(:)) ...
    std(resu.psrf_sigma(:)) ...
    quantile(resu.psrf_sigma(:),0.05) ...
    quantile(resu.psrf_sigma(:),0.95) ...
    min(resu.psrf_sigma(:)) ...
    max(resu.psrf_sigma(:)) ];

psrf_VolVar=[median(resu.psrf_VolVar(:)) ...
    std(resu.psrf_VolVar(:)) ...
    quantile(resu.psrf_VolVar(:),0.05) ...
    quantile(resu.psrf_VolVar(:),0.95) ...
    min(resu.psrf_VolVar(:))...
    max(resu.psrf_VolVar(:)) ];

psrf_volatility=[median(resu.psrf_volatility(:)) ...
    std(resu.psrf_volatility(:)) ...
    quantile(resu.psrf_volatility(:),0.05) ...
    quantile(resu.psrf_volatility(:),0.95) ...
    min(resu.psrf_volatility(:)) ...
    max(resu.psrf_volatility(:)) ];

psrf_LAM=[median(resu.psrf_LAM(:)) ...
    std(resu.psrf_LAM(:)) ...
    quantile(resu.psrf_LAM(:),0.05) ...
    quantile(resu.psrf_LAM(:),0.95) ...
    min(resu.psrf_LAM(:)) ...
    max(resu.psrf_LAM(:)) ];

psrf_VOLATILITIES=[median(resu.psrf_VOLATILITIES(:)) ...
    std(resu.psrf_VOLATILITIES(:)) ...
    quantile(resu.psrf_VOLATILITIES(:),0.05) ...
    quantile(resu.psrf_VOLATILITIES(:),0.95) ...
    min(resu.psrf_VOLATILITIES(:)) ...
    max(resu.psrf_VOLATILITIES(:)) ];

 
clear IF_ PSRF_;

for cc=1
    
IF_(:,:,cc)=[if_tetas(cc,:); if_betas(cc,:); if_sigmas(cc,:); ...
    if_vola(cc,:); if_VolVar(cc,:); if_LAM(cc,:) ;if_VOLATILITIES(cc,:)]; %#ok<SAGROW>

PSRF_(:,:,cc)=[psrf_tetas(cc,:); psrf_betas(cc,:); psrf_sigma(cc,:); ...
    psrf_volatility(cc,:); psrf_VolVar(cc,:); psrf_LAM(cc,:) ;psrf_VOLATILITIES(cc,:)]; %#ok<SAGROW>



lambda_display = 'Integrated out';


%% Display

disp('===================================================================================================================');
disp(['Tightness = ', lambda_display]);
disp('              Inefficiency factors                                  Potential Scale reduction factors' );
disp('              Med     Std      5%         95%      Min       Max |  Med     Std      5%         95%         Min       Max ');
disp('---------------------------------------------------------   -------------------------------------------------------') ;

lab=['theta (theta) ';'Psi (PHI)     ';'V (Sigma IW)  ';'vol (lambda)  ';'phi (phi)     ';'LAM (gamma)   ';'VOL (LAMBDA)  '];
space=[' |  ';' |  ';' |  ';' |  ';' |  ';' |  ';' |  '];
disp(['Code','(Paper)';]);
display([lab num2str(IF_(:,:,cc),2) space num2str(PSRF_(:,:,cc),3)]);
disp(' ') ;

disp('----Acceptance rates-----------------------------------------------------------------------------------------------') ;
display(num2str(resu.acceptance))

end % for cc=1
 
end % for showres


 
%% Save results in dropbox
for saving=1



%'Results3\' dataname '\Model_'  num2str(Model) 
name=['.\Model_'  num2str(Model) ...
'_lam_' num2str(88) '_T_' num2str(resu.T) '_' num2str(draws) '_' num2str(SEED) '_' num2str(nlags) '_lag' '.mat'];
save(name, '-STRUCT' , 'resu');

 
% save all results for full sample estimates
if T==Tfinal-1 && PARALLEL==1
    %'Results3\' dataname '\Model_'  num2str(Model) 
    name3=['.\Model_'  num2str(Model) ...
    '_lam_' num2str(88) '_full_' num2str(draws) '_' num2str(SEED) '_' num2str(nlags) '_lag.mat'];
    save(name3); 
end

end % saving=1
 
end %% of loop over seed
 
end %% end of loop over time
 
end % of loop over models