%__________________________________________________________________________
% MAKE TABLE 1       
% This processes the results for a single simulation as reported in Table 1.
% The code also produces Figures C.1 in the Online Supplement
% To get full Table 1 please 
% - Choose 'sim'  (line 39) to select the desired simulation.
% - Choose 'hor1' (line 40) to select the IRF horizon (hor1 = 1 for h=0 and 
%                                                      hor1 = 5 for h=4)
% 
% The simulations results are produced bym Sim_Main_V1_10 & Sim_Main_V1_20.
% Results are stored in files 'fn1'. These contain the following main data
% -   M_ contains the simulation parameters (see Sim_Main_V1_10)
% -   tR_ contains the draws of the data generating process
%     -  zz is the binary instrument
%     -  zv is the quantitatve isntrument as used by BVz (true innovation)
%     -  L is the standardised IRFS
%     -  LZ is the IRFs scaled by average policy shock (gamma)
%
% -   H_ is an 11 x 1 structure containing results for 11 models
%     DC - DSC - BV - BVz - SC - SC_ng - iV2 - iV5 - rV - LP - dV   
%     The abbrevations are as in Table 1. There are 4 further models that
%     are not reported in Table 1:
%     SC_ng is SC prior w/o Gibbs sampler mean adjustment
%     iV5   is a freq proxy VAR with the Jentsch-Lunsford (2016) bootstrap
%     rV    is a standard VAR adding z as an endogenous variable
%     dV    is a standard VAR adding z as an exogenous regressor @ lag 0
%     - H_.L    is the estimated standardised IRFS
%     - H_.Lz   is the estimated IRFs scaled by average policy shock (gamma)
%     - H_.S_   is several statistics on innovations (see function nVStat) 
%__________________________________________________________________________  
  clear
  clc
  cDir  = [pwd '\'];
  addpath(genpath(cDir))
  
%__________________________________________________________________________  
% LOAD THE SIMULATION RESULTS
%__________________________________________________________________________  
  sim    =  4;  
  hor1   =  1;

  switch sim
    case {1}, fn1  =  'Sim_1_10_1_1';                % Baseline 1
    case {2}, fn1  =  'Sim_1_10_1_2';                % Baseline 2
    case {3}, fn1  =  'Sim_1_20_1_1';                % Baseline 3
    case {4}, fn1  =  'Sim_1_10_2_1';                % Lagged dependency 
    case {5}, fn1  =  'Sim_1_10_4_1';                % Redundant events
    case {6}, fn1  =  'Sim_1_20_3_1';                % Unobserved events
  end

  load([cDir 'Sim_Results_1\' fn1]);
  if isempty(tR_), error(['Empty simulation: ' fn1]), end
  
%__________________________________________________________________________  
% OPTIONS AND DATA STRUCTURE (DO NOT CHANGE)
%__________________________________________________________________________  
  ser  =   1;                             % Series to be analysed                                        
  bU   =   5;                             % Index of quantile from M_.QNT                          
  bL   =   1;                             % Index of quantile from M_.QNT                            
  Q    =  [M_.QNT(bL) 0.5 M_.QNT(bU)];    % Quantiles to be reported                          

%__________________________________________________________________________  
% BASIC DEFINITIONS AND VARIOUS ADJUSTMENTS
%__________________________________________________________________________  
% Dimensions
 [irf_H, n, nq, nsim] = size(H_(1).L_);                
  
% Model names 
  Models =  cell(length(H_),1);                           
  for m = 1:length(H_) 
      Models{m}  = H_(m).mod; 
  end
  
% Invert the order of quantiles for proxy VARs
  for i = 7:8 
      H_(i).L_  =  H_(i).L_(:,:,[5 4 3 2 1],:); 
  end
  
% Rescale RecVAR to shock of size 1 
  H_(10).L_  =  H_(10).L_ / sqrt(M_.Nstar/M_.T);
  
% Rescale LP and dummy VAR to shock of size 1   
  for s = 1:nsim
      mean_z = mean(abs(tR_.zv(tR_.zv(:,s)~=0,s)));
      H_(9).L_(:,:,:,s)  =   H_(9).L_(:,:,:,s) / (mean_z *100);
      H_(11).L_(:,:,:,s) =  H_(11).L_(:,:,:,s) / (mean_z *100);
  end  
   
%__________________________________________________________________________  
% ELIMINATE DRAWS WHERE SC OR DSC DID NOT CONVERGE
%__________________________________________________________________________  
  ix    = zeros(nsim,m);
  for m = 1:length(H_)
  for d = 1:nsim    
          ix(d,m) = isnan(H_(m).L_(1,1,3,d));
  end
  end
  
  ix = sum(ix,2) > 0;
  for m = 1:length(H_)
      H_(m).L_  =  H_(m).L_(:,:,:,~ix);
      H_(m).LZ  =  H_(m).LZ(:,:,:,~ix);
      H_(m).S_  =  H_(m).S_(:,:,~ix);
  end
  tR_.L_  =  tR_.L_(:,:,~ix);
  tR_.LZ  =  tR_.LZ(:,:,~ix);
  tR_.zz  =  tR_.zz(:,~ix);
  tR_.zv  =  tR_.zv(:,~ix);
  
  nsim   =  size(H_(m).L_,4);
  disp(' ')
  disp([num2str(nsim) ' Simulations kept'])
  
%__________________________________________________________________________  
% REPORT PROPERTIES OF THE TRUE DATA PROCESS 
%__________________________________________________________________________  
  XX    =  zeros(length(Q),4);

% Basic Statistixs on ps and z
  zv_m  = nan(nsim,1);
  zv_s  = nan(nsim,1);
  for d = 1:nsim
      zv      =  tR_.zv(:,d);
      zv_m(d) =  mean(abs(zv(zv~=0)));
      zv_s(d) =  std(abs(zv(zv~=0)));
  end

  XX(:,1) =  quantile(sum(tR_.zv ~= 0,1),Q);          % Nr of non-zero ps
  XX(:,2) =  quantile(sum(tR_.zz ~= 0,1),Q);          % Nr of non-zero z
  XX(:,3) =  quantile(zv_m*100,Q);                    % Mean abs ps (ps ~=0)
  XX(:,4) =  quantile(zv_s*100,Q);                    % Std  abs ps (ps ~=0)
  
% Make Table
  CLabs  =  { '#(ps)','#(z)','m(p)','s(p)'};
  RLabs  =  {'.1' '.5', '.9'};
  
  disp(' ')
  disp('_________________________________________________________________')
  disp(['Simulation ' fn1])
  disp(['Nr of draws = ' num2str(nsim)])
  disp(['Lam_1       = ' num2str(M_.lam1)])
  disp(['Scale       = ' num2str(M_.scale)])
  disp(['std(nu)     = ' num2str(M_.sp *100)])
  disp(['std(eta)    = ' num2str(M_.eta0*100)])
  disp(['pp          = ' num2str(M_.pp)])
  disp(['pn          = ' num2str(M_.pp)])
  disp(['pA          = ' num2str(M_.pA)])
  disp(' ')
  disp(['Cols 1 & 2 show nr of nonzero policy shocks (ps) and z']);
  disp(['Cols 3 & 4 show mean & std of policy shocks abs(ps)|ps~=0']);
  disp('_________________________________________________________________')
  my_prtmat(XX,[],RLabs,CLabs)
  
  
%__________________________________________________________________________    
% REPORT STATISTICS ON INSTRUMENT AND ESTIMATED POLICY SHOCKS
%__________________________________________________________________________  
  bands  = [bL 3 bU];  

  XX     =  zeros(9,5);
  for m  = 1:length(H_)
       H_(m).Sm   = mean(H_(m).S_,3);
       XX(1:3,m)  = H_(m).Sm(1,bands)';
       XX(4:6,m)  = H_(m).Sm(2,bands)';
       XX(7:9,m)  = H_(m).Sm(3,bands)';
  end
  RLabs  = {'Corr(.1)','Corr(.5)','Corr(.9)','SC(.1)','SC(.5)','SC(.9)',...
            'mEZ(.1)','mEZ(.5)','mEZ(.9)'}; 
  
  disp(' ')
  disp('_________________________________________________________________')
  disp(['Simulation ' fn1])
  disp('Rows 1 -  3 show corr btwn z and the shock (Quantiles)');
  disp('Rows 4 -  6 show SC   btwn z and the shock (Quantiles)');
  disp('Rows 7 - 10 show mean z*shock) | z~= 0     (Quantiles)');
  disp('_________________________________________________________________')
  my_prtmat(XX,[],RLabs,Models) 
  
  
%__________________________________________________________________________  
% IRF Calculations
%__________________________________________________________________________  
% Obtain : true IRF (for fixed C0,C1 this is actually the same for all draws)
%          Bias and width of confid bands from each draw
%          Quantiles of median IRF across draws ("true" confidence bounds)
%          Means   of  IRF quantiles from each draw  (both L_ and LZ)
%          Medians of  IRF quantiles from each draw  (both L_ and LZ) 
%__________________________________________________________________________  
  IRF       =  struct('mod',Models);            % Statistics of model IRFs
  IRF_true  = squeeze(tR_.L_(:,:,1));           % Statistics of true IRF
  
% True confidence set: Quants of individual median responses ([irf_H,n,nq])
  for m = 1:length(H_)
      IRF(m).Qmdn  = squeeze(quantile(H_(m).L_(:,:,3,:) ,Q,4));
  end
  
% Bias and interquantile difference of L_  ([irf_H,n,nsim])
  for m = 1:length(H_)
      IRF(m).bias = nan([irf_H, n, nsim]); 
      for d = 1:nsim
          IRF(m).bias(:,:,d) = H_(m).L_(:,:,3,d)  - IRF_true;
      end
  end    
  
 for m = 1:length(H_)
      IRF(m).Qdif = nan([irf_H, n, nsim]);  
      for d = 1:nsim
          IRF(m).Qdif(:,:,d) = H_(m).L_(:,:,bU,d) - H_(m).L_(:,:,bL,d);
      end
  end      
  
% Mean squared error of point estimate  
  for m = 1:length(H_)
      IRF(m).mse = nan([irf_H, n, nsim]); 
      for d = 1:nsim
          IRF(m).mse(:,:,d) = (H_(m).L_(:,:,3,d) - IRF_true(:,:)).^2;
      end
  end  
  
% Coverage ratio  
  for m = 1:length(H_)
      IRF(m).cr = nan([irf_H, n, nsim]); 

      for d = 1:nsim
        IRF(m).cr(:,:,d) = (IRF_true(:,:) > H_(m).L_(:,:,bL,d)) &  ... 
                           (IRF_true(:,:) < H_(m).L_(:,:,bU,d));
      end   
     IRF(m).cr  =  mean(IRF(m).cr,3);
  end  
  
% Means of quantiles
  for m = 1:length(H_)
     IRF(m).mdn   =  squeeze(mean(H_(m).L_(:,:,3,:),4));        % Median  Q
     IRF(m).qL    =  squeeze(mean(H_(m).L_(:,:,bL,:),4));       % Lower   Q
     IRF(m).qU    =  squeeze(mean(H_(m).L_(:,:,bU,:),4));       % Upper   Q
     IRF(m).Qbs   =  squeeze(mean(IRF(m).bias(:,:,:),3));       % Bias
     IRF(m).QQd   =  squeeze(mean(IRF(m).Qdif(:,:,:),3));       % UB
     IRF(m).rmse  =  squeeze(mean(sqrt(IRF(m).mse(:,:,:)),3));  % RMSE
  end
  
%__________________________________________________________________________  
% PRINT IRF STATISTICS
%__________________________________________________________________________  
  XX     = nan(length(H_),9);
  for m  = 1:length(H_)
    XX(m,1)  =  IRF_true(hor1,ser)';
    XX(m,2)  =  IRF(m).qL(hor1,ser)';
    XX(m,3)  =  IRF(m).mdn(hor1,ser)';
    XX(m,4)  =  IRF(m).qU(hor1,ser)';

    XX(m,5)  =  IRF(m).rmse(hor1,ser);
    XX(m,6)  =  IRF(m).Qbs(hor1,ser)';
    XX(m,7)  =  IRF(m).Qmdn(hor1,ser,3) - IRF(m).Qmdn(hor1,ser,1);
    XX(m,8)  =  IRF(m).QQd(hor1,ser)';
    XX(m,9)  =  IRF(m).cr(hor1,ser);
  end
  
% Rearrange to fit Table  
  XX       =  XX([4 3 1 2 5 7 9 6 8 10 11],:);
  RLabs    =  Models([4 3 1 2 5 7 9 6 8 10 11]);
  CLabs    =  {'True','Low','Median','Upper','RMSE','Bias','IQD ','UB','CR'};
  
  disp('_________________________________________________________________')
  disp(['Simulation ' fn1])
  disp(['TABLE 1 for IRF horizon ' num2str(hor1-1)])
  disp('_________________________________________________________________')
  my_prtmat(XX(1:7,:)*100,[],RLabs(1:7),CLabs) 

  disp('_________________________________________________________________')
  disp(['Simulation ' fn1])
  disp('TABLE 1 Further models not shown in final version')
  disp('SC_ng is SC prior w/o Gibbs sampler mean adjustment')
  disp('iV5 is proxy VAR with Jentsch-Lunsford (2016) bootstrap')
  disp('rV is standard VAR adding instrument as regressor @ hor1 0 & 1')
  disp('dV is standard VAR adding instrument as regressor @ lag  0')
  disp('_________________________________________________________________')
  my_prtmat(XX(8:11,:)*100,[],RLabs(8:11),CLabs) 
  
  
%__________________________________________________________________________  
% PLOT IRFs FOR PAPER
% For the paper I must get rid of one 
%__________________________________________________________________________  
  Models   = Models(1:6);
  IRF      =    IRF(1:6);

  ModNames = {'BV_z','DC','DSC','SC','pV','LP'};
  show     =  ones(length(Models),1);
  sm       =  fn1(10:end);
  switch sm
      case {'10_1_1'},  header  = 'Baseline (1)'; 
      case {'10_1_2'},  header  = 'Baseline (2)'; 
      case {'20_1_1'},  header  = 'Baseline (3)'; 
      case {'10_2_1'},  header  = 'Lagged dependency (4)';
      case {'10_4_1'},  header  = 'Redundant events (5)';  
      case {'20_3_1'},  header  = 'Unobserved events (6)';  
      otherwise         header  = ' '; 
  end
  Sim_plot_IRF(IRF_true, IRF, ser, ModNames, header,show)

