function [H_, tR_]  = Sim_Run_1(M_) 
%__________________________________________________________________________
% Main function running simulations 1 
% Takes simulation parameters M_ as input 
% Generates the data and estimates the various models
%
% Global variable H_ should be local to Sim_Run, but is shared with 
%                    function store (defined at the bottom of this file)
%__________________________________________________________________________
  
% Model input
  yName    =  {'y1','y2'};             % Series names                            
  sName    =  {'S1','S2'};             % Innovation names
  const    =   ones(M_.T,1);           % Deterministic term         
  n        =   size(M_.A1,1);          % Nr of series
  nq       =   length(M_.QNT);         % Nr of Quantiles to be stored
    
% Specifications for DSC
  P_.type    =  '0';                   % Uninformative reduced form VAR prior
  O_.VRoot   =   1;                    % Perform VAR stability check
  O_.trend   =   0;                    % No trend in local proj
  O_.success =   1/5000;               % Maximum iterations
  irf_H      =   16;                   % Length of IRFs 
  nh         =   irf_H+1;                                
  
% SC Prior  
  P_.a     =   M_.SC_a;
  P_.b     =   M_.SC_b;
  P_.L     =   M_.SC_L;
  P_.U     =   M_.SC_U;

% Dummy restrictions  
  z        =   zeros(M_.T,1);                     
  R_       =   Add_R([],yName,sName,'y1','S1', 2,-1.0,0);
  R_       =   Add_R(R_,yName,sName,'y2','S2', 2,-1.0,0);
  
% Specifications for pV and LP  
  SPEC.NWlags  =   8;   
  SPEC.irhor   =   irf_H + 1;
  SPEC.iC      =  [0 0];                         
  SPEC.d_ix    =  [0 0]; 
  SPEC.Zflag   =   1;
  clevel       =   M_.QNT((nq+3)/2:end) * 100;

  
% _________________________________________________________________________  
% Define storage variables
% L_ stores quantiles Qd of standardised IRFs to shock S2 
% LZ stores quantiles Qd of scaled       IRFs to shock S2
% S_ stores quantiles Qd of the statistics S2 by nVStat
% _________________________________________________________________________  
% Data generating process ('true' outcomes)
  tR_.L_     =   nan(irf_H+1,n,M_.nsim);               % IRF (standardised)        
  tR_.LZ     =   nan(irf_H+1,n,M_.nsim);               % IRF (Impact of z/p)          
  tR_.S_     =   nan(6,M_.nsim);                       % Shock stat (nVStat)           
  tR_.C0     =   nan(2,2,M_.nsim);                     % Matrix C0
  tR_.lags   =   nan(1,M_.nsim);                       % Nr of lags in VAR
  tR_.zv     =   nan(M_.T,M_.nsim);                    % Policy shock
  tR_.zz     =   nan(M_.T,M_.nsim);                    % Polluted shock

% Results of model estimates  
  global H_
  model   = {'DC_F','DSC_F','BV','BVz','SC','SC_nG','pV2','pV5','LP','rV','dV'};
  H_      =  struct('mod',model);
  
  for m  = 1:length(model)
      H_(m).L_  =  nan(nh,n,nq,M_.nsim);
      H_(m).LZ  =  nan(nh,n,nq,M_.nsim);
      H_(m).S_  =  nan(6,nq,M_.nsim);
  end
  
% _________________________________________________________________________  
% MAIN LOOP  
% _________________________________________________________________________  
  rng('default');
  d = 1;
  
  while d < M_.nsim+1
    disp(['Draw ' num2str(d) ])  
      
  % Generate data and check 
    [y,xx,~,zz,zv,~,A1,C0] = Sim_Generate_1(M_);
    if std(y) > 2*std(xx),        continue, end 
    if sum(zv~=0,1) ~=  M_.Nstar, continue, end

    tR_.zz(:,d)    =   zz;              % Binary instrument
    tR_.zv(:,d)    =   zv;              % True policy shocks
    tR_.C0(:,:,d)  =   C0;              % Rotation matrix (B1)
    
  % Obtain & store true IRF  
    L_  = nan(irf_H+1,n,1);                                           
    for h = 0:irf_H
        L_(h+1,:) = A1^h * C0 * M_.c0;
    end
    tR_.L_(:,:,d) =  squeeze(L_(:,:)) * 0.01;    
    tR_.LZ(:,:,d) =  tR_.L_(:,:,d) * mean(abs(zv(zv~=0))) / 0.01;
  
  % Store nr of lags in VAR
    if M_.lags == 0
          IC    =  my_lagcrit(y,8); 
          M_.p  =  IC(2);
    else  M_.p  =  M_.lags;
    end  
    tR_.lags(d) =  M_.p; 
   
  % DC (Gibbs Sampler)
    m = 1;
    iV.z       =  zz;
    iV.priorZ  =  'F';     
    iV.lagZ    =  [0 0];    
    iV.Mean    =  'EST';
    P_.DCflag  =  1;
    P_.SCflag  =  0;
    [~,LL,E_]  = DSC_BVAR_2022_Gibbs(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
    
  % DSC (Gibbs Sampler)
    m = 2;
    iV.z       =  zz;
    iV.priorZ  =  'F';     
    iV.lagZ    =  [0 0];    
    P_.DCflag  =  1;
    P_.SCflag  =  1;
    iV.Mean    =  'EST';
    [~,LL,E_]  =  DSC_BVAR_2022_Gibbs(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
    
  % Standard Bayesian proxy VAR = DC with zero mean  & no Gibbs sampling
    m = 3;
    iV.z       =  zz;
    iV.priorZ  =  'F';     
    iV.lagZ    =  [0 0];    
    P_.DCflag  =   1;
    P_.SCflag  =   0;
    iV.Mean    =  '0';
    [~,LL,E_]  =  DSC_BVAR_2022(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
    
  % Standard Bayesian proxy VAR with true policy shock (zv)
    m = 4;
    iV.z       =  zv;
    iV.priorZ  =  'F';     
    iV.lagZ    =  [0 0];    
    P_.DCflag  =  1;
    P_.SCflag  =  0;
    iV.Mean    =  '0';
    [~,LL,E_]  =  DSC_BVAR_2022(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
    
  % SC with mean shift (Gibbs sampler)
    m = 5;
    iV.z       =  zz;
    iV.priorZ  =  'F';     
    iV.lagZ    =  [0 0];    
    P_.DCflag  =  0;
    P_.SCflag  =  1;
    [~,LL,E_]  =  DSC_BVAR_2022_Gibbs(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
   
  % SC without mean shift (no Gibbs sampler)
    m = 6;
    iV.z       = zz;
    iV.priorZ  = 'F';     
    iV.lagZ    = [0 0];    
    P_.DCflag  = 0;
    P_.SCflag  = 1;
    [~,LL,E_]  = DSC_BVAR_2022(y,const,M_.p,R_,iV,M_.nd,irf_H,P_,O_);
    store(m,d,iV.z(M_.p+1:end),E_,LL,M_.QNT,1);
  
  % Proxy VAR with bootstrap 2 (Montiel-Olea et al., 2022)
    m = 7;
    SPEC.vars          =  y;
    SPEC.DET           =  const;
    SPEC.taxshocks     =  zz;
    SPEC.p             =  M_.p;
    S_                 =  SPEC;
    S_                 =  myProxySVAR(S_); 
    L_                 =  nan(S_.irhor,n,3);
    Sc_                =  myProxySVARci(S_,clevel,2,5000);
    H_(m).L_(:,:,:,d)  =  cat(3,squeeze(Sc_.irsL),S_.irs,squeeze(Sc_.irsH));
    
  %###################################################################  
  % Not used but kept to maintain replicability of Table 1 
  % (as random numbers would change otherwise!)
    Sc_                =  myProxySVARci(S_,0.5,5,5000);
  %###################################################################  
    
  % Proxy VAR with bootstrap 5 (Jentsch and Lundsford, 2016)
    m = 8;
    SPEC.vars          =  y;
    SPEC.DET           =  const;
    SPEC.taxshocks     =  zz;
    SPEC.p             =  M_.p;
    S_                 =  SPEC;
    S_                 =  myProxySVAR(S_); 
    L_                 =  nan(S_.irhor,n,3);
    Sc_                =  myProxySVARci(S_,clevel,5,5000);
    H_(m).L_(:,:,:,d)  =  cat(3,squeeze(Sc_.irsL),S_.irs,squeeze(Sc_.irsH));
    
  %###################################################################  
  % Not used but kept to maintain replicability of Table 1
  % as random numbers would change otherwise!) 
    Sc_                =  myProxySVARci(S_,0.5,5,5000);
  %###################################################################  
  
  % Local Projections (LP)  
    m = 9;
    S_                 =  SPEC;
    S_                 =  myLPIV_3(S_); 
    Sc_                =  myLPIVci(S_,clevel,1);
    H_(m).L_(:,:,:,d)  =  cat(3,squeeze(Sc_.irsL),S_.irs,squeeze(Sc_.irsH));
    
  %###################################################################  
  % Not used but kept to maintian replicability of Table 1 (as random
  % numbers would change otherwise!)  
    Sc_                =  myLPIVci(S_,0.5,1);
  %###################################################################  
  
  % Recursive VAR (not used in final paper)
  % Adds instrument as endogneous variable to VAR
    m = 10;
   [LL,E_] = REC_BVAR(y,const,M_.p,zz,M_.nd,irf_H,O_);
    store(m,d,zz(M_.p+1:end),E_,LL,M_.QNT,1);
    
  % Dummy VAR (not used in final paper)
  % Adds instument as dummy regressor at lag 0
    m = 11;
   [LL,E_] = Dummy_VAR(y,const,M_.p,zz,M_.nd,irf_H,O_);
    store(m,d,zz(M_.p+1:end),E_,LL,M_.QNT,1);  
   
    d = d + 1;
  end
end

%____________________________________________________________________
% Local function store
% ix is index of the shock instrumented by z 
%____________________________________________________________________
 function store(m,d,z,E_,L_,Q,ix)
    global H_
    if isempty(E_), return, end

   [~,S2]   =  nVStat(z,E_,ix,[],Q,0);
    LZ      =  nan(size(L_));
    for i   =  1:size(L_,4) 
        LZ(:,:,:,i) = L_(:,:,:,i) * S2(i,4); 
    end
    
    H_(m).L_(:,:,:,d)  =  store_IRFs(L_(:,:,ix,:),Q);
    H_(m).LZ(:,:,:,d)  =  store_IRFs(LZ(:,:,ix,:),Q);
    H_(m).S_(:,:,d)    =  quantile(S2,Q)';
 end  
  
