function  [y, x, e, z, eta, eta_S, A1, B1] = Sim_Generate_2(M_)
%__________________________________________________________________________
% THIS CREATES STANDARD NORMAL INNOVATIONS AND SELECTS LARGE ONES
%
% Simulate a structural VAR for n = 2 and p = 1
%
% (1)  y(t) = c(t) + A1*y(t-1) + B1 *(c * e(t))  
%                                          e(t) ~ N(0,se*I_2)
%      where A1 = M_.A1 
%            B1 = M_.B1
%
% INPUT
%    M_.A1                                                        [2 x 2]    
%    M_.B1                                                        [2 x 2] 
%    M_.T         Nr of observations                              [1 x 1]
%
% OUTPUT
%  y       VAR data                                               [T x 2]
%  e(1,2)  VAR structural shocks                                  [T x 2]
%  z       Narrative instrument      Z(t)                         [T x 1]
%  C_0     see above
%__________________________________________________________________________
  T     =  M_.T;                                            % nr obs
  T_0   =  400;                                             % burn-in draws
  T_    =  T_0 + M_.T; 
  n     =  size(M_.A1,1);

%__________________________________________________________________ 
% Set DGP: B1 and A1 and optional constant (w break)
% If isempty(B1) or isempty(C1) => Draw!
%__________________________________________________________________ 
  A1  = M_.A1; 
  if isempty(M_.B1)       
         r   =  random('beta',1,3);
         a   =  cos(rand);
         B1  =  chol([1.0 r; r 1.0]) * [cos(a) sin(a); -sin(a) cos(a)];              
  else , B1  =  M_.B1; 
  end


  cnst = zeros(T_,1);   
  if  isfield(M_,'const') 
      cnst(T_ - fix(T./2):end) = M_.const;
  end
  
  if sum(size(A1) == [2 2]) < 2, error('Fix dimension of A1'), end
  if sum(size(B1) == [n n]) < 2, error('Fix dimension of B1'), end

%__________________________________________________________________ 
% Generate policy shocks and series
% e(:,3) is the policy shock
%__________________________________________________________________
 % Draw & initiate
   S       =  [M_.se, M_.se].^2;
   e       =  mvnrnd(zeros(n,1),diag(S),T_);
   
   y       =    nan(T_,n);                      % VAR data
   x       =    nan(T_,n);                      % VAR data w/o eta
   eta     =  zeros(T_,1);                      % Realised policy variable
   
 % Generate 
   y(1,:)  =  e(1,:);
   for t   =  2:T_
       y(t,:) = cnst(t) + y(t-1,:) * A1'  + e(t,:) * B1';  
   end
   
 % Eliminate burn-in  
   y  =  y(T_0+1:T_,:);
   e  =  e(T_0+1:T_,:); 
  
 % Select the largest M_.NStar shocks e(1,:)
%    [~, e_ix]   =  sort(e(:,2),'descend');
%     e_ix       =  e_ix(1:M_.Nstar);
%     
%     eta        =  zeros(T,1);
%     z          =  zeros(T,1);
%     eta(e_ix)  =  e(e_ix,2); 
%     z(e_ix)    =  sign(eta(e_ix)); 
 
  
    
 % Select M_.NStar from the NStar/NShare largest
  [~, e_ix]    =  sort(e(:,2),'descend');
   e_ix        =  e_ix(1:(M_.Nshare));
   
   sel_ix      =  randperm(M_.Nshare);
   sel_ix      =  sel_ix(1:M_.Nstar);
   e_ix        =  e_ix(sel_ix);
   
   eta         =  zeros(T,1);
   z           =  zeros(T,1);
   eta(e_ix)   =  e(e_ix,2); 
   z(e_ix)     =  sign(eta(e_ix)); 
  
   
% Further   
   eta_S      =  [];
   x          =  y;
 
  
   
end

