function  [y, x, e, z, eta, eta_S, A1, B1] = Sim_Generate_1(M_)
%__________________________________________________________________________
% LIKE 60 BUT LAGGED THETA* ADDED TO VAR (M_.lam2)
%
% Simulate a structural VAR for n = 2 and p = 1
%
% (1)  y(t) = c(t) + A1*y(t-1) + B1 *(c * eta(t) + e(t))  
%                                                  e(t) ~ N(0,se*I_2)
%      where A1 = M_.A1
%            B1 = M_.B1
%
% (2)  Policy interventions eta(t) are derived as the sum of two elements
% (2a) The policy function is given by the condition
%      eta(t)  =  etaP * scale(t)    if eta*(t) >   eta0  
%              =  etaN * scale(t)    if eta*(t) < - eta0  
%              =  0                  otherwise
%      where 
%      eta*(t) =  Lam_0 * e(t,2) + Lam * y(t-1,2) + nu(t) 
%      s(t)     ~ log(N(1,M_.scale)) is a random variable 
%
% (2b) Add  0 < M_.Pexo*T exogenous positive shocks of size etaP
%      Add  0 < M_.Nexo*T exogenous negative shocks of size etaP 
%      Pexo and Nexo are defined as probabilities.
%
% (3)  Define z(t) = the policy measure observed by the econometrician
%      * z(t) = sign(eta(t)) 
%      * Pollute z(t) by removing sum(z>0)  * M_.pp positive obs 
%      * Pollute z(t) by removing sum(z<0)  * M_.pn negative obs 
%      * Pollute z(t) by adding   sum(z~=0) * M_.pA obs of random sign  
%      * 0 <= M_.pp, M_.pn < 1, 0<= M_.pA
% 
% (4)  If M_.const ~= 0 then the constant c is added to the data 
%      generating process but only for the 1st half of the sample.  
%      This extension is to compare robustness of VAR and Local Proj
%
% INPUT
%    M_.A1                                                        [2 x 2]    
%    M_.B1                                                        [2 x 2] 
%    M_.T         Nr of observations                              [1 x 1]
% 
%    M_.lam1      Coeff for y(t-1,2) in policy function 
%     
%    M_.pp        Remove M_.pp percent random positive obs from eta
%    M_.pn        Remove M_.pn percent random negative obs from eta
%    M_.pA        Add    M_.pA percent random obs to eta
%                 (in both cases: percent of the actual number of nonzero)
%
% OUTPUT
%  y      VAR data                                                [T x 2]
%  e(1,2) VAR structural shocks                                   [T x 2]
%  z       Narrative instrument      Z(t)                         [T x 1]
%  eta     Realised policy shock     P(t)                         [T x 1]
%  eta_S   Latent policy variables (P*(t)                         [T x 1]
%  C_0     see above
%__________________________________________________________________________
  T     =  M_.T;                                            % nr obs
  T_0   =  400;                                             % burn-in draws
  T_    =  T_0 + M_.T; 
  n     =  size(M_.A1,1);

% Check conditions
  if M_.pp  < 0.0 || M_.pp  > 1.0,  error('M_.pp  not in [0,1]'), end
  if M_.pn  < 0.0 || M_.pn  > 1.0,  error('M_.pn  not in [0,1]'), end
  if M_.pA  < 0.0                ,  error('M_.pA  not in [0,inf]'), end

 if ~isfield(M_,'Pexo'), M_.Pexo = 0; end
 if ~isfield(M_,'Nexo'), M_.Nexo = 0; end
   
%__________________________________________________________________ 
% Set DGP: B1 and A1 and optional constant (w break)
% If isempty(B1) or isempty(C1) => Draw!
%__________________________________________________________________ 
  A1  = M_.A1; 
  if isempty(M_.B1)       
         r   =  random('beta',1,3);
         a   =  cos(rand);
         B1  =  chol([1.0 r; r 1.0]) * [cos(a) sin(a); -sin(a) cos(a)];              
  else,  B1  =  M_.B1; 
  end
  
  cnst = zeros(T_,1);   
  if  isfield(M_,'const') 
      cnst(T_ - fix(T./2):end) = M_.const;
  end
  
  if sum(size(A1) == [2 2]) < 2, error('Fix dimension of A1'), end
  if sum(size(B1) == [n n]) < 2, error('Fix dimension of B1'), end

%__________________________________________________________________ 
% Generate policy shocks and series
% e(:,3) is the policy shock
%__________________________________________________________________
 % Draw & initiate
   S        =  [M_.se, M_.se, M_.sp].^2;
   e        =  mvnrnd(zeros(n+1,1),diag(S),T_);
   eta_Scal =  exp(randn(T_,1)*M_.scale) / exp(M_.scale^2/2);
   
   y        =    nan(T_,n);                    % VAR data
   x        =    nan(T_,n);                    % VAR data w/o eta
   eta_S    =    nan(T_,1);                    % Latent   policy variable
   eta      =  zeros(T_,1);                    % Realised policy variable
   y(1,:)   =    e(1,1:2);
   x(1,:)   =    e(1,1:2);                           
   eta_S(1) =    e(1,3);
   
 % Generate 
   for t  =  2:T_
     % Policy shock 
       eta_S(t) = M_.lam1 * y(t-1,2) + e(t,3);
       if      eta_S(t) >  M_.eta0
               eta(t)   = -eta_Scal(t) * M_.etaP;
       elseif  eta_S(t) < -M_.eta0
               eta(t)   =  eta_Scal(t) * M_.etaN;
       end
     
     % Exogenous shock 
       if rand <  M_.Pexo, eta(t) = -eta_Scal(t) * M_.etaP; end
       if rand <  M_.Nexo, eta(t) =  eta_Scal(t) * M_.etaN; end 
       
     % Series
       zt     = eta(t) * M_.c0';
       y(t,:) = cnst(t) + y(t-1,:) * A1' + (zt + e(t,1:2)) * B1';  
       x(t,:) = cnst(t) + x(t-1,:) * A1' +     + e(t,1:2)  * B1';  
   end
   
 % Eliminate burn-in  
   y      =     y(T_0+1:T_,:);
   x      =     x(T_0+1:T_,:);
   e      =     e(T_0+1:T_,:); 
   eta_S  = eta_S(T_0+1:T_,:); 
   eta    =   eta(T_0+1:T_,:); 
   
 % Pollute z: remove & add unobserved events  
   z      =  sign(eta);
   sx     =  sum(z ~= 0);
   sp     =  sum(z  > 0);
   sn     =  sum(z  < 0);

   ix     =  find(z>0);
   ix     =  randsel(ix,fix(sp*M_.pp));   z(ix) = 0;    % Remove pos obs
   ix     =  find(z<0);
   ix     =  randsel(ix,fix(sn*M_.pn));   z(ix) = 0;    % Remove pos obs   
   ix     =  find(z==0);
   ix     =  randsel(ix,fix(sx*M_.pA));   z(ix) = 1;    % Add 
end

function ix  =  randsel(ix,n)
         ii  =  randperm(length(ix));
         ix  =  ix(ii(1:n));
end


