function P_ = NormWish_Minnesota(P_,s_u,m,p)
    
%__________________________________________________________________________
% P_ = NormWish_Minnesota(P_,s_u,m,p)
%__________________________________________________________________________
% Creates the Normal-Wishart Minnesota prior by Karlsson (2012)
% Forecasting with Bayesian Vector Autoregressions, section 3.2.1 p.14f
%
%  The VAR is y(t)  = Sum_(s=0)^(p) B_s y(t-s) + C * w_t
%
%  The structure of coeffs B is given from regression
%             y(t)' = x(t)'* B = u(t)'
%
%  where x(t)'= ( y(t-1)', ...,y(t-p)', z(t)') 
%
%  INPUTS
%  P_.nu        df                      (n + 2)                   [integer]
%  P_.rho       Prior diag of B_1       (ones(n,1))                 [n x 1]
%  P_.pi1       Overall tightness       (sqrt(0.1))
%  P_.pi3       Decay                   (1)                          [>= 1]
%  P_.pi4       Rel std of coeffs of C
%
%  s_u          Sample resid variances - scaling of of prior vars   [n x 1]
%  m            Nr of VAR parameters per equation                 [integer]
%  p            The number of lags in the VAR                     [integer]
%
%  OUTPUT
%  P_.B         Prior mean of vec(beta)
%  P_.V         Prior covariance matrix of vec(beta)
%  P_.S         Prior scale of Sigma
%
%  Carriero, Clark, Marcellino (2011) use pi1 = sqrt(0.1), pi3 = 1, 
%                                         pi4 = inf, nu = n+2;
%__________________________________________________________________________ 
  n  = length(s_u);

% Standard prior settings
  if ~isfield(P_,'nu')  
      P_.nu   =  n + 2;                
   %  warning(['Input P_.nu not set - default = ' num2str(P_.nu)]) 
  end
  if ~isfield(P_,'rho')
      P_.rho  =  ones(n,1);            
   %  warning(['Input P_.rho not set - default = ' num2str(P_.rho)]) 
  end

  if ~isfield(P_,'pi1')  
      P_.pi1  =  sqrt(0.1);            
   %  warning(['Input P_.pi1 not set - default = ' num2str(P_.pi1)]) 
  end
  
  if ~isfield(P_,'pi3')  
      P_.pi3  =  1 ;                   
   %  warning(['Input P_.pi3 not set - default = ' num2str(P_.pi3)]) 
  end
  if ~isfield(P_,'pi4')
      P_.pi4  =  1000;                 
   %  warning(['Input P_.pi4 not set - default = ' num2str(P_.pi4)]) 
  end

  if length(P_.rho) == 1; P_.rho  = P_.rho * ones(n,1);    end 
  if P_.nu < n+2;         P_.nu   =  n + 2;                end
  
  
% S & B
  P_.S  = (P_.nu - n - 1) * diag(s_u);
  P_.B  = [diag(P_.rho); zeros(m-n,n)]; 
   
% P_.V
  P_.V = zeros(m,m);
  for i = 1:n
  for s = 1:p
       ix  = (s-1)*n+i;
       P_.V(ix,ix) = (P_.pi1 * s^(-P_.pi3))^2 / s_u(i);
  end
  end

  for i = n*p:m
      P_.V(i,i) = (P_.pi1*P_.pi4)^2;
  end


