
function [LH_,EH_,XH_,BH_] = dummy_VAR(Y,X,p,iV,nd,h_IR,O_)
                                               
%__________________________________________________________________________
% [LH_,EH_,XH_,BH_] = dummy_VAR_V1(Y,X,p,iV,nd,h_IR,P_,O_)
%__________________________________________________________________________
% Estimates a recursive structural VAR with a bootstrap.
% That is
% Estimate VAR of length p for [iV Y] and deterministic X 
% Obtain the IRF to first shock from a Choleski decomposition
% Bootstrap
% *  Sample from the residuals
% *  Generate the new data 
% *  Rerun the above estimation
%__________________________________________________________________________
%
% INPUTS
%   Y           :  Endogenous variables                             [T x n]
%   X           :  Exogenous variables                              [T x m]
%   p           :  Number of lags of y                            [integer]  
%
%   iV          :  Instrumental variables                           [T x n]              
%   nd          :  The number of draws in MC chain                [integer]
%   h_IR        :  Number of horizons to report                   [integer]
%
% OUTPUT
%   L_          :  Draws of IRFs (h_IR+1 x n x nshock x nd)
%   E_          :  Draws of the shocks
%
%   L(:,:,:,1) contains the central estimate! 
%__________________________________________________________________________
% NOTATION
%   y(t) = x(t)*Aplus + u(t)
%   y(t) = F y(t-1)   + u(t)
%        = Sum_h=0^inf B(h) u(t-h) 
%   u(t) = A0 e(t)
%__________________________________________________________________________
% Adjust data for lags
  Y_0   =   Y;
  X_0   =  [lagmatrix(Y_0,1:p) X iV];  
  Y_0   =  Y_0(p+1:end,:);
  X_0   =  X_0(p+1:end,:);
 [T,n]  =  size(Y_0);

%_________________________________________________________________________
% STEP II: Report model specification
%_________________________________________________________________________
  if ~isfield(O_,'prt')     ; O_.prt      =   0;   end  
  if ~isfield(O_,'VRoot')   ; O_.VRoot    =   1;   end

  if O_.VRoot == 0
     disp('WARNING: VAR stability test not enabled')
  end
  if O_.prt
     disp('Summary of the SVAR model:')
     disp(' ') 
     disp(['Sample size                :  ', num2str(T)])
     disp(['Nr of endogenous variables :  ', num2str(n)])
     disp(['Nr of lags:                   ', num2str(p)])
     disp(['Nr of exog vars incl. const:  ', num2str(size(X_0,2)-n*p)])
  end
  disp(['Iteration & success: ' num2str([1 0])]) 

%_________________________________________________________________________
% STEP III: Run the basic estimate
%_________________________________________________________________________
  LH_   =  nan(h_IR+1,n,1,nd);    
  EH_   =  nan(T,1,nd);

% OLS estimates
  B_0      = (X_0'*X_0)\(X_0'*Y_0); 
  U_0      =  Y_0 - X_0*B_0;
  S_0      =  U_0'*U_0 / (T-1);
  
% Obtain IRF   
  J     =  [eye(n);repmat(zeros(n),p-1,1)];
  F    =  Build_Ff(B_0(1:n*p,:),eye(n),h_IR,J);
  
  L_    = nan(h_IR+1,n,n);
  for h = 1:(h_IR+1)
      L_(h,:,1) = (J'*((F^(h-1)))*J)' * B_0(end,:)';  
  end  
  
  LH_(:,:,:,1) =  L_(:,:,1);
  EH_(:,:,1)   =  nan;
  
  
%_________________________________________________________________________
% STEP IV: Bootstrap
%          We have iV(t) = U1(t) <-> B_0(:,1) = 0 
%          Under this condition iV is simply reshuffled while its
%          discrete nature is maintained. 
%_________________________________________________________________________
  for d = 2:nd
   % Sample from U_0 
   % Generate data with fixed Z and fixed exogenous   
     Y   =  nan(size(Y_0));
     X   =  nan(size(X_0));
     ix  =  fix(T*rand(T,1) + 1); 
     U   =  U_0(ix,:);
   % Z   =  Y_0(ix,1);
     
     X(1,:) = X_0(1,:);  
     Y(1,:) = X(1,:) * B_0 + U(1,:); 
     for t  = 2:T
        X(t,:)  = [Y(t-1,:) X(t-1,1:n*(p-1)) X_0(t,n*p+1:end)];
        Y(t,:)  =  X(t,:) * B_0 + U(t,:);
     end
     
   % Re-estimate
     B      = (X'*X)\(X'*Y); 
     U      =  Y - X*B;
     S      =  U'*U / (T-1);

   % Obtain IRF   
     LC     =  chol(S,'lower');
     A0     =  inv(LC');
     F      =  Build_Ff(B(1:n*p,:),eye(n),h_IR,J);

     L_     =  nan(h_IR+1,n,n);
     for h  =  1:(h_IR+1)
         L_(h,:,1) = (J'*((F^(h-1)))*J)' *  B(end,:)';
     end  
     LH_(:,:,1,d)  =  L_(:,:,1);
     EH_(:,:,d)    =  U * A0(:,1);
     
  end
  QH_  = [];
  XH_  = [];
  BH_  = [];
      
end

