 
function [QH_,LH_,EH_,ZH_,lag_] = DSC_BVAR_2022(Y,X,p,RR_,iV,nd,h_IR,P_,O_)
                                               
%__________________________________________________________________________
% [QQ, L_, E_, X_, B_] = BVAR_DC_2022(Y,X,p,RR_,iV,nV,nd,h_IR,P_,O_,add)
%__________________________________________________________________________
% Estimates the structural BVAR identified via sign & zero restrictions
% This adds a T x n matrix of proxy variables iV to the system, for which 
% instrumental and/or narrative restrictions might be defined. Column j
% of matrix iV.z may contain a proxy for shock j. The code checks iV.ix and 
% iV.nV for the types of restrictions to be imposed on shock j. 
% * iV.ix(i,j) = 1 defines instrumental variable restrictions. The code 
%   imposes a zero correlation of iV.z(:,j) with shock i plus a positive 
%   sign restriction on the correlation of iV(:,j) with shock j.  
% * iV.nV(j) > 0 defines narrative restrictions (see below)
% 
% This adds (again) the SC reliability prior
%__________________________________________________________________________
%
% INPUTS
%   Y           :  Endogenous variables                             [T x n]
%   X           :  Exogenous variables                              [T x m]
%   p           :  Number of lags of y                            [integer]  
%   RR_         :  Matrix of restrictions:                          [? x 4]
%                    - (1) Shock number 
%                    - (2) Variable with response restriction
%                    = (3) Restriction type
%                          0 = zero restriction       (value =  0)
%                          1 = sign restriction       (value =  1/-1)
%                          2 = max FEVD restriction   (value = (0 100))
%                          3 = min FEVD restriction   (value = (0 100))
%                    - (4) Value
%                    - (5) Horizon where the restriction holds
%                          Set h = 100 for long-run zero restriction
%
%   iV          :  Structure on iV restrictions                
%   iV.z        :  Instrumental variable                            [T x n]
%   iV.zlag     :
%   iV.Mean     :  'est':  Estimate constant term in DC regression 
%                  'fix':  Mean adjustment of z* with m/T
%                  'non':  neither constant nor mean adjustment  
%
%   nd          :  The number of draws in MC chain                [integer]
%   h_IR        :  Number of horizons to report                   [integer]
%
%   P_          :  Reduced form VAR prior
%   P_.type     :  The type of prior            
%                 '0' = uninformative            (Karlsson 2012, p 5f)
%                 'M' = Normal-Wishart Minnesota (Karlsson 2012, p 14f)
%   P_.[]       :  Hyperparameters for Minnesota prior  [3 x 1]
%                  (see NormWish_Minnesota.m for details)       
% OUTPUT
%   QH_          :  The matrix with restrictions for N repetitions
%   LH_          :  Draws of IRFs (h_IR+1 x n x nshock x nd)
%   EH_          :  Draws of the shocks
%   ZH_
%   lag_
%
%__________________________________________________________________________
% NOTATION
%   y(t) = x(t)*Aplus + u(t)
%   y(t) = F y(t-1)   + u(t)
%        = Sum_h=0^inf B(h) u(t-h) 
%   u(t) = A0 e(t)
%__________________________________________________________________________


%__________________________________________________________________________
% 0. Adjust data for lags
%__________________________________________________________________________
  X     =  [lagmatrix(Y,1:p) X];  
  Y     =     Y(p+1:end,:);
  X     =     X(p+1:end,:);
  iV.z  =  iV.z(p+1:end,:);

 [T,n]  =   size(Y);
  
%__________________________________________________________________________
% STEP I: Construct matrices with zero and sign and FEVD restrictions
%         The inputs to this block are RR_ and n
%         This uses 'assignin' to store matrices Z1, Z2, ... and S1, S2..
%         in workspace base. They are retrieved from 'evalin' later on
%         What is used below is Zj and Sj
%__________________________________________________________________________
% Initialisation
  indx   =  (RR_(:,3)==0); ZeroRestr = RR_(indx,:); Zshock = [];
  indx   =  (RR_(:,3)==1); SignRestr = RR_(indx,:); Sshock = [];
  indx   =  (RR_(:,3) >1); FevdRestr = RR_(indx,:); Fshock = [];

% Horizon vector
  h8 = (RR_(:,5)==100);
  if h8==0;   h_RR  =   (0:max(RR_(:,5)))';  
  else        h_RR  =  [(0:max(RR_(~h8,5)))'; 100]; 
  end
  nh = size(h_RR,1);
  
% Zero restrictions
  nz = size(ZeroRestr,1);
  if nz > 0 
     Zshock = unique(ZeroRestr(:,1),'rows'); 
    
     for z = 1:size(Zshock,1) 
         j       = Zshock(z,1);  
         Z_name  = ['Z',num2str(j)];
         z_index = (ZeroRestr(:,1)==j); 
         Restr   =  ZeroRestr(z_index,:);       
         Zj      = zeros(size(Restr,1),n*nh);
         for r   = 1:size(Restr,1)
             hz  = find(h_RR==Restr(r,5), 1, 'first'); 
             hz  = (hz-1)*n;
             Zj(r,hz+Restr(r,2)) = 1; 
         end
         assignin('base',Z_name,Zj)
     end 
  end

% Sign restrictions
  ns = size(SignRestr,1);
  if ns > 0 
     Sshock = unique(SignRestr(:,1),'rows'); 
    
     for s = 1:size(Sshock,1) 
         j       =  Sshock(s,1);  
         S_name  = ['S' num2str(j)];
         s_index = (SignRestr(:,1)==j); 
         Restr   =  SignRestr(s_index,:);
          
         Sj      =  zeros(size(Restr,1),n*nh);
         for r   = 1:size(Restr,1)
             hs  = find(h_RR==Restr(r,5), 1, 'first'); 
             hs  = (hs-1)*n;
             Sj(r,hs+Restr(r,2)) = Restr(r,4); 
         end
         assignin('base',S_name,Sj);
     end
  end

% FEVD restrictions
  nf = size(FevdRestr,1);
  if nf > 0 
     Fshock  = unique(FevdRestr(:,1),'rows'); 
    
     for s = 1:size(Fshock,1) 
         j       = Fshock(s,1);  
         F_name  = ['F' num2str(j)];
         f_index = (FevdRestr(:,1)==j); 
         Restr   =  FevdRestr(f_index,:);
          
         Fj      =  zeros(size(Restr,1),n*nh);
         for r   = 1:size(Restr,1)
             hf  = find(h_RR==Restr(r,5), 1, 'first'); 
             hf  = (hf-1)*n;
             Fj(r,hf+Restr(r,2)) = Restr(r,4); 
         end
         assignin('base',F_name,Fj);
     end
  end
  
% Find maximum of SC prior (for scaling of the rejection sampling step)
  P_.sumZ = sum(abs(iV.z));
  if P_.SCflag
  if P_.a > 0 && P_.b > 0
     P_.fmax = 0;
     for d = 1:200
         pi0 =  prior_pi(P_.a,P_.b,P_.L,P_.U);
         f   =  SC_Accept(fix(pi0*P_.sumZ),P_.sumZ,pi0,1);
         P_.fmax = max([P_.fmax, f]);
     end
     P_.fmax = P_.fmax * 1.05;
  else   
     P_.fmax = []; 
  end
  end
  
%__________________________________________________________________________
% STEP II: Report model specification
%__________________________________________________________________________
  if ~isfield(O_,'prt')     ; O_.prt      =   0;   end    % Detailed printout
  if ~isfield(O_,'VRoot')   ; O_.VRoot    =   1;   end    % Check VAR stability
  if ~isfield(O_,'maxiter') ; O_.maxiter  =  1e10; end    % Stop if iter/nd > maxiter
  if ~isfield(O_,'success') ; O_.success  =  1e-3; end    % Stop crit success rate


  if O_.VRoot == 0
     disp('WARNING: VAR stability test not enabled')
  end
  if O_.prt
     disp('Summary of the SVAR model:')
     
     disp(' ') 
     disp(['Sample size                :  ', num2str(T)])
     disp(['Nr of endogenous variables :  ', num2str(n)])
     disp(['Nr of lags:                   ', num2str(p)])
     disp(['Nr of exog vars incl. const:  ', num2str(size(X,2)-n*p)])
     disp(['Nr of sign restrictions:      ', num2str(ns)])
     disp(['Nr of zero restrictions:      ', num2str(nz)])
     disp(' ')
     if ns > 0 
     disp(['Sign restr on shocks:         ', num2str(Sshock')]) 
     end
     if nz > 0 
     disp(['Zero restr on shocks:         ', num2str(Zshock')])
     end 
     if nf > 0 
     disp(['FEVD restr on shocks:         ', num2str(Fshock')])
     end  
     disp(['Restrictions are at horizons: ', num2str(h_RR')])
  end
  disp(['Iteration & success: ' num2str([1 0])]) 
  
  
  
%__________________________________________________________________________
% STEP III: Initialise the draws
%   The notation is a bit of a pain - [] refer to notation in Karlsson
%           P_.B    Prior mean of VAR coeffs        (m x n)         [Gamma]
%           P_.V    Prior var  of VAR coeffs        (m x m)         [Omega]
%           P_.S    Prior mean of covariance matrix (n x n)             [S]
%
%           B_      Posterior mean of parameters B
%           S_      Posterior covariance of U (mean of Wishart)
%           V_      Posterior covariance of B  
%__________________________________________________________________________
% OLS estimates
  B_0    = (X'*X)\(X'*Y); 
  U_0    =  Y - X*B_0;
  S_0    =  U_0'*U_0;
  m      =  size(B_0,1);       
  J      =  [eye(n);repmat(zeros(n),p-1,1)];
  
  switch P_.type
    case '0'   % Uninformative (Jeffrey)                                          
        B_    =  B_0;
        V_    =  inv(X'*X);
        nu_   =  T - m*n;
        S_    =  S_0;
        
    case 'M'   % Minnesota                                                      
        P_    =  NormWish_Minnesota(P_,diag(S_0)./T,m,p);
        P_.iV =  inv(P_.V);
        
        V_    =  inv(P_.iV + X'*X);
        B_    =  V_ * (P_.iV*P_.B + X'*Y);
        nu_   =  P_.nu + T;
        S_    =  P_.S + Y'*Y + P_.B'*P_.iV*P_.B - B_'*inv(V_)*B_; 
  end
  S_  = inv(S_);  
 
  
%__________________________________________________________________________
% STEP IV: Main loop
%          - Draw from reduced form VAR
%          - P_.DCflag = 1 then estimate DC regression
%            else draw alpha randomly
%          - Check sign concordance
%          - Draw from orthonormal Q and check conditions
%__________________________________________________________________________
  k     =  max([1 Sshock' Fshock' Zshock']);
  
  QH_   =  nan(n,k,nd);
  LH_   =  nan(h_IR+1,n,k,nd);    
  EH_   =  nan(T,k,nd);
  ZH_   =  nan(T,nd);
  lag_  =  nan(2,nd);

  alg2  =  0; 
  iter  =  0;
  tic

  while alg2 < nd
    iter = iter + 1;   
    if  mod(iter,10000) == 0
        disp(['Iteration & success: ' num2str([iter alg2])])
    end
    if fix(iter / nd) > O_.maxiter
       warning('Maximum iteration limit reached')
       QH_ = [];  LH_ = [];  EH_ = [];  XH_ = [];
       return
    end
   
    if (alg2 + 10)/(iter + 10)  < O_.success
       warning('Min success criterion below threshold - raise O_.success')
       QH_ = [];  LH_ = [];  EH_ = [];  XH_ = [];
       return 
    end
    
  % __________________________________________________________
  % Sample S and B, check roots & get U
    S      =  inv(wish(S_,nu_));
    S_B    =  kron(S,V_);
    B      =  mvnrnd(B_(:),S_B);
    B      =  reshape(B,m,n); 
    U      =  Y - X*B; 
    if  O_.VRoot == 1 && ~Var_Stable(B(1:n*p,:)') 
        continue
    end
    
  % Preliminaries for Arias-RubioRamirez-Waggoner algorithm: cov(U) = L*L'    
    LC     =  chol(cov(U),'lower');
    A0     =  inv(LC');
    Aplus  =  B(1:n*p,:) * A0;
    [F,f]  =  Build_Ff(Aplus,A0,h_RR,J);
    f2     =  Build_FEV(f,1,h_RR);
    
  % __________________________________________________________
  % Make instrument ZZ
  % Draw the nr of lags of Z
    switch iV.priorZ
        case   {'F'},  lagZ      =  iV.lagZ; 
        case   {'M'},  lagZ      =  [nan nan]; 
                       lagZ(2)   =  randi(iV.lagZ(1)+iV.lagZ(2)+1) - iV.lagZ(1)-1;
                       lagZ(1)   = -lagZ(2);
        otherwise   ,  error('Set M_.priorZ')
    end
    ix   =  find(iV.z ~= 0);
    ZZ   =  zeros(size(iV.z));
   
    for j = 1:length(ix)
       h1  =  min(lagZ(1),ix(j)-1);                  % Adjust for sample start
       h2  =  min(lagZ(2),length(iV.z)-ix(j));       % Adjust for sample end
       ZZ(ix(j)-h1:ix(j)+h2) = iV.z(ix(j));          % Fill Zc with leads/lags
    end
   
  % Draw alpha from DC regression  
    if P_.DCflag
          delta =  1 - 2*(ZZ<0);
          switch iV.Mean
            case {'FIX'}   
                            Z_star  =  ZZ - (sum(abs(ZZ))/T) * delta;
                            U_star  =  U;
            case {'EST'}   
                            Z_star  =  ZZ;
                            U_star  = [delta U];
            case {'0'}  
                            Z_star  =  ZZ;
                            U_star  =  U;
          end
          
          a_    =  (U_star'*U_star)  \ (U_star'*Z_star); 
          r     =  Z_star - U_star * a_;
          se    =  gamrnd(T/2,2*std(r)^2/T);
          S_a   =  se * inv(U_star'*U_star);
          a     =  mvnrnd(a_,S_a)';
          
          if strcmp(iV.Mean,'EST')
             a  =  a(2:end);
          end
    end
  
  % Set 1st column of rotation matrix Q1 either from alpha or randomly 
    Q   =  [];
    if P_.DCflag,    Qj   =  LC' * a;
    else,            Qj   =  randn(n,1);
    end
    Qj  =  Qj/norm(Qj); 

  % Check Sign Concordance 
    if P_.SCflag
       SC   =  ZZ .* (U*A0*Qj);
       SC   =  sum(SC > 0);
       if   ~isempty(P_.fmax)
             pi0 =  prior_pi(P_.a,P_.b,P_.L,P_.U);
       else  pi0 =  P_.L;
       end
       if ~SC_Accept(SC,P_.sumZ*(sum(lagZ)+1),pi0,P_.fmax)
           continue
       end
    end     
    
  % __________________________________________________________
  % Process any remaining (zero, sign, FEVD) restrictions 
  % Construct matrix Q from Gram-Schmidt & check restrictions 
    for j = 1:k
       Xj = randn(n,1);
       
     % Zero restrictions
       if j > 1
        if nz > 0 && sum(Zshock==j) > 0 
             Zj  = evalin('base', ['Z',num2str(j)]);
             Rj  = [Q';Zj*f]; 
        else Rj  =  Q'; 
        end
        if size(Rj,1)==0
             Nj = eye(n); 
        else Nj = null(Rj);
        end
        Qj = Nj*Nj'*Xj/norm(Nj'*Xj); 
       end 
       
     % Sign restrictions
       if  ns > 0 && sum(Sshock==j)>0
             Sj    = evalin('base',['S',num2str(j)]); 
             irfsj = Sj*f*Qj;  
       else  irfsj = 1; 
       end  
       if     sum(irfsj>0) == size(irfsj,1)    
       elseif sum(irfsj<0) == size(irfsj,1) && j > 1
              Qj = -Qj;
       else   break
       end 
        
     % FEVD restrictions
       if nf > 0 && sum(Fshock==j)>0
          f2j  = Build_FEV(f,Qj,h_RR);
          f2j  = reshape(f2j./f2,1,n*nh);
          f2j  = kron(ones(size(Fj,1),1),f2j);
           
          Fj   = evalin('base',['F',num2str(j)]);
          crit = (abs(Fj)./f2j).^sign(Fj) > 1;
          if  sum(sum(crit)) > 0
              break
          end
       end
       Q  = [Q Qj]; 
    end
    
  % ____________________________________________________
  % Store IRFs L, shocks E, and X
    if size(Q,2) == k
       alg2          =  alg2+1;
       
       lag_(:,alg2)  =  lagZ;
       ZH_(:,alg2)   =  ZZ;
       QH_(:,:,alg2) =      (A0*Q); 
       EH_(:,:,alg2) =  U * (A0*Q);

       for h = 1:(h_IR+1)
           LH_(h,:,:,alg2) = (J'*((F^(h-1)))*J)' *  inv(A0)'*Q;
       end  
     end
  end
  toc

       



